/*
 * Serial Sound Interface (I2S) support for SH7764
 *
 * Copyright (c) 2009 ALPHAPROJECT
 * Copyright (c) 2007 Manuel Lauss <mano@roarinelk.homelinux.net>
 *
 *  licensed under the terms outlined in the file COPYING at the root
 *  of the linux kernel sources.
 *
 * dont forget to set IPSEL/OMSEL register bits (in your board code) to
 * enable SSI output pins!
 */

/*
 * LIMITATIONS:
 *	The SSI unit has only one physical data line, so full duplex is
 *	impossible.  This can be remedied  on the  SH7760 by  using the
 *	other SSI unit for recording; however the SH7780 has only 1 SSI
 *	unit, and its pins are shared with the AC97 unit,  among others.
 *
 * FEATURES:
 *	The SSI features "compressed mode": in this mode it continuously
 *	streams PCM data over the I2S lines and uses LRCK as a handshake
 *	signal.  Can be used to send compressed data (AC3/DTS) to a DSP.
 *	The number of bits sent over the wire in a frame can be adjusted
 *	and can be independent from the actual sample bit depth. This is
 *	useful to support TDM mode codecs like the AD1939 which have a
 *	fixed TDM slot size, regardless of sample resolution.
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/interrupt.h>
#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/initval.h>
#include <sound/soc.h>
#include <asm/io.h>

#define SSICH0_IRQ	65
#define SSICH1_IRQ 	66
#define SSICH2_IRQ	67
#define SSICH3_IRQ	70
#define SSICH4_IRQ	71
#define SSICH5_IRQ	72

#define SSICR	0x00
#define SSISR	0x04

#define CR_UIEN	0x08000000
#define CR_OIEN	0x04000000
#define CR_IIEN	0x02000000
#define CR_DIEN	0x01000000

#define CR_DMEN		(1 << 28)
#define CR_CHNL_SHIFT	22
#define CR_CHNL_MASK	(3 << CR_CHNL_SHIFT)
#define CR_DWL_SHIFT	19
#define CR_DWL_MASK	(7 << CR_DWL_SHIFT)
#define CR_SWL_SHIFT	16
#define CR_SWL_MASK	(7 << CR_SWL_SHIFT)
#define CR_SCK_MASTER	(1 << 15)	/* bitclock master bit */
#define CR_SWS_MASTER	(1 << 14)	/* wordselect master bit */
#define CR_SCKP		(1 << 13)	/* I2Sclock polarity */
#define CR_SWSP		(1 << 12)	/* LRCK polarity */
#define CR_SPDP		(1 << 11)
#define CR_SDTA		(1 << 10)	/* i2s alignment (msb/lsb) */
#define CR_PDTA		(1 << 9)	/* fifo data alignment */
#define CR_DEL		(1 << 8)	/* delay data by 1 i2sclk */
#define CR_BREN		(1 << 7)	/* clock gating in burst mode */
#define CR_CKDIV_SHIFT	4
#define CR_CKDIV_MASK	(7 << CR_CKDIV_SHIFT)	/* bitclock divider */
#define CR_MUTE		(1 << 3)	/* SSI mute */
#define CR_CPEN		(1 << 2)	/* compressed mode */
#define CR_TRMD		(1 << 1)	/* transmit/receive select */
#define CR_EN		(1 << 0)	/* enable SSI */

#define SR_UIRQ	0x08000000
#define SR_OIRQ	0x04000000
#define SR_IIRQ	0x02000000
#define SR_DIRQ	0x01000000


#define SSIREG(reg)	(*(volatile u32 *)(ssi->mmio + (reg)))

struct sh7764_ssi_priv {
	int irq;
	unsigned long mmio;
	unsigned long sysclk;
	int inuse;
	char *devname;
} ssi_cpu_data[] = {
	{
		.irq		= SSICH0_IRQ,
		.mmio		= 0xff402000,
		.devname	= "SSI_CH0"
	}, {
		.irq		= SSICH1_IRQ,
		.mmio		= 0xff403000,
		.devname	= "SSI_CH1"
	}, {
		.irq		= SSICH2_IRQ,
		.mmio		= 0xff404000,
		.devname	= "SSI_CH2"
	}, {
		.irq		= SSICH3_IRQ,
		.mmio		= 0xff502000,
		.devname	= "SSI_CH3"
	}, {
		.irq		= SSICH4_IRQ,
		.mmio		= 0xff503000,
		.devname	= "SSI_CH4"
	}, {
		.irq		= SSICH5_IRQ,
		.mmio		= 0xff504000,
		.devname	= "SSI_CH5"
	}
};

static irqreturn_t sh7764_ssi_intr(int irq, void *ptr)
{
	struct sh7764_ssi_priv *ssi = ptr;
	u32 ssicr = SSIREG(SSICR);
	u32 ssisr = SSIREG(SSISR);

	if (ssisr & SR_IIRQ) {
		SSIREG(SSICR) &= ~CR_IIEN;
	} else {
		if (ssisr & SR_UIRQ)
			printk(KERN_WARNING "SH7764-SSI UIRQ\n");
		if (ssisr & SR_OIRQ)
			printk(KERN_WARNING "SH7764-SSI OIRQ\n");
	}
	SSIREG(SSISR) &= ~(SR_UIRQ | SR_OIRQ);

	return IRQ_HANDLED;
}

/*
 * track usage of the SSI; it is simplex-only so prevent attempts of
 * concurrent playback + capture. FIXME: any locking required?
 */
static int sh7764_ssi_startup(struct snd_pcm_substream *substream,
			      struct snd_soc_dai *dai)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct sh7764_ssi_priv *ssi = &ssi_cpu_data[rtd->dai->cpu_dai->id];
	int err;

	if (ssi->inuse) {
		pr_debug("ssi: already in use!\n");
		return -EBUSY;
	} else
		ssi->inuse = 1;

	if ((err = request_irq(ssi->irq, sh7764_ssi_intr, IRQF_DISABLED,
			       ssi->devname, ssi))) {
		goto err;
	}

	return 0;
err:
	ssi->inuse = 0;
	return err;
}

static void sh7764_ssi_shutdown(struct snd_pcm_substream *substream,
				struct snd_soc_dai *dai)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct sh7764_ssi_priv *ssi = &ssi_cpu_data[rtd->dai->cpu_dai->id];

	free_irq(ssi->irq, ssi);

	ssi->inuse = 0;
}

static int sh7764_ssi_trigger(struct snd_pcm_substream *substream, int cmd,
		       struct snd_soc_dai *dai)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct sh7764_ssi_priv *ssi = &ssi_cpu_data[rtd->dai->cpu_dai->id];

	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
		SSIREG(SSICR) |= CR_DMEN | CR_UIEN | CR_OIEN | CR_EN;
		break;
	case SNDRV_PCM_TRIGGER_STOP:
		SSIREG(SSICR) &= ~CR_DMEN;
		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
			while (!(SSIREG(SSISR) & SR_DIRQ)) {
				cpu_relax();
				pr_debug(" -> %08x\n", SSIREG(SSISR));
			}
		SSIREG(SSICR) &= ~(CR_UIEN | CR_OIEN | CR_EN);
		SSIREG(SSICR) |= CR_IIEN;
		break;
	default:
		return -EINVAL;
	}
	return 0;
}

static int sh7764_ssi_hwparams(struct snd_pcm_substream *substream,
			 struct snd_pcm_hw_params *params,
			 struct snd_soc_dai *dai)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct sh7764_ssi_priv *ssi = &ssi_cpu_data[rtd->dai->cpu_dai->id];
	unsigned long ssicr = SSIREG(SSICR);
	unsigned int bits, channels, swl, recv, i;

	channels = params_channels(params);
	bits = params->msbits;
	recv = (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) ? 0 : 1;

	pr_debug(" sh7764_ssi_hwparams() enter\nssicr was    %08lx\n", ssicr);
	pr_debug(" bits: %d channels: %d\n", bits, channels);

	ssicr &= ~(CR_TRMD | CR_CHNL_MASK | CR_DWL_MASK | CR_PDTA |
		   CR_SWL_MASK);

	/* direction (send/receive) */
	if (!recv)
		ssicr |= CR_TRMD;	/* transmit */

	/* channels */
	if ((channels < 2) || (channels > 8) || (channels & 1)) {
		pr_debug("ssi: invalid number of channels\n");
		return -EINVAL;
	}
	ssicr |= ((channels >> 1) - 1) << CR_CHNL_SHIFT;

	/* DATA WORD LENGTH (DWL): databits in audio sample */
	i = 0;
	switch (bits) {
	case 32: ++i;
	case 24: ++i;
	case 22: ++i;
	case 20: ++i;
	case 18: ++i;
	case 16: ++i;
		 ssicr |= i << CR_DWL_SHIFT;
	case 8:	 break;
	default:
		pr_debug("ssi: invalid sample width\n");
		return -EINVAL;
	}

	/*
	 * SYSTEM WORD LENGTH: size in bits of half a frame over the I2S
	 * wires. This is usually bits_per_sample x channels/2;  i.e. in
	 * Stereo mode  the SWL equals DWL.  SWL can  be bigger than the
	 * product of (channels_per_slot x samplebits), e.g.  for codecs
	 * like the AD1939 which  only accept 32bit wide TDM slots.  For
	 * "standard" I2S operation we set SWL = chans / 2 * DWL here.
	 * Waiting for ASoC to get TDM support ;-)
	 */
	if ((bits > 16) && (bits <= 24)) {
		bits = 24;	/* these are padded by the SSI */
		/*ssicr |= CR_PDTA;*/ /* cpu/data endianness ? */
	}
	i = 0;
	swl = (bits * channels) / 2;
	switch (swl) {
	case 256: ++i;
	case 128: ++i;
	case 64:  ++i;
	case 48:  ++i;
	case 32:  ++i;
	case 16:  ++i;
		  ssicr |= i << CR_SWL_SHIFT;
	case 8:   break;
	default:
		pr_debug("ssi: invalid system word length computed\n");
		return -EINVAL;
	}

	SSIREG(SSICR) = ssicr;

	pr_debug("sh7764_ssi_hwparams() leave\nssicr is now %08lx\n", ssicr);
	return 0;
}

static int sh7764_ssi_set_sysclk(struct snd_soc_dai *cpu_dai, int clk_id,
			  unsigned int freq, int dir)
{
	struct sh7764_ssi_priv *ssi = &ssi_cpu_data[cpu_dai->id];

	ssi->sysclk = freq;

	return 0;
}

/*
 * This divider is used to generate the SSI_SCK (I2S bitclock) from the
 * clock at the HAC_BIT_CLK ("oversampling clock") pin.
 */
static int sh7764_ssi_set_slkdiv(struct snd_soc_dai *dai, int did, int div)
{
	struct sh7764_ssi_priv *ssi = &ssi_cpu_data[dai->id];
	unsigned long ssicr;
	int i;

	i = 0;
	ssicr = SSIREG(SSICR) & ~CR_CKDIV_MASK;
	switch (div) {
	case 16: ++i;
	case 8:  ++i;
	case 4:  ++i;
	case 2:  ++i;
		 SSIREG(SSICR) = ssicr | (i << CR_CKDIV_SHIFT);
	case 1:  break;
	default:
		pr_debug("ssi: invalid sck divider %d\n", div);
		return -EINVAL;
	}

	return 0;
}

static int sh7764_ssi_set_fmt(struct snd_soc_dai *dai, unsigned int fmt)
{
	struct sh7764_ssi_priv *ssi = &ssi_cpu_data[dai->id];
	unsigned long ssicr = SSIREG(SSICR);

	pr_debug("sh7764_ssi_set_fmt()\nssicr was    0x%08lx\n", ssicr);

	ssicr &= ~(CR_DEL | CR_PDTA | CR_BREN | CR_SWSP | CR_SCKP |
		   CR_SWS_MASTER | CR_SCK_MASTER);

	switch (fmt & SND_SOC_DAIFMT_FORMAT_MASK) {
	case SND_SOC_DAIFMT_I2S:
		break;
	case SND_SOC_DAIFMT_RIGHT_J:
		ssicr |= CR_DEL | CR_PDTA;
		break;
	case SND_SOC_DAIFMT_LEFT_J:
		ssicr |= CR_DEL;
		break;
	default:
		pr_debug("ssi: unsupported format\n");
		return -EINVAL;
	}

	switch (fmt & SND_SOC_DAIFMT_CLOCK_MASK) {
	case SND_SOC_DAIFMT_CONT:
		break;
	case SND_SOC_DAIFMT_GATED:
		ssicr |= CR_BREN;
		break;
	}

	switch (fmt & SND_SOC_DAIFMT_INV_MASK) {
	case SND_SOC_DAIFMT_NB_NF:
		ssicr |= CR_SCKP;	/* sample data at low clkedge */
		break;
	case SND_SOC_DAIFMT_NB_IF:
		ssicr |= CR_SCKP | CR_SWSP;
		break;
	case SND_SOC_DAIFMT_IB_NF:
		break;
	case SND_SOC_DAIFMT_IB_IF:
		ssicr |= CR_SWSP;	/* word select starts low */
		break;
	default:
		pr_debug("ssi: invalid inversion\n");
		return -EINVAL;
	}

	switch (fmt & SND_SOC_DAIFMT_MASTER_MASK) {
	case SND_SOC_DAIFMT_CBM_CFM:
		break;
	case SND_SOC_DAIFMT_CBS_CFM:
		ssicr |= CR_SCK_MASTER;
		break;
	case SND_SOC_DAIFMT_CBM_CFS:
		ssicr |= CR_SWS_MASTER;
		break;
	case SND_SOC_DAIFMT_CBS_CFS:
		ssicr |= CR_SWS_MASTER | CR_SCK_MASTER;
		break;
	default:
		pr_debug("ssi: invalid master/slave configuration\n");
		return -EINVAL;
	}

	SSIREG(SSICR) = ssicr;
	pr_debug("sh7764_ssi_set_fmt() leave\nssicr is now 0x%08lx\n", ssicr);

	return 0;
}

/* the SSI depends on an external clocksource (at HAC_BIT_CLK) even in
 * Master mode,  so really this is board specific;  the SSI can do any
 * rate with the right bitclk and divider settings.
 */
#define SSI_RATES	\
	SNDRV_PCM_RATE_8000_96000

/* the SSI can do 8-32 bit samples, with 8 possible channels */
#define SSI_FMTS	\
	(SNDRV_PCM_FMTBIT_S8      | SNDRV_PCM_FMTBIT_U8      |	\
	 SNDRV_PCM_FMTBIT_S16_LE  | SNDRV_PCM_FMTBIT_U16_LE  |	\
	 SNDRV_PCM_FMTBIT_S20_3LE | SNDRV_PCM_FMTBIT_U20_3LE |	\
	 SNDRV_PCM_FMTBIT_S24_3LE | SNDRV_PCM_FMTBIT_U24_3LE |	\
	 SNDRV_PCM_FMTBIT_S32_LE  | SNDRV_PCM_FMTBIT_U32_LE)

struct snd_soc_dai sh7764_ssi_dai[] = {
	{
		.name			= "SSI0",
		.id			= 0,
		.playback = {
			.rates		= SSI_RATES,
			.formats	= SSI_FMTS,
			.channels_min	= 2,
			.channels_max	= 8,
		},
		.capture = {
			.rates		= SSI_RATES,
			.formats	= SSI_FMTS,
			.channels_min	= 2,
			.channels_max	= 2,
		},
		.ops = {
			.startup	= sh7764_ssi_startup,
			.shutdown	= sh7764_ssi_shutdown,
			.trigger	= sh7764_ssi_trigger,
			.hw_params	= sh7764_ssi_hwparams,
			.set_sysclk	= sh7764_ssi_set_sysclk,
			.set_clkdiv	= sh7764_ssi_set_slkdiv,
			.set_fmt	= sh7764_ssi_set_fmt,
		},
	}, {
		.name			= "SSI1",
		.id			= 1,
		.playback = {
			.rates		= SSI_RATES,
			.formats	= SSI_FMTS,
			.channels_min	= 2,
			.channels_max	= 8,
		},
		.capture = {
			.rates		= SSI_RATES,
			.formats	= SSI_FMTS,
			.channels_min	= 2,
			.channels_max	= 2,
		},
		.ops = {
			.startup	= sh7764_ssi_startup,
			.shutdown	= sh7764_ssi_shutdown,
			.trigger	= sh7764_ssi_trigger,
			.hw_params	= sh7764_ssi_hwparams,
			.set_sysclk	= sh7764_ssi_set_sysclk,
			.set_clkdiv	= sh7764_ssi_set_slkdiv,
			.set_fmt	= sh7764_ssi_set_fmt,
		},
	},
};
EXPORT_SYMBOL_GPL(sh7764_ssi_dai);

static int __init sh7764_ssi_init(void)
{
	return snd_soc_register_dais(sh7764_ssi_dai,
				     ARRAY_SIZE(sh7764_ssi_dai));
}
module_init(sh7764_ssi_init);

static void __exit sh7764_ssi_exit(void)
{
	snd_soc_unregister_dais(sh7764_ssi_dai, ARRAY_SIZE(sh7764_ssi_dai));
}
module_exit(sh7764_ssi_exit);

MODULE_AUTHOR("ALPHAPROJECT");
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("SuperH onchip SSI (I2S) audio driver");
