/*
 * Alpha Project AP-SH4A-4A Support.
 *
 * Copyright (C) 2012 Nobuhiro Iwamatsu <nobuhiro.iwamatsu.yh@renesas.com>
 * Copyright (C) 2012 Renesas Solutions Corp.
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 */
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/io.h>
#include <linux/interrupt.h>
#include <linux/usb/r8a66597.h>
#include <linux/usb/renesas_usbhs.h>
#include <linux/platform_data/ehci-sh.h>
#include <linux/mtd/physmap.h>
#include <linux/sh_eth.h>
#include <linux/i2c.h>
#include <linux/gpio.h>
#include <linux/irq.h>
#include <linux/clk.h>
#include <linux/delay.h>
#include <cpu/sh7734.h>
#include <asm/machvec.h>
#include <asm/sizes.h>

static struct mtd_partition nor_flash_partitions[] = {
	{
		.name		= "boot loader",
		.offset		= 0x00000000,
		.size		= SZ_512K,
                .mask_flags     = MTD_WRITEABLE,  /* force read-only */
	},
	{
		.name		= "bootenv",
		.offset		= MTDPART_OFS_APPEND,
		.size		= SZ_512K,
                .mask_flags     = MTD_WRITEABLE,  /* force read-only */
	},
	{
		.name		= "kernel",
		.offset		= MTDPART_OFS_APPEND,
		.size		= SZ_4M,
	},
	{
		.name		= "data",
		.offset		= MTDPART_OFS_APPEND,
		.size		= MTDPART_SIZ_FULL,
	},
};

static struct physmap_flash_data nor_flash_data = {
	.width		= 2,
	.parts		= nor_flash_partitions,
	.nr_parts	= ARRAY_SIZE(nor_flash_partitions),
};

static struct resource nor_flash_resources[] = {
	[0] = {
		.start  = 0x00000000,
		.end    = 0x00000000 + SZ_16M - 1,
		.flags  = IORESOURCE_MEM,
	}
};

static struct platform_device nor_flash_device = {
	.name       = "physmap-flash",
	.dev        = {
		.platform_data  = &nor_flash_data,
	},
	.num_resources  = ARRAY_SIZE(nor_flash_resources),
	.resource   = nor_flash_resources,
};

static struct resource sh_eth_resources[] = {
	{
		.start  = 0xFEE00000,
		.end    = 0xFEE007FF,
		.flags  = IORESOURCE_MEM,
	}, {
		/* TSU */
		.start  = 0xFEE01800,
		.end    = 0xFEE01FFF,
		.flags  = IORESOURCE_MEM,
	}, {
		.start  = evt2irq(0xCA0),
		.end    = evt2irq(0xCA0),
		.flags  = IORESOURCE_IRQ,
	},
};

static struct sh_eth_plat_data sh_eth_pdata = {
	.phy = 0,
	.edmac_endian = EDMAC_LITTLE_ENDIAN,
	.register_type = SH_ETH_REG_GIGABIT,
	.phy_interface = PHY_INTERFACE_MODE_GMII,
	.ether_link_active_low = 1,
};

static struct platform_device sh_eth_device = {
	.name		= "sh-eth",
	.resource	= sh_eth_resources,
	.id		= 0,
	.num_resources	= ARRAY_SIZE(sh_eth_resources),
	.dev		= {
		.platform_data = &sh_eth_pdata,
	},
};

/* USB */
#define USBPCTRL1       (0xFFE70804)
#define USBST           (0xFFE70808)
#define USBEH0          (0xFFE7080C)
#define USBOH0          (0xFFE7081C)
#define USBCTL0         (0xFFE70858)

#define USBPCTRL1_RST           (1 << 31)
#define USBPCTRL1_PHYENB        (1 << 0)
#define USBPCTRL1_PLLENB        (1 << 1)
#define USBPCTRL1_PHYRST        (1 << 2)
#define USBST_ACT                       (1 << 31)
#define USBST_PPL                       (1 << 30)

#define USB_MAGIC_REG0  (0xFFE70094)
#define USB_MAGIC_REG1  (0xFFE7009C)

#define USBCTL0_CLKSEL  (1 << 7) /* Use external clock */
#define USBPCTRL0       0xFFE70800
#define USBPCTRL0_PORT1_FUNCTION        (1 << 0)
#define USBPCTRL0_OVC1 (1 << 1)

/* USB host */
static void usb_ehci_phy_init(void)
{
	__raw_writel(USBPCTRL1_RST, USBPCTRL1);
	__raw_writel(USBPCTRL1_PHYENB|USBPCTRL1_PLLENB, USBPCTRL1);

	while (__raw_readl(USBST) != (USBST_ACT|USBST_PPL))
		cpu_relax();

	__raw_writel(USBPCTRL1_PHYENB|USBPCTRL1_PLLENB|USBPCTRL1_PHYRST, USBPCTRL1);

	__raw_writel(0x00FF0040, USB_MAGIC_REG0);
	__raw_writel(0x00000001, USB_MAGIC_REG1);

	gpio_request(GPIO_GP_4_17,  NULL);
	gpio_direction_output(GPIO_GP_4_17, 1);
	__raw_writel(USBPCTRL0_PORT1_FUNCTION | USBPCTRL0_OVC1, USBPCTRL0);
}

static struct ehci_sh_platdata usb_ehci_pdata = {
	.phy_init = &usb_ehci_phy_init,
};

static struct resource usb_ehci_resources[] = {
	[0] = {
		.start  = 0xFFE70000,
		.end    = 0xFFE70058 - 1,
		.flags  = IORESOURCE_MEM,
	},
	[1] = {
		.start  = evt2irq(0x580),
		.end    = evt2irq(0x580),
		.flags  = IORESOURCE_IRQ,
	},
};

static u64 usb_ehci_dma_mask = 0xffffffffUL;
static struct platform_device usb_ehci_device = {
	.name   = "sh_ehci",
	.id             = -1,
	.dev = {
		.dma_mask = &usb_ehci_dma_mask,
		.coherent_dma_mask = DMA_BIT_MASK(32),
		.platform_data      = &usb_ehci_pdata,
	},
	.num_resources  = ARRAY_SIZE(usb_ehci_resources),
	.resource       = usb_ehci_resources,
};

/* USB function */
static struct r8a66597_platdata usb_function_data = {
	.buswait = 0x01,
	.on_chip = 1,
};

static struct resource usb_function_resources[] = {
	[0] = {
		.start  = 0xffe60000,
		.end    = 0xffe60104 - 1,
		.flags  = IORESOURCE_MEM,
	},
	[1] = {
		.start  = evt2irq(0x580),
		.end    = evt2irq(0x580),
		.flags  = IORESOURCE_IRQ | IRQF_TRIGGER_LOW,
	},
};

static struct platform_device usb_function_device = {
	.name = "r8a66597_udc",
	.id             = 1,
	.dev = {
		.dma_mask               = NULL,         /*  not use dma */
		.coherent_dma_mask      = 0xffffffff,
		.platform_data          = &usb_function_data,
	},
	.num_resources  = ARRAY_SIZE(usb_function_resources),
	.resource       = usb_function_resources,
};

static struct platform_device *apsh4a4a_devices[] __initdata = {
	&nor_flash_device,
	&sh_eth_device,
	&usb_ehci_device,
	&usb_function_device,
};

#define I2C_SUPPORT 1
#if I2C_SUPPORT
#define EEPROM_ADDR 0x50
static u8 mac_read(struct i2c_adapter *a, u8 command)
{
	struct i2c_msg msg[2];
	u8 buf;
	int ret;

	msg[0].addr  = EEPROM_ADDR;
	msg[0].flags = 0;
	msg[0].len   = 1;
	msg[0].buf   = &command;

	msg[1].addr  = EEPROM_ADDR;
	msg[1].flags = I2C_M_RD;
	msg[1].len   = 1;
	msg[1].buf   = &buf;

	ret = i2c_transfer(a, msg, 2);
	if (ret < 0) {
		printk(KERN_ERR "error %d\n", ret);
		buf = 0xff;
	}

	return buf;
}

static void __init apsh4a4a_get_macaddress(unsigned char *mac)
{
	struct i2c_adapter *i2c = i2c_get_adapter(0);
	int i;
	if (!i2c) {
		pr_err("can not get I2C 0\n");
		return;
	}

	/* Read MAC address from EEPROM */
	/* For sh-eth */
	for (i = 0; i < 6; i++) {
		mac[i] = mac_read(i2c, (u8)i);
		msleep(10);
	}

	i2c_put_adapter(i2c);
}
#else /* I2C_SUPPORT */
static void __init apsh4a4a_get_macaddress(unsigned char *mac)
{
	mac[0] = 0x00;
	mac[1] = 0x0c;
	mac[2] = 0x7b;
	mac[3] = 0x34;
	mac[4] = 0x01;
	mac[5] = 0x06;
}
#endif /* I2C_SUPPORT */

static int __init devices_setup(void)
{
	unsigned char mac[6];

	apsh4a4a_get_macaddress(mac);
	memcpy(sh_eth_pdata.mac_addr, &mac[0], sizeof(mac));

	return 0;
}
device_initcall(devices_setup);

static int __init apsh4a4a_arch_init(void)
{
	/* GETHER */
	__raw_writel(0x3, 0xFEE00790);	/* RMII_MII reg for GMII */

	return platform_add_devices(apsh4a4a_devices,
			    ARRAY_SIZE(apsh4a4a_devices));
}
arch_initcall(apsh4a4a_arch_init);

static void __init apsh4a4a_setup(char **cmdline_p)
{
	pr_info("Alpha Project AP-SH4A-4A support:\n");
}

/* The Machine Vector */
static struct sh_machine_vector mv_apsh4a4a __initmv = {
	.mv_name		= "AP-SH4A-4A",
	.mv_setup		= apsh4a4a_setup,
};
