/*
 * Renesas R0P7734C00000RZ Support.
 *
 * Copyright (C) 2012 Nobuhiro Iwamatsu <nobuhiro.iwamatsu.yh@renesas.com>
 * Copyright (C) 2012 Renesas Solutions Corp.
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 */

#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/irq.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/usb/r8a66597.h>
#include <linux/usb/renesas_usbhs.h>
#include <linux/platform_data/ehci-sh.h>
#include <linux/mtd/physmap.h>
#include <linux/sh_eth.h>
#include <linux/smsc911x.h>
#include <linux/mmc/host.h>
#include <linux/mmc/sh_mmcif.h>
#include <linux/mmc/sh_mobile_sdhi.h>
#include <linux/i2c.h>
#include <linux/gpio.h>
#include <linux/delay.h>
#include <cpu/sh7734.h>
#include <asm/sizes.h>

static struct mtd_partition nor_flash_partitions[] = {
	{
		.name		= "boot loader",
		.offset		= 0x00000000,
		.size		= SZ_512K,
                .mask_flags     = MTD_WRITEABLE,  /* force read-only */
	},
	{
		.name		= "bootenv",
		.offset		= MTDPART_OFS_APPEND,
		.size		= SZ_512K,
                .mask_flags     = MTD_WRITEABLE,  /* force read-only */
	},
	{
		.name		= "kernel",
		.offset		= MTDPART_OFS_APPEND,
		.size		= SZ_4M,
	},
	{
		.name		= "data",
		.offset		= MTDPART_OFS_APPEND,
		.size		= MTDPART_SIZ_FULL,
	},
};

static struct physmap_flash_data nor_flash_data = {
	.width		= 2,
	.parts		= nor_flash_partitions,
	.nr_parts	= ARRAY_SIZE(nor_flash_partitions),
};

static struct resource nor_flash_resources[] = {
	[0] = {
		.start  = 0x00000000,
		.end    = 0x00000000 + SZ_64M - 1,
		.flags  = IORESOURCE_MEM,
	}
};

static struct platform_device nor_flash_device = {
	.name       = "physmap-flash",
	.dev        = {
		.platform_data  = &nor_flash_data,
	},
	.num_resources  = ARRAY_SIZE(nor_flash_resources),
	.resource   = nor_flash_resources,
};

static struct resource sh_eth_resources[] = {
	{
		.start  = 0xFEE00000,
		.end    = 0xFEE007FF,
		.flags  = IORESOURCE_MEM,
	}, {
		/* TSU */
		.start  = 0xFEE01800,
		.end    = 0xFEE01FFF,
		.flags  = IORESOURCE_MEM,
	}, {
		.start  = evt2irq(0xCA0),
		.end    = evt2irq(0xCA0),
		.flags  = IORESOURCE_IRQ,
	},
};

static struct sh_eth_plat_data sh_eth_pdata = {
	.phy = 0,
	.edmac_endian = EDMAC_LITTLE_ENDIAN,
	.register_type = SH_ETH_REG_GIGABIT,
	.phy_interface = PHY_INTERFACE_MODE_RMII,
};

static struct platform_device sh_eth_device = {
	.name		= "sh-eth",
	.resource	= sh_eth_resources,
	.id		= 0,
	.num_resources	= ARRAY_SIZE(sh_eth_resources),
	.dev		= {
		.platform_data = &sh_eth_pdata,
	},
};

/*  LAN89218 */
static struct smsc911x_platform_config smsc911x_config = {
	.phy_interface	= PHY_INTERFACE_MODE_MII,
	.irq_polarity	= SMSC911X_IRQ_POLARITY_ACTIVE_LOW,
	.irq_type	= SMSC911X_IRQ_TYPE_OPEN_DRAIN,
	.flags		= SMSC911X_USE_16BIT | SMSC911X_FORCE_INTERNAL_PHY,
};

static struct resource smsc911x_resources[] = {
	[0] = {
		.start  = 0x04000000,
		.end    = 0x04000100 - 1 ,
		.flags  = IORESOURCE_MEM,
	},
	[1] = {
		.start  = evt2irq(0x280),
		.end    = evt2irq(0x280),
		.flags  = IORESOURCE_IRQ,
	}
};

static struct platform_device smsc911x_device = {
	.name       = "smsc911x",
	.id			= -1,
	.num_resources  = ARRAY_SIZE(smsc911x_resources),
	.resource	= smsc911x_resources,
	.dev		= {
		.platform_data = &smsc911x_config,
	},
};

/* SDHI1 */
static void sdhi1_set_pwr(struct platform_device *pdev, int state)
{
	gpio_set_value(GPIO_GP_2_8, state);
}

static struct sh_mobile_sdhi_info sdhi1_info = {
	.dma_slave_tx   = SHDMA_SLAVE_SDHI1_TX,
	.dma_slave_rx   = SHDMA_SLAVE_SDHI1_RX,
	.tmio_caps      = MMC_CAP_SDIO_IRQ | MMC_CAP_POWER_OFF_CARD |
			  MMC_CAP_NEEDS_POLL,
	.set_pwr	= sdhi1_set_pwr,
};

static struct resource sdhi1_resources[] = {
	[0] = {
		.name	= "SDHI1",
		.start  = 0xffe4d000,
		.end    = 0xffe4d0ff,
		.flags  = IORESOURCE_MEM,
	},
	[1] = {
		.start  = evt2irq(0xB00),
		.end    = evt2irq(0xB00),
		.flags  = IORESOURCE_IRQ,
	},
};

static struct platform_device sdhi1_device = {
	.name           = "sh_mobile_sdhi",
	.num_resources  = ARRAY_SIZE(sdhi1_resources),
	.resource       = sdhi1_resources,
	.id             = 1,
	.dev	= {
		.platform_data	= &sdhi1_info,
	},
};

/* SDHI2 */
static void sdhi2_set_pwr(struct platform_device *pdev, int state)
{
	gpio_set_value(GPIO_GP_2_30, state);
}

static struct sh_mobile_sdhi_info sdhi2_info = {
	.dma_slave_tx   = SHDMA_SLAVE_SDHI2_TX,
	.dma_slave_rx   = SHDMA_SLAVE_SDHI2_RX,
	.tmio_caps      = MMC_CAP_SDIO_IRQ | MMC_CAP_POWER_OFF_CARD |
			  MMC_CAP_NEEDS_POLL,
	.set_pwr	= sdhi2_set_pwr,
};

static struct resource sdhi2_resources[] = {
	[0] = {
		.name	= "SDHI2",
		.start  = 0xffe4e000,
		.end    = 0xffe4e0ff,
		.flags  = IORESOURCE_MEM,
	},
	[1] = {
		.start  = evt2irq(0xCC0),
		.end    = evt2irq(0xCC0),
		.flags  = IORESOURCE_IRQ,
	},
};

static struct platform_device sdhi2_device = {
	.name           = "sh_mobile_sdhi",
	.num_resources  = ARRAY_SIZE(sdhi2_resources),
	.resource       = sdhi2_resources,
	.id             = 2,
	.dev	= {
		.platform_data	= &sdhi2_info,
	},
};

/* USB */
#define USBPCTRL1       (0xFFE70804)
#define USBST           (0xFFE70808)
#define USBEH0          (0xFFE7080C)
#define USBOH0          (0xFFE7081C)
#define USBCTL0         (0xFFE70858)

#define USBPCTRL1_RST           (1 << 31)
#define USBPCTRL1_PHYENB        (1 << 0)
#define USBPCTRL1_PLLENB        (1 << 1)
#define USBPCTRL1_PHYRST        (1 << 2)
#define USBST_ACT                       (1 << 31)
#define USBST_PPL                       (1 << 30)

#define USB_MAGIC_REG0  (0xFFE70094)
#define USB_MAGIC_REG1  (0xFFE7009C)

#define USBCTL0_CLKSEL  (1 << 7) /* Use external clock */
#define USBPCTRL0       0xFFE70800
#define USBPCTRL0_PORT1_FUNCTION        (1 << 0)
#define USBPCTRL0_OVC1 (1 << 1)

/* USB host */
static void usb_ehci_phy_init(void)
{
	__raw_writel(USBPCTRL1_RST, USBPCTRL1);
	__raw_writel(USBPCTRL1_PHYENB|USBPCTRL1_PLLENB, USBPCTRL1);

	while (__raw_readl(USBST) != (USBST_ACT|USBST_PPL))
		cpu_relax();

	__raw_writel(USBPCTRL1_PHYENB|USBPCTRL1_PLLENB|USBPCTRL1_PHYRST, USBPCTRL1);

	__raw_writel(0x00FF0040, USB_MAGIC_REG0);
	__raw_writel(0x00000001, USB_MAGIC_REG1);

	__raw_writel(__raw_readl(USBCTL0) | USBCTL0_CLKSEL, USBCTL0);

	if (gpio_get_value(GPIO_GP_5_1)) {
		printk(KERN_INFO "USB1 function is selected\n");
		__raw_writel(USBPCTRL0_PORT1_FUNCTION | USBPCTRL0_OVC1, USBPCTRL0);
	} else
		printk(KERN_INFO "USB1 host is selected\n");
}

static struct ehci_sh_platdata usb_ehci_pdata = {
	.phy_init = &usb_ehci_phy_init,
};

static struct resource usb_ehci_resources[] = {
	[0] = {
		.start  = 0xFFE70000,
		.end    = 0xFFE70058 - 1,
		.flags  = IORESOURCE_MEM,
	},
	[1] = {
		.start  = evt2irq(0x580),
		.end    = evt2irq(0x580),
		.flags  = IORESOURCE_IRQ,
	},
};

static u64 usb_ehci_dma_mask = 0xffffffffUL;
static struct platform_device usb_ehci_device = {
	.name   = "sh_ehci",
	.id             = -1,
	.dev = {
		.dma_mask = &usb_ehci_dma_mask,
		.coherent_dma_mask = DMA_BIT_MASK(32),
		.platform_data      = &usb_ehci_pdata,
	},
	.num_resources  = ARRAY_SIZE(usb_ehci_resources),
	.resource       = usb_ehci_resources,
};

/* USB function */
static void usb_port_power(int port, int power)
{
	gpio_set_value(GPIO_GP_2_5, power);
}
static struct r8a66597_platdata usb_function_data = {
	.buswait = 0x01,
	.on_chip = 1,
	.port_power = usb_port_power,
};

static struct resource usb_function_resources[] = {
	[0] = {
		.start  = 0xffe60000,
		.end    = 0xffe60104 - 1,
		.flags  = IORESOURCE_MEM,
	},
	[1] = {
		.start  = evt2irq(0x580),
		.end    = evt2irq(0x580),
		.flags  = IORESOURCE_IRQ | IRQF_TRIGGER_LOW,
	},
};

static struct platform_device usb_function_device = {
	.name = "r8a66597_udc",
	.id             = 1,
	.dev = {
		.dma_mask               = NULL,         /*  not use dma */
		.coherent_dma_mask      = 0xffffffff,
		.platform_data          = &usb_function_data,
	},
	.num_resources  = ARRAY_SIZE(usb_function_resources),
	.resource       = usb_function_resources,
};

static struct platform_device *r0p7734_devices[] __initdata = {
	&nor_flash_device,
	&sh_eth_device,
	&smsc911x_device,
	&sdhi1_device,
	&sdhi2_device,
	&usb_ehci_device,
	&usb_function_device,
};
#if 1
#define EEPROM_ADDR 0x50
static u8 mac_read(struct i2c_adapter *a, u8 command)
{
	struct i2c_msg msg[2];
	u8 buf;
	int ret;

	msg[0].addr  = EEPROM_ADDR;
	msg[0].flags = 0;
	msg[0].len   = 1;
	msg[0].buf   = &command;

	msg[1].addr  = EEPROM_ADDR;
	msg[1].flags = I2C_M_RD;
	msg[1].len   = 1;
	msg[1].buf   = &buf;

	ret = i2c_transfer(a, msg, 2);
	if (ret < 0) {
		printk(KERN_ERR "error %d\n", ret);
		buf = 0xff;
	}

	return buf;
}
#endif

static void __init r0p7734_get_macaddress(unsigned char *mac)
{
#if 1
	struct i2c_adapter *i2c = i2c_get_adapter(0);
	int i, j = 0;
	if (!i2c) {
		pr_err("can not get I2C 0\n");
		return;
	}

	/* Read MAC address from EEPROM */
	/* For sh-eth */
	for (i = 0; i < 6; i++, j++) {
		mac[j] = mac_read(i2c, 0x10 + i);
		//pr_info("mac[%d] %02x\n", j , mac[j]);
		printk("%s: mac[%d] %02x\n", __func__, j , mac[j]);
		msleep(10);
	}

	/* For SMSC LAN89218 */
	for (i = 0; i < 6; i++, j++) {
		mac[j] = mac_read(i2c, 0x20 + i);
		printk("%s: mac[%d] %02x\n", __func__, j , mac[j]);
		msleep(10);
	}

	i2c_put_adapter(i2c);
#else
	mac[0] = 0x74;
	mac[1] = 0x90;
	mac[2] = 0x50;
	mac[3] = 0x00;
	mac[4] = 0x33;
	mac[5] = 0x47;

	mac[6] = 0x74;
	mac[7] = 0x90;
	mac[8] = 0x50;
	mac[9] = 0x00;
	mac[10] = 0x33;
	mac[11] = 0x48;
#endif
}

static int __init devices_setup(void)
{
	unsigned char mac[12];

	r0p7734_get_macaddress(mac);
	memcpy(sh_eth_pdata.mac_addr, &mac[0], (sizeof(mac)/2));
	memcpy(smsc911x_config.mac, &mac[6], (sizeof(mac)/2));

	return 0;
}
device_initcall(devices_setup);

/*
 *                   DS2-2 DS2-1
 * LBSC              *     ON
 * FLCTL/            ON    OFF
 * SCIF2/RQSPI       OFF   OFF
 *
 *                   DS2-5 DS2-4 DS2-3
 * RMII              ON    *      ON
 * VI0               ON    ON     OFF
 * HSCIF             ON    OFF    OFF
 * RMII/SCIF1/SDHI2  OFF   *      ON
 * SCIF1/SDHI2       OFF   ON     OFF
 * HSCIF/SCIF1/SDHI2 OFF   OFF    OFF
 *
 *                   DS2-7  DS2-6
 * DU0-DVI           *      ON
 * DU0-LCD/CN13      ON     OFF
 * DU0-DU0/CN14      OFF    OFF
 *
 *                   DS2-8
 * CAN0/CAN1         ON
 * MediaLB           OFF
 *
 *                   DS3-1
 * VI1/SSI0-SCK      ON
 * AUDATA            OFF
 *
 *                   DS3-2
 * SSI->AudioDAC     ON
 * SSI->MediaLB      OFF
 */

#define PUPCTL2		0xFFFC0108
#define MSTPCR1		0xFFC80034
#define IRQ1		evt2irq(0x280)

static int __init r0p7734_arch_init(void)
{
	/* LAN 89218 */
	gpio_request(GPIO_FN_IRQ1_A,  NULL);
	irq_set_irq_type(IRQ1, IRQ_TYPE_LEVEL_LOW);

	/* USB */
	gpio_request(GPIO_GP_5_1,  NULL);
	gpio_request(GPIO_GP_2_5,  NULL);
	gpio_request(GPIO_GP_4_30,  NULL);
	gpio_direction_input(GPIO_GP_5_1);
	gpio_direction_output(GPIO_GP_2_5, 1);
	gpio_direction_input(GPIO_GP_4_30);

	/* SDHI */
	gpio_request(GPIO_GP_2_8, NULL);
	gpio_direction_output(GPIO_GP_2_8, 0);
	gpio_request(GPIO_GP_2_30, NULL);
	gpio_direction_output(GPIO_GP_2_30, 0);
	__raw_writel(__raw_readl(PUPCTL2) & 0x003FFFFF, PUPCTL2);

	/* IPSR3 */
	gpio_request(GPIO_FN_D15, NULL);
	gpio_request(GPIO_FN_CS1_A26, NULL);
	gpio_request(GPIO_FN_RX3_B, NULL);
	gpio_request(GPIO_FN_TX3_B, NULL);
	gpio_request(GPIO_FN_SD1_CD_A, NULL);
	gpio_request(GPIO_FN_SD1_WP_A, NULL);
	gpio_request(GPIO_FN_SD1_CMD_A, NULL);
	gpio_request(GPIO_FN_RD_WR, NULL);
	gpio_request(GPIO_FN_EX_WAIT0, NULL);
	gpio_request(GPIO_FN_SD1_DAT0_A, NULL);
	gpio_request(GPIO_FN_SD1_DAT1_A, NULL);
	gpio_request(GPIO_FN_SD1_DAT2_A, NULL);

	return platform_add_devices(r0p7734_devices,
			    ARRAY_SIZE(r0p7734_devices));
}
arch_initcall(r0p7734_arch_init);

/* Initialize IRQ setting */
void __init init_r0p7734_IRQ(void)
{
	plat_irq_setup_pins(IRQ_MODE_IRQ3210);
}

static void __init r0p7734_setup(char **cmdline_p)
{
	printk(KERN_INFO "Renesas R0P7734C00000RZ support.\n");

}

/* The Machine Vector */
static struct sh_machine_vector mv_r0p7734 __initmv = {
	.mv_name		= "r0p7734",
	.mv_setup		= r0p7734_setup,
	.mv_init_irq	= init_r0p7734_IRQ,
};
