package sharin.unlinq;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Queue;
import java.util.Set;
import java.util.Map.Entry;

import sharin.util.ArrayUtils;

public class BasicEnumerable<T> implements Enumerable<T> {

    private final Iterable<T> iterable;

    public BasicEnumerable(Iterable<T> iterable) {

        if (iterable == null) {
            iterable = Collections.<T> emptyList();
        }

        this.iterable = iterable;
    }

    public Iterator<T> iterator() {
        return iterable.iterator();
    }

    public static <E> Enumerable<E> from(Iterable<E> iterable) {
        return new BasicEnumerable<E>(iterable);
    }

    public static <E> Enumerable<E> from(E... objects) {

        if (objects == null) {
            return new BasicEnumerable<E>(null);
        }

        return new BasicEnumerable<E>(Arrays.asList(objects));
    }

    public static Enumerable<Boolean> fromBoolean(boolean... values) {
        return new BasicEnumerable<Boolean>(Arrays.asList(ArrayUtils
                .toWrappers(values)));
    }

    public static Enumerable<Byte> fromByte(byte... values) {
        return new BasicEnumerable<Byte>(Arrays.asList(ArrayUtils
                .toWrappers(values)));
    }

    public static Enumerable<Character> fromChar(char... values) {
        return new BasicEnumerable<Character>(Arrays.asList(ArrayUtils
                .toWrappers(values)));
    }

    public static Enumerable<Short> fromShort(short... values) {
        return new BasicEnumerable<Short>(Arrays.asList(ArrayUtils
                .toWrappers(values)));
    }

    public static Enumerable<Integer> from(int... values) {
        return new BasicEnumerable<Integer>(Arrays.asList(ArrayUtils
                .toWrappers(values)));
    }

    public static Enumerable<Long> fromLong(long... values) {
        return new BasicEnumerable<Long>(Arrays.asList(ArrayUtils
                .toWrappers(values)));
    }

    public static Enumerable<Float> fromFloat(float... values) {
        return new BasicEnumerable<Float>(Arrays.asList(ArrayUtils
                .toWrappers(values)));
    }

    public static Enumerable<Double> fromDouble(double... values) {
        return new BasicEnumerable<Double>(Arrays.asList(ArrayUtils
                .toWrappers(values)));
    }

    public T aggregate(Func2<T, T, T> func) {
        Iterator<T> iterator = iterable.iterator();
        T seed = iterator.next();
        return doAggregate(seed, func, iterator);
    }

    public <A> A aggregate(A seed, Func2<A, T, A> func) {
        return doAggregate(seed, func, iterable.iterator());
    }

    public <A, R> R aggregate(A seed, Func2<A, T, A> func,
            Func<A, R> resultSelector) {

        A a = doAggregate(seed, func, iterable.iterator());
        return resultSelector.call(a);
    }

    private <A> A doAggregate(A seed, Func2<A, T, A> func, Iterator<T> iterator) {
        A a = seed;

        while (iterator.hasNext()) {
            a = func.call(a, iterator.next());
        }

        return a;
    }

    public Boolean all(Func<T, Boolean> predicate) {

        for (T t : iterable) {

            if (!predicate.call(t)) {
                return false;
            }
        }

        return true;
    }

    public Boolean any() {
        return iterable.iterator().hasNext();
    }

    public Boolean any(Func<T, Boolean> predicate) {

        for (T t : iterable) {

            if (predicate.call(t)) {
                return true;
            }
        }

        return false;
    }

    public double average() {
        Iterator<T> iterator = iterable.iterator();
        double sum = (Integer) iterator.next();
        int count = 1;

        while (iterator.hasNext()) {
            sum += (Integer) iterator.next();
            count++;
        }

        return sum / count;
    }

    public double averageLong() {
        Iterator<T> iterator = iterable.iterator();
        double sum = (Long) iterator.next();
        int count = 1;

        while (iterator.hasNext()) {
            sum += (Long) iterator.next();
            count++;
        }

        return sum / count;
    }

    public float averageFloat() {
        Iterator<T> iterator = iterable.iterator();
        float sum = (Float) iterator.next();
        int count = 1;

        while (iterator.hasNext()) {
            sum += (Float) iterator.next();
            count++;
        }

        return sum / count;
    }

    public double averageDouble() {
        Iterator<T> iterator = iterable.iterator();
        double sum = (Double) iterator.next();
        int count = 1;

        while (iterator.hasNext()) {
            sum += (Double) iterator.next();
            count++;
        }

        return sum / count;
    }

    public double average(Func<T, Integer> selector) {
        Iterator<T> iterator = iterable.iterator();
        double sum = selector.call(iterator.next());
        int count = 1;

        while (iterator.hasNext()) {
            sum += selector.call(iterator.next());
            count++;
        }

        return sum / count;
    }

    public double averageLong(Func<T, Long> selector) {
        Iterator<T> iterator = iterable.iterator();
        double sum = selector.call(iterator.next());
        int count = 1;

        while (iterator.hasNext()) {
            sum += selector.call(iterator.next());
            count++;
        }

        return sum / count;
    }

    public float averageFloat(Func<T, Float> selector) {
        Iterator<T> iterator = iterable.iterator();
        float sum = selector.call(iterator.next());
        int count = 1;

        while (iterator.hasNext()) {
            sum += selector.call(iterator.next());
            count++;
        }

        return sum / count;
    }

    public double averageDouble(Func<T, Double> selector) {
        Iterator<T> iterator = iterable.iterator();
        double sum = selector.call(iterator.next());
        int count = 1;

        while (iterator.hasNext()) {
            sum += selector.call(iterator.next());
            count++;
        }

        return sum / count;
    }

    public <R> Enumerable<R> cast(final Class<R> resultClass) {
        return new BasicEnumerable<R>(new Iterable<R>() {

            public Iterator<R> iterator() {
                return new Iterator<R>() {

                    private final Iterator<T> iterator = iterable.iterator();

                    public boolean hasNext() {
                        return iterator.hasNext();
                    }

                    public R next() {
                        return resultClass.cast(iterator.next());
                    }

                    public void remove() {
                        throw new UnsupportedOperationException();
                    }
                };
            }
        });
    }

    public Enumerable<T> concat(final Enumerable<T> second) {
        return new BasicEnumerable<T>(new Iterable<T>() {

            public Iterator<T> iterator() {
                return new QueuedIterator<T, T, T>(new ConcatenatedIterator<T>(
                        iterable.iterator(), second.iterator())) {

                    @Override
                    protected void addElement(Queue<T> queue, T t) {
                        queue.add(t);
                    }

                    @Override
                    protected T toResult(T e) {
                        return e;
                    }
                };
            }
        });
    }

    public boolean contains(Object o) {

        for (T t : iterable) {

            if (equals(o, t)) {
                return true;
            }
        }

        return false;
    }

    private boolean equals(Object o1, Object o2) {

        if (o2 == null) {

            if (o1 == null) {
                return true;
            }

        } else {

            if (o2.equals(o1)) {
                return true;
            }
        }

        return false;
    }

    public int count() {
        int c = 0;

        for (Iterator<T> it = iterable.iterator(); it.hasNext(); it.next()) {
            c++;
        }

        return c;
    }

    public int count(Func<T, Boolean> predicate) {
        int c = 0;

        for (T t : iterable) {

            if (predicate.call(t)) {
                c++;
            }
        }

        return c;
    }

    public Enumerable<T> defaultIfEmpty() {
        return defaultIfEmpty(null);
    }

    public Enumerable<T> defaultIfEmpty(T defaultValue) {

        if (iterable.iterator().hasNext()) {
            return new BasicEnumerable<T>(iterable);
        }

        List<T> list = new ArrayList<T>();
        list.add(defaultValue);
        return new BasicEnumerable<T>(list);
    }

    public Enumerable<T> distinct() {
        return new BasicEnumerable<T>(new Iterable<T>() {

            public Iterator<T> iterator() {
                return new QueuedIterator<T, T, T>(iterable.iterator()) {

                    private final Set<T> set = new HashSet<T>();

                    @Override
                    protected void addElement(Queue<T> queue, T t) {

                        if (!set.contains(t)) {
                            queue.add(t);
                            set.add(t);
                        }
                    }

                    @Override
                    protected T toResult(T e) {
                        return e;
                    }
                };
            }
        });
    }

    public T elementAt(int index) {
        Iterator<T> iterator = iterable.iterator();

        for (int i = 0; i < index; i++) {
            iterator.next();
        }

        return iterator.next();
    }

    public T elementAtOrDefault(int index) {
        return elementAtOrDefault(index, null);
    }

    public T elementAtOrDefault(int index, T defaultValue) {
        Iterator<T> iterator = iterable.iterator();

        for (int i = 0; i < index; i++) {

            if (!iterator.hasNext()) {
                return defaultValue;
            }

            iterator.next();
        }

        if (!iterator.hasNext()) {
            return defaultValue;
        }

        return iterator.next();
    }

    public static <R> Enumerable<R> empty() {
        return new BasicEnumerable<R>((Iterable<R>) null);
    }

    public Enumerable<T> except(final Enumerable<T> second) {
        return new BasicEnumerable<T>(new Iterable<T>() {

            public Iterator<T> iterator() {
                return new QueuedIterator<T, T, T>(iterable.iterator()) {

                    private final Set<T> firstSet = new HashSet<T>();

                    private final Set<T> secondSet = new HashSet<T>(second
                            .toList());

                    @Override
                    protected void addElement(Queue<T> queue, T t) {

                        if (!secondSet.contains(t) && !firstSet.contains(t)) {
                            queue.add(t);
                            firstSet.add(t);
                        }
                    }

                    @Override
                    protected T toResult(T e) {
                        return e;
                    }
                };
            }
        });
    }

    public T first() {
        return iterable.iterator().next();
    }

    public T first(Func<T, Boolean> predicate) {

        for (T t : iterable) {

            if (predicate.call(t)) {
                return t;
            }
        }

        throw new NoSuchElementException();
    }

    public T firstOrDefault() {
        return firstOrDefault((T) null);
    }

    public T firstOrDefault(T defaultValue) {
        Iterator<T> iterator = iterable.iterator();

        if (!iterator.hasNext()) {
            return defaultValue;
        }

        return iterator.next();
    }

    public T firstOrDefault(Func<T, Boolean> predicate) {
        return firstOrDefault(predicate, null);
    }

    public T firstOrDefault(Func<T, Boolean> predicate, T defaultValue) {

        for (T t : iterable) {

            if (predicate.call(t)) {
                return t;
            }
        }

        return defaultValue;
    }

    public <K> Enumerable<Grouping<K, T>> groupBy(Func<T, K> keySelector) {
        return groupBy(keySelector, getPassThroughFunc());
    }

    public <K, R> Enumerable<R> groupBy(Func<T, K> keySelector,
            Func2<K, Enumerable<T>, R> resultSelector) {

        return groupBy(keySelector, getPassThroughFunc(), resultSelector);
    }

    public <K, E> Enumerable<Grouping<K, E>> groupBy(Func<T, K> keySelector,
            Func<T, E> elementSelector) {

        LinkedHashMap<K, List<E>> map = doGroupBy(keySelector, elementSelector);
        List<Grouping<K, E>> groupList = new ArrayList<Grouping<K, E>>();

        for (Entry<K, List<E>> entry : map.entrySet()) {
            groupList.add(new BasicGrouping<K, E>(entry.getKey(),
                    new BasicEnumerable<E>(entry.getValue())));
        }

        return new BasicEnumerable<Grouping<K, E>>(groupList);
    }

    public <K, E, R> Enumerable<R> groupBy(Func<T, K> keySelector,
            Func<T, E> elementSelector,
            Func2<K, Enumerable<E>, R> resultSelector) {

        LinkedHashMap<K, List<E>> map = doGroupBy(keySelector, elementSelector);
        List<R> resultList = new ArrayList<R>();

        for (Entry<K, List<E>> entry : map.entrySet()) {
            resultList.add(resultSelector.call(entry.getKey(),
                    new BasicEnumerable<E>(entry.getValue())));
        }

        return new BasicEnumerable<R>(resultList);
    }

    private <E, K> LinkedHashMap<K, List<E>> doGroupBy(Func<T, K> keySelector,
            Func<T, E> elementSelector) {

        LinkedHashMap<K, List<E>> map = new LinkedHashMap<K, List<E>>();

        for (T entity : iterable) {
            K key = keySelector.call(entity);
            List<E> list = map.get(key);

            if (list == null) {
                list = new ArrayList<E>();
                map.put(key, list);
            }

            list.add(elementSelector.call(entity));
        }

        return map;
    }

    public <I, K, R> Enumerable<R> groupJoin(final Enumerable<I> inner,
            final Func<T, K> outerKeySelector,
            final Func<I, K> innerKeySelector,
            final Func2<T, Enumerable<I>, R> resultSelector) {

        return new BasicEnumerable<R>(new Iterable<R>() {

            public Iterator<R> iterator() {

                return new Iterator<R>() {

                    private final Iterator<T> iterator = iterable.iterator();

                    private final Map<K, List<I>> innerListMap;

                    {
                        innerListMap = new HashMap<K, List<I>>();

                        for (I innerEntity : inner) {
                            K innerKey = innerKeySelector.call(innerEntity);
                            List<I> innerList = innerListMap.get(innerKey);

                            if (innerList == null) {
                                innerList = new ArrayList<I>();
                                innerListMap.put(innerKey, innerList);
                            }

                            innerList.add(innerEntity);
                        }
                    }

                    public boolean hasNext() {
                        return iterator.hasNext();
                    }

                    public R next() {
                        T outerEntity = iterator.next();
                        K outerKey = outerKeySelector.call(outerEntity);
                        List<I> innerList = innerListMap.get(outerKey);
                        return resultSelector.call(outerEntity,
                                new BasicEnumerable<I>(innerList));
                    }

                    public void remove() {
                        throw new UnsupportedOperationException();
                    }
                };
            }
        });
    }

    public Enumerable<T> intersect(final Enumerable<T> second) {
        return new BasicEnumerable<T>(new Iterable<T>() {

            public Iterator<T> iterator() {
                return new QueuedIterator<T, T, T>(iterable.iterator()) {

                    private final Set<T> firstSet = new HashSet<T>();

                    private final Set<T> secondSet = new HashSet<T>(second
                            .toList());

                    @Override
                    protected void addElement(Queue<T> queue, T t) {

                        if (secondSet.contains(t) && !firstSet.contains(t)) {
                            queue.add(t);
                            firstSet.add(t);
                        }
                    }

                    @Override
                    protected T toResult(T e) {
                        return e;
                    }
                };
            }
        });
    }

    public <I, K, R> Enumerable<R> join(final Enumerable<I> inner,
            final Func<T, K> outerKeySelector,
            final Func<I, K> innerKeySelector,
            final Func2<T, I, R> resultSelector) {

        return new BasicEnumerable<R>(new Iterable<R>() {

            public Iterator<R> iterator() {
                return new QueuedIterator<T, Pair<T, I>, R>(iterable.iterator()) {

                    private final Map<K, I> innerMap;

                    {
                        innerMap = new HashMap<K, I>();

                        for (I innerEntity : inner) {
                            K innerKey = innerKeySelector.call(innerEntity);
                            innerMap.put(innerKey, innerEntity);
                        }
                    }

                    @Override
                    protected void addElement(Queue<Pair<T, I>> queue, T t) {
                        K outerKey = outerKeySelector.call(t);
                        I innerEntity = innerMap.get(outerKey);

                        if (innerEntity != null) {
                            queue.add(new Pair<T, I>(t, innerEntity));
                        }
                    }

                    @Override
                    protected R toResult(Pair<T, I> e) {
                        return resultSelector.call(e.a, e.b);
                    }
                };
            }
        });
    }

    public T last() {
        Iterator<T> iterator = iterable.iterator();

        if (!iterator.hasNext()) {
            throw new NoSuchElementException();
        }

        T last = null;

        while (iterator.hasNext()) {
            last = iterator.next();
        }

        return last;
    }

    public T last(Func<T, Boolean> predicate) {
        T last = null;
        boolean found = false;

        for (T t : iterable) {

            if (predicate.call(t)) {
                last = t;

                if (!found) {
                    found = true;
                }
            }
        }

        if (!found) {
            throw new NoSuchElementException();
        }

        return last;
    }

    public T lastOrDefault() {
        return lastOrDefault((T) null);
    }

    public T lastOrDefault(T defaultValue) {
        Iterator<T> iterator = iterable.iterator();

        if (!iterator.hasNext()) {
            return defaultValue;
        }

        T last = null;

        while (iterator.hasNext()) {
            last = iterator.next();
        }

        return last;
    }

    public T lastOrDefault(Func<T, Boolean> predicate) {
        return lastOrDefault(predicate, null);
    }

    public T lastOrDefault(Func<T, Boolean> predicate, T defaultValue) {
        T last = null;
        boolean found = false;

        for (T t : iterable) {

            if (predicate.call(t)) {
                last = t;

                if (!found) {
                    found = true;
                }
            }
        }

        if (!found) {
            return defaultValue;
        }

        return last;
    }

    public long longCount() {
        long c = 0;

        for (Iterator<T> it = iterable.iterator(); it.hasNext(); it.next()) {
            c++;
        }

        return c;
    }

    public long longCount(Func<T, Boolean> predicate) {
        long c = 0;

        for (T t : iterable) {

            if (predicate.call(t)) {
                c++;
            }
        }

        return c;
    }

    @SuppressWarnings("unchecked")
    public T max() {
        Iterator<T> iterator = iterable.iterator();
        T max = iterator.next();

        while (iterator.hasNext()) {
            T t = iterator.next();

            if (((Comparable<T>) t).compareTo(max) > 0) {
                max = t;
            }
        }

        return max;
    }

    public <R extends Comparable<R>> R max(Func<T, R> selector) {
        Iterator<T> iterator = iterable.iterator();
        R max = selector.call(iterator.next());

        while (iterator.hasNext()) {
            R r = selector.call(iterator.next());

            if (r.compareTo(max) > 0) {
                max = r;
            }
        }

        return max;
    }

    @SuppressWarnings("unchecked")
    public T min() {
        Iterator<T> iterator = iterable.iterator();
        T min = iterator.next();

        while (iterator.hasNext()) {
            T t = iterator.next();

            if (((Comparable<T>) t).compareTo(min) < 0) {
                min = t;
            }
        }

        return min;
    }

    public <R extends Comparable<R>> R min(Func<T, R> selector) {
        Iterator<T> iterator = iterable.iterator();
        R min = selector.call(iterator.next());

        while (iterator.hasNext()) {
            R r = selector.call(iterator.next());

            if (r.compareTo(min) < 0) {
                min = r;
            }
        }

        return min;
    }

    public <R> Enumerable<R> ofType(final Class<R> resultClass) {
        return new BasicEnumerable<R>(new Iterable<R>() {

            public Iterator<R> iterator() {
                return new QueuedIterator<T, T, R>(iterable.iterator()) {

                    @Override
                    protected void addElement(Queue<T> queue, T t) {

                        if (resultClass.isInstance(t)) {
                            queue.add(t);
                        }
                    }

                    @Override
                    protected R toResult(T e) {
                        return resultClass.cast(e);
                    }
                };
            }
        });
    }

    public <K> OrderedEnumerable<T> orderBy(Func<T, K> keySelector) {
        return new BasicOrderedEnumerable<T>(this, keySelector, false);
    }

    public <K> OrderedEnumerable<T> orderBy(Func<T, K> keySelector,
            Comparator<K> comparator) {

        return new BasicOrderedEnumerable<T>(this, keySelector, false,
                comparator);
    }

    public <K> OrderedEnumerable<T> orderByDescending(Func<T, K> keySelector) {
        return new BasicOrderedEnumerable<T>(this, keySelector, true);
    }

    public <K> OrderedEnumerable<T> orderByDescending(Func<T, K> keySelector,
            Comparator<K> comparator) {

        return new BasicOrderedEnumerable<T>(this, keySelector, true,
                comparator);
    }

    public static Enumerable<Integer> range(final int start, final int count) {
        return new BasicEnumerable<Integer>(new Iterable<Integer>() {

            public Iterator<Integer> iterator() {

                return new Iterator<Integer>() {

                    private int index;

                    private int value = start;

                    public boolean hasNext() {
                        return index < count;
                    }

                    public Integer next() {

                        if (index >= count) {
                            throw new NoSuchElementException();
                        }

                        index++;
                        return value++;
                    }

                    public void remove() {
                        throw new UnsupportedOperationException();
                    }
                };
            }
        });
    }

    public static <R> Enumerable<R> repeat(final R element, final int count) {
        return new BasicEnumerable<R>(new Iterable<R>() {

            public Iterator<R> iterator() {

                return new Iterator<R>() {

                    private int index;

                    public boolean hasNext() {
                        return index < count;
                    }

                    public R next() {

                        if (index >= count) {
                            throw new NoSuchElementException();
                        }

                        index++;
                        return element;
                    }

                    public void remove() {
                        throw new UnsupportedOperationException();
                    }
                };
            }
        });
    }

    public Enumerable<T> reverse() {
        List<T> list = toList();
        Collections.reverse(list);
        return new BasicEnumerable<T>(list);
    }

    public <R> Enumerable<R> select(final Func<T, R> selector) {
        return new BasicEnumerable<R>(new Iterable<R>() {

            public Iterator<R> iterator() {

                return new Iterator<R>() {

                    private final Iterator<T> iterator = iterable.iterator();

                    public boolean hasNext() {
                        return iterator.hasNext();
                    }

                    public R next() {
                        return selector.call(iterator.next());
                    }

                    public void remove() {
                        throw new UnsupportedOperationException();
                    }
                };
            }
        });
    }

    public <R> Enumerable<R> select(final Func2<T, Integer, R> selector) {
        return select(new Func<T, R>() {

            private int index;

            public R call(T arg1) {
                return selector.call(arg1, index++);
            }
        });
    }

    public <R> Enumerable<R> selectMany(Func<T, Enumerable<R>> selector) {
        return selectMany(selector, new Func2<T, R, R>() {

            public R call(T arg1, R arg2) {
                return arg2;
            }
        });
    }

    public <C, R> Enumerable<R> selectMany(
            final Func<T, Enumerable<C>> collectionSelector,
            final Func2<T, C, R> resultSelector) {

        return new BasicEnumerable<R>(new Iterable<R>() {

            public Iterator<R> iterator() {
                return new QueuedIterator<T, Pair<T, C>, R>(iterable.iterator()) {

                    @Override
                    protected void addElement(Queue<Pair<T, C>> queue, T t) {

                        for (C c : collectionSelector.call(t)) {
                            queue.add(new Pair<T, C>(t, c));
                        }
                    }

                    @Override
                    protected R toResult(Pair<T, C> e) {
                        return resultSelector.call(e.a, e.b);
                    }
                };
            }
        });
    }

    public <R> Enumerable<R> selectMany(
            Func2<T, Integer, Enumerable<R>> selector) {

        return selectMany(selector, new Func2<T, R, R>() {

            public R call(T arg1, R arg2) {
                return arg2;
            }
        });
    }

    public <C, R> Enumerable<R> selectMany(
            final Func2<T, Integer, Enumerable<C>> collectionSelector,
            Func2<T, C, R> resultSelector) {

        return selectMany(new Func<T, Enumerable<C>>() {

            private int index;

            public Enumerable<C> call(T arg1) {
                return collectionSelector.call(arg1, index++);
            }
        }, resultSelector);
    }

    public Boolean sequenceEqual(Enumerable<T> second) {
        Iterator<T> iterator = second.iterator();

        for (T f : iterable) {

            if (!iterator.hasNext()) {
                return false;
            }

            if (!equals(f, iterator.next())) {
                return false;
            }
        }

        if (iterator.hasNext()) {
            return false;
        }

        return true;
    }

    public T single() {
        Iterator<T> iterator = iterable.iterator();
        T t = iterator.next();

        if (iterator.hasNext()) {
            throw new NoSuchElementException();
        }

        return t;
    }

    public T single(Func<T, Boolean> predicate) {
        T single = null;
        boolean found = false;

        for (T t : iterable) {

            if (predicate.call(t)) {

                if (found) {
                    found = false;
                    break;
                }

                single = t;
                found = true;
            }
        }

        if (!found) {
            throw new NoSuchElementException();
        }

        return single;
    }

    public T singleOrDefault() {
        return singleOrDefault((T) null);
    }

    public T singleOrDefault(T defaultValue) {
        Iterator<T> iterator = iterable.iterator();

        if (!iterator.hasNext()) {
            return defaultValue;
        }

        T t = iterator.next();

        if (iterator.hasNext()) {
            return defaultValue;
        }

        return t;
    }

    public T singleOrDefault(Func<T, Boolean> predicate) {
        return singleOrDefault(predicate, (T) null);
    }

    public T singleOrDefault(Func<T, Boolean> predicate, T defaultValue) {
        T single = null;
        boolean found = false;

        for (T t : iterable) {

            if (predicate.call(t)) {

                if (found) {
                    found = false;
                    break;
                }

                single = t;
                found = true;
            }
        }

        if (!found) {
            return defaultValue;
        }

        return single;
    }

    public Enumerable<T> skip(final int count) {
        return new BasicEnumerable<T>(new Iterable<T>() {

            public Iterator<T> iterator() {
                return new Iterator<T>() {

                    private final Iterator<T> iterator = iterable.iterator();

                    private boolean skipped;

                    private boolean skip() {

                        if (skipped) {
                            return true;
                        }

                        skipped = true;

                        for (int i = 0; i < count; i++) {

                            if (!iterator.hasNext()) {
                                return false;
                            }

                            iterator.next();
                        }

                        return true;
                    }

                    public boolean hasNext() {

                        if (!skip()) {
                            return false;
                        }

                        return iterator.hasNext();
                    }

                    public T next() {

                        if (!skip()) {
                            throw new NoSuchElementException();
                        }

                        return iterator.next();
                    }

                    public void remove() {
                        throw new UnsupportedOperationException();
                    }
                };
            }
        });
    }

    public Enumerable<T> skipWhile(final Func<T, Boolean> predicate) {
        return new BasicEnumerable<T>(new Iterable<T>() {

            public Iterator<T> iterator() {
                return new QueuedIterator<T, T, T>(iterable.iterator()) {

                    private boolean skipped;

                    @Override
                    protected void addElement(Queue<T> queue, T t) {

                        if (!skipped) {

                            if (predicate.call(t)) {
                                return;
                            }

                            skipped = true;
                        }

                        queue.add(t);
                    }

                    @Override
                    protected T toResult(T e) {
                        return e;
                    }
                };
            }
        });
    }

    public Enumerable<T> skipWhile(final Func2<T, Integer, Boolean> predicate) {
        return skipWhile(new Func<T, Boolean>() {

            private int index;

            public Boolean call(T arg1) {
                return predicate.call(arg1, index++);
            }
        });
    }

    public int sum() {
        int sum = 0;

        for (T t : iterable) {
            sum += (Integer) t;
        }

        return sum;
    }

    public long sumLong() {
        long sum = 0;

        for (T t : iterable) {
            sum += (Long) t;
        }

        return sum;
    }

    public float sumFloat() {
        float sum = 0;

        for (T t : iterable) {
            sum += (Float) t;
        }

        return sum;
    }

    public double sumDouble() {
        double sum = 0;

        for (T t : iterable) {
            sum += (Double) t;
        }

        return sum;
    }

    public int sum(Func<T, Integer> selector) {
        int sum = 0;

        for (T t : iterable) {
            sum += selector.call(t);
        }

        return sum;
    }

    public long sumLong(Func<T, Long> selector) {
        long sum = 0;

        for (T t : iterable) {
            sum += selector.call(t);
        }

        return sum;
    }

    public float sumFloat(Func<T, Float> selector) {
        float sum = 0;

        for (T t : iterable) {
            sum += selector.call(t);
        }

        return sum;
    }

    public double sumDouble(Func<T, Double> selector) {
        double sum = 0;

        for (T t : iterable) {
            sum += selector.call(t);
        }

        return sum;
    }

    public Enumerable<T> take(final int count) {
        return new BasicEnumerable<T>(new Iterable<T>() {

            public Iterator<T> iterator() {
                return new Iterator<T>() {

                    private final Iterator<T> iterator = iterable.iterator();

                    private int index;

                    public boolean hasNext() {

                        if (index >= count) {
                            return false;
                        }

                        return iterator.hasNext();
                    }

                    public T next() {

                        if (index >= count) {
                            throw new NoSuchElementException();
                        }

                        index++;
                        return iterator.next();
                    }

                    public void remove() {
                        throw new UnsupportedOperationException();
                    }
                };
            }
        });
    }

    public Enumerable<T> takeWhile(final Func<T, Boolean> predicate) {
        return new BasicEnumerable<T>(new Iterable<T>() {

            public Iterator<T> iterator() {
                return new QueuedIterator<T, T, T>(iterable.iterator()) {

                    private boolean done;

                    @Override
                    protected void addElement(Queue<T> queue, T t) {

                        if (done) {
                            return;
                        }

                        if (!predicate.call(t)) {
                            done = true;
                            return;
                        }

                        queue.add(t);
                    }

                    @Override
                    protected T toResult(T e) {
                        return e;
                    }
                };
            }
        });
    }

    public Enumerable<T> takeWhile(final Func2<T, Integer, Boolean> predicate) {
        return takeWhile(new Func<T, Boolean>() {

            private int index;

            public Boolean call(T arg1) {
                return predicate.call(arg1, index++);
            }
        });
    }

    public Object[] toArray() {
        return toList().toArray();
    }

    @SuppressWarnings("unchecked")
    public <R> R[] toArray(Class<R> resultClass) {
        List<T> list = toList();
        R[] array = (R[]) Array.newInstance(resultClass, list.size());
        return list.toArray(array);
    }

    public List<T> toList() {
        ArrayList<T> list = new ArrayList<T>();

        for (T e : iterable) {
            list.add(e);
        }

        return list;
    }

    public EnumerableList<T> toEnumerableList() {
        return new BasicEnumerableList<T>(toList());
    }

    public <K> Map<K, List<T>> toListMap(Func<T, K> keySelector) {
        return toListMap(keySelector, getPassThroughFunc());
    }

    public <K, E> Map<K, List<E>> toListMap(Func<T, K> keySelector,
            Func<T, E> elementSelector) {

        Map<K, List<E>> map = new LinkedHashMap<K, List<E>>();

        for (T t : iterable) {
            K k = keySelector.call(t);
            List<E> list = map.get(k);

            if (list == null) {
                list = new ArrayList<E>();
                map.put(k, list);
            }

            list.add(elementSelector.call(t));
        }

        return map;
    }

    public <K> Lookup<K, T> toLookup(Func<T, K> keySelector) {
        return toLookup(keySelector, getPassThroughFunc());
    }

    public <K, E> Lookup<K, E> toLookup(Func<T, K> keySelector,
            Func<T, E> elementSelector) {

        return new BasicLookup<K, E>(toListMap(keySelector, elementSelector));
    }

    public <K> Map<K, T> toMap(Func<T, K> keySelector) {
        return toMap(keySelector, getPassThroughFunc());
    }

    public <K, E> Map<K, E> toMap(Func<T, K> keySelector,
            Func<T, E> elementSelector) {

        Map<K, E> map = new LinkedHashMap<K, E>();

        for (T t : iterable) {
            K k = keySelector.call(t);

            if (map.containsKey(k)) {
                throw new IllegalArgumentException();
            }

            map.put(k, elementSelector.call(t));
        }

        return map;
    }

    public <K> Dictionary<K, T> toDictionary(Func<T, K> keySelector) {
        return toDictionary(keySelector, getPassThroughFunc());
    }

    public <K, E> Dictionary<K, E> toDictionary(Func<T, K> keySelector,
            Func<T, E> elementSelector) {

        return new BasicDictionary<K, E>(toMap(keySelector, elementSelector));
    }

    public Enumerable<T> union(final Enumerable<T> second) {
        return new BasicEnumerable<T>(new Iterable<T>() {

            public Iterator<T> iterator() {
                return new QueuedIterator<T, T, T>(new ConcatenatedIterator<T>(
                        iterable.iterator(), second.iterator())) {

                    private final Set<T> set = new HashSet<T>();

                    @Override
                    protected void addElement(Queue<T> queue, T t) {

                        if (!set.contains(t)) {
                            queue.add(t);
                            set.add(t);
                        }
                    }

                    @Override
                    protected T toResult(T e) {
                        return e;
                    }
                };
            }
        });
    }

    public Enumerable<T> where(final Func<T, Boolean> predicate) {
        return new BasicEnumerable<T>(new Iterable<T>() {

            public Iterator<T> iterator() {
                return new QueuedIterator<T, T, T>(iterable.iterator()) {

                    @Override
                    protected void addElement(Queue<T> queue, T t) {

                        if (predicate.call(t)) {
                            queue.add(t);
                        }
                    }

                    @Override
                    protected T toResult(T e) {
                        return e;
                    }
                };
            }
        });
    }

    public Enumerable<T> where(final Func2<T, Integer, Boolean> predicate) {
        return where(new Func<T, Boolean>() {

            private int index;

            public Boolean call(T arg1) {
                return predicate.call(arg1, index++);
            }
        });
    }

    private Func<T, T> getPassThroughFunc() {
        return new Func<T, T>() {

            public T call(T arg1) {
                return arg1;
            }
        };
    }
}
