package sharin.unlinq;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

public class BasicOrderedEnumerable<T> implements OrderedEnumerable<T> {

    private final Enumerable<T> enumerable;

    private final List<List<T>> listList;

    public <K> BasicOrderedEnumerable(Enumerable<T> enumerable,
            Func<T, K> keySelector, boolean descending) {

        this(enumerable, keySelector, descending, null);
    }

    public <K> BasicOrderedEnumerable(Enumerable<T> enumerable,
            Func<T, K> keySelector, boolean descending, Comparator<K> comparator) {

        List<List<T>> listList = null;

        if (enumerable instanceof BasicOrderedEnumerable<?>) {
            listList = apply(toBasicOrderedEnumerable(enumerable).listList,
                    keySelector, descending, comparator);

        } else {
            listList = new ArrayList<List<T>>();
            listList.add(enumerable.toList());
            listList = apply(listList, keySelector, descending, comparator);
        }

        this.listList = listList;

        List<T> all = new ArrayList<T>();

        for (List<T> list : listList) {
            all.addAll(list);
        }

        this.enumerable = new BasicEnumerable<T>(all);
    }

    private BasicOrderedEnumerable<T> toBasicOrderedEnumerable(
            Enumerable<T> enumerable) {

        return (BasicOrderedEnumerable<T>) enumerable;
    }

    private <K> List<List<T>> apply(List<List<T>> prevListList,
            Func<T, K> keySelector, boolean descending, Comparator<K> comparator) {

        List<List<T>> currListList = new ArrayList<List<T>>();

        for (List<T> prevList : prevListList) {
            TreeMap<K, List<T>> map = createTreeMap(comparator);

            for (T entity : prevList) {
                K key = keySelector.call(entity);
                List<T> currList = map.get(key);

                if (currList == null) {
                    currList = new ArrayList<T>();
                    map.put(key, currList);
                }

                currList.add(entity);
            }

            Collection<List<T>> values = map.values();

            if (descending) {
                List<List<T>> list = new ArrayList<List<T>>(values);
                Collections.reverse(list);
                values = list;
            }

            currListList.addAll(values);
        }

        return currListList;
    }

    private <K> TreeMap<K, List<T>> createTreeMap(Comparator<K> comparator) {

        if (comparator == null) {
            return new TreeMap<K, List<T>>();
        }

        return new TreeMap<K, List<T>>(comparator);
    }

    public <K> OrderedEnumerable<T> thenBy(Func<T, K> keySelector) {
        return new BasicOrderedEnumerable<T>(this, keySelector, false);
    }

    public <K> OrderedEnumerable<T> thenBy(Func<T, K> keySelector,
            Comparator<K> comparator) {

        return new BasicOrderedEnumerable<T>(this, keySelector, false,
                comparator);
    }

    public <K> OrderedEnumerable<T> thenByDescending(Func<T, K> keySelector) {
        return new BasicOrderedEnumerable<T>(this, keySelector, true);
    }

    public <K> OrderedEnumerable<T> thenByDescending(Func<T, K> keySelector,
            Comparator<K> comparator) {

        return new BasicOrderedEnumerable<T>(this, keySelector, true,
                comparator);
    }

    public <A, R> R aggregate(A seed, Func2<A, T, A> func,
            Func<A, R> resultSelector) {
        return enumerable.aggregate(seed, func, resultSelector);
    }

    public <A> A aggregate(A seed, Func2<A, T, A> func) {
        return enumerable.aggregate(seed, func);
    }

    public T aggregate(Func2<T, T, T> func) {
        return enumerable.aggregate(func);
    }

    public Boolean all(Func<T, Boolean> predicate) {
        return enumerable.all(predicate);
    }

    public Boolean any() {
        return enumerable.any();
    }

    public Boolean any(Func<T, Boolean> predicate) {
        return enumerable.any(predicate);
    }

    public Enumerable<T> concat(Enumerable<T> second) {
        return enumerable.concat(second);
    }

    public boolean contains(Object value) {
        return enumerable.contains(value);
    }

    public int count() {
        return enumerable.count();
    }

    public int count(Func<T, Boolean> predicate) {
        return enumerable.count(predicate);
    }

    public Enumerable<T> defaultIfEmpty() {
        return enumerable.defaultIfEmpty();
    }

    public Enumerable<T> defaultIfEmpty(T defaultValue) {
        return enumerable.defaultIfEmpty(defaultValue);
    }

    public Enumerable<T> distinct() {
        return enumerable.distinct();
    }

    public T elementAt(int index) {
        return enumerable.elementAt(index);
    }

    public T elementAtOrDefault(int index) {
        return enumerable.elementAtOrDefault(index);
    }

    public Enumerable<T> except(Enumerable<T> second) {
        return enumerable.except(second);
    }

    public T first() {
        return enumerable.first();
    }

    public T first(Func<T, Boolean> predicate) {
        return enumerable.first(predicate);
    }

    public T firstOrDefault() {
        return enumerable.firstOrDefault();
    }

    public T firstOrDefault(Func<T, Boolean> predicate) {
        return enumerable.firstOrDefault(predicate);
    }

    public <K, E, R> Enumerable<R> groupBy(Func<T, K> keySelector,
            Func<T, E> elementSelector,
            Func2<K, Enumerable<E>, R> resultSelector) {

        return enumerable.groupBy(keySelector, elementSelector, resultSelector);
    }

    public <K, E> Enumerable<Grouping<K, E>> groupBy(Func<T, K> keySelector,
            Func<T, E> elementSelector) {

        return enumerable.groupBy(keySelector, elementSelector);
    }

    public <K, R> Enumerable<R> groupBy(Func<T, K> keySelector,
            Func2<K, Enumerable<T>, R> resultSelector) {
        return enumerable.groupBy(keySelector, resultSelector);
    }

    public <K> Enumerable<Grouping<K, T>> groupBy(Func<T, K> keySelector) {
        return enumerable.groupBy(keySelector);
    }

    public <I, K, R> Enumerable<R> groupJoin(Enumerable<I> inner,
            Func<T, K> outerKeySelector, Func<I, K> innerKeySelector,
            Func2<T, Enumerable<I>, R> resultSelector) {

        return enumerable.groupJoin(inner, outerKeySelector, innerKeySelector,
                resultSelector);
    }

    public Enumerable<T> intersect(Enumerable<T> second) {
        return enumerable.intersect(second);
    }

    public Iterator<T> iterator() {
        return enumerable.iterator();
    }

    public <I, K, R> Enumerable<R> join(Enumerable<I> inner,
            Func<T, K> outerKeySelector, Func<I, K> innerKeySelector,
            Func2<T, I, R> resultSelector) {
        return enumerable.join(inner, outerKeySelector, innerKeySelector,
                resultSelector);
    }

    public T last() {
        return enumerable.last();
    }

    public T last(Func<T, Boolean> predicate) {
        return enumerable.last(predicate);
    }

    public T lastOrDefault() {
        return enumerable.lastOrDefault();
    }

    public T lastOrDefault(Func<T, Boolean> predicate) {
        return enumerable.lastOrDefault(predicate);
    }

    public long longCount() {
        return enumerable.longCount();
    }

    public long longCount(Func<T, Boolean> predicate) {
        return enumerable.longCount(predicate);
    }

    public T max() {
        return enumerable.max();
    }

    public <R extends Comparable<R>> R max(Func<T, R> selector) {
        return enumerable.max(selector);
    }

    public T min() {
        return enumerable.min();
    }

    public <R extends Comparable<R>> R min(Func<T, R> selector) {
        return enumerable.min(selector);
    }

    public <R> Enumerable<R> ofType(Class<R> resultClass) {
        return enumerable.ofType(resultClass);
    }

    public <K> OrderedEnumerable<T> orderBy(Func<T, K> keySelector,
            Comparator<K> comparator) {
        return enumerable.orderBy(keySelector, comparator);
    }

    public <K> OrderedEnumerable<T> orderBy(Func<T, K> keySelector) {
        return enumerable.orderBy(keySelector);
    }

    public <K> OrderedEnumerable<T> orderByDescending(Func<T, K> keySelector,
            Comparator<K> comparator) {
        return enumerable.orderByDescending(keySelector, comparator);
    }

    public <K> OrderedEnumerable<T> orderByDescending(Func<T, K> keySelector) {
        return enumerable.orderByDescending(keySelector);
    }

    public Enumerable<T> reverse() {
        return enumerable.reverse();
    }

    public <R> Enumerable<R> select(Func<T, R> selector) {
        return enumerable.select(selector);
    }

    public <R> Enumerable<R> select(Func2<T, Integer, R> selector) {
        return enumerable.select(selector);
    }

    public <C, R> Enumerable<R> selectMany(
            Func<T, Enumerable<C>> collectionSelector,
            Func2<T, C, R> resultSelector) {
        return enumerable.selectMany(collectionSelector, resultSelector);
    }

    public <R> Enumerable<R> selectMany(Func<T, Enumerable<R>> selector) {
        return enumerable.selectMany(selector);
    }

    public <C, R> Enumerable<R> selectMany(
            Func2<T, Integer, Enumerable<C>> collectionSelector,
            Func2<T, C, R> resultSelector) {
        return enumerable.selectMany(collectionSelector, resultSelector);
    }

    public <R> Enumerable<R> selectMany(
            Func2<T, Integer, Enumerable<R>> selector) {
        return enumerable.selectMany(selector);
    }

    public Boolean sequenceEqual(Enumerable<T> second) {
        return enumerable.sequenceEqual(second);
    }

    public T single() {
        return enumerable.single();
    }

    public T single(Func<T, Boolean> predicate) {
        return enumerable.single(predicate);
    }

    public T singleOrDefault() {
        return enumerable.singleOrDefault();
    }

    public T singleOrDefault(Func<T, Boolean> predicate) {
        return enumerable.singleOrDefault(predicate);
    }

    public Enumerable<T> skip(int count) {
        return enumerable.skip(count);
    }

    public Enumerable<T> skipWhile(Func<T, Boolean> predicate) {
        return enumerable.skipWhile(predicate);
    }

    public Enumerable<T> skipWhile(Func2<T, Integer, Boolean> predicate) {
        return enumerable.skipWhile(predicate);
    }

    public int sum() {
        return enumerable.sum();
    }

    public int sum(Func<T, Integer> selector) {
        return enumerable.sum(selector);
    }

    public double sumDouble() {
        return enumerable.sumDouble();
    }

    public double sumDouble(Func<T, Double> selector) {
        return enumerable.sumDouble(selector);
    }

    public float sumFloat() {
        return enumerable.sumFloat();
    }

    public float sumFloat(Func<T, Float> selector) {
        return enumerable.sumFloat(selector);
    }

    public long sumLong() {
        return enumerable.sumLong();
    }

    public long sumLong(Func<T, Long> selector) {
        return enumerable.sumLong(selector);
    }

    public Enumerable<T> take(int count) {
        return enumerable.take(count);
    }

    public Enumerable<T> takeWhile(Func<T, Boolean> predicate) {
        return enumerable.takeWhile(predicate);
    }

    public Enumerable<T> takeWhile(Func2<T, Integer, Boolean> predicate) {
        return enumerable.takeWhile(predicate);
    }

    public Object[] toArray() {
        return enumerable.toArray();
    }

    public <R> R[] toArray(Class<R> resultClass) {
        return enumerable.toArray(resultClass);
    }

    public List<T> toList() {
        return enumerable.toList();
    }

    public EnumerableList<T> toEnumerableList() {
        return enumerable.toEnumerableList();
    }

    public <K, E> Lookup<K, E> toLookup(Func<T, K> keySelector,
            Func<T, E> elementSelector) {
        return enumerable.toLookup(keySelector, elementSelector);
    }

    public <K> Lookup<K, T> toLookup(Func<T, K> keySelector) {
        return enumerable.toLookup(keySelector);
    }

    public <K, E> Dictionary<K, E> toDictionary(Func<T, K> keySelector,
            Func<T, E> elementSelector) {
        return enumerable.toDictionary(keySelector, elementSelector);
    }

    public <K> Dictionary<K, T> toDictionary(Func<T, K> keySelector) {
        return enumerable.toDictionary(keySelector);
    }

    public Enumerable<T> union(Enumerable<T> second) {
        return enumerable.union(second);
    }

    public Enumerable<T> where(Func<T, Boolean> predicate) {
        return enumerable.where(predicate);
    }

    public Enumerable<T> where(Func2<T, Integer, Boolean> predicate) {
        return enumerable.where(predicate);
    }

    public double average() {
        return enumerable.average();
    }

    public double average(Func<T, Integer> selector) {
        return enumerable.average(selector);
    }

    public double averageDouble() {
        return enumerable.averageDouble();
    }

    public double averageDouble(Func<T, Double> selector) {
        return enumerable.averageDouble(selector);
    }

    public float averageFloat() {
        return enumerable.averageFloat();
    }

    public float averageFloat(Func<T, Float> selector) {
        return enumerable.averageFloat(selector);
    }

    public double averageLong() {
        return enumerable.averageLong();
    }

    public double averageLong(Func<T, Long> selector) {
        return enumerable.averageLong(selector);
    }

    public T firstOrDefault(Func<T, Boolean> predicate, T defaultValue) {
        return enumerable.firstOrDefault(predicate, defaultValue);
    }

    public T firstOrDefault(T defaultValue) {
        return enumerable.firstOrDefault(defaultValue);
    }

    public T lastOrDefault(Func<T, Boolean> predicate, T defaultValue) {
        return enumerable.lastOrDefault(predicate, defaultValue);
    }

    public T lastOrDefault(T defaultValue) {
        return enumerable.lastOrDefault(defaultValue);
    }

    public T singleOrDefault(Func<T, Boolean> predicate, T defaultValue) {
        return enumerable.singleOrDefault(predicate, defaultValue);
    }

    public T singleOrDefault(T defaultValue) {
        return enumerable.singleOrDefault(defaultValue);
    }

    public T elementAtOrDefault(int index, T defaultValue) {
        return enumerable.elementAtOrDefault(index, defaultValue);
    }

    public <R> Enumerable<R> cast(Class<R> resultClass) {
        return enumerable.cast(resultClass);
    }

    public <K, E> Map<K, E> toMap(Func<T, K> keySelector,
            Func<T, E> elementSelector) {
        return enumerable.toMap(keySelector, elementSelector);
    }

    public <K> Map<K, T> toMap(Func<T, K> keySelector) {
        return enumerable.toMap(keySelector);
    }

    public <K, E> Map<K, List<E>> toListMap(Func<T, K> keySelector,
            Func<T, E> elementSelector) {
        return enumerable.toListMap(keySelector, elementSelector);
    }

    public <K> Map<K, List<T>> toListMap(Func<T, K> keySelector) {
        return enumerable.toListMap(keySelector);
    }
}
