﻿#region Copyright (c) 2002-2003, James W. Newkirk, Michael C. Two, Alexei A. Vorontsov, Charlie Poole, Philip A. Craig
/************************************************************************************
	'
	' Copyright © 2002-2003 James W. Newkirk, Michael C. Two, Alexei A. Vorontsov, Charlie Poole
	' Copyright © 2000-2003 Philip A. Craig
	'
	' This software is provided 'as-is', without any express or implied warranty. In no
	' event will the authors be held liable for any damages arising from the use of this
	' software.
	'
	' Permission is granted to anyone to use this software for any purpose, including
	' commercial applications, and to alter it and redistribute it freely, subject to the
	' following restrictions:
	'
	' 1. The origin of this software must not be misrepresented; you must not claim that
	' you wrote the original software. If you use this software in a product, an
	' acknowledgment (see the following) in the product documentation is required.
	'
	' Portions Copyright © 2003 James W. Newkirk, Michael C. Two, Alexei A. Vorontsov, Charlie Poole
	' or Copyright © 2000-2003 Philip A. Craig
	'
	' 2. Altered source versions must be plainly marked as such, and must not be
	' misrepresented as being the original software.
	'
	' 3. This notice may not be removed or altered from any source distribution.
	'
	'***********************************************************************************/
#endregion

// This version of NUnit-console is modified to support running single test methods using the "testMethodName" command line argument.

namespace NUnit.ConsoleRunner
{
	using System;
	using System.Collections.Specialized;
	using System.IO;
	using System.Reflection;
	using System.Xml;
	using System.Resources;
	using System.Text;
	using System.Text.RegularExpressions;
	using System.Diagnostics;
	using System.Runtime.InteropServices;
	using NUnit.Core;
	using NUnit.Util;
	
	/// <summary>
	/// Summary description for ExtendedConsoleUi.
	/// </summary>
	public class ExtendedConsoleUi
	{
		[STAThread]
		public static int Main(string[] args)
		{
			ExtendedConsoleOptions options = new ExtendedConsoleOptions(args);
			
			if(!options.nologo)
				WriteCopyright();
			
			if(options.help)
			{
				options.Help();
				return 0;
			}
			
			if(options.NoArgs)
			{
				Console.Error.WriteLine("fatal error: no inputs specified");
				options.Help();
				return 0;
			}
			
			if(!options.Validate())
			{
				Console.Error.WriteLine("fatal error: invalid arguments");
				options.Help();
				return 2;
			}
			
			try
			{
				ExtendedConsoleUi consoleUi = new ExtendedConsoleUi();
				return consoleUi.Execute( options );
			}
			catch( FileNotFoundException ex )
			{
				Console.WriteLine( ex.Message );
				return 2;
			}
			catch( BadImageFormatException ex )
			{
				Console.WriteLine( ex.Message );
				return 2;
			}
			catch( Exception ex )
			{
				Console.WriteLine( "Unhandled Exception:\n{0}", ex.ToString() );
				return 2;
			}
			finally
			{
				if(options.wait)
				{
					Console.Out.WriteLine("\nHit <enter> key to continue");
					Console.ReadLine();
				}
			}
		}
		
		private static XmlTextReader GetTransformReader(ConsoleOptions parser)
		{
			return (XmlTextReader)typeof(ConsoleUi).InvokeMember("GetTransformReader",
			                                                     BindingFlags.Static | BindingFlags.NonPublic | BindingFlags.InvokeMethod,
			                                                     null, null, new object[] { parser });
		}
		
		private static void WriteCopyright()
		{
			typeof(ConsoleUi).InvokeMember("WriteCopyright",
			                               BindingFlags.Static | BindingFlags.NonPublic | BindingFlags.InvokeMethod,
			                               null, null, null);
		}
		
		private static Test MakeTestFromCommandLine(TestDomain testDomain, ConsoleOptions parser)
		{
			return (Test)typeof(ConsoleUi).InvokeMember("MakeTestFromCommandLine",
			                                            BindingFlags.Static | BindingFlags.NonPublic | BindingFlags.InvokeMethod,
			                                            null, null, new object[] { testDomain, parser });
		}
		
		public int Execute( ExtendedConsoleOptions options )
		{
			XmlTextReader transformReader = GetTransformReader(options);
			if(transformReader == null) return 3;
			
			TextWriter outWriter = options.isOut
				? new StreamWriter( options.output )
				: Console.Out;
			
			TextWriter errorWriter = options.isErr
				? new StreamWriter( options.err )
				: Console.Error;
			
			// TODO: Use other kinds of runners
			TestDomain testDomain = new TestDomain();
			TestRunner testRunner = testDomain;
			
			if ( options.noshadow  ) testDomain.ShadowCopyFiles = false;
			
			Test test = MakeTestFromCommandLine(testDomain, options);
			
			if(test == null)
			{
				Console.Error.WriteLine("Unable to locate fixture {0}", options.fixture);
				return 2;
			}
			
			EventListener collector = CreateEventCollector( options, outWriter, errorWriter );
			
			if (options.HasInclude)
			{
				Console.WriteLine( "Included categories: " + options.include );
				testRunner.Filter = new CategoryFilter( options.IncludedCategories );
			}
			else if ( options.HasExclude )
			{
				Console.WriteLine( "Excluded categories: " + options.exclude );
				testRunner.Filter = new CategoryFilter( options.ExcludedCategories, true );
			}
			
			TestResult result = null;
			using( new DirectorySwapper() )
			{
				try
				{
					if (options.testMethodName != null)
						result = testRunner.Run( collector, new string[] { options.testMethodName } )[0];
					else
						result = testRunner.Run( collector );
				}
				finally
				{
					if ( options.isOut )
						outWriter.Close();
					if ( options.isErr )
						errorWriter.Close();
				}
			}
			
			Console.WriteLine();
			
			string xmlOutput = CreateXmlOutput( result );
			
			if (options.xmlConsole)
			{
				Console.WriteLine(xmlOutput);
			}
			else
			{
				try
				{
					//CreateSummaryDocument(xmlOutput, transformReader );
					XmlResultTransform xform = new XmlResultTransform( transformReader );
					xform.Transform( new StringReader( xmlOutput ), Console.Out );
				}
				catch( Exception ex )
				{
					Console.WriteLine( "Error: {0}", ex.Message );
					return 3;
				}
			}
			
			// Write xml output here
			string xmlResultFile = options.IsXml ? options.xml : "TestResult.xml";
			
			using ( StreamWriter writer = new StreamWriter( xmlResultFile ) )
			{
				writer.Write(xmlOutput);
			}
			
			if ( testRunner != null )
				testRunner.Unload();
			
			
			return result.IsFailure ? 1 : 0;
		}
		
		static EventListener CreateEventCollector(ConsoleOptions options, TextWriter outWriter, TextWriter errorWriter)
		{
			Type eventCollector = typeof(ConsoleUi).GetNestedType("EventCollector", BindingFlags.NonPublic);
			return (EventListener)Activator.CreateInstance(eventCollector, options, outWriter, errorWriter);
		}
		
		private string CreateXmlOutput( TestResult result )
		{
			StringBuilder builder = new StringBuilder();
			XmlResultVisitor resultVisitor = new XmlResultVisitor(new StringWriter( builder ), result);
			result.Accept(resultVisitor);
			resultVisitor.Write();
			
			return builder.ToString();
		}
	}
}

