open ScmUtil
open ScmTypes

exception UnknownCmd of string
exception Quit
exception SyntaxError

type eval_method = Lazy_eval | Eager_eval

let current_input_port = ref (datum_of_in_channel stdin)
let current_output_port = ref (datum_of_out_channel stdout)

let print_datum d = print_endline (string_of_datum d)

module Environment =
struct
  type t = ScmTypes.environment


  let global_add env k v = 
    let vr = ref v in
      Hashtbl.add env.top k vr; vr

  let lookup (env : environment) k =
    let rec loop p =
      match p with
	  shd::stl, vhd::vtl ->
	    if shd = k then vhd
	    else loop (stl,vtl)
	| [],_ -> raise Not_found
	| _ -> failwith "lookup"
    in 
    let rec loop1 =
      function
	  (s,v)::tl ->
	    (try loop (s, v)
	     with Not_found -> loop1 tl)
	| _ -> 
	    Hashtbl.find env.top k
    in loop1 env.stack


  let define env k v = Hashtbl.replace env.top k v
  let set env k v = 
    let vr = lookup env k in vr := v
      
  let get env k = 
    let vr = lookup env k in !vr
      
  (* let rec bind f =
    function
      Nil,Nil -> ()
      | Pair p1,Pair p2 ->
	  f (string_of_symbol p1.car) p2.car;
	  if is_symbol p1.cdr then
	    f (string_of_symbol p1.cdr) p2.cdr
	  else
	    bind f (p1.cdr, p2.cdr)
      | Symbol x, r -> f x r
      | a,b -> invalid_arg 
	  ("cannot bind " ^ (string_of_datum b) ^ " on " ^ (string_of_datum a)) *)

  let rec bind signature params prefs =
    let rec loop =
      function
	| Pair sp,Pair pp,hd::tl ->
	    hd := pp.car;
	    loop (sp.cdr,pp.cdr,tl)
	| Symbol _, v, hd::[] ->
	    hd := v
	| Nil,Nil,[] -> ()
	| _ -> failwith ("bind, sig:" ^ string_of_datum signature)
    in loop (signature,params,prefs)

  let add env k v = 
    let r = ref v in
      {env with stack = ([k],[r])::env.stack }, r
    
  let add_ref_binding env ks vrs = 
    {env with stack = (ks,vrs)::env.stack }

  let add_binding env ks vs = 
    let vrs = List.map (fun v -> ref v) vs in
      add_ref_binding env ks vrs
end


let scm_quit _ = raise Quit


let scm_list arg = copy arg
let concat l1 l2 = 
  let rec search_tail =
    function
      | Nil -> scm_list l2
      | Pair p -> cons (p.car) (search_tail p.cdr)
      | datum  -> invalid_arg ("concat: " ^ string_of_datum datum)
  in search_tail l1


let simple_cmd_1 f args = f (car args)
let simple_cmd_2 f args = f (car args) (cadr args)
let simple_cmd_3 f args = f (car args) (cadr args) (caddr args)

let predicate f args = Boolean (f args)

let predicate f args = 
  let var = car args in Boolean (f var)

let predicate2 f args =
  Boolean (f (car args) (cadr args))

let scm_boolean_p = predicate is_boolean
let scm_number_p = predicate is_number
let scm_symbol_p = predicate is_symbol
let scm_procedure_p = predicate is_procedure
let scm_pair_p = predicate is_pair
let scm_vector_p = predicate is_vector
let scm_string_p = predicate is_string    
let scm_port_p = predicate is_port
let scm_char_p = predicate is_char

let scm_input_port_p = predicate is_input_port
let scm_output_port_p = predicate is_output_port

let num_test test = 
  function
    | Number n -> test n
    | _ -> false

let num_cmd f =
  function
    | Number n -> Number (f n)
    | _ -> invalid_arg "num_cmd"

let scm_complex = predicate (num_test ScmDynnum.is_complex)
let scm_real = predicate (num_test ScmDynnum.is_real)
let scm_rational = predicate (num_test ScmDynnum.is_rational)
let scm_integer = predicate (num_test ScmDynnum.is_integer)
let scm_exact = predicate (num_test ScmDynnum.is_exact)
let scm_inexact = predicate (num_test ScmDynnum.is_inexact)

let scm_number_of_string args = 
  let radix =
    if cdr args = Nil then ScmDynnum.default_radix
    else number_of_datum (cadr args) in
    Number (ScmDynnum.of_string ~radix (string_of_scm_string (car args)))

let scm_string_of_number args = 
  let radix =
    if cdr args = Nil then ScmDynnum.default_radix
    else number_of_datum (cadr args) in
    String (ScmDynnum.string_of ~radix (number_of_datum (car args)))

let scm_exact_of_inexact = simple_cmd_1 (num_cmd ScmDynnum.exact)
let scm_inexact_of_exact = simple_cmd_1 (num_cmd ScmDynnum.inexact)

let scm_list_p = predicate is_list

let lambda env signature body =
  let siglist = list_of_improper_list signature in
  let names = List.map string_of_symbol siglist in
  let refs = List.map 
	       (fun _ -> ref Unspecified) siglist
	       
  in let env1 = Environment.add_ref_binding env names refs in
    Procedure (signature,body,env1,refs)

let scm_lambda env cont args =   
  let signature = car args
  and body = cdr args in
    cont (lambda env signature body)


let scm_eqv_p = predicate2 eqv
let scm_eq_p = predicate2 eq
let scm_equal_p = predicate2 equal

let scm_memq = simple_cmd_2 memq
let scm_memv = simple_cmd_2 memv
let scm_member = simple_cmd_2 member

let scm_assoc = simple_cmd_2 assoc
let scm_assq = simple_cmd_2 assq
let scm_assv = simple_cmd_2 assv

let scm_null_p = predicate (fun a -> a = Nil)
let scm_quote env cont args = cont (car args)

let quote args = cons (gen_symbol "quote") args



let rec eval (env : environment) (cont : datum -> datum) = 
  function
    | Symbol s -> 
	cont (try Environment.get env s 
	      with Not_found -> failwith ("Unknown: " ^ s))
    | SymbolInEnv (s,senv) ->
	cont (try Environment.get senv s
	      with Not_found -> failwith ("Unknown: " ^ s))
    | Pair c as l ->
	let rec eval_list f ev_done =
	  function
	    | Nil -> 
		f (reverse ev_done)
	    | Pair c ->
		eval env (fun v ->
			    eval_list f (cons v ev_done) c.cdr) c.car
	    | datum -> failwith ("improper list ends with " ^ string_of_datum datum) in
	let evaluate f = 
	  (match f with
	     | ML_EagerProcedure f -> 
		 eval_list (fun v -> cont (f v)) Nil c.cdr
	     | Procedure (signature,body,env,prefs) ->
		 let apply_proc args =
		   Environment.bind signature args prefs;
		   scm_begin env cont body 
		 in eval_list apply_proc Nil c.cdr
	     | ML_LazyProcedure f -> 
		 f env cont c.cdr 
	     | datum -> failwith ("expected procedure, got " ^ string_of_datum datum))
	in 
	  (* print_datum l; *)
	  eval env (fun v -> evaluate v) (c.car)
    | value -> cont value
and scm_begin env cont  =
  function
    | Pair c ->
	if c.cdr = Nil then eval env cont c.car
	else eval env (fun _ -> scm_begin env cont c.cdr) c.car
    | Nil -> cont Nil
    | datum ->  failwith ("improper list ends with " ^ string_of_datum datum)
	(* reduce (fun a b -> eval env b) Unspecified args *)
      
let scm_if env cont args =
  let test = car args in
    eval env (fun v ->
		let select =
		  if is_true v then cdr args
		  else cddr args in
		  if select = Nil then cont Unspecified
		  else eval env cont (car select)) test


let scm_cond env cont args =
  let rec loop = 
    function
      | Pair cond_clausel ->
	  let c = cond_clausel.car in
	  let test = car c 
	  and rest = cdr c in
	  let test_current test_result =
	    if is_true test_result then 
	      if rest = Nil then cont Nil
	      else
		match car rest with
		  (* | Nil -> cont Nil *)
		  | Symbol "=>" ->
		      eval env cont (cons (caddr c) test_result)
		  | _ -> eval env cont (cadr c)
	    else
	      loop cond_clausel.cdr 
	  in
	    if test = Symbol "else" then test_current scm_true
	    else eval env test_current test
      | Nil -> cont scm_false 
      | _ -> invalid_arg "cond"
  in loop args

let scm_case env cont args =
  let rec test cases key =
    match cases with
      | Nil -> cont Unspecified
      | Pair p -> 
	  let clause = p.car in
	  let ckey_list = car clause in
	    if ckey_list = Symbol "else" || 
	      is_true (exists (fun k -> Boolean (k = key)) ckey_list) then scm_begin env cont (cdr clause)
	    else test p.cdr key
      | _ -> invalid_arg "case" in
    eval env (test (cdr args)) (car args) 


let clamp a = cons a Nil

let rec scm_define env cont args = 
  let key = car args 
  and value = cdr args in
    if is_pair key then
      let signature = cdr key
      and scm_fun_name =  car key in
      let fun_name = string_of_symbol scm_fun_name in
	let lambda = scm_lambda env (fun x -> x) (cons signature value) in
	let _ = Environment.global_add env fun_name lambda in
	  print_endline fun_name;
	  cont scm_fun_name;
    else
      let var_name = string_of_symbol key in
      let r = Environment.global_add env var_name Unspecified in
	eval env (fun v ->
		    r := v;
		    (* if Environment.toplevel env then Environment.global_add env k_str v
		       else Environment.local_add env k_str v; *)
		    cont key) (car value)

let scm_and_p env cont args = 
  let rec test args v =
    if is_true v then 
      if args = Nil then cont v
      else eval env (test (cdr args)) (car args)
    else cont scm_false 
  in
    eval env (test args) scm_true
	
let scm_or_p env cont args =
  let rec test args v =
    if is_true v then cont v
    else if args = Nil then cont scm_false
    else eval env (test (cdr args)) (car args)
  in
    eval env (test args) scm_false
	
let scm_not_p = 
  let not_p = function
      Boolean false -> true
    | _ -> false
  in predicate not_p

let scm_set_var env cont args = 
  let var = string_of_symbol (car args) in
    eval env (fun value -> 
		Environment.set env var value;
		cont Unspecified) (cadr args) 

let let_fun f env cont args =
  let vars = car args 
  and body = cdr args in
    f env cont vars body



let let_with_late_binding env cont vars body =
  let let_env ks vs =   Environment.add_binding env ks vs 
  in
    if is_pair vars then
      let rec eval_var (ks,(vs : ScmTypes.datum list)) vars =
	if vars = Nil then 
	  let env1 = let_env ks vs in
	    scm_begin env1 cont body
	else
	  let n = car vars in (* TODO: eval (cdr n) ?? *)
	  let k = string_of_symbol (car n) in
	    eval env 
	      (fun v ->
		 eval_var ((k::ks),(v::vs))
		 (cdr vars)) (cadr n)
      in eval_var ([],[]) vars
    else (* R5RS 4.2.4 *)
      let vname = string_of_symbol vars 
      and bindings = car body 
      and body = cdr body in
      let signature = map car bindings
      and values = map cadr bindings in
      let env1, vname_ref = Environment.add env vname Unspecified in
      let lambda_val = lambda env1 signature body in
	  vname_ref := lambda_val;
	  let pseudo_proc = cons lambda_val values in
	    eval env1 cont pseudo_proc 



let let_with_sequencially_binding env cont vars body =
  let rec bind env =
    function 
      | Nil -> scm_begin env cont body
      | Pair p ->  
	  let varname = string_of_symbol (car p.car)
	  and value = cadr p.car in
	    eval env (fun v ->
			let env1,_ = Environment.add env varname v in
			  bind env1 p.cdr) value
      | _ -> failwith "bind_struct"
  in bind env vars



let let_with_recursive_binding env cont vars body =
  let keys, values = 
    reduce
      (fun (k,v) b ->
	 let var = string_of_symbol (car b)
	 and value = cadr b in
	   (var::k,value::v)) ([],[]) vars in
  let refs = List.map (fun _ -> ref Unspecified) keys in
  let env1 = Environment.add_ref_binding env keys refs in
  let rec bind = function
      |  r::rtl,v::vtl ->
	  eval env1 (fun value ->
		       r := value;
		       bind (rtl,vtl)) v
      | _ ->  scm_begin env1 cont body 
    in bind (refs,values)




let scm_do env cont args =
  let how_to_perfom = car args
  and test_line = cadr args
  and body = cddr args in
  let rec init keys values steps = 
    function 
      | Nil -> 
	  let refs = List.map (fun v -> ref v) values in
	  let env1 = Environment.add_ref_binding env keys refs in
	  let test = car test_line in
	  let rec iterate v =
	    if is_true v then scm_begin env1 cont (cdr test_line)
	    else
	      scm_begin env1
		(fun _ ->
		   let rec incr_vars = 
		     function
		       | r::rtl,step::steptl ->
			   if step  = Nil then incr_vars (rtl,steptl)
			   else eval env1 
			     (fun v -> 
				r := v;
				incr_vars (rtl,steptl))
			     step
		       | _ -> eval env1 iterate test 
		   in incr_vars (refs,steps)) body
	  in eval env1 iterate test
      | Pair p ->
	  let init_line = p.car in
	  let key = string_of_symbol (car init_line)
	  and init_value = cadr init_line
	  and step_block = cddr init_line in
	  let step = 
	    if step_block = Nil then Nil 
	  else car step_block
	  in eval env (fun v -> 
			 init (key::keys) (v::values) (step::steps) p.cdr)  init_value
    | _ -> raise SyntaxError
  in init [] [] [] how_to_perfom
       





let scm_delay env cont args =
  (* let local_storage = Environment.copy_local_storage env  *)
  let result = ref None in
    cont (Delayed (fun () ->  
		     match !result with
		       | None ->
			   eval env 
			     (fun v -> result := Some v; v) (car args)
		       | Some v -> v))

let scm_force args =
  match (car args) with
    | Delayed f -> f ()
	(* let rec sick_scm_loop v =
	   (try Lazy.force v
	   with  Lazy.Undefined -> failwith "recusive forcing isn't implemented")
	   in sick_scm_loop v *)
    | _ -> invalid_arg "force"



let scm_quasiquote env cont args =
  let rec check_for_unquote level rest item =
    if is_pair item then
      let carsym = car item in
	match carsym with
	  | Symbol "unquote" ->
	      let value =
		if level = 0 then 
		  eval env id (cadr item) 
		else 
		  let pre_build = (build_list (level-1) (cadr item)) in
		    cons carsym (check_for_unquote (level-1) Nil pre_build) 
	      in
		  cons value rest
	  | Symbol "unquote-splicing" ->
	      if level = 0 then 
		eval env (fun v -> concat v rest) (cadr item) 
	      else 
		let this = cons carsym (build_list level (cadr item)) in
		  cons this rest
	  | e -> 
	      let value = build_list level item in
		cons value rest
    else
      cons item rest
  and build_list level  =
    function
      | Nil -> Nil
      | Pair p ->
	  let rest = 
	    if level = 0 && is_pair p.cdr && (car p.cdr) = Symbol "unquote" then
	      (* after dot. eg `(a . ,(car '(1))) *)
	      eval env id (cadr p.cdr)
	    else if p.car = Symbol "quasiquote" then
	      build_list (succ level) p.cdr
(*	    else if level > 0  && p.car = Symbol "unquote" then
	      (print_endline "hit";
	       build_list (pred level) p.cdr) *)
	    else build_list level p.cdr in
	    check_for_unquote level rest p.car
      | Vector _ as vec ->
	  (* this is an exotic case. Therefore we keep it as simple as possible. *)
	  vector_of_list (build_list 0 (list_of_vector vec))
      | atom -> atom
  in cont (build_list 0 (car args))
  


let apply env cont f args =
  let rec build_args =
    function
      | Nil -> Nil
      | Pair p ->
	  if p.cdr = Nil then p.car
	  else cons p.car (build_args p.cdr)
      | _ -> invalid_arg "apply"
  in let bargs = build_args args in
  let code = cons f bargs in
    (* print_endline (string_of_datum code); *)
    eval env cont code


let scm_apply env cont args = 
  let rec build_args accum =
    function 
      | Nil -> eval env cont (cons (car args) (reverse accum))
      | Pair p -> 
	  eval env 
	    (fun v -> build_args (concat v accum) p.cdr) p.car
      | datum -> invalid_arg "scm_apply"
  in
    build_args Nil (cdr args)
	      
(*   apply env cont (car args) (map (eval env) (cdr args)) ) *)
	       
let scm_map env cont args =
  let rec iter f accum =
    function
      | Nil -> cont (reverse accum)
      | Pair p as l ->
	  eval env (fun v -> iter f (cons v accum) p.cdr) (cons f (clamp (quote (clamp p.car))))
      | _ -> raise SyntaxError
  in eval env 
       (fun f -> 
	  eval env (fun v -> iter f Nil v) (cadr args))
       (car args)


let scm_for_each env cont args =
  scm_map env (fun v -> cont Unspecified) args

let scm_abs = simple_cmd_1 (fun a -> Number (ScmDynnum.abs (number_of_datum a)))


let num_command_of_list default f args = 
  if args = Nil then default
  else Number (reduce (fun a n -> f a (number_of_datum n)) (number_of_datum (car args)) (cdr args))

let one_element_list l =
  l <> Nil && cdr l = Nil 

let scm_add = num_command_of_list scm_zero ScmDynnum.add
let scm_sub args = 
  let nargs =
    if one_element_list args then cons scm_zero args
    else args 
  in num_command_of_list scm_zero ScmDynnum.sub nargs

let scm_mul = num_command_of_list scm_one ScmDynnum.mul
let scm_div args =
  let nargs =
    if one_element_list args then cons scm_one args
    else args 
  in num_command_of_list scm_one ScmDynnum.div nargs

let predicate_of_list conv f args =
  let rec cmp_all last =
    function
      | Nil -> true
      | Pair p ->
	let num_val = conv p.car in
	  if f last num_val then cmp_all num_val p.cdr
	  else false
      | _ -> invalid_arg "cmp_all" in
    Boolean (cmp_all (conv (car args)) (cdr args))

let num_predicate_of_list = predicate_of_list number_of_datum
let char_predicate_of_list = predicate_of_list char_of_datum
let string_predicate_of_list = predicate_of_list string_of_scm_string



let num_predicate f args =  Boolean (f (number_of_datum (car args)))
let num_simple_cmd f args = Number (f (number_of_datum (car args)))
let num_simple_cmd2 f args = Number (f (number_of_datum (car args)) (number_of_datum (cadr args)))

let scm_greater_p = num_predicate_of_list ScmDynnum.greater
let scm_less_p = num_predicate_of_list ScmDynnum.less
let scm_greater_or_equal_p = num_predicate_of_list ScmDynnum.greater_or_equal
let scm_less_or_equal_p = num_predicate_of_list ScmDynnum.less_or_equal
(* let scm_not_equal_p = num_pred ScmDynnum.unequal *)
let scm_num_equal_p = num_predicate_of_list ScmDynnum.equal



let scm_zero_p = num_predicate ScmDynnum.is_zero
let scm_positive_p = num_predicate ScmDynnum.is_positive
let scm_negative_p = num_predicate ScmDynnum.is_negative
let scm_odd_p = num_predicate  ScmDynnum.is_odd
let scm_even_p = num_predicate ScmDynnum.is_even


let scm_sqrt = num_simple_cmd ScmDynnum.sqrt  
let scm_sin = num_simple_cmd ScmDynnum.sin
let scm_cos = num_simple_cmd ScmDynnum.cos
let scm_tan = num_simple_cmd ScmDynnum.tan
let scm_asin = num_simple_cmd ScmDynnum.asin
let scm_acos = num_simple_cmd ScmDynnum.acos
(* let scm_atan args = 
  if cdr args = Nil then num_simple_cmd ScmDynnum.atan args
  else num_simple_cmd2
    (fun y z -> 
       ScmDynnum.angle (ScmDynnum.make_rectangular x y)) args *)
let scm_exp = num_simple_cmd ScmDynnum.exp
let scm_log = num_simple_cmd ScmDynnum.log
let scm_expt = num_simple_cmd2 ScmDynnum.pow

let scm_im_part = num_simple_cmd ScmDynnum.im_part
let scm_re_part = num_simple_cmd ScmDynnum.re_part


let scm_max = num_command_of_list scm_zero ScmDynnum.max 
let scm_min = num_command_of_list scm_zero ScmDynnum.min

let scm_quotient = num_simple_cmd2 ScmDynnum.quotient
let scm_remainder = num_simple_cmd2 ScmDynnum.remainder
let scm_modulo = num_simple_cmd2 ScmDynnum.modulo

let scm_gcd = num_command_of_list scm_zero ScmDynnum.gcd
let scm_lcm = num_command_of_list scm_one ScmDynnum.lcm

let scm_numerator = num_simple_cmd ScmDynnum.numerator
let scm_denominator = num_simple_cmd ScmDynnum.denominator       
    
let scm_truncate = num_simple_cmd ScmDynnum.truncate
(*let scm_round = num_simple_cmd ScmDynnum.round*)

let scm_set_car args = simple_cmd_2 set_car args; Unspecified
let scm_set_cdr args = simple_cmd_2 set_cdr args; Unspecified

let scm_car = simple_cmd_1 car
let scm_cdr = simple_cmd_1 cdr


let scm_cadr = simple_cmd_1 cadr
let scm_caar = simple_cmd_1 caar
let scm_cddr = simple_cmd_1 cddr
let scm_cdar = simple_cmd_1 cdar

let scm_caaar = simple_cmd_1 caaar
let scm_caadr = simple_cmd_1 caadr
let scm_cadar = simple_cmd_1 cadar
let scm_cdaar = simple_cmd_1 cdaar
let scm_caddr = simple_cmd_1 caddr
let scm_cdadr = simple_cmd_1 cdadr
let scm_cddar = simple_cmd_1 cddar
let scm_cdddr = simple_cmd_1 cdddr

let scm_caaaar = simple_cmd_1 caaaar
let scm_caaadr = simple_cmd_1 caaadr
let scm_caadar = simple_cmd_1 caadar
let scm_cadaar = simple_cmd_1 cadaar
let scm_cdaaar = simple_cmd_1 cdaaar
let scm_caaddr = simple_cmd_1 caaddr
let scm_cadadr = simple_cmd_1 cadadr
let scm_caddar = simple_cmd_1 caddar
let scm_cdaadr = simple_cmd_1 cdaadr
let scm_cdadar = simple_cmd_1 cdadar
let scm_cddaar = simple_cmd_1 cddaar
let scm_cadddr = simple_cmd_1 cadddr
let scm_cdaddr = simple_cmd_1 cdaddr
let scm_cddadr = simple_cmd_1 cddadr
let scm_cdddar = simple_cmd_1 cdddar
let scm_cddddr = simple_cmd_1 cddddr

let scm_length args = datum_of_int (simple_cmd_1 length args)

let scm_append args =
  let rec iter = 
    function
      | Nil -> Nil
      | Pair p ->
	  if p.cdr = Nil then p.car
	  else append p.car (iter p.cdr)
      | _ -> invalid_arg "scm_append"
  in iter args

let scm_reverse = simple_cmd_1 reverse
let scm_list_tail = 
  let call_internal list k = list_tail list (exact_int_of_datum k) in
    simple_cmd_2 call_internal

let scm_list_ref = 
  let call_internal list k = nth list (exact_int_of_datum k) in
    simple_cmd_2 call_internal

let scm_symbol_of_string = simple_cmd_1 (fun s -> gen_symbol (string_of_scm_string s))
let scm_string_of_symbol = simple_cmd_1 (fun s -> String (string_of_symbol s))

let scm_char_eq_p = char_predicate_of_list (=)
let scm_char_greater_p = char_predicate_of_list (>)
let scm_char_less_p = char_predicate_of_list (<)
let scm_char_greater_or_equal_p = char_predicate_of_list (>=)
let scm_char_less_or_eqal_p = char_predicate_of_list (<=)

let char_predicate f args = Boolean (f (char_of_datum (car args)))

let scm_char_alphabetic_p =
  char_predicate
    (function 
       | 'a' .. 'z' | 'A' .. 'Z' -> true 
       | _ -> false)

let scm_char_numeric_p =
  char_predicate
    (function
       | '0' .. '9' -> true
       | _ -> false)

let scm_char_whitespace_p =
  char_predicate
    (function
       | ' ' | '\t' | '\n' | '\r' -> true
       | _ -> false)

let scm_char_uppercase_p =
  char_predicate (fun c -> c = Char.uppercase c)
       
let scm_char_lowercase_p =
  char_predicate (fun c -> c = Char.lowercase c)

let chr_ci_cmp cmp c1 c2 = 
  cmp (Char.lowercase c1) (Char.lowercase c2)

let scm_char_ci_eq_p = char_predicate_of_list (chr_ci_cmp (=))
let scm_char_ci_greater_p = char_predicate_of_list (chr_ci_cmp (>))
let scm_char_ci_less_p = char_predicate_of_list (chr_ci_cmp (<))
let scm_char_ci_greater_or_equal_p = char_predicate_of_list (chr_ci_cmp (>=))
let scm_char_ci_less_or_eqal_p = char_predicate_of_list (chr_ci_cmp (<=))

let scm_integer_of_char =
  let conv arg =
    let i = Char.code (char_of_datum arg) in
      datum_of_int i
  in simple_cmd_1 conv

let scm_char_of_datum_integer =
  let conv arg =
    let c = Char.chr (exact_int_of_datum arg) in
      Character c
  in simple_cmd_1 conv

let scm_make_string args =
  if cdr args = Nil then
    String (String.create (exact_int_of_datum (car args)))
  else
    String (String.make (exact_int_of_datum (car args)) (char_of_datum (cadr args)))

let scm_string args =
  String (reduce (fun a b -> a ^ Char.escaped (char_of_datum b)) "" args)

let scm_string_of_list = simple_cmd_1 scm_string

let scm_list_of_string =
  let build_list scm_s =
    let s = string_of_scm_string scm_s in
    let length = String.length s in
    let rec iter accum idx =
      if idx < 0 then accum
      else iter (cons (Character (String.get s idx)) accum) (idx-1)
    in iter Nil (length-1)
  in simple_cmd_1 build_list
	  
		    
let scm_string_length =
  simple_cmd_1 (fun s -> datum_of_int (String.length (string_of_scm_string s)))

let scm_string_ref =
  simple_cmd_2 (fun s k -> Character (String.get (string_of_scm_string s) (exact_int_of_datum k)))

let scm_string_set = 
  simple_cmd_3 (fun s k c -> 
		  String.set (string_of_scm_string s) (exact_int_of_datum k) (char_of_datum c);
		  Unspecified)

let scm_string_eq_p = string_predicate_of_list (=)
let scm_string_greater_p = string_predicate_of_list (>)
let scm_string_less_p = string_predicate_of_list (<)
let scm_string_greater_or_equal_p = string_predicate_of_list (>=)
let scm_string_less_or_equal_p = string_predicate_of_list (<=)

let str_ci_cmp cmp s1 s2 =
  cmp (String.lowercase s1) (String.lowercase s2)

let scm_string_ci_eq_p = string_predicate_of_list (str_ci_cmp (=))
let scm_string_ci_greater_p = string_predicate_of_list (str_ci_cmp (>))
let scm_string_ci_less_p = string_predicate_of_list (str_ci_cmp (<))
let scm_string_ci_greater_or_equal_p = string_predicate_of_list (str_ci_cmp (>=))
let scm_string_ci_less_or_equal_p = string_predicate_of_list (str_ci_cmp (<=))

let scm_substring =
  simple_cmd_3 (fun s a o ->
		  let ai = exact_int_of_datum a and oi = exact_int_of_datum o in
		  let len = oi - ai in
		    String (String.sub (string_of_scm_string s) ai len))

let scm_string_append args =
  String (reduce (fun a b -> a ^ (string_of_scm_string b)) "" args)

let scm_string_copy = simple_cmd_1 (fun s -> String (String.copy (string_of_scm_string s)))
let scm_string_fill = simple_cmd_2 
			   (fun s c -> 
			      let str = string_of_scm_string s in
				String.fill (str) 0 (String.length str) (char_of_datum c); 
				Unspecified)

let scm_vector = vector_of_list 

let exc_id = ref 0
exception Exit_with_id_and_args of int * datum


let scm_call_with_current_continuation env cont args =
    let cmd = car args in
    let nargs = clamp (ML_LazyProcedure 
			 (fun env _ args -> 
			    let eval_args_f = cons (ML_EagerProcedure (fun v -> car v)) args in
			      eval env cont eval_args_f)) in
    let code = cons cmd nargs in
      eval env cont code 




let scm_values args = values_of_list args


let scm_call_with_values env cont args =
  eval env 
    (fun values -> 
    let to_list = 
      function
	| Values v -> Pair v
	| a -> clamp a in
    let code = cons (cadr args) (to_list values) in
      eval env cont code) 
    (clamp (car args)) 



let scm_report_environment args = Symbol "default-environment"

let scm_make_vector args =
  let size = car args
  and initial = 
    if simple_cell args then Nil
    else cadr args in
    Vector (Array.create (exact_int_of_datum size) initial)

let scm_getenv =
  simple_cmd_1 
    (fun s -> 
       try String (Sys.getenv (string_of_scm_string s))
       with Not_found -> scm_false)

let scm_vector_ref args =
  let v = car args
  and k = cadr args in
    match v with
	Vector v -> v.(exact_int_of_datum k)
      | _ -> invalid_arg "vector_ref"  

let scm_vector_set args =
  let vec = car args
  and idx = cadr args 
  and value = caddr args in
    match vec with
      | Vector v -> 
	  v.(exact_int_of_datum idx) <- value;
	  Unspecified
      | _ -> invalid_arg "vector-set!"

let scm_vector_of_list = simple_cmd_1 vector_of_list
let scm_list_of_vector = simple_cmd_1 list_of_vector
let scm_vector_length = 
  simple_cmd_1 
    (fun v -> datum_of_int (Array.length (vector_of_datum v)))


let scm_call_with_input_file env cont =
  let do_it file proc = 
    let  cont_with_file file_name =
      let ich = open_in (string_of_scm_string file_name) in
	eval env 
	  (fun v -> 
	     close_in ich;
	     cont v) 
	  (cons proc (clamp (InputPort ich))) 
    in
      eval env cont_with_file file 
  in
      simple_cmd_2 do_it

let scm_call_with_output_file env cont args =
  let do_it file  = 
    let cont_with_file proc =   
      let och = open_out (string_of_scm_string file) in
	eval env 
	  (fun v -> 
	     close_out och;
	     cont v)
	  (cons proc (clamp (OutputPort och))) 
    in
      eval env cont_with_file (cadr args)  
  in
    eval env do_it (car args)


let scm_current_input_port _  = !current_input_port
let scm_current_output_port _  = !current_output_port

let scm_with_input_from_file env cont args =
  let do_it file thunk =
    let file_name = string_of_scm_string file in
    let ich = open_in file_name in
    let old_cip = !current_input_port in
      current_input_port := InputPort ich;
      scm_begin env 
	(fun v -> 
	   close_in ich;
	   current_input_port := old_cip;
	   cont v)
	thunk 
  in 
    do_it (car args) (cdr args)


let scm_with_output_from_file env cont args =
  let do_it file thunk =
    let file_name = string_of_scm_string file in
    let och = open_out file_name in
    let old_cop = !current_output_port in
      current_output_port := OutputPort och;
      scm_begin env 
	(fun v -> 
	   close_out och;
	   current_output_port := old_cop;
	   cont v)
	thunk 
  in 
    do_it (car args) (cdr args)



let scm_open_input_file =
  let do_it scm_string =
    let ich = open_in (string_of_scm_string scm_string) in
      InputPort ich
  in simple_cmd_1 do_it

let scm_open_output_file =
  let do_it scm_string =
    let och = open_out (string_of_scm_string scm_string) in
      OutputPort och
  in simple_cmd_1 do_it

let scm_close =
  let do_it =
    function
      | InputPort ich -> 
	  close_in ich;
	  Unspecified
      | OutputPort och -> 
	  close_out och;
	  Unspecified
      | _ -> invalid_arg "close"
  in simple_cmd_1 do_it

let input_channel_of_args args =
in_channel_of_datum
  (if args = Nil then !current_input_port
   else car args)

let output_channel_of_args args =
  out_channel_of_datum
    (if args = Nil then !current_output_port
     else car args)

let scm_read args =
  let ich = input_channel_of_args args in
  let lexbuf = Lexing.from_channel ich in
    try ScmParser.item ScmLexer.token lexbuf
    with End_of_file -> Eof

let scm_read_char args =
  let ich = input_channel_of_args args in
    try Character (input_char ich)
    with End_of_file -> Eof

let scm_peek_char args =
  let ich = input_channel_of_args args in
  let ipos = pos_in ich in
  let result = 
    try Character (input_char ich) 
    with End_of_file -> Eof
  in
    seek_in ich ipos;
    result

let scm_eof_object_p = predicate is_eof

let scm_char_ready_p _ = scm_true

let display_string_of_datum =
  function
      String s -> s
    | d -> string_of_datum d

let scm_write args = 
  let och = output_channel_of_args (cdr args) in
    output_string och (display_string_of_datum (car args));
    flush och;
    Unspecified

let scm_display args =
  let och = output_channel_of_args (cdr args) in
    (match (car args) with
       | String s -> output_string och s
       | Character c ->  output_char och c
       | atom -> 
	   let s = string_of_datum atom in
	     output_string och s);
  flush och;
  Unspecified

let scm_newline args =
  let och = output_channel_of_args args in
    output_char och '\n';
    flush och;
    Unspecified

let scm_write_char args =
  let och = output_channel_of_args args in
    output_char och (char_of_datum (car args));
    Unspecified



let load env ?(prompt = fun v -> v) stream =
  let result = ref Unspecified in
  let rec rep () =
    let read_block = ScmParser.item ScmLexer.token stream in
      result := eval env prompt read_block;
      rep ()
  in 
    try rep ()
    with Exit -> !result


let scm_load env cont args =
  let do_it scm_string =
    let file_name = string_of_scm_string scm_string in
    let stream = Lexing.from_channel (open_in file_name) in
      cont (load env ~prompt:cont stream)
  in eval env do_it (car args)



let define_scm_var env name value =
  Environment.global_add env name value
    

let register_scm_fun env name func =
  Environment.add env name (ML_EagerProcedure func)  

let rec command_list = 
  ["+", ML_EagerProcedure scm_add;
   "-", ML_EagerProcedure scm_sub;
   "*", ML_EagerProcedure scm_mul;
   "/", ML_EagerProcedure scm_div;
   "eval", ML_LazyProcedure scm_eval;
   "read", ML_EagerProcedure scm_read;
   "write", ML_EagerProcedure scm_write;
   "make-vector", ML_EagerProcedure scm_make_vector;
   "vector-ref", ML_EagerProcedure scm_vector_ref;
   "vector-length", ML_EagerProcedure scm_vector_length;
   "list", ML_EagerProcedure scm_list;
   "car", ML_EagerProcedure (simple_cmd_1 car);
   "cdr", ML_EagerProcedure (simple_cmd_1 cdr);
   "list->vector", ML_EagerProcedure scm_vector_of_list;
   "vector->list", ML_EagerProcedure scm_list_of_vector;
   "define", ML_LazyProcedure scm_define;
   "lambda", ML_LazyProcedure scm_lambda; 
   "quote", ML_LazyProcedure scm_quote; 
   "symbol?", ML_EagerProcedure scm_symbol_p;
   "boolean?", ML_EagerProcedure scm_boolean_p;
   "number?", ML_EagerProcedure scm_number_p;
   "vector?", ML_EagerProcedure scm_vector_p;
   "pair?", ML_EagerProcedure scm_pair_p;
   "string?", ML_EagerProcedure scm_string_p;
   "port?", ML_EagerProcedure scm_port_p;
   "char?", ML_EagerProcedure scm_char_p;
   "procedure?", ML_EagerProcedure scm_procedure_p;
   "eqv?", ML_EagerProcedure scm_eqv_p;
   "eq?", ML_EagerProcedure scm_eq_p;
   "equal?", ML_EagerProcedure scm_equal_p;
   "begin", ML_LazyProcedure scm_begin;
   "if", ML_LazyProcedure scm_if;
   "cond", ML_LazyProcedure scm_cond;
   ">", ML_EagerProcedure scm_greater_p;
   "<", ML_EagerProcedure scm_less_p;
   ">=", ML_EagerProcedure scm_greater_or_equal_p;
   "<=", ML_EagerProcedure scm_less_or_equal_p;
   "=", ML_EagerProcedure scm_num_equal_p;
   "set!", ML_LazyProcedure scm_set_var;
   "case", ML_LazyProcedure scm_case;
   "and", ML_LazyProcedure scm_and_p;
   "or", ML_LazyProcedure scm_or_p;
   "display", ML_EagerProcedure scm_display;
   "quit", ML_EagerProcedure scm_quit;
   "exit", ML_EagerProcedure scm_quit;
   "let", ML_LazyProcedure (let_fun let_with_late_binding);
   "let*", ML_LazyProcedure (let_fun let_with_sequencially_binding);
   "letrec", ML_LazyProcedure (let_fun let_with_recursive_binding);  
   "zero?", ML_EagerProcedure scm_zero_p;
   "do", ML_LazyProcedure scm_do;  
   "vector-set!", ML_EagerProcedure scm_vector_set;
   "null?", ML_EagerProcedure scm_null_p;
   "cons", ML_EagerProcedure (simple_cmd_2 cons);
   "delay", ML_LazyProcedure scm_delay; 
   "force", ML_EagerProcedure scm_force; 
   "quasiquote", ML_LazyProcedure scm_quasiquote; 
   "map", ML_LazyProcedure scm_map; 
   "apply", ML_LazyProcedure scm_apply; 
   "abs", ML_EagerProcedure scm_abs;
   "concat", ML_EagerProcedure (simple_cmd_2 concat);
   "sqrt", ML_EagerProcedure scm_sqrt;
   "positive?", ML_EagerProcedure scm_positive_p;
   "negative?", ML_EagerProcedure scm_negative_p;
   "odd?", ML_EagerProcedure scm_odd_p;
   "even?", ML_EagerProcedure scm_even_p;
   "min", ML_EagerProcedure scm_min;
   "max", ML_EagerProcedure scm_max;
   "modulo", ML_EagerProcedure scm_modulo;
   "quotient", ML_EagerProcedure scm_quotient;
   "remainder", ML_EagerProcedure scm_remainder;
   "list?", ML_EagerProcedure scm_list_p;
   "set-car!", ML_EagerProcedure scm_set_car;
   "set-cdr!", ML_EagerProcedure scm_set_cdr;
   "complex?", ML_EagerProcedure scm_complex;
   "real?", ML_EagerProcedure scm_real;
   "rational?", ML_EagerProcedure scm_rational;
   "integer?", ML_EagerProcedure scm_integer;
   "exact?", ML_EagerProcedure scm_exact;
   "inexact?", ML_EagerProcedure scm_inexact;
   "exact->inexact", ML_EagerProcedure scm_inexact_of_exact;
   "inexact->exact", ML_EagerProcedure scm_exact_of_inexact;
   "sin", ML_EagerProcedure scm_sin;
   "cos", ML_EagerProcedure scm_cos;
   "tan", ML_EagerProcedure scm_tan;
   "asin", ML_EagerProcedure scm_asin;
   "acos", ML_EagerProcedure scm_acos;
   "exp", ML_EagerProcedure scm_exp;
   "log", ML_EagerProcedure scm_log;
   "expt", ML_EagerProcedure scm_expt;
   "string->number", ML_EagerProcedure scm_number_of_string;
   "number->string", ML_EagerProcedure scm_string_of_number;
   "not", ML_EagerProcedure scm_not_p;
   "caaaar", ML_EagerProcedure scm_caaaar;
   "caaadr", ML_EagerProcedure scm_caaadr;
   "caadar", ML_EagerProcedure scm_caadar;
   "cadaar", ML_EagerProcedure scm_cadaar;
   "cdaaar", ML_EagerProcedure scm_cdaaar;
   "caaddr", ML_EagerProcedure scm_caaddr;
   "cadadr", ML_EagerProcedure scm_cadadr;
   "caddar", ML_EagerProcedure scm_caddar;
   "cdaadr", ML_EagerProcedure scm_cdaadr;
   "cdadar", ML_EagerProcedure scm_cdadar;
   "cddaar", ML_EagerProcedure scm_cddaar;
   "cadddr", ML_EagerProcedure scm_cadddr;
   "cdaddr", ML_EagerProcedure scm_cdaddr;
   "cddadr", ML_EagerProcedure scm_cddadr;
   "cdddar", ML_EagerProcedure scm_cdddar;
   "cddddr", ML_EagerProcedure scm_cddddr;
   "caaar", ML_EagerProcedure scm_caaar;
   "caadr", ML_EagerProcedure scm_caadr;
   "cadar", ML_EagerProcedure scm_cadar;
   "cdaar", ML_EagerProcedure scm_cdaar;
   "caddr", ML_EagerProcedure scm_caddr;
   "cdadr", ML_EagerProcedure scm_cdadr;
   "cddar", ML_EagerProcedure scm_cddar;
   "cdddr", ML_EagerProcedure scm_cdddr;
   "caar", ML_EagerProcedure scm_caar;
   "cadr", ML_EagerProcedure scm_cadr;
   "cdar", ML_EagerProcedure scm_cdar;
   "cddr", ML_EagerProcedure scm_cddr;
   "length", ML_EagerProcedure scm_length;
   "append", ML_EagerProcedure scm_append;
   "reverse", ML_EagerProcedure scm_reverse;
   "list-tail", ML_EagerProcedure scm_list_tail;
   "list-ref", ML_EagerProcedure scm_list_ref;
   "memq", ML_EagerProcedure scm_memq;
   "memv", ML_EagerProcedure scm_memv;
   "member", ML_EagerProcedure scm_member;
   "assoc", ML_EagerProcedure scm_assoc;
   "assq", ML_EagerProcedure scm_assq;
   "assv", ML_EagerProcedure scm_assv;
   "string->symbol", ML_EagerProcedure scm_symbol_of_string;
   "symbol->string", ML_EagerProcedure scm_string_of_symbol;
   "char=?", ML_EagerProcedure scm_char_eq_p;
   "char>=?", ML_EagerProcedure scm_char_greater_or_equal_p;
   "char>?", ML_EagerProcedure scm_char_greater_p;
   "char<?", ML_EagerProcedure scm_char_less_p;
   "char<=?", ML_EagerProcedure scm_char_less_or_eqal_p;
   "char-ci=?", ML_EagerProcedure scm_char_ci_eq_p;
   "char-ci>=?", ML_EagerProcedure scm_char_ci_greater_or_equal_p;
   "char-ci>?", ML_EagerProcedure scm_char_ci_greater_p;
   "char-ci<?", ML_EagerProcedure scm_char_ci_less_p;
   "char-ci<=?", ML_EagerProcedure scm_char_ci_less_or_eqal_p;
   "integer->char", ML_EagerProcedure scm_char_of_datum_integer;
   "char->integer", ML_EagerProcedure scm_integer_of_char;
   "make-string", ML_EagerProcedure scm_make_string;
   "string", ML_EagerProcedure scm_string;
   "string-length", ML_EagerProcedure scm_string_length;
   "string-set!", ML_EagerProcedure scm_string_set;
   "string-ref", ML_EagerProcedure scm_string_ref;
   "string=?", ML_EagerProcedure scm_string_eq_p;
   "string>?", ML_EagerProcedure scm_string_greater_p;
   "string>=?", ML_EagerProcedure scm_string_greater_or_equal_p;
   "string<=?", ML_EagerProcedure scm_string_less_or_equal_p;
   "string<?", ML_EagerProcedure scm_string_less_p;
   "string-ci=?", ML_EagerProcedure scm_string_ci_eq_p;
   "string-ci>?", ML_EagerProcedure scm_string_ci_greater_p;
   "string-ci>=?", ML_EagerProcedure scm_string_ci_greater_or_equal_p;
   "string-ci<=?", ML_EagerProcedure scm_string_ci_less_or_equal_p;
   "string-ci<?", ML_EagerProcedure scm_string_ci_less_p;
   "substring", ML_EagerProcedure scm_substring;
   "string-append", ML_EagerProcedure scm_string_append;
   "string-copy", ML_EagerProcedure scm_string_copy;
   "string-fill", ML_EagerProcedure scm_string_fill;
   "vector", ML_EagerProcedure scm_vector;
   "for-each", ML_LazyProcedure scm_for_each; 
   "call-with-current-continuation", ML_LazyProcedure scm_call_with_current_continuation; 
   "call/cc", ML_LazyProcedure scm_call_with_current_continuation; 
   "values", ML_EagerProcedure scm_values;
   "call-with-values", ML_LazyProcedure scm_call_with_values; 
   "char-alphabetic?", ML_EagerProcedure scm_char_alphabetic_p;
   "char-numeric?", ML_EagerProcedure scm_char_numeric_p;
   "char-whitespace?", ML_EagerProcedure scm_char_whitespace_p;
   "char-upper-case?", ML_EagerProcedure scm_char_uppercase_p;
   "char-lower-case?", ML_EagerProcedure scm_char_lowercase_p;
   "getenv", ML_EagerProcedure scm_getenv;
   "list->string", ML_EagerProcedure scm_string_of_list;
   "string->list", ML_EagerProcedure scm_list_of_string;
   "gcd", ML_EagerProcedure scm_gcd;
   "lcm", ML_EagerProcedure scm_lcm;
   "numerator", ML_EagerProcedure scm_numerator;
   "denominator", ML_EagerProcedure scm_denominator;
   "call-with-input-file", ML_LazyProcedure scm_call_with_input_file; 
   "call-with-output-file", ML_LazyProcedure scm_call_with_output_file; 
   "input-port?", ML_EagerProcedure scm_input_port_p;
   "ouput-port?", ML_EagerProcedure scm_output_port_p;
   "current-input-port", ML_EagerProcedure scm_current_input_port;
   "current-output-port", ML_EagerProcedure scm_current_output_port;
   "with-input-from-file", ML_LazyProcedure scm_with_input_from_file; 
   "with-output-to-file", ML_LazyProcedure scm_with_output_from_file; 
   "open-input-file", ML_EagerProcedure scm_open_input_file;
   "open-output-file", ML_EagerProcedure scm_open_output_file;
   "close-input-port", ML_EagerProcedure scm_close;
   "close-output-port", ML_EagerProcedure scm_close;
   "char-ready?", ML_EagerProcedure scm_char_ready_p;
   "eof-object?", ML_EagerProcedure scm_eof_object_p;
   "read-char", ML_EagerProcedure scm_read_char;
   "peek-char", ML_EagerProcedure scm_peek_char;
   "load", ML_LazyProcedure scm_load; 
   "newline", ML_EagerProcedure scm_newline;
   "write-char", ML_EagerProcedure scm_write_char;
   "scheme-report-environment" , ML_EagerProcedure scm_report_environment;
   "truncate", ML_EagerProcedure scm_truncate;
  ]  
and scm_eval env cont args =
  let eval_code code =
    (* let env = init_environment () in *)
    eval env cont code
  in eval env eval_code (car args)
       (* eval env cont (map (eval env) (car args)) *)
and init_environment () = 
  let top = Hashtbl.create 256 in
  let ks, vs = List.split command_list in
  let env = { (* stack = [ks, List.map (fun v -> ref v) vs]; *)
    stack = [];
    top = top } in
    List.iter (fun (k,v) -> ignore( Environment.global_add env k v )) command_list;
    env


       
