(**************************************************************************)
(*  Mana : A kana(romaji)-kanji conversion engine using ChaSen algorithm.    *)
(*  Copyright (C) 2003, 2004  Yamagata Yoriyuki                           *)
(*                                                                        *)
(*  This program is free software; you can redistribute it and/or modify  *)
(*  it under the terms of the GNU General Public License as published by  *)
(*  the Free Software Foundation; either version 2 of the License, or (at  *)
(*  your option) any later version.                                       *)
(*                                                                        *)
(*  This program is distributed in the hope that it will be useful, but   *)
(*  WITHOUT ANY WARRANTY; without even the implied warranty of            *)
(*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *)
(*  General Public License for more details.                              *)
(**************************************************************************)

let rec scm_list = function
    [] -> ScmTypes.Nil
  | v :: r ->
      ScmTypes.cons v (scm_list r)

let rec scm_path = function
  | [] -> ScmTypes.Nil
  | (mrph, p, l, state, cost) :: path ->
      let converted = Conv.surface_form mrph in
      let x = 
	scm_list
	  [ScmTypes.String converted;
	   ScmTypes.datum_of_int p;
	   ScmTypes.datum_of_int l;
	   ScmTypes.datum_of_int state;
	   ScmTypes.datum_of_int cost] in
      ScmTypes.cons x (scm_path path)

let scm_best_path args =
  try
    let string = ScmTypes.string_of_scm_string (ScmTypes.car args) in
    let state = ScmTypes.exact_int_of_datum (ScmTypes.cadr args) in
    let pos = ScmTypes.exact_int_of_datum (ScmTypes.caddr args) in
    let len = ScmTypes.exact_int_of_datum (ScmTypes.cadddr args) in
    let parse_obj = Parse.parse string ~state ~pos ~len in
    let path = Parse.best_path parse_obj (pos + len) in
    scm_path path
  with Not_found -> ScmTypes.Nil

let rec scm_candidates = function
  | [] -> ScmTypes.Nil
  | (mrph, p, l) :: path ->
      let converted = Conv.surface_form mrph in
      let x = 
	scm_list
	  [ScmTypes.String converted;
	   ScmTypes.datum_of_int p;
	   ScmTypes.datum_of_int l] in
      ScmTypes.cons x (scm_candidates path)

let scm_list_candidates args =
  let string = ScmTypes.string_of_scm_string (ScmTypes.car args) in
  let state = ScmTypes.exact_int_of_datum (ScmTypes.cadr args) in
  let pos = ScmTypes.exact_int_of_datum (ScmTypes.caddr args) in
  let mrph_len = ScmTypes.exact_int_of_datum (ScmTypes.cadddr args) in
  let len = ScmTypes.exact_int_of_datum (ScmTypes.nth args 4) in
  scm_candidates 
    (Parse.list_candidates string ~state ~pos ~mrph_len ~len)

let scm_add_new_word args =
  let kaki = ScmTypes.string_of_scm_string (ScmTypes.car args) in
  let yomi = ScmTypes.string_of_scm_string (ScmTypes.cadr args) in
  PersonalDict.add_new_word ~kaki ~yomi;
  ScmTypes.Nil

(* Some code comes from Schoca *)
let _ =
  let prompt v =
    print_string (ScmTypes.string_of_datum v ^ "\n"); 
    flush stdout;
    v in
  let lexbuf = Lexing.from_channel stdin in
  let env = ScmEval.init_environment () in
  let env, _= ScmEval.register_scm_fun env "mana-best-path" scm_best_path in
  let env, _ = 
    ScmEval.register_scm_fun env "mana-list-candidates" scm_list_candidates in
  let env, _= 
    ScmEval.register_scm_fun env "mana-add-new-word" scm_add_new_word in
  let rec exc_loop () =
    try ignore( ScmSchoca.parse_stream_with_environment env ~prompt lexbuf );
    with 
    | ScmEval.Quit | Exit -> ()
  in exc_loop ()
