
(* type datum  *)
(** Scheme is dynamic typed. The type datum can represents all 
all data which Scheme knows.
 *)

type environment = {
  stack: ((string list) * (datum ref list)) list;
  top: (string,datum ref) Hashtbl.t;
}
and datum =
    Symbol of string
  | SymbolInEnv of string * environment
  | Ref of datum ref
  | Pair of pair
  | Vector of datum array
  | Nil
  | Number of ScmDynnum.t
  | String of string
  | Character of char
  | Boolean of bool
  | Procedure of datum * datum * environment * (datum ref list)
  | ML_EagerProcedure of (datum -> datum)
  | ML_LazyProcedure of (environment -> (datum -> datum) -> datum -> datum)
  | Delayed of (unit -> datum)
  | Values of pair
  | InputPort of in_channel 
  | OutputPort of out_channel
  | Eof
  | Unspecified
and pair = { 
  mutable car: datum;
  mutable cdr: datum
}




(*s some tests on datum. *)
val is_pair : datum -> bool
val is_boolean : datum -> bool
val is_symbol : datum -> bool
val is_number : datum -> bool
val is_vector : datum -> bool
val is_procedure : datum -> bool
val is_port : datum -> bool
val is_input_port : datum -> bool
val is_output_port : datum -> bool
val is_eof : datum -> bool
val is_string : datum -> bool
val is_list : datum -> bool
val is_char : datum -> bool

(*s Some datum constants. *)
val scm_false : datum
val scm_true : datum
val scm_zero : datum
val scm_one : datum
val scm_nil : datum

(*s Constructor functions. *)
val gen_symbol : string -> datum
(** generates a new symbol. *)



val char_of_datum : datum -> char
val datum_of_char : char -> datum
val string_of_datum : datum -> string
val datum_of_string: string -> datum
val string_of_symbol : datum -> string
val datum_of_in_channel : in_channel -> datum
val datum_of_out_channel : out_channel -> datum
val in_channel_of_datum : datum -> in_channel
val out_channel_of_datum : datum -> out_channel
val datum_of_bool : bool -> datum
val bool_of_datum : datum -> bool
val string_of_scm_string: datum -> string
val number_of_datum : datum -> ScmDynnum.t
val symbol_of_datum : datum -> datum
val datum_of_int : int -> datum
val datum_of_number : ScmDynnum.t -> datum
val exact_int_of_datum : datum -> int
val vector_of_datum : datum -> datum array
val vector_of_list : datum -> datum
val list_of_vector : datum -> datum
val values_of_list : datum -> datum
val list_of_values : datum -> datum
val list_of_improper_list : datum -> datum list


(* List functions. *)
val cons : datum -> datum -> datum
val car : datum -> datum
val cdr : datum -> datum
val cadr : datum -> datum
val caar : datum -> datum
val cddr : datum -> datum
val cdar : datum -> datum
val caaar : datum -> datum
val caadr : datum -> datum
val cadar : datum -> datum
val cdaar : datum -> datum
val caddr : datum -> datum
val cdadr : datum -> datum
val cddar : datum -> datum
val cdddr : datum -> datum
val caaaar : datum -> datum
val caaadr : datum -> datum
val caadar : datum -> datum
val cadaar : datum -> datum
val cdaaar : datum -> datum
val caaddr : datum -> datum
val cadadr : datum -> datum
val caddar : datum -> datum
val cdaadr : datum -> datum
val cdadar : datum -> datum
val cddaar : datum -> datum
val cadddr : datum -> datum
val cdaddr : datum -> datum
val cddadr : datum -> datum
val cdddar : datum -> datum
val cddddr : datum -> datum

val set_car : datum -> datum -> unit
val set_cdr : datum -> datum -> unit

val list_of_datum : datum -> datum list

val simple_cell : datum -> bool

(* test *)
val is_true : datum -> bool


(* Highlevl list functions *)
val reduce : ('a -> datum -> 'a) -> 'a -> datum -> 'a
val exists : (datum -> datum) -> datum -> datum
val for_all : (datum -> datum) -> datum -> datum
val reduce2 : ('a -> datum -> datum -> 'a) -> 'a -> datum -> datum -> 'a
val append : datum -> datum -> datum
val map : (datum -> datum) -> datum -> datum
val copy : datum -> datum
val iter : (datum -> unit) -> datum -> unit
val mem : datum -> datum -> bool
val length : datum -> int
val reverse : datum -> datum
val list_tail : datum -> int -> datum
val nth : datum -> int -> datum
val eqv : datum -> datum -> bool
val eq : datum -> datum -> bool
val equal : 'a -> 'a -> bool
val member_search : ('a -> datum -> bool) -> 'a -> datum -> datum
val memq : datum -> datum -> datum
val memv : datum -> datum -> datum
val member : datum -> datum -> datum
val assoc_search : (datum -> 'a -> bool) -> 'a -> datum -> datum
val assq : datum -> datum -> datum
val assv : datum -> datum -> datum
val assoc : datum -> datum -> datum

