<?php
/**
 * 簡易的なKey-Value機構を提供する.<br>
 * 複数スレッドからのデータ取得、保存、削除に対応しており<br>
 * データをスレッドセーフに扱うことが可能である.<br>
 * 方法としては、ファイルのロック機構を採用している.<br>
 * 保存するデータはファイルにシリアライズして保存しているので、<br>
 * シリアライズ出来ないデータは保存出来ない.<br>
 * またキー値に使用可能な値は、文字列もしくは数値のみである<br>
 * 配列などは不可となる.<br>
 *
 *
 * 本PHPを使用するにあたって以下の定数を事前に使用する部分て定義すると<br>
 * 動きを制御することが可能である.<br>
 *
 * "SIMPLECACHE_LOCK_FILE_PATH":スレッドセーフを実現する為の、ロックファイルのパス.<br>
 * 宣言しない場合は、カレントディレクトリに".lock"というファイルを作成する.<br>
 * ファイルが作成出来ない場合又は、使用出来ない場合はエラーとなる.<br>
 *
 * "SIMPLECACHE_DATA_FILE_DIR":データを保存するディレクトリのパス.<br>
 * 宣言時は最後の「/」は不要である<br>
 * 宣言しない場合は、カレントディレクトリに"data"というディレクトリを作成する<br>
 * ディレクトリが作成出来ない場合又は使用出来ない場合はエラーとなる<br>
 *
 * "SIMPLECACHE_DATA_ALGORITHM":データを保存するディレクトリの階層数.<br>
 * 宣言可能な値は数値のみ、最大値は3<br>
 * 宣言しない場合は、0階層となる<br>
 * 1階層に作成されるファイル及び、ディレクトリの名前は0～9とa～fの計16の名前の<br>
 * ファイルとディレクトリが作成される<br>
 * 一旦データを保存した後でアルゴリズムを変更することは出来ない.<br>
 * 変更する場合は現在のデータを一度全て削除した後、再度作成する.<br>
 * 削除する場合は"SIMPLECACHE_DATA_FILE_DIR"で宣言したディレクトリごと削除すればよい.<br>
 * アルゴリズムの数値を大きな値にすれば保存は遅くなるが、取得が高速になる<br>
 * 大量のデータ(数十万件)を扱う場合は2程度の値にすると取得スピードに差が出る.<br>
 *
 * "SIMPLECACHE_DATA_FILE_DIR"と"SIMPLECACHE_DATA_ALGORITHM"は1セットとして使用するスタイルとなる.<br>
 * 例) アルゴリズム 1(ディレクトリ名の横には":D"がファイル名の横には":F"と記す).<br>
 *     /var/tmp配下に本クラスを配置し"SIMPLECACHE_DATA_FILE_DIR"を"./data1"とし、<br>
 *     "SIMPLECACHE_DATA_ALGORITHM"を"1"にした場合、/var/tmp/data1というディレクトリが作成され、<br>
 *     data1配下には以下の構成でディレクトリとファイルが作成される.<br>
 * ---------------------例 Start ------------------------------------
 * 0:D------0:F
 *        |-1:F
 *        |-2:F
 *        |-3:F
 *        |-4:F
 *       ・
 *       ・以下fまで続く
 *       ・
 *        |_f:F
 *
 * 1:D------0:F
 *        |-1:F
 *        |-2:F
 *        |-3:F
 *        |-4:F
 *       ・
 *       ・以下fまで続く
 *       ・
 *        |_f:F
 * ・
 * ・以下fまで続く
 * ・
 * f:D------0:F
 *        |-1:F
 *        |-2:F
 *        |-3:F
 *        |-4:F
 *       ・
 *       ・以下fまで続く
 *       ・
 *        |_f:F
 * ---------------------例 End ------------------------------------
 *
 * 使用方法<br>
 *  // Config 設定
 *  define('SIMPLECACHE_DATA_FILE_DIR', './data1');
 *  define('SIMPLECACHE_LOCK_FILE_PATH', '.lock1');
 *  define('SIMPLECACHE_DATA_ALGORITHM', '1');
 *
 *  // 読み込み
 *  require_once("SimpleCache.class.php");
 *
 *  // instance 取得
 *  $dataMgr = SimpleCache::getInstance();
 *
 *  // データをセット (キー, 値)
 *  $dataMgr->setData("key_1", "data_1");
 *
 *  // データをゲット
 *  $dataMgr->getData("key_1");
 *
 * @author T.Okuyama
 * @license GPL(Lv3)
 */ 
class SimpleCache {

    // ロックファイルパス
    private static $lockFilePath = ".lock"; 

    // ロックファイルのポインタ
    private static $lockFp = null;

    // 自身のインスタンス
    private static $_instance = null;

    // データ格納用
    private $dataMap = null;

    // データファイルを格納するディレクトリ
    private static $DATA_FILE_DIR = "./data/";

    // データを格納するディレクリの階層構造数
    // デフォルト0階層
    // この値を変更する場合は、一度現在のデータを削除してから使用する。
    private static $dataDirAlgorithm = 0;

    /**
     * privateコンストラクタ.<br>
     */
    private function __construct() {

        // new 禁止
        $this->dataMap = array();

        // ロックファイルのパスを調べる
        if (defined('SIMPLECACHE_LOCK_FILE_PATH')) {
            SimpleCache::$lockFilePath = SIMPLECACHE_LOCK_FILE_PATH;
        }

        // データファイル保存先ディレクトリ
        if (defined('SIMPLECACHE_DATA_FILE_DIR')) {
            SimpleCache::$DATA_FILE_DIR = SIMPLECACHE_DATA_FILE_DIR . "/";
        }

        // データを格納するディレクリの階層構造数を調べる
        if (defined('SIMPLECACHE_DATA_ALGORITHM')) {
            if (is_numeric(SIMPLECACHE_DATA_ALGORITHM) && SIMPLECACHE_DATA_ALGORITHM < 4) {
                SimpleCache::$dataDirAlgorithm = SIMPLECACHE_DATA_ALGORITHM;
            } else {
                throw new Exception("SIMPLECACHE_DATA_ALGORITHM declaration is illegal : Only the numerical value is permitted");
            }

        }

        // データ保存ディレクトリの存在を調べる
        if (!is_dir(SimpleCache::$DATA_FILE_DIR)) {
            mkdir(SimpleCache::$DATA_FILE_DIR, 700);
        }
    }


    /**
     * インスタンス取得.<br>
     *
     */
    public static function getInstance () {
        if (SimpleCache::$_instance == null) {
            SimpleCache::$_instance = new SimpleCache();
        } 
        return SimpleCache::$_instance;
    }


    /**
     * データ取得.<br>
     *
     * @param $key キー値 文字列もしくは数字であること
     * @return mixed データ
     */ 
    public function getData ($key) {
        $val = null;
        try {
            if (!$this->checkKeyValue($key)) 
                throw new Exception("key value is illegal : Only the character string or the numerical value");

            $this->synchronizedStart();

            $val = $this->restoreData($this->createDataPath($key), $key);

            $this->synchronizedEnd();
        } catch (Exception $e) {
            throw $e;
        }
        return $val;
    }


    /**
     * データ格納.<br>
     *
     * @param $key キー値 文字列もしくは数字であること
     * @param $val 値
     */
    public function setData ($key, $val) {
        try {
            if (!$this->checkKeyValue($key)) 
                throw new Exception("key value is illegal : Only the character string or the numerical value");

            $this->synchronizedStart();

            // データを格納するディレクトリを作成
            $this->makeDataDir($key);
            $this->saveData($this->createDataPath($key), $key, $val);

            $this->synchronizedEnd();
        } catch (Exception $e) {
            throw $e;
        }
    }


    /** 
     * ファイルのロックを使用して、全PHPでの同期化を行う.<br>
     * 同期化開始.<br>
     */
    private function synchronizedStart () {
        try {

            SimpleCache::$lockFp = fopen(SimpleCache::$lockFilePath,"w");
            flock(SimpleCache::$lockFp,LOCK_EX);
        } catch (Exception $e) {
            throw $e;
        }
    }

    /** 
     * ファイルのロックを使用して、全PHPでの同期化を行う.<br>
     * 同期化終了.<br>
     */
    private function synchronizedEnd () {
        try {

            fclose(SimpleCache::$lockFp);
            SimpleCache::$lockFp = null;
        } catch (Exception $e) {
            throw $e;
        }
    }

    /**
     * データファイルのパスを作成する.<br>
     *
     * @param $key キー値
     * @return string データファイルのパス
     */
    private function createDataPath ($key) {

        $hashStr = sha1($key);
        $index = 0;

        $dirPath = SimpleCache::$DATA_FILE_DIR;
        try {

            for (; $index < SimpleCache::$dataDirAlgorithm; $index++) {

                $dirName = substr($hashStr, $index, 1);
                $dirPath =  $dirPath . $dirName . "/";

            }

            
            $dirPath = $dirPath . substr($hashStr, $index, 1);


        } catch (Exception $e) {
            throw $e;
        }

        return $dirPath;

    }


    /**
     * データファイルを作成するディレクトリを作成する.<br>
     * 変数dataDirAlgorithmに指定された階層数分作成する.<br>
     * キー値の先頭から階層数分の文字列名でディレクトリを作成する.<br>
     * すでにディレクトリが存在する場合は何もしない.<br>
     *
     * @param $key キー値
     */
    private function makeDataDir ($key) {
        try {
            $hashStr = sha1($key);
            $dirPath = SimpleCache::$DATA_FILE_DIR;


            for ($index = 0; $index < SimpleCache::$dataDirAlgorithm; $index++) {

                $dirName = substr($hashStr, $index, 1);
                $dirPath =  $dirPath . $dirName . "/";
            }

            if (!is_dir($dirPath)) {

                $dirPath = SimpleCache::$DATA_FILE_DIR;

                for ($index = 0; $index < SimpleCache::$dataDirAlgorithm; $index++) {

                    $dirName = substr($hashStr, $index, 1);
                    $dirPath =  $dirPath . $dirName . "/";

                    if (!is_dir($dirPath)) {
                        mkdir($dirPath, 700);
                    }
                }
            }
        } catch (Exception $e) {
            throw $e;
        }
    }


    /**
     * 引数のデータをシリアライズし、ファイルに保存する.<br>
     * 引数のデータを連想配列に保存し、シリアライズする.<br>
     * 引数のファイルが存在する場合はファイルから既に保存されている、<br>
     * データを取り出し(連想配列)、その連想配列に対してデータ格納してから<br>
     * 再度ファイルにシリアライズして保存する.<br>
     * ファイルが存在しない場合は新規で作成する.<br>
     *
     * @param $saveFilePath シリアライズデータを保存する対象のファイルパス
     * @param $key キー値
     * @param $data 対象データ
     */
    private function saveData ($saveFilePath, $key, $data) {
        $fp = null;
        try {

            if (file_exists($saveFilePath)) {

                // ファイルは存在する
                $oldDataStr = file_get_contents($saveFilePath);

                $oldData = unserialize($oldDataStr);

                $oldData[$key] = $data;

                $fp = fopen($saveFilePath,"w"); 
                fwrite($fp, serialize($oldData));
            } else {

                // ファイルは存在しない
                $fp = fopen($saveFilePath,"w"); 
                $newData = array();

                $newData[$key] = $data;
                fwrite($fp, serialize($newData));
            }

            if ($fp !== null) {
                fclose($fp); 
                $fp = null;
            }
        } catch (Exception $e) {
            if ($fp !== null) {
                // ファイルを閉じる
                try {
                    fclose($fp); 
                    $fp = null;
                } catch (Exception $e2) {
                    print $e2;
                }
            }
            throw $e;
        }
    }


    /**
     * データをファイルからデシリアライズし、キー値に紐付くデータを取り出す.<br>
     * ファイルのデータは必ず連想配列になる想定.<br>
     * ファイルが存在しない、連想配列にデータが存在しない場合はNULLを返す.<br>
     *
     * @param $saveFilePath デシリアライズ対象のファイルパス
     * @param $key キー値
     * @return mixed データ
     */
    private function restoreData ($saveFilePath, $key) {
        $retVal = null;

        try {

            if (file_exists($saveFilePath)) {

                // ファイルは存在する
                $dataStr = file_get_contents($saveFilePath);

                $dataVal = unserialize($dataStr);

                $retVal = $dataVal[$key];
            }
        } catch (Exception $e) {
            throw $e;
        }
        return $retVal;
    }

    /**
     * キー値が正しいかをチェックする.<br>
     *
     * @param $key チェック対象のキー値
     * @return boolean true:正しい false:不正
     */
    private function checkKeyValue ($key) {
        $ret = true;
        if (!is_string($key)) {
            if (!is_numeric($key)) $ret = false;

        }
        return $ret;
    }


    function __destruct() {
        //print "SimpleCache - Destroying";
        try {
            if (SimpleCache::$lockFp != null) {
               $this->synchronizedEnd();
            }
        } catch (Exception $e) {
            throw $e;
        }
    }
}
?>