/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "local.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <sys/time.h>
#include "lispmgrp.h"
#include "cstring.h"

static	Boolean	lispMgr_initialize 				(TLispManager*) ;
static	Boolean	lispMgr_collectOtherGarbage		(TLispManager*, TLispEntity**) ;
static	Boolean	lispMgr_markEntityRecursively	(TLispManager*, TLispEntity*) ;
static	Boolean	lispMgr_registerEntityToList    (TLispManager*, TLispEntity**, TLispEntity*) ;

/*	inline functions */
inline	Boolean
lispMgr_markedEntityp (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;

	return	(pEntity->m_iMarker == pLispMgr->m_iMarker)? True : False ;
}

inline	Boolean
lispMgr_markEntity (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;

	pEntity->m_iMarker	= pLispMgr->m_iMarker ;
	return	True ;
}

/*	Ʊ̾/ƱǤäƤ eq Ƚ nil ֤ entity Ĥʤ*/
static inline	Boolean
lispMgr_registerEntityToList (
	register TLispManager*	pLispMgr,
	register TLispEntity**	ppEntTop,
	register TLispEntity*	pEntity)
{
	assert (pLispMgr != NULL) ;
	assert (ppEntTop != NULL) ;
	assert (pEntity  != NULL) ;

	pEntity->m_pRight			= *ppEntTop ;
	pEntity->m_pLeft			= NULL ;
	if (*ppEntTop != NULL)
		(*ppEntTop)->m_pLeft	= pEntity ;
	*ppEntTop	= pEntity ;
	return	True ;
}

static inline	Boolean
lispMgr_unregisterEntityFromList (
	register TLispManager*	pLispMgr,
	register TLispEntity**	ppEntTop,
	register TLispEntity*	pEntity)
{
	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;

	if (pEntity->m_pLeft != NULL) {
		pEntity->m_pLeft->m_pRight	= pEntity->m_pRight ;
	} else {
		assert (pEntity == *ppEntTop) ;
		*ppEntTop	= pEntity->m_pRight ;
	}
	if (pEntity->m_pRight != NULL) 
		pEntity->m_pRight->m_pLeft	= pEntity->m_pLeft ;

	pEntity->m_pRight			= NULL ;
	pEntity->m_pLeft			= NULL ;

	return	True ;
}

/*	global functions */
Boolean
TLispMgr_Create (
	register TLispManager** const	ppLispMgr)
{
	register TLispManager*	pLispMgr ;
	register int			i ;
	
	assert (ppLispMgr != NULL) ;

	pLispMgr	= MALLOC (sizeof (TLispManager)) ;
	if (pLispMgr == NULL)
		return	False ;

	for (i = 0 ; i < NELEMENTS (pLispMgr->m_apSymbolListTop) ; i ++)
		pLispMgr->m_apSymbolListTop [i]		= NULL ;
	for (i = 0 ; i < NELEMENTS (pLispMgr->m_apIntegerListTop) ; i ++)
		pLispMgr->m_apIntegerListTop [i]	= NULL ;
	for (i = 0 ; i < NELEMENTS (pLispMgr->m_apSubrListTop) ; i ++)
		pLispMgr->m_apSubrListTop [i]		= NULL ;
	pLispMgr->m_pEntListNamedMutex	= NULL ;
	pLispMgr->m_pEntListNoNameMutex	= NULL ;
	pLispMgr->m_pEntMiscListTop		= NULL ;
	pLispMgr->m_pEntVoid			= NULL ;
	pLispMgr->m_pEntEmpty			= NULL ;
	for (i = 0 ; i < LISPMGR_SIZE_RESERVED ; i ++)
		pLispMgr->m_apEntReserved [i]	= NULL ;

	if (TFAILED (TVarbuffer_Initialize (&pLispMgr->m_vbufLocalSymbol, sizeof (TLispEntity*))))
		goto	error ;
	if (TFAILED (lispMgr_initialize (pLispMgr)))
		goto	error ;
	*ppLispMgr			= pLispMgr ;
	return	True ;

  error:
	FREE (pLispMgr) ;
	return	False ;
}

Boolean
TLispMgr_Destroy (
	register TLispManager*	pLispMgr)
{
	/*	˴λ...*/
	return	False ;
}

/*	ȤƤʤ ENTITY 롣*/
Boolean
lispMgr_CollectGarbage (
	register TLispManager*	pLispMgr)
{
	register TLispEntity*	pEntity ;
#if defined (DEBUG) || 0
	struct timeval	start, end ;
	long			lEllapsed ;
	gettimeofday (&start, NULL) ;
#endif

	pLispMgr->m_iMarker	++ ;

	/*	ENTITY λȥ󥿤֤˸ơï⻲ȤƤʤ¸ߤ
	 *	롣
	 *	Vector  Conscell ˤäƻȤƤϻȥ
	 *	äƤʤΤդ뤳ȡ
	 *	äơVector  Conscell õƻ˥ޡƤȤ
	 *	ɬפȤʤ롣
	 */
	pEntity	= pLispMgr->m_pEntMiscListTop ;
	while (pEntity != NULL) {
		if (pEntity->m_lReferCount > 0) 
			lispMgr_markEntityRecursively (pLispMgr, pEntity) ;
		pEntity	= pEntity->m_pRight ;
	}

	lispMgr_CollectIntegerGarbage (pLispMgr) ;
	/*	̾դ MUTEX  GC оݤˤϷ褷Ƥʤʤ顢̾դ
	 *	MUTEX ɤɤȡ꥽ġդ뤳
	 *	ȡ*/
	lispMgr_collectOtherGarbage (pLispMgr, &pLispMgr->m_pEntListNoNameMutex) ;
	lispMgr_collectOtherGarbage (pLispMgr, &pLispMgr->m_pEntMiscListTop) ;

#if defined (DEBUG) && defined (WIN32)
	fprintf (stderr, "GarbageCollect ... %ld ms\n", GetTickCount () - dwStart) ;
#endif
#if defined (DEBUG) || 0
	gettimeofday (&end, NULL) ;
	lEllapsed	= (end.tv_sec - start.tv_sec) * 1000 + (end.tv_usec - start.tv_usec) / 1000 ;
	fprintf (stderr, "GC (%ld msec)\n", lEllapsed) ;
#endif
	return	True ;
}

Boolean
lispMgr_UnregisterMisc (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	assert (0 < pEntity->m_iType && pEntity->m_iType < MAX_LISPENTITY_TYPE &&
			pEntity->m_iType != LISPENTITY_INTEGER) ;

	return	lispMgr_unregisterEntityFromList (pLispMgr, &pLispMgr->m_pEntMiscListTop, pEntity) ;
}

Boolean
lispMgr_initialize (
	register TLispManager*	pLispMgr)
{
	/*	졹ľʤƺѤ褦ϿƤ*/
	static const char*	apReservedSymbols [LISPMGR_SIZE_RESERVED]	= {
		"nil",					"t",					"-",
		"lambda",				"macro",				"&optional",
		"&rest",				"keymap",				"error",
		"exit",					"quit",					"quit-flag",
		"inhibit-quit",			"features",				"interactive",
		"this-command",			"last-command",			"unread-command-events",
		"pre-command-hook",		"post-command-hook",	"last-command-char",
		"last-command-event",	"unread-command-char",	"global-map",
		"minibuffer-local-map",	"minor-mode-map-alist",	"wrong-type-argument",
		"buffer-file-name",		"mode-line-format",		"load-path",
		"kill-ring",			"kill-region",			"kill-ring-yank-pointer",
		"yank",				

		"coding-system-for-read",	"coding-system-for-write",	"euc-japan",
		"euc-jp",				"shift-jis",			"japanese-shift-jis",
		"sjis",					"junet",				"iso-2022-jp",
		"iso-2022-jp-2",		"ctext",				"x-ctext",
		"compound-text",	

		/*	for skk */
		"skk-server-host",		"skk-server-list",		"skk-portnum",
		"skk-process-okuri-early",	"j-henkan-key",		"j-search-key",
		"j-okuri-ari",			"j-henkan-okurigana",	"j-henkan-okuri-strictly",

		/* for skk10 */
		"skk-servers-list",
	} ;
	/*	default  buffer-local-symbol ȤѰդ륷ܥ롣
	 */
	static int	rnDefaultBufferSymbols []	= {
		LISPMGR_INDEX_BUFFER_FILE_NAME,
	} ;
	register const char**	pPtr ;
	register int			i ;
	TLispEntity*			pEntity ;
	TLispEntity*			pEntVoid ;
	TLispEntity*			pEntEmpty ;

	pPtr	= apReservedSymbols ;
	for (i = 0 ; i < NELEMENTS (apReservedSymbols) ; i ++) {
		if (TFAILED (lispMgr_InternSymbolA (pLispMgr, *pPtr, strlen (*pPtr), &pEntity)))
			return	False ;
		lispEntity_AddRef (pLispMgr, pEntity) ;
		pLispMgr->m_apEntReserved [i]	= pEntity ;
		pPtr	++ ;
	}
	if (TFAILED (lispMgr_CreateSymbolA (pLispMgr, "void", 4, &pEntVoid)))
		return	False ;
	pEntVoid->m_iType		= LISPENTITY_VOID ;
	pLispMgr->m_pEntVoid	= pEntVoid ;
	lispEntity_AddRef (pLispMgr, pEntVoid) ;
	if (TFAILED (lispMgr_CreateSymbolA (pLispMgr, "empty", 4, &pEntEmpty)))
		return	False ;
	pEntEmpty->m_iType		= LISPENTITY_EMPTY ;
	pLispMgr->m_pEntEmpty	= pEntEmpty ;

	/*	default  buffer-local ˤʤ symbol ϿƤߤΤȤ
	 *		buffer-file-name,
	 *	ꤷƤ롣
	 */
	for (i = 0 ; i < NELEMENTS (rnDefaultBufferSymbols) ; i ++) {
		pEntity	= pLispMgr->m_apEntReserved [rnDefaultBufferSymbols [i]] ;
		lispMgr_AddSymbolToLocalSymbols (pLispMgr, pEntity) ;
	}
	return	True ;
}

Boolean
lispMgr_collectOtherGarbage (
	register TLispManager*	pLispMgr,
	register TLispEntity**	ppEntTop)
{
	register TLispEntity*	pNextEntity ;
	register TLispEntity*	pEntity ;

	assert (pLispMgr != NULL) ;

	pEntity	= *ppEntTop ;
	while (pEntity != NULL) {
		pNextEntity	= pEntity->m_pRight ;
		if (pEntity->m_lReferCount == 0 &&
			pEntity->m_iMarker != pLispMgr->m_iMarker) {
			lispMgr_unregisterEntityFromList (pLispMgr, ppEntTop, pEntity) ;
			lispMgr_DestroyEntity (pLispMgr, pEntity) ;
		} else {
			pEntity->m_iMarker	= pLispMgr->m_iMarker ;
		}
		pEntity	= pNextEntity ;
	}
	return	True ;
}


Boolean
lispMgr_markEntityRecursively (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	lispMgr_markEntity (pLispMgr, pEntity) ;
	
	switch (pEntity->m_iType) {
	case	LISPENTITY_CONSCELL:
	{
		TLispEntity*	pCar ;
		TLispEntity*	pCdr ;

		lispEntity_GetCar (pLispMgr, pEntity, &pCar) ;
		lispEntity_GetCdr (pLispMgr, pEntity, &pCdr) ;
		if (TFAILED (lispMgr_markedEntityp (pLispMgr, pCar)))
			lispMgr_markEntityRecursively (pLispMgr, pCar) ;
		if (TFAILED (lispMgr_markedEntityp (pLispMgr, pCdr)))
			lispMgr_markEntityRecursively (pLispMgr, pCdr) ;
		break ;
	}
		
	case	LISPENTITY_VECTOR:
	{
		TLispEntity**	ppElement ;
		int				nElement ;

		lispEntity_GetVectorValue (pLispMgr, pEntity, &ppElement, &nElement) ;
		while (nElement > 0) {
			if (TFAILED (lispMgr_markedEntityp (pLispMgr, *ppElement)))
				lispMgr_markEntityRecursively (pLispMgr, *ppElement) ;
			ppElement	++ ;
			nElement	-- ;
		}
		break ;
	}
		
	default:
		break ;
	}
	return	True ;
}

#if defined (DEBUG)
void
lispMgr_CheckEntity (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	register TLispEntity*	pNode ;

	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;

	fprintf (stderr, " Searching: Entity(%p),Count(%ld),Type(%d) ...",
			 pEntity, pEntity->m_lReferCount, pEntity->m_iType) ;

	switch (pEntity->m_iType) {
	case	LISPENTITY_INTEGER:
	case	LISPENTITY_SYMBOL:
		fprintf (stderr, "not check\n") ;
		break ;

	default:
		pNode	= pLispMgr->m_pEntMiscListTop ;
		while (pNode != NULL) {
			if (pNode == pEntity) {
				fprintf (stderr, "hit\n") ;
				//lispMgr_CheckMark (pLispMgr, pEntity) ;
				return ;
			}
			pNode	= pNode->m_pRight ;
		}
		fprintf (stderr, "miss\n") ;
		break ;
	}
	return ;
}

void
lispMgr_CheckMark (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	register TLispEntity*	pNode ;

	if (pEntity->m_iType != 2)
		return ;

	fprintf (stderr, "Target(%p): Mark = %d -> ", 
			 pEntity, pEntity->m_iMarker) ;
	pLispMgr->m_iMarker	++ ;
	pNode	= pLispMgr->m_pEntMiscListTop ;
	while (pNode != NULL) {
		if (pNode->m_lReferCount > 0) 
			lispMgr_markEntityRecursively (pLispMgr, pNode) ;
		pNode	= pNode->m_pRight ;
	}
	fprintf (stderr, "%d/%d\n",
			 pEntity->m_iMarker, pLispMgr->m_iMarker) ;
	return ;
}
#endif

