/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "local.h"
#include <stdio.h>
#include <assert.h>
#include "lispmgrp.h"

#define	lispEntity_GetLongPtr(ptr)	((long *)((TLispEntity *)(ptr) + 1))
#define	lispEntity_GetFloatPtr(ptr)	((float *)((TLispEntity *)(ptr) + 1))

static	Boolean	lispMgr_searchInteger         (TLispManager*, long, TLispEntity** const) ;
static	Boolean	lispMgr_registerInteger       (TLispManager*, TLispEntity*) ;
static	Boolean	lispMgr_unregisterInteger     (TLispManager*, TLispEntity*) ;
static	Boolean	lispMgr_collectIntegerGarbage (TLispManager*, TLispEntity*) ;
static	Boolean	lispMgr_mathCompareNumber	(TLispManager*, TLispEntity*, TLispEntity*, Boolean (*)(TLispNumber*,TLispNumber*), Boolean*) ;
static	Boolean	lispMgr_mathTestGreaterThan	(TLispNumber*, TLispNumber*) ;
static	Boolean	lispMgr_mathTestGreaterEqual(TLispNumber*, TLispNumber*) ;
static	Boolean	lispMgr_mathTestLessThan	(TLispNumber*, TLispNumber*) ;
static	Boolean	lispMgr_mathTestLessEqual	(TLispNumber*, TLispNumber*) ;
static	Boolean	lispMgr_mathTestEqual		(TLispNumber*, TLispNumber*) ;

/*	inline functions */
inline	int
lispMgr_createIntegerHashkey (register long lValue)
{
#if defined (DEBUG)
	int	iRetval	= ((unsigned)lValue) % LISPMGR_SIZE_INTEGER_HASH ;
	assert (0 <= iRetval && iRetval < LISPMGR_SIZE_INTEGER_HASH) ;
	return	iRetval ;
#else
	return	((unsigned)lValue) % LISPMGR_SIZE_INTEGER_HASH ;
#endif
}


/*	global functions */
Boolean
lispMgr_CreateInteger (
	register TLispManager*			pLispMgr,
	register long					lValue,
	register TLispEntity** const	ppEntReturn)
{
	TLispEntity*	pEntity ;

	assert (pLispMgr    != NULL) ;
	assert (ppEntReturn != NULL) ;

#if defined (DEBUG_LV99)
	fprintf (stderr, "Garbage collecting ...") ;
	fflush (stderr) ;
	lispMgr_CollectGarbage (pLispMgr) ;
	fprintf (stderr, "done.\n") ;
#endif	
	if (TFAILED (lispMgr_searchInteger (pLispMgr, lValue, &pEntity)))
		return	False ;
	
	if (pEntity == NULL) {
		register long*	plValue ;

		if (TFAILED (lispMgr_AllocateEntity (pLispMgr, sizeof (long), &pEntity)))
			return	False ;

		pEntity->m_iType	= LISPENTITY_INTEGER ;
		plValue				= lispEntity_GetLongPtr (pEntity) ;
		*plValue			= lValue ;
		lispMgr_registerInteger (pLispMgr, pEntity) ;
	}

	*ppEntReturn	= pEntity ;
	return	True ;
}

Boolean
lispMgr_CreateFloat (
	register TLispManager*			pLispMgr,
	register float					fValue,
	register TLispEntity** const	ppEntReturn)
{
	TLispEntity*		pEntity ;
	register float*		pfValue ;

	assert (pLispMgr    != NULL) ;
	assert (ppEntReturn != NULL) ;

#if defined (DEBUG_LV99)
	fprintf (stderr, "Garbage collecting ...") ;
	fflush (stderr) ;
	lispMgr_CollectGarbage (pLispMgr) ;
	fprintf (stderr, "done.\n") ;
#endif	
	if (TFAILED (lispMgr_AllocateEntity (pLispMgr, sizeof (float), &pEntity)))
		return	False ;
		
	pEntity->m_iType		= LISPENTITY_FLOAT ;
	pEntity->m_lReferCount	= 0 ;	/* ľϻȥ󥿤 0*/
	pfValue					= lispEntity_GetFloatPtr (pEntity) ;
	*pfValue				= fValue ;
	lispMgr_RegisterMisc (pLispMgr, pEntity) ;
	*ppEntReturn			= pEntity ;
	return	True ;
}

Boolean
lispMgr_CollectIntegerGarbage (
	register TLispManager* pLispMgr)
{
	register int		i ;
	
	for (i = 0 ; i < LISPMGR_SIZE_INTEGER_HASH ; i ++)
		lispMgr_collectIntegerGarbage (pLispMgr, pLispMgr->m_apIntegerListTop [i]) ;
	return	True ;
}

Boolean
lispMgr_MathGreaterThan (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pNum1,
	register TLispEntity*	pNum2,
	register Boolean*		pfResult)
{
	return	lispMgr_mathCompareNumber (pLispMgr, pNum1, pNum2, &lispMgr_mathTestGreaterThan, pfResult) ;
}

Boolean
lispMgr_MathGreaterEqual (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pNum1,
	register TLispEntity*	pNum2,
	register Boolean*		pfResult)
{
	return	lispMgr_mathCompareNumber (pLispMgr, pNum1, pNum2, &lispMgr_mathTestGreaterEqual, pfResult) ;
}

Boolean
lispMgr_MathLessThan (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pNum1,
	register TLispEntity*	pNum2,
	register Boolean*		pfResult)
{
	return	lispMgr_mathCompareNumber (pLispMgr, pNum1, pNum2, &lispMgr_mathTestLessThan, pfResult) ;
}

Boolean
lispMgr_MathLessEqual (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pNum1,
	register TLispEntity*	pNum2,
	register Boolean*		pfResult)
{
	return	lispMgr_mathCompareNumber (pLispMgr, pNum1, pNum2, &lispMgr_mathTestLessEqual, pfResult) ;
}

Boolean
lispMgr_MathEqual (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pNum1,
	register TLispEntity*	pNum2,
	register Boolean*		pfResult)
{
	return	lispMgr_mathCompareNumber (pLispMgr, pNum1, pNum2, &lispMgr_mathTestEqual, pfResult) ;
}

/*
 *	private functions
 */
Boolean
lispMgr_searchInteger (
	register TLispManager*			pLispMgr,
	register long					lSearchKey,
	register TLispEntity** const	ppEntReturn)
{
	register TLispEntity*	pEntity ;
	register long*			plValue ;
	register int			iHashkey ;

	assert (pLispMgr    != NULL) ;
	assert (ppEntReturn != NULL) ;

	iHashkey	= lispMgr_createIntegerHashkey (lSearchKey) ;
	pEntity		= pLispMgr->m_apIntegerListTop [iHashkey] ;

	while (pEntity != NULL) {
		assert (pEntity->m_iType == LISPENTITY_INTEGER) ;
		plValue	= lispEntity_GetLongPtr (pEntity) ;
		if (*plValue == lSearchKey) {
			*ppEntReturn	= pEntity ;
			return	True ;
		}
		if (*plValue > lSearchKey) 
			break ;
		pEntity	= pEntity->m_pRight ;
	}
	*ppEntReturn	= NULL ;
	return	True ;
}

Boolean
lispMgr_registerInteger (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pNewEntity)
{
	register const long*	plValue ;
	register long			lNewValue ;
	register int			iHashkey ;
	register TLispEntity*	pPrevEntity ;
	register TLispEntity*	pNextEntity ;

	assert (pLispMgr   != NULL) ;
	assert (pNewEntity != NULL) ;
	assert (pNewEntity->m_iType == LISPENTITY_INTEGER) ;

	plValue		= lispEntity_GetLongPtr (pNewEntity) ;
	lNewValue	= *plValue ;
	iHashkey	= lispMgr_createIntegerHashkey (lNewValue) ;

	pNextEntity	= pLispMgr->m_apIntegerListTop [iHashkey] ;
	pPrevEntity	= NULL ;
	while (pNextEntity != NULL) {
		assert (pNextEntity->m_iType == LISPENTITY_INTEGER) ;
			
		plValue		= lispEntity_GetLongPtr (pNextEntity) ;
		if (*plValue > lNewValue)
			break ;
			
		pPrevEntity	= pNextEntity ;
		pNextEntity	= pNextEntity->m_pRight ;
	}

	if (pPrevEntity == NULL) {
		pLispMgr->m_apIntegerListTop [iHashkey]	= pNewEntity ;
	} else {
		pPrevEntity->m_pRight	= pNewEntity ;
	}
	if (pNextEntity != NULL) {
		pNextEntity->m_pLeft	= pNewEntity ;
	}

	pNewEntity->m_pLeft		= pPrevEntity ;
	pNewEntity->m_pRight	= pNextEntity ;

	return	True ;
}


Boolean
lispMgr_unregisterInteger (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	assert (pLispMgr != NULL) ;
	assert (pEntity  != NULL) ;
	assert (pEntity->m_iType == LISPENTITY_INTEGER) ;

	if (pEntity->m_pLeft != NULL) {
		pEntity->m_pLeft->m_pRight	= pEntity->m_pRight ;
	} else {
		register const long*	plValue ;
		register int			iHashkey ;

		plValue		= lispEntity_GetLongPtr (pEntity) ;
		iHashkey	= lispMgr_createIntegerHashkey (*plValue) ;
		assert (pLispMgr->m_apIntegerListTop [iHashkey] == pEntity) ;
		pLispMgr->m_apIntegerListTop [iHashkey] = pEntity->m_pRight ;
	}

	if (pEntity->m_pRight != NULL) {
		pEntity->m_pRight->m_pLeft	= pEntity->m_pLeft ;
	} 

	pEntity->m_pLeft	= NULL ;
	pEntity->m_pRight	= NULL ;

	return	True ;
}

Boolean
lispMgr_collectIntegerGarbage (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity)
{
	register TLispEntity*	pNextEntity ;

	assert (pLispMgr != NULL) ;

	while (pEntity != NULL) {
		assert (pEntity != NULL) ;
		assert (pEntity->m_iType == LISPENTITY_INTEGER) ;

		pNextEntity	= pEntity->m_pRight ;

		if (pEntity->m_lReferCount == 0 && pEntity->m_iMarker != pLispMgr->m_iMarker) {
			lispMgr_unregisterInteger (pLispMgr, pEntity) ;
			lispMgr_DestroyEntity (pLispMgr, pEntity) ;
		} else {
			pEntity->m_iMarker	= pLispMgr->m_iMarker ;
		}
		pEntity	= pNextEntity ;
	}
	return	True ;
}

Boolean
lispMgr_mathCompareNumber (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pNum1,
	register TLispEntity*	pNum2,
	register Boolean		(*pCompFunc)(TLispNumber*,TLispNumber*),
	register Boolean*		pfResult)
{
	TLispNumber	num1, num2 ;

	if (TFAILED (lispEntity_GetNumberValueOrMarkerPosition (pLispMgr, pNum1, &num1)) ||
		TFAILED (lispEntity_GetNumberValueOrMarkerPosition (pLispMgr, pNum2, &num2))) 
		return	False ;
	*pfResult	= (*pCompFunc)(&num1, &num2) ;
	return	True ;
}

Boolean
lispMgr_mathTestGreaterThan (
	register TLispNumber*	pNum1,
	register TLispNumber*	pNum2)
{
	register Boolean	fGreaterThan ;

	if (pNum1->m_fFloatp) {
		if (pNum2->m_fFloatp) {
			fGreaterThan	= (pNum1->m_Value.m_fFloat > pNum2->m_Value.m_fFloat) ;
		} else {
			fGreaterThan	= (pNum1->m_Value.m_fFloat > pNum2->m_Value.m_lLong) ;
		}
	} else {
		if (pNum2->m_fFloatp) {
			fGreaterThan	= (pNum1->m_Value.m_lLong > pNum2->m_Value.m_fFloat) ;
		} else {
			fGreaterThan	= (pNum1->m_Value.m_lLong > pNum2->m_Value.m_lLong) ;
		}
	}
	return	fGreaterThan ;
}

Boolean
lispMgr_mathTestGreaterEqual (
	register TLispNumber*	pNum1,
	register TLispNumber*	pNum2)
{
	register Boolean	fGreaterEqual ;

	if (pNum1->m_fFloatp) {
		if (pNum2->m_fFloatp) {
			fGreaterEqual	= (pNum1->m_Value.m_fFloat >= pNum2->m_Value.m_fFloat) ;
		} else {
			fGreaterEqual	= (pNum1->m_Value.m_fFloat >= pNum2->m_Value.m_lLong) ;
		}
	} else {
		if (pNum2->m_fFloatp) {
			fGreaterEqual	= (pNum1->m_Value.m_lLong >= pNum2->m_Value.m_fFloat) ;
		} else {
			fGreaterEqual	= (pNum1->m_Value.m_lLong >= pNum2->m_Value.m_lLong) ;
		}
	}
	return	fGreaterEqual ;
}

Boolean
lispMgr_mathTestLessThan (
	register TLispNumber*	pNum1,
	register TLispNumber*	pNum2)
{
	register Boolean	fLessThan ;

	if (pNum1->m_fFloatp) {
		if (pNum2->m_fFloatp) {
			fLessThan	= (pNum1->m_Value.m_fFloat < pNum2->m_Value.m_fFloat) ;
		} else {
			fLessThan	= (pNum1->m_Value.m_fFloat < pNum2->m_Value.m_lLong) ;
		}
	} else {
		if (pNum2->m_fFloatp) {
			fLessThan	= (pNum1->m_Value.m_lLong < pNum2->m_Value.m_fFloat) ;
		} else {
			fLessThan	= (pNum1->m_Value.m_lLong < pNum2->m_Value.m_lLong) ;
		}
	}
	return	fLessThan ;
}

Boolean
lispMgr_mathTestLessEqual (
	register TLispNumber*	pNum1,
	register TLispNumber*	pNum2)
{
	register Boolean	fLessEqual ;

	if (pNum1->m_fFloatp) {
		if (pNum2->m_fFloatp) {
			fLessEqual	= (pNum1->m_Value.m_fFloat <= pNum2->m_Value.m_fFloat) ;
		} else {
			fLessEqual	= (pNum1->m_Value.m_fFloat <= pNum2->m_Value.m_lLong) ;
		}
	} else {
		if (pNum2->m_fFloatp) {
			fLessEqual	= (pNum1->m_Value.m_lLong <= pNum2->m_Value.m_fFloat) ;
		} else {
			fLessEqual	= (pNum1->m_Value.m_lLong <= pNum2->m_Value.m_lLong) ;
		}
	}
	return	fLessEqual ;
}

Boolean
lispMgr_mathTestEqual (
	register TLispNumber*	pNum1,
	register TLispNumber*	pNum2)
{
	register Boolean	fEqual ;

	if (pNum1->m_fFloatp) {
		if (pNum2->m_fFloatp) {
			fEqual	= (pNum1->m_Value.m_fFloat == pNum2->m_Value.m_fFloat) ;
		} else {
			fEqual	= (pNum1->m_Value.m_fFloat == (float)pNum2->m_Value.m_lLong) ;
		}
	} else {
		if (pNum2->m_fFloatp) {
			fEqual	= ((float)pNum1->m_Value.m_lLong == pNum2->m_Value.m_fFloat) ;
		} else {
			fEqual	= (pNum1->m_Value.m_lLong == pNum2->m_Value.m_lLong) ;
		}
	}
	return	fEqual ;
}

