/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "local.h"
#include <stdio.h>
#include <assert.h>
#include "lmachinep.h"
#include "cstring.h"

static	TLMRESULT	lispMachineState_makeIntArgStart (TLispMachine*, TLispEntity*, LMCMDINFO const*, TLMRESULT (*)(TLispMachine *)) ;
static	TLMRESULT	lispMachineState_makeIntArg (TLispMachine*) ;
static	TLMRESULT	lispMachineState_makeIntArgWithMinibuf  (TLispMachine*) ;
static	TLMRESULT	lispMachineState_makeIntArgWithMinibuf1 (TLispMachine*) ;
static	TLMRESULT	lispMachineState_makeIntArgVar (TLispMachine*) ;
static	TLMRESULT	lispMachineState_makeIntArgNum (TLispMachine*) ;
static	TLMRESULT	lispMachineState_callIntBuiltin (TLispMachine*) ;
static	TLMRESULT	lispMachineState_callIntLambda1 (TLispMachine*) ;
static	TLMRESULT	lispMachineState_callIntLambda2 (TLispMachine*) ;

static	Boolean		lispMachine_getIntArg (TLispManager*, TLispEntity*, TLispEntity**) ;
static	Boolean		lispMachine_addEntityToArglist (TLispMachine*, TLispEntity**, TLispEntity*) ;
static	Boolean		lispMachine_commandp (TLispMachine*, TLispEntity*, TLispEntity**) ;

/*
 *	Interactiveness ι eval, funcall, apply, call-interactively
 *	ǹԤɤǤeval  (xxx) Ƚ񤤤˼ưŪɾ
 *	Τ (eval XXX) Ƚ񤤤Ƽ¹ԤΤξޤǤ롣
 */

/*
 *	Code letters available are:
 *	a -- ؿ̾ ؿäƤ륷ܥ
 *		ؿ̾Ԥˤʤ롣minibuffer ǡ
 *	b -- ¸ߤƤХåե̾
 *		(default: *scratch) Τ褦˽Фơbuffer ̾Ϥˤʤ롣
 *	B -- 餯¸ߤƤʤХåե̾
 *	c -- 饯 (no input method is used).
 *		1ʸԤˤʤ롣λˤ input method ϻȤʤ
 *	C -- ޥ̾ interactive function äƤ륷ܥ롣
 *		ޥ̾Ԥˤʤ롣minibuffer ǡ
 *	d -- ͤȤƤΥݥȤ͡I/O Ϥʤ
 *	D -- ǥ쥯ȥ̾
 *	e -- Υޥɤѥ᡼줿٥ȡ
 *		⤷ٰʾȤ줿Τʤ顢N ܤ `e'  N ܤΥѥ᡼
 *		줿٥Ȥ֤
 *		ϿޤϥܥΥ٥Ȥ򥹥åפ롣
 *		(㤨Сkey event ʤɤ command ƤƤĤȤ)
 *	f -- ¸ߤƤե̾
 *	F -- ¿ʬ¸ߤƤʤե̾
 *	i -- ̵뤵롣ʤ nilϤϤʤ
 *	k -- 󥹡(饹ȥ٥Ȥ downcase )
 *	K -- 륭󥹡
 *	m -- ͤȤƤΥޡ͡I/O Ϥʤ
 *	M -- Ǥդʸ󡣸ߤ input method Ѿ롣
 *	n -- ߥ˥ХåեȤäɤ߹ޤ줿͡
 *	N -- Raw prefix arg ⤷ʤä顢`n' Τ褦ư롣
 *	p -- Ѵ줿ץեI/O Ϥʤ
 *	P -- ȤΤޤޤΥץեI/O Ϥʤ
 *	r -- ꡼󡧥ݥȤȥޡ2ĤΰȤƤȤ롣
 *		I/O Ϥʤ
 *	s -- Ǥդʸinput method ϷѾʤ
 *	S -- ǤդΥܥ롣
 *	v -- ѿ̾user-variable-p ǤȤΥܥ롣
 *	x -- ɤ߹ޤ줿ΤɾƤʤ Lisp 
 *	X -- ɤ߹ޤ줫ɾ줿 Lisp 
 *	z -- Coding system.
 *	Z -- Coding system, nil if no prefix arg.
 *	äʸ * ǤϤޤäƤ顢Хåե read-only 
 *	饨顼 signal 롣
 *	ϰɤޤ˵롣
 *	⤷ʸ `@' ǤϤޤäƤ顢Emacs ϺǽΥޥ
 *	åͿ줿ޥɤ invoke 륭󥹤õ
 *	Emacs Ǥդΰɤޤ˥ɥ򤹤롣
 *	`@'  `*' ξȤäƤ顢о줹֤˽
 *
 *	(interactive ARGS)
 *
 *
 *	interactive μ̤ƤɬפϤʤʤΤǡ
 *	ɬפʤΤΤ߼Ƥ롣
 */
TLMRESULT
lispMachineState_Interactive (
	register TLispMachine* pLM)
{
	return	LMR_RETURN ;
}

/*	built-in function:
 *		(interactive-p)
 *
 *	δؿƤؿ interactive ˸ƤӽФƤ
 *	ʤ t ֤Ϥδؿ call-interactively ˤäƸƤ
 *	Ф줿Ȥ̣롣
 */
TLMRESULT
lispMachineState_Interactivep (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pArglist ;
	TLispEntity*			pEntRetval ;
	Boolean					fInteractive ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetInteractive (pLM, &fInteractive) ;
	if (TSUCCEEDED (fInteractive)) {
		lispMgr_CreateT   (pLispMgr, &pEntRetval) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	}
	assert (pEntRetval != NULL) ;
#if defined (DEBUG_LV99) 
	fprintf (stderr, "(interactive-p) = ") ;
	lispEntity_Print (pLispMgr, pEntRetval) ;
	fprintf (stderr, "\n") ;
#endif
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*
 *	(commandp FUNCTION)
 *
 *	⤷ FUNCTION  interactive ƤӽФΤѰդ򤷤Ƥ
 *	ʤ T ֤ϡɤΤ褦˰ɤɤΤε
 *	 FUNCTION äƤ뤳Ȥ̣롣nil ʤ function 
 *	Ƥʤ symbol ⤷ invalid  function Ǥ롣
 */
TLMRESULT
lispMachineState_Commandp (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pEntArglist ;
	TLispEntity*			pEntFunc ;
	TLispEntity*			pEntRetval ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntFunc))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispMachine_commandp (pLM, pEntFunc, &pEntRetval)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*
 *	(prefix-numeric-value RAW)
 *
 *	raw prefix  RAW οȤƤΰ̣֤
 */
TLMRESULT
lispMachineState_PrefixNumericValue (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*			pEntArglist ;
	TLispEntity*			pEntRAW ;
	TLispEntity*			pEntRetval ;

	assert (pLM != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntRAW) ;
	if (TSUCCEEDED (lispEntity_Numberp (pLispMgr, pEntRAW))) {
		pEntRetval	= pEntRAW ;
	} else {
		if (TFAILED (lispEntity_Symbolp (pLispMgr, pEntRAW))) {
			register TLispEntity*	pEntMinus ;
			
			pEntMinus	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_MINUS) ;
			if (TSUCCEEDED (lispEntity_Eq (pLispMgr, pEntMinus, pEntRAW))) {
				lispMgr_CreateInteger (pLispMgr, -1, &pEntRetval) ;
				goto	exitfunc ;
			}
		} else if (TSUCCEEDED (lispEntity_Consp (pLispMgr, pEntRAW))) {
			lispEntity_GetCar (pLispMgr, pEntRAW, &pEntRetval) ;
			goto	exitfunc ;
		}
		lispMgr_CreateInteger (pLispMgr, 1, &pEntRetval) ;
	}
  exitfunc:
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*	built-in function:
 *		(command-execute CMD &optional RECORD-FLAG KEYS SPECIAL)
 *
 *	CMD 򥨥ǥΥޥɤȤƼ¹Ԥ롣CMD  commandp ­
 *	ܥǤʤФʤʤʹߤΰϺ̵뤹롣ȡ
 *	Ū SPECIAL  nil ꤷƤΤǡprefix argument ϥ
 *	롣
 */
TLMRESULT
lispMachineState_CommandExecute (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pEntArglist ;
	TLispEntity*			pEntCMD ;
	register TLispEntity*	pEntNil ;
	TLispEntity*			pEntNewArglist ;

	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar    (pLispMgr, pEntArglist, &pEntCMD))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	pEntNil	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_NIL) ;
	if (TFAILED (lispMgr_CreateConscell (pLispMgr, pEntCMD, pEntNil, &pEntNewArglist)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntNewArglist) ;
	lispMachineCode_SetState (pLM, &lispMachineState_CallInteractively) ;
	return	LMR_CONTINUE ;
}

/*
 *	(call-interactively FUNCTION &optional RECORD-FLAG KEYS)
 *
 *	ŪƤӽФΥڥå˽äưѰդơFUNCTION ƤӽФ
 *	FUNCTION ֤֤ͤ롣
 *	ؿϤΰɤΤ褦ɤΤλͤޤǤʤФʤʤ
 *	桼ؿξˤϡϴؿ body ΰŷդ˴ؿ
 *	`interactive' θƤӽФ֤ȤǤʤ롣
 *
 *	ץ2ܤΰ RECORD FLAG  non-nil ȡΥޥ
 *	̵˥ޥ put 뤳Ȥ̣롣Ǥʤ
 *	minibuffer Ȥäưɤʤ顢put 롣
 *
 *
 * ()
 *	ʳ prefix-argument ꤵƤС
 *	ȤϤˤʤġĤɤ˺ƤΤʡ
 */
TLMRESULT
lispMachineState_CallInteractively (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pEntArglist ;
	TLispEntity*			pFUNCTION ;
	TLispEntity*			pRECORDFLAG ;
	TLispEntity*			pEntString ;
	TLispEntity*			pEntValue ;
	LMCMDINFO const*		pProcInfo ;

	assert (pLM != NULL) ;

	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

#if defined (DEBUG_LV99) 
	fprintf (stderr, "call-interactive ()/0\n") ;
#endif
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pFUNCTION)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pEntArglist, &pRECORDFLAG)) ||
		TFAILED (lispEntity_Symbolp (pLispMgr, pFUNCTION))) 
		goto	error ;
#if defined (DEBUG_LV99) 
	fprintf (stderr, "call-interactive ()/1\n") ;
#endif

	/*
	 *	FUNCTION ΥåԤ
	 */
	if (TFAILED (lispMachine_GetSymbolFunctionValue (pLM, pFUNCTION, &pEntValue)))
		goto	error ;
#if defined (DEBUG_LV99) 
	fprintf (stderr, "call-interactive ()/1.5\n") ;
#endif
	if (TSUCCEEDED (lispEntity_Symbolp (pLispMgr, pEntValue)) &&
		TFAILED (lispMachine_GetFinalSymbolFunctionValue (pLM, pEntValue, &pEntValue))) {
#if defined (DEBUG) || 0
		fprintf (stderr, "symbol's function: ") ;
		lispEntity_Print (pLispMgr, pFUNCTION) ;
		fprintf (stderr, "is void.\n") ;
#endif
		goto	error ;
	}
	if (TSUCCEEDED (lispEntity_Consp (pLispMgr, pEntValue))) {
		TLispEntity*	pIntArg ;
		TLispEntity*	pEntSubr ;

		/*	interactive õinteractive ƬǤɾʤ
		 *	ոȡinteractive Ƭϡɾפ롣ա*/ 
		if (TFAILED (lispMachine_getIntArg (pLispMgr, pEntValue, &pIntArg))) {
			/*	ޥɰʳϼ¹ԤǤʤ*/
#if defined (DEBUG)  || 0
			fprintf (stderr, "symbol's function: ") ;
			lispEntity_Print (pLispMgr, pFUNCTION) ;
			fprintf (stderr, " is not interactive.\n") ;
#endif
			lispMachineCode_SetError (pLM) ;
			return	LMR_RETURN ;
		}
		/*	lambda ʤΤǥ顼ˤʤ뤳ȤϤʤȻפΤġ*/
		if (TFAILED (lispMachine_GetSymbolFunctionValue (pLM, pEntValue, &pEntSubr)) ||
			TFAILED (lispSubr_GetProc (pLispMgr, pEntSubr, &pProcInfo))) {
			goto	error ;
		}
#if defined (DEBUG) || 0
		fprintf (stderr, "symbol-function = ") ;
		lispEntity_Print (pLispMgr, pEntValue) ;
		fprintf (stderr, "\n") ;
#endif

		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntValue) ;
		/*	interactive ΰʸ󤫡 ʸǤʤСɾ롣
		 *	ɾ̤ϥꥹȤǤʤФʤʤ*/
		if (TSUCCEEDED (lispEntity_Stringp (pLispMgr, pIntArg))) {
			lispMachineCode_PushLReg (pLM, LM_LREG_ACC) ;
			return	lispMachineState_makeIntArgStart (pLM, pIntArg, pProcInfo, &lispMachineState_callIntLambda1) ;
		} else {
			lispMachineCode_PushLReg (pLM, LM_LREG_ACC) ;
			lispMachineCode_PushVReg (pLM, LM_VREG_1) ;
			lispMachineCode_SetVRegP (pLM, LM_VREG_1, (void *)pProcInfo) ;
			lispMachineCode_Evaln (pLM, pIntArg, &lispMachineState_callIntLambda2) ;
			return	LMR_CONTINUE ;
		}
	}
#if defined (DEBUG)
	fprintf (stderr, "(call-interactive ") ;
	lispEntity_Print (pLispMgr, pFUNCTION) ;
	fprintf (stderr, ")\n") ;
#endif
	/*	ؿ ``'' õФΥå򤹤롣*/
	if (TFAILED (lispSubr_GetProc (pLispMgr, pEntValue, &pProcInfo)) ||
		pProcInfo->m_pInteractive == NULL) 
		goto	error ;
#if defined (DEBUG)
	fprintf (stderr, "call-interactive ()/3\n") ;
#endif
	if (TFAILED (lispMgr_CreateString (pLispMgr, pProcInfo->m_pInteractive, Cstrlen (pProcInfo->m_pInteractive), &pEntString)))
		return	LMR_ERROR ;
#if defined (DEBUG)
	fprintf (stderr, "call-interactive ()/4\n") ;
#endif
	return	lispMachineState_makeIntArgStart (pLM, pEntString, pProcInfo, &lispMachineState_callIntBuiltin) ;

  error:
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_makeIntArgStart (
	register TLispMachine*		pLM,
	register TLispEntity*		pEntString,
	register LMCMDINFO const*	pProcInfo,
	TLMRESULT					(*pReturnState)(TLispMachine *))
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntNil ;
	TLispEntity*	pEntInteger ;

	assert (pLM != NULL) ;
	assert (pEntString != NULL) ;
	assert (pProcInfo != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;

	lispMachineCode_PushLReg  (pLM, LM_LREG_1) ;
	lispMachineCode_PushLReg  (pLM, LM_LREG_3) ;
	lispMachineCode_PushLReg  (pLM, LM_LREG_4) ;
	lispMachineCode_PushLReg  (pLM, LM_LREG_5) ;
	lispMachineCode_PushVReg  (pLM, LM_VREG_1) ;
	lispMachineCode_SetVRegP  (pLM, LM_VREG_1, (void *)pProcInfo) ;
	lispMachineCode_PushState (pLM, pReturnState) ;
	lispMachineCode_SetState (pLM, &lispMachineState_makeIntArg) ;

	lispMgr_CreateNil (pLispMgr, &pEntNil) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_1, pEntNil) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_3, pEntString) ;
	if (TFAILED (lispMgr_CreateInteger (pLM->m_pLispMgr, 0, &pEntInteger)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_4, pEntInteger) ;
	return	LMR_CONTINUE ;
}

/*
 *	REG_1 ˤϡֺΰꥹȡפ
 *	REG_3 ˤϡֲϤƤ ENTITY (ʸ)פ
 *	REG_4 ˤϡֲʸޤʸɤ()פ
 *	롣
 */
TLMRESULT
lispMachineState_makeIntArg (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntString ;
	TLispEntity*	pStrIndexEntity ;
	TLispEntity*	pEntNil ;
	TLispEntity*	pBufferEntity ;
	TLispEntity*	pEntity ;
	TLispEntity*	pNewNode ;
	const Char*		pString ;
	const Char*		ptr ;
	int				nLength ;
	long			lIndex ;
	register int	nRest ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

#if defined (DEBUG) 
	fprintf (stderr, "make-int-arg ()/0\n") ;
#endif
	/*	쥸ͤȴФ*/
	lispMachineCode_GetLReg (pLM, LM_LREG_1, &pEntArglist) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_3, &pEntString) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_4, &pStrIndexEntity) ;

	/*	interactive (ʸ)ȲޤǤϤȴФ*/
	lispEntity_GetStringValue  (pLispMgr, pEntString, &pString, &nLength) ;
	lispEntity_GetIntegerValue (pLispMgr, pStrIndexEntity, &lIndex) ; 

	/*	餯ˤѤǤ Entity ѰդƤ*/
	lispMgr_CreateNil (pLispMgr, &pEntNil) ;
	lispMachineCode_GetCurrentBuffer (pLM, &pBufferEntity) ;

	/*	interactive ɾ (ǽ餫 or Ƴ) */
	ptr		= pString + lIndex ;
	nRest	= nLength - lIndex ;
	while (nRest > 0) {
		switch (*ptr) {
		case	'*':	/* read-only buffer ξĺϴطʤ*/
			ptr	  ++ ;
			nRest -- ;
			continue ;

		case	'B':
		case	'b':
		case	'c':
		case	'e':

		case	'P':	/* P -- ȤΤޤޤΥץե*/
			pEntity	= pEntNil ;
			ptr		++ ;
			nRest	-- ;
			break ;
		case	'p':	/* p -- Ѵ줿ץե*/
			lispMgr_CreateInteger (pLispMgr, 1, &pEntity) ;
			ptr		++ ;
			nRest	-- ;
			break ;
		case	'r':	/* r -- ꡼󡧥ݥȤȥޡ2ĤΰȤơ*/
		{
			TLispEntity*	apMarker [2] ;
			lispBuffer_PointMarker (pLispMgr, pBufferEntity, &apMarker [0]) ;
			lispBuffer_MarkMarker  (pLispMgr, pBufferEntity, &apMarker [1]) ;
			lispMgr_CreateList (pLispMgr, apMarker, 2, &pEntity) ;
			ptr		++ ;
			nRest	-- ;
			break ;
		}

		case	'n':	/* */
		{
			lispMachineCode_PushState (pLM, &lispMachineState_makeIntArgNum) ;
			lispMachineCode_SetState  (pLM, &lispMachineState_makeIntArgWithMinibuf) ;
			goto	exit_loop ;
		}

		case	'v':	/* v -- ѿ̾*/
		{
			lispMachineCode_PushState (pLM, &lispMachineState_makeIntArgVar) ;
			lispMachineCode_SetState  (pLM, &lispMachineState_makeIntArgWithMinibuf) ;
			goto	exit_loop ;
		}

		default:
			lispMachineCode_SetError (pLM) ;
			return	LMR_RETURN ;
		}

		/*	 Node ɲä롣*/
		lispEntity_AddRef (pLispMgr, pEntity) ;
		if (TFAILED (lispMgr_CreateConscell (pLispMgr, pEntity, pEntNil, &pNewNode)))
			return	LMR_ERROR ;
		lispEntity_Release (pLispMgr, pEntity) ;

		if (TFAILED (lispMachine_addEntityToArglist (pLM, &pEntArglist, pNewNode))) {
			lispMachineCode_SetError (pLM) ;
			return	LMR_RETURN ;
		}
	}
 exit_loop:
	/*	LM_LREG_1 ꤷľɬפϤʤϡSetCdr ѤΤᡣ*/
	/*	λȽԤ*/
	if (nRest == 0) {
		return	LMR_RETURN ;
	} else {
		/*	Index ľ*/
		lispMgr_CreateInteger (pLispMgr, nLength - nRest, &pStrIndexEntity) ;
		lispMachineCode_SetLReg (pLM, LM_LREG_4, pStrIndexEntity) ;
		return	LMR_CONTINUE ;
	}
}

TLMRESULT
lispMachineState_makeIntArgWithMinibuf (
	register TLispMachine* pLM)
{
	TLispEntity*			pEntString ;
	TLispEntity*			pEntStrIndex ;
	const Char*				pString ;
	long					lIndex ;
	int						nLength ;
	register TLispManager*	pLispMgr ;
	register const Char*	pStrPrompt ;
	register const Char*	ptr ;
	register int			nLenPrompt, nRest ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_3, &pEntString) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_4, &pEntStrIndex) ;

	/*	interactive (ʸ)ȲޤǤϤȴФ*/
	lispEntity_GetStringValue  (pLispMgr, pEntString, &pString, &nLength) ;
	lispEntity_GetIntegerValue (pLispMgr, pEntStrIndex, &lIndex) ; 

	ptr			= pString + lIndex + 1;
	nRest		= nLength - lIndex ;
	pStrPrompt	= ptr ;
	while (nRest > 0 && *ptr != '\n') {
		ptr		++ ;
		nRest	-- ;
	}
	nLenPrompt	= ptr - pStrPrompt ;
	if (nRest > 0 && *ptr == '\n') {
		ptr		++ ;
		nRest	-- ;
	}
	lIndex		= ptr - pString ;

	/*	Index ľ*/
	lispMgr_CreateInteger (pLispMgr, nLength - nRest, &pEntStrIndex) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_4, pEntStrIndex) ;

	if (TFAILED (lispMgr_CreateString (pLispMgr, pStrPrompt, nLenPrompt, &pEntString)))
		return	LMR_ERROR ;

	lispMachineCode_SetLReg   (pLM, LM_LREG_5, pEntString) ;
	return	lispMachineState_makeIntArgWithMinibuf1 (pLM) ;
}

TLMRESULT
lispMachineState_makeIntArgWithMinibuf1 (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	register TLispEntity*	pEntKeymap ;
	TLispEntity*			pEntKeymapValue ;
	TLispEntity*			pEntString ;
	const Char*				pString ;
	int						nString ;

	lispMachineCode_GetLReg (pLM, LM_LREG_5, &pEntString) ;
	if (TFAILED (lispEntity_GetStringValue (pLispMgr, pEntString, &pString, &nString))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}

	/*	minibuffer-local-map ΤޤޤǤɤʤȻפ⤷ȡ
	 *	minor-mode ˲դΤ⤷ʤ*/
	pEntKeymap	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_MINIBUFFER_LOCAL_MAP) ;
	if (TFAILED (lispMachine_GetCurrentSymbolValue (pLM, pEntKeymap, &pEntKeymapValue)) ||
		pEntKeymapValue == NULL ||
		TSUCCEEDED (lispEntity_Voidp (pLispMgr, pEntKeymapValue))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	return	lispMachine_ReadStringStart (pLM, pString, nString, pEntKeymap) ;
}

TLMRESULT
lispMachineState_makeIntArgVar (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pEntString ;
	TLispEntity*			pEntArglist ;
	TLispEntity*			pEntRetval ;
	const Char*				pString ;
	int						nLength ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	
	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM))
		return	LMR_RETURN ;

	/*	쥸ͤȴФ*/
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntString) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_1, &pEntArglist) ;

	lispEntity_GetStringValue (pLispMgr, pEntString, &pString, &nLength) ;
	if (TFAILED (lispMgr_InternSymbol (pLispMgr, pString, nLength, &pEntRetval)))
		return	LMR_ERROR ;

	if (TFAILED (lispMachine_addEntityToArglist (pLM, &pEntArglist, pEntRetval))) 
		lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_makeIntArgNum (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pEntString ;
	TLispEntity*			pEntArglist ;
	TLispEntity*			pEntRetval ;
	const Char*				pString ;
	int						nLength ;
	long					lValue ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	
	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM))
		return	LMR_RETURN ;

	/*	쥸ͤȴФ*/
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntString) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_1, &pEntArglist) ;

	lispEntity_GetStringValue (pLispMgr, pEntString, &pString, &nLength) ;
	if (TFAILED (catoi (pString, nLength, &lValue)))
		goto	retry ;
	if (TFAILED (lispMgr_CreateInteger (pLispMgr, lValue, &pEntRetval)))
		return	LMR_ERROR ;

	lispMachineCode_GetLReg (pLM, LM_LREG_1, &pEntArglist) ;
	if (TFAILED (lispMachine_addEntityToArglist (pLM, &pEntArglist, pEntRetval))) 
		lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;

  retry:
	lispMachineCode_SetState (pLM, &lispMachineState_makeIntArgWithMinibuf1) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_callIntBuiltin (
	register TLispMachine* pLM)
{
	LMCMDINFO const*	pProcInfo ;

#if defined (DEBUG) 
	fprintf (stderr, "call-int-builtin ()/0\n") ;
#endif
	lispMachineCode_MoveLReg (pLM, LM_LREG_ACC, LM_LREG_1) ;
	lispMachineCode_GetVRegP (pLM, LM_VREG_1,   (void *)&pProcInfo) ;
	lispMachineCode_PopVReg  (pLM, LM_VREG_1) ;
	lispMachineCode_PopLReg  (pLM, LM_LREG_5) ;
	lispMachineCode_PopLReg  (pLM, LM_LREG_4) ;
	lispMachineCode_PopLReg  (pLM, LM_LREG_3) ;
	lispMachineCode_PopLReg  (pLM, LM_LREG_1) ;

	assert (pProcInfo != NULL) ;

	/*	顼⤷㳰ȯƤ롣*/
	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM))
		return	LMR_RETURN ;

	lispMachineCode_SetInteractive (pLM, True) ;
	lispMachineCode_SetState (pLM, pProcInfo->m_pProc) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_callIntLambda1 (
	register TLispMachine* pLM)
{
	register TLispManager*	pLispMgr ;
	LMCMDINFO const*		pProcInfo ;
	TLispEntity*			pEntArglist ;
	TLispEntity*			pFunc ;
	TLispEntity*			pTarget ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

#if defined (DEBUG) 
	fprintf (stderr, "call-int-lambda1 ()/0\n") ;
#endif
	lispMachineCode_GetLReg  (pLM, LM_LREG_1, &pEntArglist) ;
	lispMachineCode_GetVRegP (pLM, LM_VREG_1, (void *)&pProcInfo) ;
	lispMachineCode_PopVReg  (pLM, LM_VREG_1) ;
	lispMachineCode_PopLReg  (pLM, LM_LREG_5) ;
	lispMachineCode_PopLReg  (pLM, LM_LREG_4) ;
	lispMachineCode_PopLReg  (pLM, LM_LREG_3) ;
	lispMachineCode_PopLReg  (pLM, LM_LREG_1) ;
	lispMachineCode_PopLReg  (pLM, LM_LREG_ACC) ;
	lispMachineCode_GetLReg  (pLM, LM_LREG_ACC, &pFunc) ;

	assert (pProcInfo != NULL) ;

	/*	顼⤷㳰ȯƤ롣*/
	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) 
		return	LMR_RETURN ;

	lispEntity_AddRef (pLispMgr, pEntArglist) ;
	if (TFAILED (lispMgr_CreateConscell (pLispMgr, pFunc, pEntArglist, &pTarget))) 
		return	LMR_ERROR ;
	lispEntity_Release (pLispMgr, pEntArglist) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pTarget) ;
#if defined (DEBUG) 
	fprintf (stderr, "Target = ") ;
	lispEntity_Print (pLispMgr, pTarget) ;
	fprintf (stderr, "\n") ;
#endif

	lispMachineCode_SetInteractive (pLM, True) ;
	lispMachineCode_SetState (pLM, pProcInfo->m_pProc) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_callIntLambda2 (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	LMCMDINFO const*		pProcInfo ;
	TLispEntity*			pEntArglist ;
	TLispEntity*			pFunc ;
	TLispEntity*			pTarget ;

	lispMachineCode_GetLReg   (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispMachineCode_GetVRegP  (pLM, LM_VREG_1,  (void *)&pProcInfo) ;
	lispMachineCode_PopVReg   (pLM, LM_VREG_1) ;
	lispMachineCode_PopLReg   (pLM, LM_LREG_ACC) ;
	lispMachineCode_GetLReg   (pLM, LM_LREG_ACC, &pFunc) ;
	assert (pProcInfo != NULL) ;

	/*	顼⤷㳰ȯƤ롣*/
	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) 
		return	LMR_RETURN ;

	lispEntity_AddRef (pLispMgr, pEntArglist) ;
	if (TFAILED (lispMgr_CreateConscell (pLispMgr, pFunc, pEntArglist, &pTarget))) 
		return	LMR_ERROR ;
	lispEntity_Release (pLispMgr, pEntArglist) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pTarget) ;
#if defined (DEBUG) 
	fprintf (stderr, "Target = ") ;
	lispEntity_Print (pLispMgr, pTarget) ;
	fprintf (stderr, "\n") ;
#endif

	lispMachineCode_SetInteractive (pLM, True) ;
	lispMachineCode_SetState (pLM, pProcInfo->m_pProc) ;
	return	LMR_CONTINUE ;
}

/*[ǽ]
 *	lambda 椫 (interactive ...) Ƚ񤫤줿 LIST õ
 *
 *	ɤΤȤǽ餫֤õʳˡϤʤlambda
 *	 LIST CAAR Ȥäơ줬 SYMBOL interactive 
 *	 CAR ֤
 *	pEntity  lambda  (LIST) ǡppReturn ˸դä interactive
 *	()֤롣
 */
Boolean
lispMachine_getIntArg (
	register TLispManager*	pLispMgr,
	register TLispEntity*	pEntity,
	register TLispEntity**	ppEntReturn)
{
	TLispEntity*	pEntIntact ;
	TLispEntity*	pEntCar ;
	TLispEntity*	pEntCaar ;
	TLispEntity*	pEntNext ;

	pEntIntact	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_INTERACTIVE) ;
	while (TFAILED (lispEntity_Nullp (pLispMgr, pEntity))) {
		if (TSUCCEEDED (lispEntity_GetCar (pLispMgr, pEntity, &pEntCar)) &&
			TSUCCEEDED (lispEntity_GetCar (pLispMgr, pEntCar, &pEntCaar)) &&
			TSUCCEEDED (lispEntity_Eq (pLispMgr, pEntCaar, pEntIntact))) {
			return	lispEntity_GetCadr (pLispMgr, pEntCar, ppEntReturn) ;
		}
		if (TFAILED (lispEntity_GetCdr (pLispMgr, pEntity, &pEntNext)))
			break ;
		pEntity	= pEntNext ;
	}
	return	False ;
}

Boolean
lispMachine_addEntityToArglist (
	register TLispMachine*	pLM,
	register TLispEntity**	ppEntArglist,
	register TLispEntity*	pEntNewEntity)
{
	register TLispManager*	pLispMgr ;
	register TLispEntity*	pEntArglist ;

	assert (pLM != NULL) ;
	assert (ppEntArglist  != NULL) ;
	assert (pEntNewEntity != NULL) ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	
	pEntArglist	= *ppEntArglist ;

	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntArglist))) {
		lispMachineCode_SetLReg (pLM, LM_LREG_1, pEntNewEntity) ;
		pEntArglist	= pEntNewEntity ;
	} else {
		TLispEntity*	pNextNode ;
		
		/*	List κǸ򸡺ɲä롣*/
		for ( ; ; ) {
			if (TFAILED (lispEntity_GetCdr (pLispMgr, pEntArglist, &pNextNode))) 
				return	False ;

			if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pNextNode))) {
				lispEntity_SetCdr (pLispMgr, pEntArglist, pEntNewEntity) ;
				pEntArglist	= pEntNewEntity ;
				break ;
			}
			pEntArglist	= pNextNode ;
		}
	}
	*ppEntArglist	= pEntArglist ;
	return	True ;
}

/*
 *
 */
Boolean
lispMachine_commandp (
	register TLispMachine*	pLM,
	register TLispEntity*	pEntARG,
	register TLispEntity**	ppEntRetval)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	register TLispEntity*	pEntInt ;
	LMCMDINFO const*		pCmdInfo ;
	TLispEntity*			pEntValue ;
	TLispEntity*			arList [2] ;
	int						nType ;

	pEntInt		= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_INTERACTIVE) ;
	arList [0]	= pEntInt ;

	if (TFAILED (lispEntity_GetType (pLispMgr, pEntARG, &nType)))
		return	False ;

	switch (nType) {
	case	LISPENTITY_SYMBOL:
		if (TFAILED (lispMachine_GetSymbolFunctionValue (pLM, pEntARG, &pEntValue))) {
			return	lispMgr_CreateNil (pLispMgr, ppEntRetval) ;
		}
		if (TSUCCEEDED (lispSubr_GetProc (pLispMgr, pEntValue, &pCmdInfo))) 
			goto	entity_is_subr ;
		if (TFAILED (lispMachine_getIntArg (pLispMgr, pEntValue, &arList [1])))
			return	lispMgr_CreateNil (pLispMgr, ppEntRetval) ;
		return	lispMgr_CreateList (pLispMgr, arList, 2, ppEntRetval) ;

	case	LISPENTITY_CONSCELL:
	{
		TLispEntity*	pEntCar ;

		if (TSUCCEEDED (lispEntity_GetCar  (pLispMgr, pEntARG, &pEntCar)) &&
			TSUCCEEDED (lispEntity_Lambdap (pLispMgr, pEntCar))) 
			if (TSUCCEEDED (lispMachine_getIntArg (pLispMgr, pEntARG, &arList [1])))
				return	lispMgr_CreateList (pLispMgr, arList, 2, ppEntRetval) ;
		break ;
	}
	case	LISPENTITY_SUBR:
	  entity_is_subr:
	{
		register const Char*	pString ;
		register int			nString ;

		if (TFAILED (lispSubr_GetProc (pLispMgr, pEntARG, &pCmdInfo))) 
			break ;
		if (pCmdInfo->m_pInteractive == NULL) 
			return	lispMgr_CreateNil (pLispMgr, ppEntRetval) ;
		pString	= pCmdInfo->m_pInteractive ;
		nString	= Cstrlen (pString) ;
		if (TFAILED (lispMgr_CreateString (pLispMgr, pString, nString, &arList [1])))
			return	False ;
		arList [0]	= pEntInt ;
		return	lispMgr_CreateList (pLispMgr, arList, 2, ppEntRetval) ;
	}
	default:
		break ;
	}
	return	lispMgr_CreateNil (pLispMgr, ppEntRetval) ;
}

