/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "local.h"
#include <stdio.h>
#include <assert.h>
#include "lmachinep.h"
#include "regex.h"

static	TLMRESULT	lispMachine_search (TLispMachine*, Boolean (*)(TLispMachine*, TLispEntity*, const Char*, int, int, int), int) ;

/*
 *	string-match is a built-in function.
 *	(string-match REGEXP STRING &optional START)
 *	
 *	Return index of start of first match for REGEXP in STRING, or nil.
 *	Case is ignored if `case-fold-search' is non-nil in the current buffer.
 *	If third arg START is non-nil, start search at that index in STRING.
 *	For index of first char beyond the match, do (match-end 0).
 *	`match-end' and `match-beginning' also give indices of substrings
 *	matched by parenthesis constructs in the pattern.
 */
TLMRESULT
lispMachineState_StringMatch (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntREGEXP ;
	TLispEntity*	pEntSTRING ;
	TLispEntity*	pEntSTART ;
	TLispEntity*	pEntRetval ;
	const Char*		pStrREGEXP ;
	int				nStrREGEXP ;
	const Char*		pStrSTRING ;
	int				nStrSTRING ;
	long			lStart ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntREGEXP)) ||
		TFAILED (lispEntity_Stringp (pLispMgr, pEntREGEXP)) ||
		TFAILED (lispEntity_GetCdr  (pLispMgr, pEntArglist, &pEntArglist)) ||
		TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntSTRING)) ||
		TFAILED (lispEntity_Stringp (pLispMgr, pEntSTRING)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntSTART))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispEntity_GetStringValue (pLispMgr, pEntREGEXP, &pStrREGEXP, &nStrREGEXP) ;
	lispEntity_GetStringValue (pLispMgr, pEntSTRING, &pStrSTRING, &nStrSTRING) ;
	if (TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntSTART, &lStart)))
		lStart	= 0 ;
	if (lStart > nStrSTRING) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_SetRegmatchTarget (pLM, pEntSTRING) ;
	if (TFAILED (lispMachineCode_StringMatch (pLM, pStrREGEXP, nStrREGEXP, pStrSTRING, nStrSTRING, lStart))) {
		lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	} else {
		int		nPosition ;
		lispMachineCode_MatchBeginning (pLM, 0, &nPosition) ;
		lispMgr_CreateInteger (pLispMgr, nPosition, &pEntRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*
 *	re-search-forward is an interactive built-in function.
 *	(re-search-forward REGEXP &optional BOUND NOERROR COUNT)
 *
 *	Search forward from point for regular expression REGEXP.
 *	Set point to the end of the occurrence found, and return point.
 *	An optional second argument bounds the search; it is a buffer position.
 *	The match found must not extend after that position.
 *	Optional third argument, if t, means if fail just return nil (no error).
 *	If not nil and not t, move to limit of search and return nil.
 *	Optional fourth argument is repeat count--search for successive occurrences.
 *	See also the functions `match-beginning', `match-end', `match-string',
 *	and `replace-match'.
 */
TLMRESULT
lispMachineState_ReSearchForward (
	register TLispMachine*	pLM)
{
	return	lispMachine_search (pLM, &lispMachineCode_ReSearchForward, 1) ;
}

TLMRESULT
lispMachineState_ReSearchBackward (
	register TLispMachine*	pLM)
{
	return	lispMachine_search (pLM, &lispMachineCode_ReSearchBackward, - 1) ;
}

/*
 *	match-beginning is a built-in function.
 *	(match-beginning SUBEXP)
 *
 *	Ǹ search ˤäƥޥåƥȤκǽΰ֤֤
 *	Return position of start of text matched by last search.
 *	SUBEXP, a number, specifies which parenthesized expression in the last
 *	regexp.
 *	Value is nil if SUBEXPth pair didn't match, or there were less than
 *	SUBEXP pairs.
 *	Zero means the entire text matched by the whole regexp or whole string.
 */
TLMRESULT
lispMachineState_MatchBeginning (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntSUBEXP ;
	TLispEntity*	pEntRetval ;
	long			lNumber ;
	int				nPosition ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntSUBEXP)) ||
		TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntSUBEXP, &lNumber))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispMachineCode_MatchBeginning (pLM, lNumber, &nPosition))) {
		lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	} else {
		lispMgr_CreateInteger (pLispMgr, nPosition, &pEntRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*
 *	match-end is a built-in function.
 *	(match-end SUBEXP)
 *
 *	Return position of end of text matched by last search.
 *	SUBEXP, a number, specifies which parenthesized expression in the last
 *	regexp.
 *	Value is nil if SUBEXPth pair didn't match, or there were less than
 *	SUBEXP pairs.
 *	Zero means the entire text matched by the whole regexp or whole string.
 */
TLMRESULT
lispMachineState_MatchEnd (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntSUBEXP ;
	TLispEntity*	pEntRetval ;
	long			lNumber ;
	int				nPosition ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntSUBEXP)) ||
		TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntSUBEXP, &lNumber))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispMachineCode_MatchEnd (pLM, lNumber, &nPosition))) {
		lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	} else {
		lispMgr_CreateInteger (pLispMgr, nPosition, &pEntRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*
 *	match-data is a built-in function.
 *	(match-data &optional INTEGERS REUSE)
 *
 *	Return a list containing all info on what the last search matched.
 *	Element 2N is `(match-beginning N)'; element 2N + 1 is `(match-end N)'.
 *	All the elements are markers or nil (nil if the Nth pair didn't match)
 *	if the last match was on a buffer; integers or nil if a string was matched.
 *	Use `store-match-data' to reinstate the data in this list.
 *
 *	If INTEGERS (the optional first argument) is non-nil, always use integers
 *	(rather than markers) to represent buffer positions.
 *	If REUSE is a list, reuse it as part of the value.  If REUSE is long enough
 *	to hold all the values, and if INTEGERS is non-nil, no consing is done.
 */
TLMRESULT
lispMachineState_MatchData (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pEntArglist ;
	TLispEntity*			pEntINTEGERS ;
	TLispEntity*			pEntREUSE ;
	TLispEntity*			pEntNode ;
	register TLispEntity*	pEntNil ;
	register TLispEntity*	pEntBuffer	= NULL ;
	TLispConscell			conslst ;
	register int			i ;
	register regmatch_t*	pMatch ;
	register Boolean		fInteger ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntINTEGERS)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntREUSE))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	fInteger	= lispEntity_Nullp (pLispMgr, pEntINTEGERS) ;
	if (!fInteger &&
		pLM->m_pEntRegMatch != NULL && 
		TSUCCEEDED (lispEntity_Bufferp (pLispMgr, pLM->m_pEntRegMatch))) {
		pEntBuffer	= pLM->m_pEntRegMatch ;
	} else {
		fInteger	= True ;
	}
		
	pEntNil	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_NIL) ;
	conslst.m_pCar	= conslst.m_pCdr	= NULL ;
	pMatch	= pLM->m_aRegMatch ;
	for (i = 0 ; i < MAX_REGEXP_MATCHES ; i ++, pMatch ++) {
		if (pMatch->rm_so < 0)
			break ;
		if (fInteger) {
			if (TFAILED (lispMgr_CreateInteger (pLispMgr, (long) pMatch->rm_so, &pEntNode)))
				return	LMR_ERROR ;
			lispEntity_Push2List (pLispMgr, &conslst, pEntNode) ;
			if (TFAILED (lispMgr_CreateInteger (pLispMgr, (long) pMatch->rm_eo, &pEntNode)))
				return	LMR_ERROR ;
			lispEntity_Push2List (pLispMgr, &conslst, pEntNode) ;
		} else {
			if (TFAILED (lispMgr_CreateMarker (pLispMgr, &pEntNode)))
				return	LMR_ERROR ;
			lispEntity_Push2List (pLispMgr, &conslst, pEntNode) ;
			lispBuffer_AddMarker (pLispMgr, pEntBuffer, pEntNode) ;
			lispMarker_SetBufferPosition (pLispMgr, pEntNode, pEntBuffer, pMatch->rm_so) ;
			if (TFAILED (lispMgr_CreateMarker (pLispMgr, &pEntNode)))
				return	LMR_ERROR ;
			lispEntity_Push2List (pLispMgr, &conslst, pEntNode) ;
			lispBuffer_AddMarker (pLispMgr, pEntBuffer, pEntNode) ;
			lispMarker_SetBufferPosition (pLispMgr, pEntNode, pEntBuffer, pMatch->rm_eo) ;
		}
	}
	if (! fInteger) {
		while (i < MAX_REGEXP_MATCHES) {
			lispEntity_Push2List (pLispMgr, &conslst, pEntNil) ;
			lispEntity_Push2List (pLispMgr, &conslst, pEntNil) ;
			i	++ ;
		}
	}
	if (conslst.m_pCar != NULL) {
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, conslst.m_pCar) ;
		lispEntity_Release (pLispMgr, conslst.m_pCar) ;
		conslst.m_pCar	= NULL ;
	} else {
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntNil) ;
	}
	return	LMR_RETURN ;
}

/*	(set-match-data LIST)
 *
 *	Set internal data on last search match from elements of LIST.
 *	LIST should have been created by calling `match-data' previously.
 */
TLMRESULT
lispMachineState_SetMatchData (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*			pEntArglist ;
	TLispEntity*			pEntLIST ;
	TLispEntity*			pEntBuffer	= NULL ;
	register TLispEntity*	pEntNil ;
	register int			nMatch ;
	register regmatch_t*	pMatch ;

	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar (pLispMgr, pEntArglist, &pEntLIST)) ||
		TFAILED (lispEntity_Listp  (pLispMgr, pEntLIST))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	
	pEntNil	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_NIL) ;
	pMatch	= pLM->m_aRegMatch ;
	nMatch	= 0 ;
	while (nMatch < MAX_REGEXP_MATCHES && TFAILED (lispEntity_Nullp (pLispMgr, pEntLIST))) {
		TLispEntity*	pEntSO ;
		TLispEntity*	pEntEO ;
		TLispNumber		num ;

		if (TFAILED (lispEntity_GetCar (pLispMgr, pEntLIST, &pEntSO))   ||
			TFAILED (lispEntity_GetCdr (pLispMgr, pEntLIST, &pEntLIST)) ||
			TFAILED (lispEntity_GetCar (pLispMgr, pEntLIST, &pEntEO))   ||
			TFAILED (lispEntity_GetCdr (pLispMgr, pEntLIST, &pEntLIST))) {
			lispMachineCode_SetError (pLM) ;
			return	LMR_RETURN ;
		}
		if (pEntBuffer == NULL &&
			TSUCCEEDED (lispEntity_Markerp (pLispMgr, pEntSO))) {
			int	nPosition ;
			(void) lispMarker_GetBufferPosition (pLispMgr, pEntSO, &pEntBuffer, &nPosition) ;
		}
		if (TFAILED (lispEntity_GetNumberOrMarkerValue (pLispMgr, pEntSO, &num))) {
			pMatch->rm_so	= -1 ;
		} else {
			pMatch->rm_so	= (num.m_fFloatp)? -1 : num.m_Value.m_lLong ;
		}
		if (TFAILED (lispEntity_GetNumberOrMarkerValue (pLispMgr, pEntEO, &num))) {
			pMatch->rm_eo	= -1 ;
		} else {
			pMatch->rm_eo	= (num.m_fFloatp)? -1 : num.m_Value.m_lLong ;
		}
		pMatch	++ ;
		nMatch	++ ;
	}
	while (nMatch < MAX_REGEXP_MATCHES) {
		pMatch->rm_so = pMatch->rm_eo	= -1 ;
		pMatch	++ ;
		nMatch	++ ;
	}
	lispMachineCode_SetRegmatchTarget (pLM, pEntBuffer) ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntNil) ;
	return	LMR_RETURN ;
}

/*	search-forward is an interactive built-in function.
 *	(search-forward STRING &optional BOUND NOERROR COUNT)
 *	
 *	Search forward from point for STRING.
 *	Set point to the end of the occurrence found, and return point.
 *	An optional second argument bounds the search; it is a buffer position.
 *	The match found must not extend after that position.  nil is equivalent
 *	to (point-max).
 *	Optional third argument, if t, means if fail just return nil (no error).
 *	If not nil and not t, move to limit of search and return nil.
 *	Optional fourth argument is repeat count--search for successive occurrences.
 *	
 *	Search case-sensitivity is determined by the value of the variable
 *	`case-fold-search', which see.
 *	
 *	See also the functions `match-beginning', `match-end' and `replace-match'.
 */
TLMRESULT
lispMachineState_SearchForward (
	register TLispMachine*	pLM)
{
	return	lispMachine_search (pLM, &lispMachineCode_SearchForward, 1) ;
}

/*
 */
TLMRESULT
lispMachineState_SearchBackward (
	register TLispMachine*	pLM)
{
	return	lispMachine_search (pLM, &lispMachineCode_SearchBackward, - 1) ;
}

/*	built-in function:
 *		(regexp-quote STRING)
 *
 *	STRING Τ˥ޥåƤ¾ˤϥޥåʤɽʸ֤
 *
 *	Ĥޤϡ[]  *, ., ? ʤɤδʤʸ quote ȤȤʤ
 *	 TCL/TK äơä regex library ʸäƲ
 *	äʡemacs Ȥ quote δطդä褦ʡ
 *
 *	ΤǡTCL/TK ǤΤѤƤط塢"[", "?" ʤɤɬä
 *	ȤȤɬסȡ
 */
TLMRESULT
lispMachineState_RegexpQuote (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr			= pLM->m_pLispMgr ;
	TVarbuffer				vbufSTRING ;
	static const char		rchQuoteChars []	= ".*+?^$()\\[]{}|" ;
	static const Char		chQuote				= '\\' ;
	register TLMRESULT		nResult				= LMR_ERROR ;
	TLispEntity*			pEntArglist ;
	TLispEntity*			pEntSTRING ;
	const Char*				pStrSTRING ;
	int						nStrSTRING ;
	register const Char*	pStrHEAD ;
	register const Char*	pStrResult ;
	register int			nStrResult ;
	TLispEntity*			pEntRetval ;

	assert (pLM != NULL) ;
	assert (pLispMgr != NULL) ;

	if (TFAILED (TVarbuffer_Initialize (&vbufSTRING, sizeof (Char))))
		return	LMR_ERROR ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntSTRING) ;
	if (TFAILED (lispEntity_GetStringValue (pLispMgr, pEntSTRING, &pStrSTRING, &nStrSTRING))) {
		lispMachineCode_SetError (pLM) ;
		goto	error ;
	}
	/*	".*+?^$()\\[]{}|" ȡ quote 롣*/
	while (nStrSTRING > 0) {
		pStrHEAD	= pStrSTRING ;
		while (nStrSTRING > 0 &&
			   (!Char_IsAscii (*pStrSTRING) ||
				memchr (rchQuoteChars, (char)*pStrSTRING, NELEMENTS(rchQuoteChars)) == NULL)) {
			pStrSTRING	++ ;
			nStrSTRING	-- ;
		}
		if (pStrHEAD < pStrSTRING)
			if (TFAILED (TVarbuffer_Add (&vbufSTRING, pStrHEAD, pStrSTRING - pStrHEAD)))
				goto	error ;
		if (nStrSTRING > 0) {
			if (TFAILED (TVarbuffer_Add (&vbufSTRING, &chQuote, 1)) ||
				TFAILED (TVarbuffer_Add (&vbufSTRING, pStrSTRING, 1)))
				goto	error ;
			pStrSTRING	++ ;
			nStrSTRING	-- ;
		}
	}
	pStrResult	= TVarbuffer_GetBuffer (&vbufSTRING) ;
	nStrResult	= TVarbuffer_GetUsage  (&vbufSTRING) ;
	if (TFAILED (lispMgr_CreateString (pLispMgr, pStrResult, nStrResult, &pEntRetval)))
		goto	error ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	nResult	= LMR_RETURN ;
  error:
	TVarbuffer_Uninitialize (&vbufSTRING) ;
	return	nResult ;
}

/*========================================================================
 *	private functions
 */
TLMRESULT
lispMachine_search (
	register TLispMachine*	pLM,
	register Boolean		(*pCmd)(TLispMachine*, TLispEntity*, const Char*, int, int, int),
	register int			nDir)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntBuffer ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntSTRING ;
	TLispEntity*	pEntBOUND ;
	TLispEntity*	pEntNOERROR ;
	TLispEntity*	pEntCOUNT ;
	const Char*		pStrSTRING ;
	int				nStrSTRING ;
	int				nCOUNT, nBOUND ;
	TLispEntity*	pEntRetval ;

	assert (pLM  != NULL) ;
	assert (pCmd != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	assert (pEntBuffer != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;

	lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntSTRING) ;
	if (TFAILED (lispEntity_GetStringValue (pLispMgr, pEntSTRING, &pStrSTRING, &nStrSTRING))) 
		goto	error ;
	lispEntity_GetCdr  (pLispMgr, pEntArglist, &pEntArglist) ;

	/*	default  bound  point-max/min Ǥ뤬̤˻ꤵƤϤ
	 *	¤ǤϤʤ*/
	lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntBOUND) ;
	if (nDir > 0) {
		lispBuffer_PointMax (pLispMgr, pEntBuffer, &nBOUND) ;
	} else {
		lispBuffer_PointMin (pLispMgr, pEntBuffer, &nBOUND) ;
	}
	if (TFAILED (lispEntity_Nullp (pLispMgr, pEntBOUND))) {
		TLispNumber	numBOUND ;
		if (TFAILED (lispEntity_GetNumberOrMarkerValue (pLispMgr, pEntBOUND, &numBOUND)) ||
			numBOUND.m_fFloatp)
			goto	error ;
		nBOUND	= numBOUND.m_Value.m_lLong ;
	}
	lispEntity_GetCdr  (pLispMgr, pEntArglist, &pEntArglist) ;
	lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntNOERROR) ;

	/*	η֤ default ǤϰǤ롣*/
	nCOUNT	= 1 ;
	lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntCOUNT) ;
	if (TFAILED (lispEntity_Nullp (pLispMgr, pEntCOUNT))) {
		long	lCOUNT ;
		if (TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntCOUNT, &lCOUNT)))
			goto	error ;
		nCOUNT	= lCOUNT ;
	}
	lispMachineCode_SetRegmatchTarget (pLM, pEntBuffer) ;
	if (TFAILED ((*pCmd) (pLM, pEntBuffer, pStrSTRING, nStrSTRING, nBOUND, nCOUNT))) {
		if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntNOERROR)))
			goto	error ;
		lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
		if (TFAILED (lispEntity_Tp (pLispMgr, pEntNOERROR))) {
			/*	ɤ٤ʤΤ ɬפϤΤ */
		}
	} else {
		TLispEntity*	pEntPoint ;
		int				nPosition ;

		if (nDir > 0) {
			(void) lispMachineCode_MatchEnd       (pLM, 0, &nPosition) ;
		} else {
			(void) lispMachineCode_MatchBeginning (pLM, 0, &nPosition) ;
		}
		if (TFAILED (lispMgr_CreateInteger (pLispMgr, nPosition, &pEntRetval)))
			return	LMR_ERROR ;

		/*	backward ξˤϡposition  beginning ΰ֤ˤʤ롣
		 *	forward  ξˤϡposition  end       ΰ֤ˤʤ롣
		 */
		lispBuffer_PointMarker (pLispMgr, pEntBuffer, &pEntPoint) ;
		lispMarker_SetBufferPosition (pLispMgr, pEntPoint, pEntBuffer, nPosition) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;

  error:
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

