/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "local.h"
#include <stdio.h>
#include <assert.h>
#include "lmachinep.h"

static	TLMRESULT	lispMachine_killRegion	(TLispMachine*, TLispEntity*, TBufStringMarker*, int, int) ;
static	Boolean		lispMachine_currentKill	(TLispMachine*, int, Boolean, TLispEntity**) ;

/*
 *	(transporse-chars ARG)
 */
TLMRESULT
lispMachineState_TransposeChars (
	register TLispMachine* pLM)
{
	return	LMR_RETURN ;
}

/*
 */
TLMRESULT
lispMachineState_SetMark (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntPOS ;
	TLispEntity*	pEntBuffer ;
	TLispEntity*	pEntMark ;

	assert (pLM      != NULL) ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	lispEntity_GetCar     (pLispMgr, pEntArglist, &pEntPOS) ;
	lispBuffer_MarkMarker (pLispMgr, pEntBuffer, &pEntMark) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntPOS))) {
		/*	nil ʤХåե*/
		lispBuffer_RemoveMarker (pLispMgr, pEntMark) ;
	} else {
		TLispNumber		numPOS ;
		register int	nPOS ;
		int				nBufferTop, nBufferEnd ;

		if (TFAILED (lispEntity_GetNumberValueOrMarkerPosition (pLispMgr, pEntPOS, &numPOS)) ||
			numPOS.m_fFloatp) {
			lispMachineCode_SetError (pLM) ;
			return	LMR_RETURN ;
		}
		nPOS	= numPOS.m_Value.m_lLong ;
		/*	Хåեɲäơ*/
		lispBuffer_AddMarker (pLispMgr, pEntBuffer, pEntMark) ;
		/*	Хåեΰ֤Ĵ롣*/
		lispBuffer_PointBufferTop (pLispMgr, pEntBuffer, &nBufferTop) ;
		lispBuffer_PointBufferEnd (pLispMgr, pEntBuffer, &nBufferEnd) ;
		if (nPOS < nBufferTop)
			nPOS	= nBufferTop ;
		if (nPOS > nBufferEnd)
			nPOS	= nBufferEnd ;
		lispMarker_SetBufferPosition (pLispMgr, pEntMark, pEntBuffer, nPOS) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntMark) ;
	return	LMR_RETURN ;
}

/*	(mark &optional FORCE)
 */
TLMRESULT
lispMachineState_Mark (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntCurBuffer ;
	TLispEntity*	pEntMark ;
	TLispEntity*	pEntBuffer ;
	TLispEntity*	pEntFORCE ;
	int				nPos ;
	TLispEntity*	pEntRetval ;

	assert (pLM      != NULL) ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	lispMachineCode_GetCurrentBuffer (pLM, &pEntCurBuffer) ;
	assert (pEntCurBuffer != NULL) ;
	lispEntity_GetCar     (pLispMgr, pEntArglist, &pEntFORCE) ;
	lispBuffer_MarkMarker (pLispMgr, pEntCurBuffer, &pEntMark) ;
	assert (pEntMark != NULL) ;
	lispMarker_GetBufferPosition (pLispMgr, pEntMark, &pEntBuffer, &nPos) ;
	if (pEntBuffer != pEntCurBuffer || nPos < 1) {
		/*	inactive */
		if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntFORCE))) {
			lispMachineCode_SetError (pLM) ;
			return	LMR_RETURN ;
		}
		lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	} else {
		/*	active */
		if (TFAILED (lispMgr_CreateInteger (pLispMgr, nPos, &pEntRetval)))
			return	LMR_ERROR ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*	(count-lines START END)
 */
TLMRESULT
lispMachineState_CountLines (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*		pEntArglist ;
	TLispEntity*		pEntCurBuffer ;
	TLispEntity*		pEntSTART ;
	TLispEntity*		pEntEND ;
	TLispEntity*		pEntRetval ;
	TBufStringMarker	mk ;
	register Char		cc ;
	TLispNumber			numSTART, numEND ;
	register int		nStart, nEnd, nUsage, nLines ;
	int					nBufferTop, nBufferEnd, nLength ;

	assert (pLM      != NULL) ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	lispMachineCode_GetCurrentBuffer (pLM, &pEntCurBuffer) ;
	assert (pEntCurBuffer != NULL) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntSTART)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntEND)) ||
		TFAILED (lispEntity_GetNumberValueOrMarkerPosition (pLispMgr, pEntSTART, &numSTART)) ||
		numSTART.m_fFloatp ||
		TFAILED (lispEntity_GetNumberValueOrMarkerPosition (pLispMgr, pEntEND,   &numEND)) ||
		numEND.m_fFloatp) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	nStart	= numSTART.m_Value.m_lLong ;
	nEnd	= numEND.m_Value.m_lLong ;

	lispBuffer_GetFullString  (pLispMgr, pEntCurBuffer, &mk, &nLength) ;
	lispBuffer_PointBufferTop (pLispMgr, pEntCurBuffer, &nBufferTop) ;
	lispBuffer_PointBufferEnd (pLispMgr, pEntCurBuffer, &nBufferEnd) ;
	if (nEnd < nBufferTop || nStart> nEnd || nStart > nBufferEnd) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}

	TBufStringMarker_Forward (&mk, nStart - nBufferTop) ;
	nUsage	= nEnd - nStart ;
	nLines	= 0 ;
	while (nUsage -- > 0) {
		cc		= TBufStringMarker_GetChar (&mk) ;
		TBufStringMarker_Forward (&mk, 1) ;
		if (cc == '\n' || cc == '\r') 
			nLines	++ ;
	}
	if (TFAILED (lispMgr_CreateInteger (pLispMgr, nLines, &pEntRetval)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*	(kill-line &optional ARG)
 *
 *		ߤιԤλĤ kill 롣⤷blank ä顢newline  kill
 *	롣prefix argument ˤϡpoint ʣιԤ kill 롣
 *	negative argument ä顢 kill 롣
 *
 *	program ɤ硢nil ϡְʤפ̣롣 prefix
 *	arg Ȥƿ롣
 */
TLMRESULT
lispMachineState_KillLine (
	register TLispMachine*		pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*		pEntArglist ;
	TLispEntity*		pEntArg ;
	TLispEntity*		pEntBuffer ;
	register int		nArg, nOrigin ;
	register Char		cc ;
	TBufStringMarker	mk, mkOrigin ;
	int					nPoint, nPointTop, nPointEnd, nPointMin, nPointMax, nLength ;

	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntArg))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntArg))) {
		nArg	= 1 ;
	} else {
		TLispNumber	numArg ;

		if (TFAILED (lispEntity_IntegerOrMarkerp (pLispMgr, pEntArg))) {
			lispMachineCode_SetError (pLM) ;
			return	LMR_RETURN ;
		}
		(void) lispEntity_GetNumberValueOrMarkerPosition (pLispMgr, pEntArg, &numArg) ;
		nArg	= numArg.m_Value.m_lLong ;
	}

	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	assert (pEntBuffer != NULL) ;
	lispBuffer_GetString (pLispMgr, pEntBuffer, &mk, &nLength) ;
	lispBuffer_Point     (pLispMgr, pEntBuffer, &nPoint) ;
	lispBuffer_PointBufferTop (pLispMgr, pEntBuffer, &nPointTop) ;
	lispBuffer_PointBufferEnd (pLispMgr, pEntBuffer, &nPointEnd) ;
	lispBuffer_PointBufferTop (pLispMgr, pEntBuffer, &nPointMin) ;
	lispBuffer_PointBufferEnd (pLispMgr, pEntBuffer, &nPointMax) ;
	TBufStringMarker_Forward  (&mk, nPoint - nPointTop) ;
	mkOrigin	= mk ;
	nOrigin		= nPoint ;
	if (nArg <= 0) {
		do {
			while (nPoint > nPointMin) {
				TBufStringMarker_Backward (&mk, 1) ;
				nPoint	-- ;
				cc = TBufStringMarker_GetChar (&mk) ;
				if (cc == '\n') {
					TBufStringMarker_Forward (&mk, 1) ;
					nPoint	++ ;
					break ;
				}
			}
			nArg	++ ;
		}	while (nArg <= 0) ;

	} else {
		register int	nTmp ;

		do {
			while (nPoint < nPointMax) {
				TBufStringMarker_Forward (&mk, 1) ;
				nPoint	++ ;
				cc = TBufStringMarker_GetChar (&mk) ;
				if (cc == '\n') 
					break ;
			}
			nArg	-- ;
		}	while (nArg > 0) ;

		nTmp	= nOrigin ;
		nOrigin	= nPoint ;
		nPoint	= nTmp ;
		mk		= mkOrigin ;
	}
	return	lispMachine_killRegion (pLM, pEntBuffer, &mk, nPoint, nOrigin) ;
}

/*	(kill-region BEG END)
 *
 *	point  mark δ֤ kill 롣줿 text  kill ring ¸
 *	롣C-y ǤФȤǤ롣
 */
TLMRESULT
lispMachineState_KillRegion (
	register TLispMachine*		pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*		pEntArglist ;
	TLispEntity*		pEntBuffer ;
	TLispEntity*		pEntBegin ;
	TLispEntity*		pEntEnd ;
	TBufStringMarker	mk ;
	int					nPoint, nPointTop, nPointEnd, nPointMin, nPointMax, nLength ;
	register int		nBegin, nEnd ;

	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntBegin)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntEnd))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TSUCCEEDED (lispEntity_IntegerOrMarkerp (pLispMgr, pEntBegin)) ||
		TSUCCEEDED (lispEntity_IntegerOrMarkerp (pLispMgr, pEntEnd))) {
		TLispNumber	numBegin, numEnd ;

		(void) lispEntity_GetNumberValueOrMarkerPosition (pLispMgr, pEntBegin, &numBegin) ;
		(void) lispEntity_GetNumberValueOrMarkerPosition (pLispMgr, pEntEnd,   &numEnd) ;
		if (numBegin.m_Value.m_lLong < numEnd.m_Value.m_lLong) {
			nBegin	= numBegin.m_Value.m_lLong ;
			nEnd	= numEnd.m_Value.m_lLong ;
		} else {
			nBegin	= numEnd.m_Value.m_lLong ;
			nEnd	= numBegin.m_Value.m_lLong ;
		}
	} else {
		/*	number-or-integer-p, XXX */
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}

	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	assert (pEntBuffer != NULL) ;
	lispBuffer_GetString (pLispMgr, pEntBuffer, &mk, &nLength) ;
	lispBuffer_Point     (pLispMgr, pEntBuffer, &nPoint) ;
	lispBuffer_PointBufferTop (pLispMgr, pEntBuffer, &nPointTop) ;
	lispBuffer_PointBufferEnd (pLispMgr, pEntBuffer, &nPointEnd) ;
	lispBuffer_PointBufferTop (pLispMgr, pEntBuffer, &nPointMin) ;
	lispBuffer_PointBufferEnd (pLispMgr, pEntBuffer, &nPointMax) ;
	TBufStringMarker_Forward  (&mk, nBegin - nPointTop) ;

	if (nBegin < nPointMin || nPointMax < nEnd) {
		/*	out-of-range, XXX */
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	return	lispMachine_killRegion (pLM, pEntBuffer, &mk, nBegin, nEnd) ;
}

TLMRESULT
lispMachineState_Yank (
	register TLispMachine*		pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*			pEntArglist ;
	TLispEntity*			pEntArg ;
	TLispEntity*			pEntString ;
	TLispEntity*			pEntBuffer ;
	register TLispEntity*	pEntMinus ;
	const Char*				pString ;
	int						nLength, nPoint ;
	register int			nArg ;

	assert (pLispMgr != NULL) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	assert (pEntArglist != NULL) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntArg))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	pEntMinus	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_MINUS) ;
	if (TSUCCEEDED (lispEntity_Listp (pLispMgr, pEntArg))) {
		nArg	= 0 ;
	} else if (TSUCCEEDED (lispEntity_Eq (pLispMgr, pEntArg, pEntMinus))) {
		nArg	= -1 ;
	} else {
		TLispNumber	numArg ;
		if (TFAILED (lispEntity_GetNumberValueOrMarkerPosition (pLispMgr, pEntArg, &numArg))) {
			lispMachineCode_SetError (pLM) ;
			return	LMR_RETURN ;
		}
		nArg	= numArg.m_Value.m_lLong - 1 ;
	}
	lispMachineCode_GetCurrentBuffer (pLM, &pEntBuffer) ;
	assert (pEntBuffer != NULL) ;
	lispBuffer_Point (pLispMgr, pEntBuffer, &nPoint) ;
	if (TFAILED (lispMachine_currentKill (pLM, nArg, False, &pEntString)) ||
		TFAILED (lispEntity_GetStringValue (pLispMgr, pEntString, &pString, &nLength)) ||
		TFAILED (lispBuffer_InsertString (pLispMgr, pEntBuffer, nPoint, pString, nLength))) {
		lispMachineCode_SetError (pLM) ;
	} else {
		register TLispEntity*	pEntYank ;
		register TLispEntity*	pEntThisCmd ;
		register TLispEntity*	pEntNil ;

		/*	mark ΰư򤷤ʤȤʤɤ... */

		pEntYank	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_YANK) ;
		pEntThisCmd	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_THIS_COMMAND) ;
		lispMachine_SetCurrentSymbolValue (pLM, pEntThisCmd, pEntYank) ;
		pEntNil		= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_NIL) ;
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntNil) ;
	}
	return	LMR_RETURN ;
}

TLMRESULT
lispMachine_killRegion (
	register TLispMachine*		pLM,
	register TLispEntity*		pEntBuffer,
	register TBufStringMarker*	pMarker,
	register int				nStart,
	register int				nEnd)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TVarbuffer				vbufKilledText ;
	register Char*			pbuf ;
	register Char			cc ;
	register int			nbuf, n ;
	static Char				srbuf [256] ;
	register TLispEntity*	pEntKillRing ;
	register TLispEntity*	pEntKillRegion ;
	register TLispEntity*	pEntThisCommand ;
	register TLispEntity*	pEntKRYP ;
	register TLispEntity*	pEntNil ;
	TLispEntity*			pEntLastCommand ;
	TLispEntity*			pValLastCommand ;
	TLispEntity*			pValKillRing ;
	TLispEntity*			pEntKilledText ;
	register Boolean		fRetval ;

	pEntKillRegion	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_KILL_REGION) ;

	n		= nEnd - nStart ;
	if (n <= 0)
		goto	exit_func ;

	if (TFAILED (TVarbuffer_Initialize (&vbufKilledText, sizeof (Char))))
		return	LMR_ERROR ;

	pEntKRYP		= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_KILL_RING_YANK_POINTER) ;
	pEntKillRing	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_KILL_RING) ;
	lispMachine_GetCurrentSymbolValue (pLM, pEntKillRing, &pValKillRing) ;

	pEntLastCommand	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_LAST_COMMAND) ;
	lispMachine_GetCurrentSymbolValue (pLM, pEntLastCommand, &pValLastCommand) ;

	/*	ľ command  kill-region ʤ append ưˤʤ롣*/
	if (TSUCCEEDED (lispEntity_Eq (pLispMgr, pValLastCommand, pEntKillRegion))) {
		TLispEntity*	pEntCar ;
		const Char*		pString ;
		int				nLength ;
		
		if (TSUCCEEDED (lispEntity_GetCar (pLispMgr, pValKillRing, &pEntCar)) &&
			TSUCCEEDED (lispEntity_GetStringValue (pLispMgr, pEntCar, &pString, &nLength))) {
			if (TFAILED (TVarbuffer_Add (&vbufKilledText, pString, nLength)))
				return	LMR_ERROR ;
		}
	}

	pbuf	= srbuf ;
	nbuf	= NELEMENTS (srbuf) ;
	while (n > 0) {
		cc	= TBufStringMarker_GetChar (pMarker) ;
		if (nbuf > 0) {
			*pbuf ++	= cc ;
			nbuf  -- ;
		} else {
			if (TFAILED (TVarbuffer_Add (&vbufKilledText, srbuf, NELEMENTS (srbuf))))
				return	LMR_ERROR ;
			pbuf	= srbuf ;
			nbuf	= NELEMENTS (srbuf) ;
		}
		TBufStringMarker_Forward (pMarker, 1) ;
		n	-- ;
	}
	if (nbuf < NELEMENTS (srbuf)) {
		if (TFAILED (TVarbuffer_Add (&vbufKilledText, srbuf, NELEMENTS (srbuf) - nbuf)))
			return	LMR_ERROR ;
	}

	pbuf	= TVarbuffer_GetBuffer (&vbufKilledText) ;
	nbuf	= TVarbuffer_GetUsage  (&vbufKilledText) ;
	fRetval	= lispMgr_CreateString (pLispMgr, pbuf, nbuf, &pEntKilledText) ;
	TVarbuffer_Uninitialize (&vbufKilledText) ;
	if (TFAILED (fRetval))
		return	LMR_ERROR ;

	pEntNil	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_NIL) ;
	if (TFAILED (lispMgr_CreateConscell (pLispMgr, pEntKilledText, pEntNil, &pEntKilledText)))
		return	LMR_ERROR ;
	lispMachine_SetCurrentSymbolValue (pLM, pEntKillRing, pEntKilledText) ;
	lispMachine_SetCurrentSymbolValue (pLM, pEntKRYP,     pEntKilledText) ;

	pEntThisCommand	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_THIS_COMMAND) ;
	lispMachine_SetCurrentSymbolValue (pLM, pEntThisCommand, pEntKillRegion) ;

	if (TFAILED (lispBuffer_DeleteChar (pLispMgr, pEntBuffer, nStart, nEnd - nStart)))
		return	LMR_ERROR ;

  exit_func:
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntKillRegion) ;
	return	LMR_RETURN ;
}

Boolean
lispMachine_currentKill (
	register TLispMachine*	pLM,
	register int			n,
	register Boolean		fDoNotMove,
	register TLispEntity**	ppEntRetval)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	register TLispEntity*	pEntKR ;
	register TLispEntity*	pEntKRYP ;
	TLispEntity*	pValKR ;
	TLispEntity*	pValKRYP ;
	int				nLenKR, nLenKRYP ;

	pEntKR	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_KILL_RING) ;
	pEntKRYP= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_KILL_RING_YANK_POINTER) ;
	assert (pEntKR != NULL && pEntKRYP != NULL) ;
	lispMachine_GetCurrentSymbolValue (pLM, pEntKRYP, &pValKRYP) ;
	lispMachine_GetCurrentSymbolValue (pLM, pEntKR,   &pValKR) ;

	/*	(nthcdr (mod (- n (length kill-ring-yank-pionter))
	 *		 	(length kill-ring))
	 *			kill-ring)
	 */
	if (TFAILED (lispEntity_GetLength (pLispMgr, pValKRYP, &nLenKRYP)) ||
		TFAILED (lispEntity_GetLength (pLispMgr, pValKRYP, &nLenKR)))
		return	FALSE ;
	if (nLenKR == 0)
		return	FALSE ;

	n	= ((n - nLenKRYP) % nLenKR) ;
	while (n > 0 && TFAILED (lispEntity_Nullp (pLispMgr, pValKR))) {
		if (TFAILED (lispEntity_GetCdr (pLispMgr, pValKR, &pValKR)))
			return	FALSE ;
		n	-- ;
	}
	
	if (! fDoNotMove) 
		lispMachine_SetCurrentSymbolValue (pLM, pEntKRYP, pValKR) ;
	lispEntity_GetCar (pLispMgr, pValKR, ppEntRetval) ;
	return	TRUE ;
}


