/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "local.h"
#include <stdio.h>
#include <assert.h>
#include "lmachinep.h"
#include "varbuffer.h"
#include "cstring.h"

typedef struct tagEmacsCharsetDef {
	const char*		m_strName ;
	int				m_nID ;
	unsigned long	m_uOffset ;
	unsigned long	m_uMask ;
} TEmacsCharsetDef ;

static	TLMRESULT	lispMachineState_mapconcatListApply		(TLispMachine*) ;
static	TLMRESULT	lispMachineState_mapconcatListPostApply	(TLispMachine*) ;
static	TLMRESULT	lispMachineState_mapconcatVectorApply	(TLispMachine*) ;
static	TLMRESULT	lispMachineState_mapconcatVectorPostApply(TLispMachine*) ;
static	TLMRESULT	lispMachineState_mapconcatStringApply	(TLispMachine*) ;
static	TLMRESULT	lispMachineState_mapconcatStringPostApply(TLispMachine*) ;
static	TLMRESULT	lispMachineState_mapconcatPostApplyCommon(TLispMachine*, TLMRESULT (*)(TLispMachine*)) ;
static	TLMRESULT	lispMachineState_mapconcatFinalize		(TLispMachine*) ;
static	TLMRESULT	lispMachineState_tryCompletionAlist		(TLispMachine*) ;
static	TLMRESULT	lispMachineState_tryCompAlistPostPredict(TLispMachine*) ;
static	TLMRESULT	lispMachineState_tryCompAlistFin		(TLispMachine*) ;

static const char*	astrSkkinputCharset []	= {
	"ascii",					"iso8859-1",
	"iso8859-2",				"iso8859-3",
	"iso8859-4",				"iso8859-5",
	"iso8859-6",				"iso8859-7",
	"iso8859-8",				"iso8859-9",
	"iso8859-14",				"iso8859-15",
	"japanese-jisx0201-1976",	"japanese-jisx0208-1978",
	"japanese-jisx0208-1983",	"japanese-jisx0212-1990",
	"chinese-gb2312-1980",		"chinese-gb12345-90",
	"chinese-gb7589-87",		"chinese-gb7590-87",
	"chinese-gb13131-91",		"chinese-gb13132-91",
	"korean-ksc5601-1987",		"korean-ksc5601-1992",
	"chinese-cns11643-1992",
} ;

/*	ࡢemacs  version ˤä charset-list ֤ͤɼ˰㤦Ȥ
 *	դʤä*/
static TEmacsCharsetDef	arEmacsCharsets []	= {
	{ "ascii",					KCHARSET_ASCII, 			0,			0x7F, },
	{ "latin-iso8859-1",		KCHARSET_ISO8859_1,			0,			0xFF, },
	{ "latin-iso8859-2",		KCHARSET_ISO8859_2,			0,			0xFF, },
	{ "latin-iso8859-3",		KCHARSET_ISO8859_3,			0,			0xFF, },
	{ "latin-iso8859-4",		KCHARSET_ISO8859_4,			0,			0xFF, },
	{ "cyrillic-iso8859-5",		KCHARSET_ISO8859_5,			0,			0xFF, },
	{ "arabic-iso8859-6",		KCHARSET_ISO8859_6,			0,			0xFF, },
	{ "latin-iso8859-7",		KCHARSET_ISO8859_7,			0,			0xFF, },
	{ "hebrew-iso8859-8",		KCHARSET_ISO8859_8,			0,			0xFF, },
	{ "latin-iso8859-9",		KCHARSET_ISO8859_9,			0,			0xFF, },
	{ "latin-iso8859-14",		KCHARSET_ISO8859_14,		0,			0xFF, },
	{ "latin-iso8859-15",		KCHARSET_ISO8859_15,		0,			0xFF, },
	{ "latin-jisx0201",			KCHARSET_JISX0201_1976,		0,			0x7F, },
	{ "katakana-jisx0201",		KCHARSET_JISX0201_1976,		0x80,		0x7F, },
	{ "japanese-jisx0208-1978",	KCHARSET_JISX0208_1978,		0,			0x7F7F, },
	{ "japanese-jisx0208",		KCHARSET_JISX0208_1983,		0,			0x7F7F, },
	{ "japanese-jisx0212",		KCHARSET_JISX0212_1990,		0,			0x7F7F, },
	{ "korean-ksc5601",			KCHARSET_KSC5601_1987,		0,			0x7F7F, },
	{ "korean-ksc5601-1992",	KCHARSET_KSC5601_1992,		0,			0x7F7F, },
	{ "chinese-gb2312",			KCHARSET_GB2312_1980,		0,			0x7F7F,	},
	{ "chinese-cns11643-1",		KCHARSET_CNS11643_1992,		0x010000,	0x7F7F, },
	{ "chinese-cns11643-2",		KCHARSET_CNS11643_1992,		0x020000,	0x7F7F, },
	{ "chinese-cns11643-3",		KCHARSET_CNS11643_1992,		0x030000,	0x7F7F, },
	{ "chinese-cns11643-4",		KCHARSET_CNS11643_1992,		0x040000,	0x7F7F, },
	{ "chinese-cns11643-5",		KCHARSET_CNS11643_1992,		0x050000,	0x7F7F, },
	{ "chinese-cns11643-6",		KCHARSET_CNS11643_1992,		0x060000,	0x7F7F, },
	{ "chinese-cns11643-7",		KCHARSET_CNS11643_1992,		0x070000,	0x7F7F, },
} ;

TLMRESULT
lispMachineState_String (
	register TLispMachine*		pLM)
{
	register TLispManager*	pLispMgr ;
	TVarbuffer		vbufStr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pElm ;
	Char			cc ;
	long			lValue ;

	assert (pLM != NULL) ;

	if (TFAILED (TVarbuffer_Initialize (&vbufStr, sizeof (Char))))
		return	LMR_ERROR ;
	
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	while (TFAILED (lispEntity_Nullp (pLispMgr, pArglist))) {
		if (TFAILED (lispEntity_GetCar (pLispMgr, pArglist, &pElm)) ||
			TFAILED (lispEntity_GetIntegerValue (pLispMgr, pElm, &lValue))) 
			break ;
		cc	= (Char) lValue ;
		if (TFAILED (TVarbuffer_Add (&vbufStr, &cc, 1)))
			break ;
		lispEntity_GetCdr (pLispMgr, pArglist, &pArglist) ;
	}
	if (TFAILED (lispEntity_Nullp (pLispMgr, pArglist))) {
		lispMachineCode_SetError (pLM) ;
	} else {
		TLispEntity*			pResult ;
		register const Char*	pString ;
		register int			nLength ;

		pString	= TVarbuffer_GetBuffer (&vbufStr) ;
		nLength	= TVarbuffer_GetUsage  (&vbufStr) ;
		if (TSUCCEEDED (lispMgr_CreateString (pLispMgr, pString, nLength, &pResult)))
			lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pResult) ;
	}
	TVarbuffer_Uninitialize (&vbufStr) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Concat (TLispMachine* pLM)
{
	TLispEntity*	pResult ;
	TLispEntity*	pArglist ;
	
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispMgr_Concat (pLM->m_pLispMgr, pArglist, NULL, &pResult))) {
		lispMachineCode_SetError (pLM) ;
	} else {
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pResult) ;
	}
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_StringEqual (TLispMachine* pLM)
{
	TLispManager*	pLispMgr ;
	TLispEntity*	pS1 ;
	TLispEntity*	pS2 ;
	TLispEntity*	pRetval ;
	TLispEntity*	pArglist ;
	const Char*		pStringS1 ;
	int				nStringS1 ;
	const Char*		pStringS2 ;
	int				nStringS2 ;

	assert (pLM      != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pS1)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pArglist, &pS2))) {
		goto	error ;
	}
	if (TSUCCEEDED (lispEntity_Stringp (pLispMgr, pS1))) {
		lispEntity_GetStringValue (pLispMgr, pS1, &pStringS1, &nStringS1) ;
	} else if (TSUCCEEDED (lispEntity_Symbolp (pLispMgr, pS1))) {
		lispEntity_GetSymbolName  (pLispMgr, pS1, &pStringS1, &nStringS1) ;
	} else {
		goto	error ;
	}
	if (TSUCCEEDED (lispEntity_Stringp (pLispMgr, pS2))) {
		lispEntity_GetStringValue (pLispMgr, pS2, &pStringS2, &nStringS2) ;
	} else if (TSUCCEEDED (lispEntity_Symbolp (pLispMgr, pS2))) {
		lispEntity_GetSymbolName  (pLispMgr, pS2, &pStringS2, &nStringS2) ;
	} else {
		goto	error ;
	}
#if defined (DEBUG)
	fprintf (stderr, "S1: (%d):", nStringS1) ;
	lispEntity_Print (pLispMgr, pS1) ;
	fprintf (stderr, ", S2: (%d):", nStringS2) ;
	lispEntity_Print (pLispMgr, pS2) ;
	fprintf (stderr, "\n") ;
#endif
	if (nStringS1 == nStringS2 && !Cstrncmp (pStringS1, pStringS2, nStringS1)) {
		lispMgr_CreateT   (pLispMgr, &pRetval) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pRetval) ;
	}
#if defined (DEBUG)
	fprintf (stderr, "string= returns ") ;
	lispEntity_Print (pLispMgr, pRetval) ;
	fprintf (stderr, "\n") ;
#endif
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;

 error:
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Mapconcat (TLispMachine* pLM)
{
	TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pFunction ;
	TLispEntity*	pSequence ;
	TLispEntity*	pSeparator ;
	TLispEntity*	pFunclist ;
	TLispEntity*	pFunctail ;
	TLispEntity*	pNil ;
	int			iType ;
	
	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar   (pLispMgr, pArglist, &pFunction)) ||
		TFAILED (lispEntity_GetCdr   (pLispMgr, pArglist, &pArglist)) ||
		TFAILED (lispEntity_GetCar   (pLispMgr, pArglist, &pSequence)) ||
		TFAILED (lispEntity_GetCadr  (pLispMgr, pArglist, &pSeparator)) ||
		TFAILED (lispEntity_GetType  (pLispMgr, pSequence, &iType))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	switch (iType) {
	case	LISPENTITY_CONSCELL:
		pLM->m_pState	= &lispMachineState_mapconcatListApply ;
		break ;
	case	LISPENTITY_VECTOR:
		pLM->m_pState	= &lispMachineState_mapconcatVectorApply ;
		break ;
	case	LISPENTITY_STRING:
		pLM->m_pState	= &lispMachineState_mapconcatStringApply ;
		break ;
	default:
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	
	lispMachineCode_PushLReg (pLM, LM_LREG_1) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_2) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_3) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_4) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_5) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_6) ;
	
	lispMgr_CreateNil      (pLispMgr, &pNil) ;
	if (TFAILED (lispMgr_CreateConscell (pLispMgr, pNil, pNil, &pFunctail)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_2, pFunctail) ;
	if (TFAILED (lispMgr_CreateConscell (pLispMgr, pFunction, pFunctail, &pFunclist)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_1, pFunclist) ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_3, pSequence) ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_4, pNil) ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_5, pNil) ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_6, pSeparator) ;
	
	lispMachineCode_PushVReg (pLM, LM_VREG_1) ;
	lispMachineCode_SetVRegI (pLM, LM_VREG_1, 0) ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_mapconcatListApply (TLispMachine* pLM)
{
	TLispManager*	pLispMgr ;
	TLispEntity*	pFunctail ;
	TLispEntity*	pFunclist ;
	TLispEntity*	pSequence ;
	TLispEntity*	pCar ;
	
	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_3, &pSequence) ;
	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pSequence))) {
		pLM->m_pState	= &lispMachineState_mapconcatFinalize ;
		return	LMR_CONTINUE ;
	}
	if (TFAILED (lispEntity_GetCar (pLispMgr, pSequence, &pCar))) {
		lispMachineCode_SetError (pLM) ;
		pLM->m_pState	= &lispMachineState_mapconcatFinalize ;
		return	LMR_CONTINUE ;
	}
	lispMachineCode_GetLReg   (pLM, LM_LREG_2,   &pFunctail) ;
	lispEntity_SetCar         (pLispMgr, pFunctail, pCar) ;
	lispMachineCode_GetLReg   (pLM, LM_LREG_1,   &pFunclist) ;
#if defined (DEBUG_LV99)
	fprintf (stderr, "pCar = ") ;
	lispEntity_Print (pLispMgr, pCar) ;
	fprintf (stderr, ", pFunctail = ") ;
	lispEntity_Print (pLispMgr, pFunctail) ;
	fprintf (stderr, ", pFunctlist = ") ;
	lispEntity_Print (pLispMgr, pFunclist) ;
	fprintf (stderr, "\n") ;
#endif
	
	lispMachineCode_PushState (pLM, &lispMachineState_mapconcatListPostApply) ;
	lispMachineCode_Cdr       (pLM, LM_LREG_3,   LM_LREG_3) ;
	lispMachineCode_SetLReg   (pLM, LM_LREG_ACC, pFunclist) ;
	pLM->m_pState	= &lispMachineState_Funcall ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_mapconcatListPostApply (TLispMachine* pLM)
{
	return	lispMachineState_mapconcatPostApplyCommon (pLM, &lispMachineState_mapconcatListApply) ;
}

TLMRESULT
lispMachineState_mapconcatVectorApply (TLispMachine* pLM)
{
	TLispManager*	pLispMgr ;
	TLispEntity*	pSequence ;
	TLispEntity*	pElement ;
	TLispEntity*	pFunctail ;
	TLispEntity*	pFunclist ;
	long		lIndex ;
	
	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	
	lispMachineCode_GetLReg  (pLM, LM_LREG_3, &pSequence) ;
	lispMachineCode_GetVRegI (pLM, LM_VREG_1, &lIndex) ;
	if (TFAILED (lispEntity_GetVectorElement (pLispMgr, pSequence, lIndex, &pElement))) {
		/*	pSequence  VECTOR ǤΤϳǧƤġ*/
		pLM->m_pState	= &lispMachineState_mapconcatFinalize ;
		return	LMR_CONTINUE ;
	}
	lIndex	++ ;
	lispMachineCode_SetVRegI  (pLM, LM_VREG_1,   lIndex) ;
	
	lispMachineCode_GetLReg   (pLM, LM_LREG_2,   &pFunctail) ;
	lispEntity_SetCar         (pLispMgr, pFunctail, pElement) ;
	lispMachineCode_GetLReg   (pLM, LM_LREG_1,   &pFunclist) ;
	
	lispMachineCode_PushState (pLM, &lispMachineState_mapconcatVectorPostApply) ;
	lispMachineCode_SetLReg   (pLM, LM_LREG_ACC, pFunclist) ;
	pLM->m_pState	= &lispMachineState_Funcall ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_mapconcatVectorPostApply (TLispMachine* pLM)
{
	return	lispMachineState_mapconcatPostApplyCommon (pLM, &lispMachineState_mapconcatVectorApply) ;
}

TLMRESULT
lispMachineState_mapconcatStringApply (TLispMachine* pLM)
{
	TLispManager*	pLispMgr ;
	TLispEntity*	pSequence ;
	TLispEntity*	pElement ;
	TLispEntity*	pFunctail ;
	TLispEntity*	pFunclist ;
	long		lIndex ;
	Char		cc ;
	
	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;
	
	lispMachineCode_GetLReg  (pLM, LM_LREG_3, &pSequence) ;
	lispMachineCode_GetVRegI (pLM, LM_VREG_1, &lIndex) ;
	if (TFAILED (lispEntity_GetStringElement (pLispMgr, pSequence, lIndex, &cc))) {
		/*	pSequence  VECTOR ǤΤϳǧƤġ*/
		pLM->m_pState	= &lispMachineState_mapconcatFinalize ;
		return	LMR_CONTINUE ;
	}
	if (TFAILED (lispMgr_CreateInteger (pLispMgr, cc, &pElement)))
		return	LMR_ERROR ;

	lIndex	++ ;
	lispMachineCode_SetVRegI  (pLM, LM_VREG_1,   lIndex) ;
	
	lispMachineCode_GetLReg   (pLM, LM_LREG_2,   &pFunctail) ;
	lispEntity_SetCar         (pLispMgr, pFunctail, pElement) ;
	lispMachineCode_GetLReg   (pLM, LM_LREG_1,   &pFunclist) ;
	
	lispMachineCode_PushState (pLM, &lispMachineState_mapconcatStringPostApply) ;
	lispMachineCode_SetLReg   (pLM, LM_LREG_ACC, pFunclist) ;
	pLM->m_pState	= &lispMachineState_Funcall ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_mapconcatStringPostApply (TLispMachine* pLM)
{
	return	lispMachineState_mapconcatPostApplyCommon (pLM, &lispMachineState_mapconcatStringApply) ;
}

TLMRESULT
lispMachineState_mapconcatPostApplyCommon (TLispMachine* pLM, TLMRESULT (*pNextState)(TLispMachine*))
{
	TLispManager*	pLispMgr ;
	TLispEntity*	pTail ;
	TLispEntity*	pResult ;
	TLispEntity*	pNewTail ;
	TLispEntity*	pNil ;
	
#if defined (DEBUG_LV99)
	fprintf (stderr, "state = mapconcat-post-apply-common\n") ;
#endif
	
	assert (pLM != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) {
		pLM->m_pState	= &lispMachineState_mapconcatFinalize ;
		return	LMR_CONTINUE ;
	}
	lispMachineCode_GetLReg (pLM, LM_LREG_5,   &pTail) ;
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pResult) ;

#if defined (DEBUG_LV99)
	fprintf (stderr, "ACC = ") ;
	lispEntity_Print (pLispMgr, pResult) ;
	fprintf (stderr, ", REG_5 = ") ;
	lispEntity_Print (pLispMgr, pTail) ;
	fprintf (stderr, "\n") ;
#endif
	
	lispMgr_CreateNil      (pLispMgr, &pNil) ;
	lispMgr_CreateConscell (pLispMgr, pResult, pNil, &pNewTail) ;
	lispMachineCode_SetTail (pLM, LM_LREG_4, LM_LREG_5, pNewTail) ;
	pLM->m_pState	= pNextState ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_mapconcatFinalize (TLispMachine* pLM)
{
	if (!LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) {
		TLispEntity*	pResult ;
		TLispEntity*	pRetval ;
		TLispEntity*	pSeparator ;
		
		lispMachineCode_GetLReg (pLM, LM_LREG_4, &pResult) ;
		lispMachineCode_GetLReg (pLM, LM_LREG_6, &pSeparator) ;

#if defined (DEBUG_LV99)
		fprintf (stderr, "REG_4 = ") ;
		lispEntity_Print (pLM->m_pLispMgr, pResult) ;
		fprintf (stderr, ", REG_6 = ") ;
		lispEntity_Print (pLM->m_pLispMgr, pSeparator) ;
		fprintf (stderr, "\n") ;
#endif
		if (TFAILED (lispMgr_Concat (pLM->m_pLispMgr, pResult, pSeparator, &pRetval))) {
			lispMachineCode_SetError (pLM) ;
		} else {
			lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
		}
	}
	lispMachineCode_PopVReg  (pLM, LM_VREG_1) ;
	lispMachineCode_PopLReg  (pLM, LM_LREG_6) ;
	lispMachineCode_PopLReg  (pLM, LM_LREG_5) ;
	lispMachineCode_PopLReg  (pLM, LM_LREG_4) ;
	lispMachineCode_PopLReg  (pLM, LM_LREG_3) ;
	lispMachineCode_PopLReg  (pLM, LM_LREG_2) ;
	lispMachineCode_PopLReg  (pLM, LM_LREG_1) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Substring (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pArray ;
	TLispEntity*	pTo ;
	TLispEntity*	pFrom ;
	TLispEntity*	pRet ;
	long		lFrom, lTo ;
	
	pLispMgr	= pLM->m_pLispMgr ;
	
	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pArray)) ||
		TFAILED (lispEntity_GetCdr  (pLispMgr, pArglist, &pArglist)) ||
		TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pFrom)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pArglist, &pTo)) ||
		TFAILED (lispEntity_Arrayp   (pLispMgr, pArray)) ||
		TFAILED (lispEntity_GetIntegerValue (pLispMgr, pFrom, &lFrom))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispEntity_GetIntegerValue (pLispMgr, pTo, &lTo))) {
		int		iLength ;
		if (TFAILED (lispEntity_Nullp (pLispMgr, pTo)) ||
			TFAILED (lispEntity_GetLength (pLispMgr, pArray, &iLength))) {
			lispMachineCode_SetError (pLM) ;
			return	LMR_RETURN ;
		}
		lTo	= (long) iLength ;
	}
	if (TFAILED (lispMgr_Substring (pLispMgr, pArray, lFrom, lTo, &pRet))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRet) ;
	return	LMR_RETURN ;
}

/*
 *	(length SEQUENCE)
 *
 *	vector, list, string Τ줫Ǥ SEQUENCE Ĺ֤
 *	ΥХȥɴؿ֥ȤޤƤ뤬ʤΤϤʤ!
 *	⤷̩ʥХȿߤСstring-bytes Ȥȡ饯
 *	ɽΤ˲ӥåȻȤƤ뤫ϴĶ¸Ǥ롣
 */
TLMRESULT
lispMachineState_Length (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pSequence ;
	TLispEntity*	pRetval ;
	int		nLength ;
	
	pLispMgr	= pLM->m_pLispMgr ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pSequence)) ||
		TFAILED (lispEntity_GetLength (pLispMgr, pSequence, &nLength))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispMgr_CreateInteger (pLispMgr, nLength, &pRetval)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

/*
 *	 STRING 򥭥饯string κǽʸѴ롣
 *	ޥХȥ饯롣
 */
TLMRESULT
lispMachineState_StringToChar (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pString ;
	TLispEntity*	pArglist ;
	TLispEntity*	pRetval ;
	Char		cc ;

	assert (pLM      != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pString)) ||
		TFAILED (lispEntity_Stringp (pLispMgr, pString))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}

	if (TFAILED (lispEntity_GetStringElement (pLispMgr, pString, 0, &cc)))
		cc	= 0 ;
	if (TFAILED (lispMgr_CreateInteger (pLispMgr, cc, &pRetval))) 
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

/*
 *	 CHAR ʸȤƴޤʸѴ롣
 */
TLMRESULT
lispMachineState_CharToString (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pChar ;
	TLispEntity*	pArglist ;
	TLispEntity*	pRetval ;
	long		lValue ;
	Char		cc ;

	assert (pLM      != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pChar)) ||
		TFAILED (lispEntity_GetIntegerValue (pLispMgr, pChar, &lValue))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	cc	= (Char) lValue ;
	if (Char_Charset (cc) >= MAX_CHARSET) {
		if (Char_Charset (cc) != KCHARSET_XCHAR) 
			goto	error ;
		cc	= Char_MakeAscii (Char_Code (cc) & 0x7F) ;
	}
	if (TFAILED (lispMgr_CreateString (pLispMgr, &cc, 1, &pRetval))) 
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;

  error:
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_NumberToString (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pNumber ;
	TLispEntity*	pArglist ;
	TLispEntity*	pRetval ;
	Char		aChBuf [256] ;
	char		achBuf [256] ;
	int			nLen ;

	assert (pLM      != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pArglist, &pNumber)) ||
		TFAILED (lispEntity_Numberp (pLispMgr, pNumber))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TSUCCEEDED (lispEntity_Integerp (pLispMgr, pNumber))) {
		long	lNumber ;
		(void) lispEntity_GetIntegerValue (pLispMgr, pNumber, &lNumber) ;
		snprintf (achBuf, NELEMENTS (achBuf) - 1, "%ld", lNumber) ;
	} else {
		float	fNumber ;
		(void) lispEntity_GetFloatValue (pLispMgr, pNumber, &fNumber) ;
		snprintf (achBuf, NELEMENTS (achBuf) - 1, "%f", fNumber) ;
	}
	achBuf [NELEMENTS (achBuf) - 1]	= '\0' ;
	nLen	= strlen (achBuf) ;
	strtocstr (aChBuf, achBuf, nLen) ;
	if (TFAILED (lispMgr_CreateString (pLispMgr, aChBuf, nLen, &pRetval))) 
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Upcase (
	register TLispMachine*	pLM)
{
	TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pObj ;
	TLispEntity*	pRetval ;

	assert (pLM      != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar   (pLispMgr, pArglist, &pObj)) ||
		TFAILED (lispEntity_Upcase (pLispMgr, pObj, &pRetval))) {
		lispMachineCode_SetError (pLM) ;
	} else {
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	}
	return	LMR_RETURN ;
}

TLMRESULT
lispMachineState_Downcase (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pArglist ;
	TLispEntity*	pObj ;
	TLispEntity*	pRetval ;

	assert (pLM      != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pArglist) ;
	if (TFAILED (lispEntity_GetCar   (pLispMgr, pArglist, &pObj)) ||
		TFAILED (lispEntity_Downcase (pLispMgr, pObj, &pRetval))) {
		lispMachineCode_SetError (pLM) ;
	} else {
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pRetval) ;
	}
	return	LMR_RETURN ;
}

/*
 *	format
 *
 *	%s ʸɽ롣ºݤˤ `princ' ǥ֥Ȥɽ
 *	롣
 *	%d 10ʤǿɽ롣
 *	%e  exponentail ɽǿɽ롣`` e ؿ'' η
 *	%f  decimal-point ɽǿɽ롣``XXXX.XXXX'' η
 *	%g  exponential ɽޤ decimal-point ɽǿɽ롣
 *	ʤʸɽ֡
 *	%c ϥ󥰥륭饯Ȥɽ롣
 *	%S  s-expression ǥ֥Ȥɽ롣%s Ȥΰ㤤
 *	%d, %o, %x, %e, %f, %g, %c ϿͤǤʤФʤʤ
 *	%%  % ˤʤ롣
 */
TLMRESULT
lispMachineState_Format (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr ;
	TLispEntity*	pEntArglist ;
	const Char*		pFormat ;
	int				nFormat ;
	TLispEntity*	pEntRetval ;
	TLispEntity*	pEntString ;

	assert (pLM      != NULL) ;
	pLispMgr	= pLM->m_pLispMgr ;
	assert (pLispMgr != NULL) ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntString)) ||
		TFAILED (lispEntity_GetStringValue (pLispMgr, pEntString, &pFormat, &nFormat)) ||
		TFAILED (lispEntity_GetCdr  (pLispMgr, pEntArglist, &pEntArglist))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispEntity_Format (pLispMgr, pFormat, nFormat, pEntArglist, &pEntRetval))) {
		lispMachineCode_SetError (pLM) ;
	} else {
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	}
	return	LMR_RETURN ;
}

/*
 *	PREDICATE ¸ߤ硢ޥåΤ PREDICATE Ϥ
 *	ΥޥåϰƬפɤϤ̤ FUNCALL ʡ
 *	Ϥ줿Τ EVAL 櫓ǤϤʤΤǡ
 */
TLMRESULT
lispMachineState_TryCompletion (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntString ;
	TLispEntity*	pEntAlist ;
	TLispEntity*	pEntPredicate ;
	TLispEntity*	pEntNil ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntString)) ||
		TFAILED (lispEntity_Stringp (pLispMgr, pEntString)) ||
		TFAILED (lispEntity_GetCdr  (pLispMgr, pEntArglist, &pEntArglist)) ||
		TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntAlist)) ||
		TFAILED (lispEntity_Consp   (pLispMgr, pEntAlist)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntPredicate))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	lispMachineCode_PushLReg (pLM, LM_LREG_1) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_2) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_3) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_4) ;
	lispMachineCode_PushLReg (pLM, LM_LREG_5) ;
	lispMachineCode_PushVReg (pLM, LM_VREG_1) ;
	lispMachineCode_PushVReg (pLM, LM_VREG_2) ;
	lispMachineCode_PushVReg (pLM, LM_VREG_3) ;

	lispMgr_CreateNil (pLispMgr, &pEntNil) ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_1, pEntString) ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_2, pEntAlist) ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_3, pEntPredicate) ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_4, pEntNil) ;
	lispMachineCode_SetLReg  (pLM, LM_LREG_5, pEntNil) ;
	lispMachineCode_SetVRegI (pLM, LM_VREG_1, 0) ;
	lispMachineCode_SetVRegI (pLM, LM_VREG_2, 0) ;
	lispMachineCode_SetVRegI (pLM, LM_VREG_3, 0) ;
	pLM->m_pState	= &lispMachineState_tryCompletionAlist ;	
	return	LMR_CONTINUE ;
}


/*
 *	try-completion ưϡ
 *	(1)	ʬ SYMBOL ̾ PREFIX ˤʤäƤ뤫ɤ֤
 *		ɾ롣
 *	(2)	ʬ PREFIX ˤʤäƤСޤǤ PREFIX Ȥʤä
 *		Ƕ̤ʬȴФ
 *	(3)	EXACT ޥå1Ĥ䤬ʤäˤ t ֤
 */
TLMRESULT
lispMachineState_tryCompletionAlist (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*			pEntString ;
	const Char*				pString ;
	int						nString ;
	TLispEntity*			pEntMaxMatch ;
	const Char*				pStrMaxMatch ;
	int						nDummy ;
	long					lMaxMatch ;
	TLispEntity*			pEntAlist ;
	TLispEntity*			pEntPredict ;
	TLispEntity*			pEntTarget ;
	TLispEntity*			apEntities [2] ;
	TLispEntity*			pEntPredictForm ;
	long					lHit ;
	const Char*				pStrTarget ;
	int						nStrTarget ;

	lispMachineCode_GetLReg   (pLM, LM_LREG_1, &pEntString) ;
	lispEntity_GetStringValue (pLispMgr, pEntString, &pString, &nString) ;
	lispMachineCode_GetLReg   (pLM, LM_LREG_2, &pEntAlist) ;
	lispMachineCode_GetLReg   (pLM, LM_LREG_3, &pEntPredict) ;
	lispMachineCode_GetLReg   (pLM, LM_LREG_4, &pEntMaxMatch) ;
	lispMachineCode_GetVRegI  (pLM, LM_VREG_1, &lMaxMatch) ;
	lispMachineCode_GetVRegI  (pLM, LM_VREG_3, &lHit) ;

	if (TSUCCEEDED (lispEntity_Nullp (pLispMgr, pEntMaxMatch))) {
		pStrMaxMatch	= pString ;
		lMaxMatch		= nString ;
	} else {
		lispEntity_GetStringValue (pLispMgr, pEntMaxMatch, &pStrMaxMatch, &nDummy) ;
	}

	while (TFAILED (lispEntity_Nullp (pLispMgr, pEntAlist))) {
		if (TFAILED (lispEntity_GetCaar (pLispMgr, pEntAlist, &pEntTarget)))
			goto	error ;
		if (TSUCCEEDED (lispEntity_GetStringValue (pLispMgr, pEntTarget, &pStrTarget, &nStrTarget)) &&
			nStrTarget >= nString) {
			register const Char*	pPtrLeft ;
			register const Char*	pPtrRight ;
			register int			nMaxCheck ;
			register int			i ;

			nMaxCheck	= (nStrTarget < lMaxMatch)? nStrTarget : lMaxMatch ;
			pPtrLeft	= pStrMaxMatch ;
			pPtrRight	= pStrTarget ;
			for (i = 0 ; i < nMaxCheck ; i ++)
				if (*pPtrLeft ++ != *pPtrRight ++)
					break ;
			if (i >= nString) {
				lMaxMatch		= (lHit == 0)? nStrTarget : i ;
				pEntMaxMatch	= pEntTarget ;
				if (TFAILED (lispEntity_Nullp (pLispMgr, pEntPredict)))
					goto	found ;
				pStrMaxMatch	= pStrTarget ;
				lHit	++ ;
			}
		}
		if (TFAILED (lispEntity_GetCdr (pLispMgr, pEntAlist, &pEntAlist)))
			goto	error ;
	}

	if (pEntMaxMatch != pEntString)
		lispMachineCode_SetLReg (pLM, LM_LREG_4, pEntMaxMatch) ;
	lispMachineCode_SetVRegI (pLM, LM_VREG_1, lMaxMatch) ;
	lispMachineCode_SetVRegI (pLM, LM_VREG_3, lHit) ;
	pLM->m_pState	= &lispMachineState_tryCompAlistFin ;
	return	LMR_CONTINUE ;

  found:
	lispEntity_GetCar (pLispMgr, pEntAlist, &pEntTarget) ;
	lispEntity_GetCdr (pLispMgr, pEntAlist, &pEntAlist) ;

	lispMachineCode_SetLReg   (pLM, LM_LREG_2, pEntAlist) ;
	lispMachineCode_SetVRegI  (pLM, LM_VREG_2, lMaxMatch) ;
	lispMachineCode_SetLReg   (pLM, LM_LREG_5, pEntMaxMatch) ;
	lispMachineCode_PushState (pLM, &lispMachineState_tryCompAlistPostPredict) ;
	apEntities [0]	= pEntPredict ;
	apEntities [1]	= pEntTarget ;
	if (TFAILED (lispMgr_CreateList (pLispMgr, apEntities, 2, &pEntPredictForm)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg   (pLM, LM_LREG_ACC, pEntPredictForm) ;
	pLM->m_pState	= &lispMachineState_Funcall ;
	return	LMR_CONTINUE ;

  error:
	lispMachineCode_SetError (pLM) ;
	pLM->m_pState	= &lispMachineState_tryCompAlistFin ;
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_tryCompAlistPostPredict (
	register TLispMachine*	pLM)
{
	if (LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) {
		pLM->m_pState	= lispMachineState_tryCompAlistFin ;
	} else {
		register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
		TLispEntity*	pEntResult ;

		lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntResult) ;
		if (TFAILED (lispEntity_Nullp (pLispMgr, pEntResult))) {
			long	lHit ;
			lispMachineCode_MoveLReg (pLM, LM_LREG_4, LM_LREG_5) ;
			lispMachineCode_MoveVReg (pLM, LM_VREG_1, LM_VREG_2) ;
			lispMachineCode_GetVRegI (pLM, LM_VREG_3, &lHit) ;
			lHit	++ ;
			lispMachineCode_SetVRegI (pLM, LM_VREG_3, lHit) ;
		}
		pLM->m_pState	= &lispMachineState_tryCompletionAlist ;
	}
	return	LMR_CONTINUE ;
}

TLMRESULT
lispMachineState_tryCompAlistFin (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;

	if (!LISPMACHINE_EXCEPTIONORSIGNALP (pLM)) {
		TLispEntity*	pEntString ;
		TLispEntity*	pEntResult ;
		TLispEntity*	pEntMaxMatch ;
		const Char*		pString ;
		int				nString ;
		long			lLength ;

		lispMachineCode_GetLReg   (pLM, LM_LREG_1, &pEntString) ;
		lispEntity_GetStringValue (pLispMgr, pEntString, &pString, &nString) ;
		lispMachineCode_GetLReg   (pLM, LM_LREG_4, &pEntMaxMatch) ;
		lispMachineCode_GetVRegI  (pLM, LM_VREG_1, &lLength) ;
		if (TFAILED (lispEntity_Nullp (pLispMgr, pEntMaxMatch))) {
			const Char*		pStrMatch ;
			int				nStrMatch ;
			long			lHit ;
			assert (lLength > 0) ;
			lispMachineCode_GetVRegI (pLM, LM_VREG_3, &lHit) ;
			assert (lHit    > 0) ;
			lispEntity_GetStringValue (pLispMgr, pEntMaxMatch, &pStrMatch, &nStrMatch) ;
			if (lHit == 1 && lLength == nString) {
				lispMgr_CreateT (pLispMgr, &pEntResult) ;
			} else {
				if (TFAILED (lispMgr_CreateString (pLispMgr, pStrMatch, lLength, &pEntResult)))
					return	LMR_ERROR ;
			}
		} else {
			pEntResult	= pEntMaxMatch ;	/* = nil */
		}
		lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntResult) ;
	}
	lispMachineCode_PopVReg (pLM, LM_VREG_3) ;
	lispMachineCode_PopVReg (pLM, LM_VREG_2) ;
	lispMachineCode_PopVReg (pLM, LM_VREG_1) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_5) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_4) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_3) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_2) ;
	lispMachineCode_PopLReg (pLM, LM_LREG_1) ;
	return	LMR_RETURN ;
}

/*
 *
 */
TLMRESULT
lispMachineState_MakeString (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*		pEntArglist ;
	TLispEntity*		pEntLength ;
	TLispEntity*		pEntInit ;
	TLispEntity*		pEntRet ;
	long				lLength, lInit ;
	TVarbuffer			vbuf ;
	register Boolean	fRetval ;
	register Char*		pCh ;
	register int		nCh ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntLength) ;
	lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntInit) ;
	if (TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntLength, &lLength)) ||
		TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntInit,   &lInit))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (TVarbuffer_Initialize (&vbuf, sizeof (Char))) ||
		TFAILED (TVarbuffer_Require (&vbuf, lLength))) 
		return	LMR_ERROR ;

	pCh	= TVarbuffer_GetBuffer (&vbuf) ;
	nCh	= lLength ;
	while (nCh -- > 0)
		*pCh ++	= (Char) lInit ;
	fRetval	= lispMgr_CreateString (pLispMgr, TVarbuffer_GetBuffer (&vbuf), lLength, &pEntRet) ;
	TVarbuffer_Uninitialize (&vbuf) ;
	if (TFAILED (fRetval)) 
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRet) ;
	return	LMR_RETURN ;
}

/*	built-in function:
 *		(read-from-string STRING &optional START END)
 *
 *	STRING ˤäƥƥȤȤɽƤ Lisp ɽ1ɤ
 *	ࡣ
 */
TLMRESULT
lispMachineState_ReadFromString (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	 = pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntSTRING ;
	TLispEntity*	pEntObjRead ;
	TLispEntity*	pEntStrIndex ;
	TLispEntity*	pEntSTART ;
	TLispEntity*	pEntEND ;
	TLispEntity*	pEntRetval ;
	const Char*		pStrSTRING ;
	int				nStrSTRING ;
	int				nIndex ;
	long			lEnd, lStart ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntSTRING) ;
	if (TFAILED (lispEntity_GetStringValue (pLispMgr, pEntSTRING, &pStrSTRING, &nStrSTRING)))
		goto	error ;
	lispEntity_GetCdr  (pLispMgr, pEntArglist, &pEntArglist) ;
	lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntSTART) ;
	if (TFAILED (lispEntity_Nullp (pLispMgr, pEntSTART))) {
		if (TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntSTART, &lStart))) 
			goto	error ;
	} else {
		lStart	= 0 ;
	}
	lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntEND) ;
	if (TFAILED (lispEntity_Nullp (pLispMgr, pEntEND))) {
		if (TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntEND, &lEnd)))
			goto	error ;
	} else {
		lEnd	= nStrSTRING ;
	}
	if (lStart > lEnd || lEnd > nStrSTRING || lStart < 0)
		goto	error ;

	pEntObjRead	= lispMgr_ParseString (pLispMgr, pStrSTRING + lStart, lEnd - lStart, &nIndex) ;
	if (pEntObjRead == NULL) 
		goto	error ;
	lispEntity_AddRef (pLispMgr, pEntObjRead) ;
	if (TFAILED (lispMgr_CreateInteger  (pLispMgr, nIndex + lStart, &pEntStrIndex)))
		return	LMR_ERROR ;
	lispEntity_AddRef (pLispMgr, pEntStrIndex) ;
	if (TFAILED (lispMgr_CreateConscell (pLispMgr, pEntObjRead, pEntStrIndex, &pEntRetval)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	lispEntity_Release (pLispMgr, pEntObjRead) ;
	lispEntity_Release (pLispMgr, pEntStrIndex) ;
	return	LMR_RETURN ;

  error:
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

/*	built-in function:
 *		(string-to-number STRING &optional BASE)
 *
 *	μǤ floating-point value ưʤ
 */
TLMRESULT
lispMachineState_StringToNumber (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntSTRING ;
	TLispEntity*	pEntBASE ;
	const Char*		pStrSTRING ;
	int				nStrSTRING ;
	register int	n ;
	long			lBASE ;
	register long	lValue, lSign ;
	TLispEntity*	pEntRetval ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntSTRING) ;
	if (TFAILED (lispEntity_GetStringValue (pLispMgr, pEntSTRING, &pStrSTRING, &nStrSTRING)))
		goto	error ;
	lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntBASE) ;
	if (TFAILED (lispEntity_Nullp (pLispMgr, pEntBASE))) {
		if (TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntBASE, &lBASE)))
			goto	error ;
		if (lBASE < 2 || lBASE > 16)
			goto	error ;
	} else {
		lBASE	= 10 ;
	}
	lValue	= 0 ;
	lSign	= +1 ;
	if (nStrSTRING > 0) {
		if (*pStrSTRING == '-') {
			lSign	= -1 ;
			pStrSTRING	++ ;
			nStrSTRING	-- ;
		} else if (*pStrSTRING == '+') {
			lSign	= +1 ;
			pStrSTRING	++ ;
			nStrSTRING	-- ;
		}
	}
	while (nStrSTRING -- > 0) {
		if ('0' <= *pStrSTRING && *pStrSTRING <= '9') {
			n	= *pStrSTRING - '0' ;
		} else if ('a' <= *pStrSTRING && *pStrSTRING <= 'f') {
			n	= *pStrSTRING - 'a' + 10 ;
		} else if ('A' <= *pStrSTRING && *pStrSTRING <= 'F') {
			n	= *pStrSTRING - 'A' + 10 ;
		} else {
			break ;
		}
		if (n >= lBASE)
			break ;
		lValue	= lValue * lBASE + n ;
		pStrSTRING	++ ;
	}
	if (TFAILED (lispMgr_CreateInteger (pLispMgr, lValue, &pEntRetval)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;

  error:
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

/*
 *	char-charset is a built-in function.
 *	(char-charset CH)
 *
 *	Return charset of CHAR.
 */
TLMRESULT
lispMachineState_CharCharset (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntCH ;
	TLispEntity*	pEntRetval ;
	Char			cc ;
	register int	nSet ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar (pLispMgr, pEntArglist, &pEntCH) ;
	if (TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntCH, &cc))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (Char_IsAscii (cc)) {
		nSet	= KCHARSET_ASCII ;
	} else {
		nSet	= Char_Charset (cc) ;
	}
	if (nSet < KCHARSET_ASCII || nSet >= MAX_CHARSET) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispMgr_InternSymbolA (pLispMgr, astrSkkinputCharset [nSet], strlen (astrSkkinputCharset [nSet]), &pEntRetval)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*	built-in function:
 *		(string-lessp S1 S2)
 *
 *	⤷ǽΰʸ2ܤΰʸ꼭Ǿ
 *	t ֤ʸʸϽפǤ롣ܥǤ⤫ޤʤ
 *	ˤϡȤ print name Ȥ롣
 */
TLMRESULT
lispMachineState_StringLessp (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntS1 ;
	TLispEntity*	pEntS2 ;
	const Char*		pStrS1 ;
	int				nStrS1 ;
	const Char*		pStrS2 ;
	int				nStrS2 ;
	TLispEntity*	pEntRetval ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntS1) ;
	lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntS2) ;
	if ((TFAILED (lispEntity_GetStringValue (pLispMgr, pEntS1, &pStrS1, &nStrS1)) &&
		 TFAILED (lispEntity_GetSymbolName (pLispMgr, pEntS1, &pStrS1, &nStrS1))) ||
		(TFAILED (lispEntity_GetStringValue (pLispMgr, pEntS2, &pStrS2, &nStrS2)) &&
		 TFAILED (lispEntity_GetSymbolName (pLispMgr, pEntS2, &pStrS2, &nStrS2)))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (cmemcmp (pStrS1, nStrS1, pStrS2, nStrS2) < 0) {
		lispMgr_CreateT   (pLispMgr, &pEntRetval) ;
	} else {
		lispMgr_CreateNil (pLispMgr, &pEntRetval) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*	(string-width STR)
 *
 *	Return width of STRING when displayed in the current buffer.
 *	Width is measured by how many columns it occupies on the screen.
 *	When calculating width of a multibyte character in STRING,
 *	only the base leading-code is considered; the validity of
 *	the following bytes is not checked.
 */
TLMRESULT
lispMachineState_StringWidth (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntSTR ;
	register int	nWidth ;
	TLispEntity*	pEntRetval ;
	const Char*		pString ;
	int				nString ;
	register int	nCharset ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntSTR) ;
	if (TFAILED (lispEntity_GetStringValue (pLispMgr, pEntSTR, &pString, &nString))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	nWidth	= 0 ;
	while (nString > 0) {
		nWidth	++ ;
		nCharset	= Char_Charset (*pString) ;
		if (KCHARSET_2BYTES_CHARACTER <= nCharset && nCharset < MAX_CHARSET)
			nWidth	++ ;
		pString	++ ;
		nString	-- ;
	}
	if (TFAILED (lispMgr_CreateInteger (pLispMgr, (long) nWidth, &pEntRetval)))
		return	LMR_ERROR ;
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}

/*	(make-char CHARSET &optional C1 C2)
 */
TLMRESULT
lispMachineState_MakeChar (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntCharset ;
	TLispEntity*	pEntC1 ;
	TLispEntity*	pEntC2 ;
	const Char*		strSymName ;
	int				nSymName ;
	register int	i ;
	register const TEmacsCharsetDef*	pCharset ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntCharset)) ||
		TFAILED (lispEntity_Symbolp (pLispMgr, pEntCharset)) ||
		TFAILED (lispEntity_GetCdr  (pLispMgr, pEntArglist, &pEntArglist)) ||
		TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntC1)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntC2))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	(void) lispEntity_GetSymbolName (pLispMgr, pEntCharset, &strSymName, &nSymName) ;
	pCharset	= arEmacsCharsets ;
	for (i = 0 ; i < NELEMENTS (arEmacsCharsets) ; i ++, pCharset ++) {
		register int	nLength	= strlen (pCharset->m_strName) ;
		if (!cmemccmp (strSymName, nSymName, pCharset->m_strName, nLength) &&
			nSymName == nLength) {
			TLispEntity*	pEntRetval ;
			register Char	cc ;
			long			lC1, lC2 ;

			if (lispEntity_Nullp (pLispMgr, pEntC1)) {
				lC1	= 0 ;
			} else {
				if (TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntC1, &lC1)))
					goto	error ;
				lC1	&= 0x00FF ;
			}
			if (lispEntity_Nullp (pLispMgr, pEntC2)) {
				lC2	=0 ;
			} else {
				if (TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntC2, &lC2)))
					goto	error ;
				lC2	&= 0x00FF ;
			}
			cc	= Char_Make (pCharset->m_nID, pCharset->m_uOffset + (((lC1 << 8) | lC2) & pCharset->m_uMask)) ;
			if (TFAILED (lispMgr_CreateInteger (pLispMgr, cc, &pEntRetval)))
				return	LMR_ERROR ;
			lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
			return	LMR_RETURN ;
		}
	}
  error:
	lispMachineCode_SetError (pLM) ;
	return	LMR_RETURN ;
}

/*
 *	Return t if OBJECT is a valid normal character.
 *	If optional arg GENERICP is non-nil, also return t if OBJECT is
 *	a valid generic character.
 *
 *	ࡢvalid normal character  valid generic character 
 *	褬㤦
 */
TLMRESULT
lispMachineState_CharValidp (
	register TLispMachine*	pLM)
{
	register TLispManager*	pLispMgr	= pLM->m_pLispMgr ;
	TLispEntity*	pEntArglist ;
	TLispEntity*	pEntOBJECT ;
	TLispEntity*	pEntGENERICP ;
	TLispEntity*	pEntRetval ;
	Char			cc ;
	register int	nCharset ;
	register unsigned long	uCode ;
	register Boolean	fRetval, fGenericp ;

	lispMachineCode_GetLReg (pLM, LM_LREG_ACC, &pEntArglist) ;
	if (TFAILED (lispEntity_GetCar  (pLispMgr, pEntArglist, &pEntOBJECT)) ||
		TFAILED (lispEntity_GetCadr (pLispMgr, pEntArglist, &pEntGENERICP))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	if (TFAILED (lispEntity_GetIntegerValue (pLispMgr, pEntOBJECT, &cc))) {
		lispMachineCode_SetError (pLM) ;
		return	LMR_RETURN ;
	}
	fGenericp	= TFAILED (lispEntity_Nullp (pLispMgr, pEntGENERICP)) ;
	nCharset	= Char_Charset (cc) ;
	uCode		= Char_Code (cc) ;
	if (nCharset < KCHARSET_ASCII || nCharset >= MAX_CHARSET) {
		fRetval	= False ;
	} else {
		if (KCHARSET_7BIT_CHARACTER <= nCharset &&
			nCharset < KCHARSET_1BYTE_CHARACTER) {
			fRetval	= (uCode < 0x80) ;
		} else if (KCHARSET_1BYTE_CHARACTER <= nCharset &&
				   nCharset < KCHARSET_2BYTES_CHARACTER) {
			fRetval	= (uCode < 0x100) ;
		} else if (KCHARSET_2BYTES_CHARACTER <= nCharset &&
				   nCharset < KCHARSET_3BYTES_CHARACTER) {
			/*	󡢿ܤ 96^2 Υåʤܤ*/
			if (fGenericp) {
				fRetval	= (uCode < 0x10000) ;
			} else {
				register int	nLower	= (int)((uCode >> 0) & 0x00FF) ;
				register int	nUpper	= (int)((uCode >> 8) & 0x00FF) ;
				fRetval	= ((uCode < 0x10000) &&
						   (0x21 <= nLower && nLower <= 0x7E) &&
						   (0x21 <= nUpper && nUpper <= 0x7E)) ;
			}
		} else {
			if (fGenericp) {
				fRetval	= (uCode < 0x1000000) ;
			} else {
				register int	nLower	= (int)((uCode >>  0) & 0x00FF) ;
				register int	nMiddle	= (int)((uCode >>  8) & 0x00FF) ;
				register int	nUpper	= (int)((uCode >> 16) & 0x00FF) ;
				fRetval	= ((uCode < 0x1000000) &&
						   (0x21 <= nLower  && nLower  <= 0x7E) &&
						   (0x21 <= nMiddle && nMiddle <= 0x7E) &&
						   (0x21 <= nUpper  && nUpper  <= 0x7E)) ;
			}
		}
	}
	if (fRetval) {
		pEntRetval	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_T) ;
	} else {
		pEntRetval	= lispMgr_GetReservedEntity (pLispMgr, LISPMGR_INDEX_NIL) ;
	}
	lispMachineCode_SetLReg (pLM, LM_LREG_ACC, pEntRetval) ;
	return	LMR_RETURN ;
}


