/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "local.h"
#include <assert.h>
#include "kfile.h"

enum {
	CHECKSIZE			= (2048),
} ;


Boolean
KFile_Init (
	register KFILE*	pFile)
{
	assert (pFile != NULL) ;

	pFile->m_iCodingSystem	= KCODING_SYSTEM_UNKNOWN ;

	pFile->m_pf	= NULL ;
	return	True ;
}
	
Boolean
KFile_Open (
	register KFILE*			pFile,
	register const char*	pFileName,
	register int			iCodingSystem)
{
	assert (pFile != NULL) ;

	pFile->m_pf	= fopen (pFileName, "rb") ;
	if (pFile->m_pf == NULL)
		return	False ;

	if (iCodingSystem == KCODING_SYSTEM_UNKNOWN) {
		iCodingSystem	= KFile_DetectCodingSystem (pFile) ;
		if (iCodingSystem == KCODING_SYSTEM_UNKNOWN)
			iCodingSystem	= KCODING_SYSTEM_BINARY ;
	}
	pFile->m_iCodingSystem	= iCodingSystem ;
	InitializeKanjiFiniteStateMachine (&pFile->m_ksm, iCodingSystem) ;
	return	True ;
}

Boolean
KFile_Close (
	register KFILE*			pFile)
{
	assert (pFile != NULL) ;

	return	(!fclose (pFile->m_pf))? True : False ;
}

Char
KFile_Getc (
	register KFILE*			pFile)
{
	register int		cc ;
	Char	dwc ;
	
	assert (pFile != NULL) ;

	dwc	= EOF ;
	while (
		cc	= fgetc (pFile->m_pf),
		cc != EOF) {
		if (TSUCCEEDED (TransferKanjiFiniteStateMachine (&pFile->m_ksm, cc, &dwc)))
			break ;
	}
	return	(cc == EOF)? EOF : dwc ;
}

Boolean
KFile_FindsAtBeginningOfLine (
	register KFILE*			pFile,
	register const Char*	pString)
{
	register Char			cc ;
	register const Char*	ptr ;

	assert (pFile != NULL) ;

	while (cc = KFile_Getc (pFile), cc != EOF) {
		if (cc == *pString) {
			ptr	= pString + 1 ;
			while (!Char_IsNul (*ptr)) {
				cc = KFile_Getc (pFile) ;
				if (cc == EOF || cc != *ptr)
					break ;
				ptr	++ ;
			}
			if (cc == EOF)
				break ;
			if (Char_IsNul (*ptr))
				return	True ;
		}
		if (!KFile_Nextline (pFile))
			break ;
	}
	return	False ;
}

Boolean
KFile_FindsAtBeginningOfLineA (
	register KFILE*			pFile,
	register const char*	pString)
{
	register Char			cc ;
	register const char*	ptr ;

	assert (pFile != NULL) ;
	assert (pString != NULL && *pString != '\0') ;

	while (cc = KFile_Getc (pFile), cc != EOF) {
		if (!Char_DifferenceAscii (cc, *pString)) {
			ptr	= pString + 1 ;
			while (*ptr != '\0') {
				cc = KFile_Getc (pFile) ;
				if (cc == EOF || Char_DifferenceAscii (cc, *ptr))
					break ;
				ptr	++ ;
			}
			if (cc == EOF)
				break ;
			if (*ptr == '\0')
				return	True ;
		}
		if (!KFile_Nextline (pFile))
			break ;
	}
	return	False ;
}

Boolean
KFile_Cmps (
	register KFILE*			pFile,
	register const Char*	pString)
{
	register Char		cc ;

	assert (pFile != NULL) ;
	assert (pString != NULL) ;
	
	while (!Char_IsNul (*pString)) {
		cc = KFile_Getc (pFile) ;
		if (cc == EOF || *pString != cc)
			break ;
		pString	++ ;
	}
	return	Char_IsNul (*pString) ;
}

Boolean
KFile_Cmpsn (
	register KFILE*			pFile,
	register const Char*	pString,
	register int			nStrLength)
{
	register Char		cc ;

	assert (pFile != NULL) ;
	assert (pString != NULL || nStrLength <= 0) ;
	
	while (nStrLength > 0 && !Char_IsNul (*pString)) {
		cc = KFile_Getc (pFile) ;
		if (cc == EOF || *pString != cc)
			break ;
		pString		++ ;
		nStrLength	-- ;
	}
	return	(nStrLength == 0 || Char_IsNul (*pString))? True : False ;
}

Boolean
KFile_Nextline (
	register KFILE*		pFile)
{
	register Char			cc ;

	assert (pFile != NULL) ;

	while (cc = KFile_Getc (pFile), cc != EOF && cc != '\n')
		;
	
	return	(cc == EOF)? False : True ;
}

Boolean
KFile_Tell (
	register KFILE*		pFile,
	register KFILEPOS*	pPos)
{
	assert (pFile != NULL) ;
	assert (pPos  != NULL) ;

	pPos->m_lPosition	= ftell (pFile->m_pf) ;
	memcpy (&pPos->m_ksm, &pFile->m_ksm, sizeof (KANJISTATEMACHINE)) ;
	return	True ;
}

Boolean
KFile_Seek (
	register KFILE*		pFile,
	register KFILEPOS*	pPos)
{
	assert (pFile != NULL) ;
	assert (pPos  != NULL) ;

	fseek (pFile->m_pf, pPos->m_lPosition, SEEK_SET) ;
	memcpy (&pFile->m_ksm, &pPos->m_ksm, sizeof (KANJISTATEMACHINE)) ;
	return	True ;
}

Boolean
KFile_Seekn (
	register KFILE*		pFile,
	register long		lOffset,
	register int		nWhence)
{
	fseek (pFile->m_pf, lOffset, nWhence) ;
	InitializeKanjiFiniteStateMachine (&pFile->m_ksm, pFile->m_iCodingSystem) ;
	return	True ;
}

Boolean
KFile_Rewind (
	register KFILE*		pFile)
{
	assert (pFile != NULL) ;

	rewind (pFile->m_pf) ;

	InitializeKanjiFiniteStateMachine (&pFile->m_ksm, pFile->m_iCodingSystem) ;
	return	True ;
}

int
KFile_GetCodingSystem (
	register KFILE*		pFile)
{
	assert (pFile != NULL) ;

	return	pFile->m_iCodingSystem ;
}

int
KFile_DetectCodingSystem (
	register KFILE*		pFile)
{
	char			achBuf [CHECKSIZE] ;
	register int	nRead ;

	assert (pFile != NULL) ;
	
	nRead			= fread (achBuf, sizeof (char), CHECKSIZE, pFile->m_pf) ;
	return	DetectKanjiCodingSystem (achBuf, nRead, NULL) ;
}


