/* # skkinput (Simple Kana-Kanji Input)
 *
 * This file is part of skkinput.
 * Copyright (C) 2002
 * Takashi SAKAMOTO (PXG01715@nifty.ne.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "local.h"
#include <ctype.h>
#include <assert.h>
#include <X11/keysym.h>
#include "lispyevent.h"
#include "cstring.h"
#if defined (DEBUG)
#include <stdio.h>
#endif

struct StringKeySymPair {
	const char*		m_pName ;
	unsigned int	m_nSymbol ;
} ;

static const struct StringKeySymPair	reverse_lispy_function_keys []	= {
	{ "backspace",		0xff08, },	{ "backtab",		0xff74, },
	{ "begin",			0xff58, },	{ "break", 			0xff6b, },
	{ "cancel",			0xff69, },	{ "clear", 			0xff0b, },
	{ "delete", 		0xffff, },	{ "down",			0xff54, },
	{ "eisu-shift",		0xff2f, },	{ "eisu-toggle",	0xff30, },
	{ "end",			0xff57, },	{ "escape",			0xff1b, },
	{ "execute",		0xff62, },	{ "f1",				0xffbe, },
	{ "f10",			0xffc7, },	{ "f11",			0xffc8, },
	{ "f12",			0xffc9, },	{ "f13",			0xffca, },
	{ "f14",			0xffcb, },	{ "f15",			0xffcc, },
	{ "f16",			0xffcd, },	{ "f17",			0xffce, },
	{ "f18",			0xffcf, },	{ "f19",			0xffd0, },
	{ "f2",				0xffbf, },	{ "f20",			0xffd1, },
	{ "f21",			0xffd2, },	{ "f22",			0xffd3, },
	{ "f23",			0xffd4, },	{ "f24",			0xffd5, },
	{ "f25",			0xffd6, },	{ "f26",			0xffd7, },
	{ "f27",			0xffd8, },	{ "f28",			0xffd9, },
	{ "f29",			0xffda, },	{ "f3",				0xffc0, },
	{ "f30",			0xffdb, },	{ "f31",			0xffdc, },
	{ "f32",			0xffdd, },	{ "f33",			0xffde, },
	{ "f34",			0xffdf, },	{ "f35",			0xffe0, },
	{ "f4",				0xffc1, },	{ "f5",				0xffc2, },
	{ "f6",				0xffc3, },	{ "f7",				0xffc4, },
	{ "f8",				0xffc5, },	{ "f9",				0xffc6, },
	{ "find",			0xff68, },	{ "hankaku",		0xff29, },
	{ "help",			0xff6a, },	{ "henkan",			0xff23, },
	{ "hiragana",		0xff25, },	{ "hiragana-katakana", 0xff27, },
	{ "home",			0xff50, },	{ "insert",			0xff63, },
	{ "iso-center-object",				0xfe33, },
	{ "iso-continuous-underline",		0xfe30, },
	{ "iso-discontinuous-underline",	0xfe31, },
	{ "iso-emphasize",	0xfe32, },	{ "iso-enter",		0xfe34, },
	{ "iso-fast-cursor-down",			0xfe2f, },
	{ "iso-fast-cursor-left",			0xfe2c, },
	{ "iso-fast-cursor-right",			0xfe2d, },
	{ "iso-fast-cursor-up",				0xfe2e, },
	{ "iso-lefttab",	0xfe20, },	{ "iso-move-line-down",	0xfe22, },
	{ "iso-move-line-up",				0xfe21, },
	{ "iso-partial-line-down",			0xfe24, },
	{ "iso-partial-line-up",			0xfe23, },
	{ "iso-partial-space-left",			0xfe25, },
	{ "iso-partial-space-right",		0xfe26, },
	{ "iso-release-both-margins",		0xfe2b, },
	{ "iso-release-margin-left",		0xfe29, },
	{ "iso-release-margin-right",		0xfe2a, },
	{ "iso-set-margin-left",			0xfe27, },
	{ "iso-set-margin-right",			0xfe28, },
	{ "kana-A",			0x04b1, },	{ "kana-CHI",		0x04c1, },
	{ "kana-E",			0x04b4, },	{ "kana-FU",		0x04cc, },
	{ "kana-HA",		0x04ca, },	{ "kana-HE",		0x04cd, },
	{ "kana-HI",		0x04cb, },	{ "kana-HO",		0x04ce, },
	{ "kana-I",			0x04b2, },	{ "kana-KA",		0x04b6, },
	{ "kana-KE",		0x04b9, },	{ "kana-KI",		0x04b7, },
	{ "kana-KO",		0x04ba, },	{ "kana-KU",		0x04b8, },
	{ "kana-MA",		0x04cf, },	{ "kana-ME",		0x04d2, },
	{ "kana-MI",		0x04d0, },	{ "kana-MO",		0x04d3, },
	{ "kana-MU",		0x04d1, },	{ "kana-N",			0x04dd, },
	{ "kana-NA",		0x04c5, },	{ "kana-NE",		0x04c8, },
	{ "kana-NI",		0x04c6, },	{ "kana-NO",		0x04c9, },
	{ "kana-NU",		0x04c7, },	{ "kana-O",			0x04b5, },
	{ "kana-RA",		0x04d7, },	{ "kana-RE",		0x04da, },
	{ "kana-RI",		0x04d8, },	{ "kana-RO",		0x04db, },
	{ "kana-RU",		0x04d9, },	{ "kana-SA",		0x04bb, },
	{ "kana-SE",		0x04be, },	{ "kana-SHI",		0x04bc, },
	{ "kana-SO",		0x04bf, },	{ "kana-SU",		0x04bd, },
	{ "kana-TA",		0x04c0, },	{ "kana-TE",		0x04c3, },
	{ "kana-TO",		0x04c4, },	{ "kana-TSU",		0x04c2, },
	{ "kana-U",			0x04b3, },	{ "kana-WA",		0x04dc, },
	{ "kana-WO",		0x04a6, },	{ "kana-YA",		0x04d4, },
	{ "kana-YO",		0x04d6, },	{ "kana-YU",		0x04d5, },
	{ "kana-a",			0x04a7, },	{ "kana-closingbracket",	0x04a3, },
	{ "kana-comma",		0x04a4, },	{ "kana-conjunctive",		0x04a5, },
	{ "kana-e",			0x04aa, },	{ "kana-fullstop",	0x04a1, },
	{ "kana-i",			0x04a8, },	{ "kana-lock",		0xff2d, },
	{ "kana-o",			0x04ab, },	{ "kana-openingbracket", 0x04a2, },
	{ "kana-shift",		0xff2e, },	{ "kana-tsu",		0x04af, },
	{ "kana-u",			0x04a9, },	{ "kana-ya",		0x04ac, },
	{ "kana-yo",		0x04ae, },	{ "kana-yu",		0x04ad, },
	{ "kanji",			0xff21, },	{ "katakana",		0xff26, },
	{ "kp-0",			0xffb0, },	{ "kp-1",			0xffb1, },
	{ "kp-2",			0xffb2, },	{ "kp-3",			0xffb3, },
	{ "kp-4",			0xffb4, },	{ "kp-5",			0xffb5, },
	{ "kp-6",			0xffb6, },	{ "kp-7",			0xffb7, },
	{ "kp-8",			0xffb8, },	{ "kp-9",			0xffb9, },
	{ "kp-add",			0xffab, },	{ "kp-begin",		0xff9d, },
	{ "kp-decimal",		0xffae, },	{ "kp-delete",		0xff9f, },
	{ "kp-divide",		0xffaf, },	{ "kp-down",		0xff99, },
	{ "kp-end",			0xff9c, },	{ "kp-enter",		0xff8d, },
	{ "kp-equal",		0xffbd, },	{ "kp-f1",			0xff91, },
	{ "kp-f2",			0xff92, },	{ "kp-f3",			0xff93, },
	{ "kp-f4",			0xff94, },	{ "kp-home",		0xff95, },
	{ "kp-insert",		0xff9e, },	{ "kp-left",		0xff96, },
	{ "kp-multiply",	0xffaa, },	{ "kp-next",		0xff9b, },
	{ "kp-numlock",		0xff7f, },	{ "kp-prior",		0xff9a, },
	{ "kp-right",		0xff98, },	{ "kp-separator",	0xffac, },
	{ "kp-space",		0xff80, },	{ "kp-subtract",	0xffad, },
	{ "kp-tab",			0xff89, },	{ "kp-up",			0xff97, },
	{ "left",			0xff51, },	{ "linefeed",		0xff0a, },
	{ "massyo",			0xff2c, },	{ "menu",			0xff67, },
	{ "muhenkan",		0xff22, },	{ "next",			0xff56, },
	{ "overline",		0x047e, },	{ "pause",			0xff13, },
	{ "print",			0xff61, },	{ "prior",			0xff55, },
	{ "prolongedsound",	0x04b0, },	{ "redo",			0xff66, },
	{ "return",			0xff0d, },	{ "right",			0xff53, },
	{ "romaji",			0xff24, },	{ "select",			0xff60, },
	{ "semivoicedsound", 0x04df, },	{ "tab",			0xff09, },
	{ "touroku",		0xff2b, },	{ "undo",			0xff65, },
	{ "up",				0xff52, },	{ "voicedsound",	0x04de, },
	{ "zenkaku",		0xff28, },	{ "zenkaku-hankaku", 0xff2a, },
} ;

static char *lispy_kana_keys[] = {
    /* X Keysym value */
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,	/* 0x400 .. 0x40f */
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,	/* 0x410 .. 0x41f */
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,	/* 0x420 .. 0x42f */
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,	/* 0x430 .. 0x43f */
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,	/* 0x440 .. 0x44f */
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,	/* 0x450 .. 0x45f */
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,	/* 0x460 .. 0x46f */
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,"overline",0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,	/* 0x480 .. 0x48f */
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,	/* 0x490 .. 0x49f */
    0, "kana-fullstop", "kana-openingbracket", "kana-closingbracket", 
    "kana-comma", "kana-conjunctive", "kana-WO", "kana-a",
    "kana-i", "kana-u", "kana-e", "kana-o",
    "kana-ya", "kana-yu", "kana-yo", "kana-tsu",
    "prolongedsound", "kana-A", "kana-I", "kana-U",
    "kana-E", "kana-O", "kana-KA", "kana-KI",
    "kana-KU", "kana-KE", "kana-KO", "kana-SA",
    "kana-SHI", "kana-SU", "kana-SE", "kana-SO",
    "kana-TA", "kana-CHI", "kana-TSU", "kana-TE",
    "kana-TO", "kana-NA", "kana-NI", "kana-NU",
    "kana-NE", "kana-NO", "kana-HA", "kana-HI",
    "kana-FU", "kana-HE", "kana-HO", "kana-MA",
    "kana-MI", "kana-MU", "kana-ME", "kana-MO",
    "kana-YA", "kana-YU", "kana-YO", "kana-RA",
    "kana-RI", "kana-RU", "kana-RE", "kana-RO",
    "kana-WA", "kana-N", "voicedsound", "semivoicedsound",
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,	/* 0x4e0 .. 0x4ef */
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,	/* 0x4f0 .. 0x4ff */
} ;

static char *lispy_function_keys [] = {
    /* X Keysym value */

    0, 0, 0, 0, 0, 0, 0, 0,			      /* 0xff00...0f */
    "backspace", "tab", "linefeed", "clear",
    0, "return", 0, 0,
    0, 0, 0, "pause",				      /* 0xff10...1f */
    0, 0, 0, 0, 0, 0, 0, "escape",
    0, 0, 0, 0,
    0, "kanji", "muhenkan", "henkan",		      /* 0xff20...2f */
    "romaji", "hiragana", "katakana", "hiragana-katakana",
    "zenkaku", "hankaku", "zenkaku-hankaku", "touroku",
    "massyo", "kana-lock", "kana-shift", "eisu-shift",
    "eisu-toggle",				      /* 0xff30...3f */
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,   /* 0xff40...4f */

    "home", "left", "up", "right", /* 0xff50 */	/* IsCursorKey */
    "down", "prior", "next", "end",
    "begin", 0, 0, 0, 0, 0, 0, 0,
    "select",			/* 0xff60 */	/* IsMiscFunctionKey */
    "print",
    "execute",
    "insert",
    0,		/* 0xff64 */
    "undo",
    "redo",
    "menu",
    "find",
    "cancel",
    "help",
    "break",			/* 0xff6b */

    0, 0, 0, 0,
    0, 0, 0, 0, "backtab", 0, 0, 0,		/* 0xff70... */
    0, 0, 0, 0, 0, 0, 0, "kp-numlock",		/* 0xff78... */
    "kp-space",			/* 0xff80 */	/* IsKeypadKey */
    0, 0, 0, 0, 0, 0, 0, 0,
    "kp-tab",			/* 0xff89 */
    0, 0, 0,
    "kp-enter",			/* 0xff8d */
    0, 0, 0,
    "kp-f1",			/* 0xff91 */
    "kp-f2",
    "kp-f3",
    "kp-f4",
    "kp-home",			/* 0xff95 */
    "kp-left",
    "kp-up",
    "kp-right",
    "kp-down",
    "kp-prior",			/* kp-page-up */
    "kp-next",			/* kp-page-down */
    "kp-end",
    "kp-begin",
    "kp-insert",
    "kp-delete",
    0,				/* 0xffa0 */
    0, 0, 0, 0, 0, 0, 0, 0, 0,
    "kp-multiply",		/* 0xffaa */
    "kp-add",
    "kp-separator",
    "kp-subtract",
    "kp-decimal",
    "kp-divide",		/* 0xffaf */
    "kp-0",			/* 0xffb0 */
    "kp-1",	"kp-2",	"kp-3",	"kp-4",	"kp-5",	"kp-6",	"kp-7",	"kp-8",	"kp-9",
    0,		/* 0xffba */
    0, 0,
    "kp-equal",			/* 0xffbd */
    "f1",			/* 0xffbe */	/* IsFunctionKey */
    "f2",
    "f3", "f4", "f5", "f6", "f7", "f8",	"f9", "f10", /* 0xffc0 */
    "f11", "f12", "f13", "f14", "f15", "f16", "f17", "f18",
    "f19", "f20", "f21", "f22", "f23", "f24", "f25", "f26", /* 0xffd0 */
    "f27", "f28", "f29", "f30", "f31", "f32", "f33", "f34",
    "f35", 0, 0, 0, 0, 0, 0, 0,	/* 0xffe0 */
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,     /* 0xfff0 */
    0, 0, 0, 0, 0, 0, 0, "delete"
} ;

static char *iso_lispy_function_keys[] = {
    0, 0, 0, 0, 0, 0, 0, 0,	/* 0xfe00 */
    0, 0, 0, 0, 0, 0, 0, 0,	/* 0xfe08 */
    0, 0, 0, 0, 0, 0, 0, 0,	/* 0xfe10 */
    0, 0, 0, 0, 0, 0, 0, 0,	/* 0xfe18 */
    "iso-lefttab",		/* 0xfe20 */
    "iso-move-line-up", "iso-move-line-down", 
    "iso-partial-line-up", "iso-partial-line-down", 
    "iso-partial-space-left", "iso-partial-space-right", 
    "iso-set-margin-left", "iso-set-margin-right", /* 0xffe27, 28 */
    "iso-release-margin-left", "iso-release-margin-right",
    "iso-release-both-margins",
    "iso-fast-cursor-left", "iso-fast-cursor-right",
    "iso-fast-cursor-up", "iso-fast-cursor-down",
    "iso-continuous-underline", "iso-discontinuous-underline", /* 0xfe30, 31 */
    "iso-emphasize", "iso-center-object", "iso-enter", /* ... 0xfe34 */
} ;

static const char*	lispy_mask_strings []	= {
	"S-", "L-", "C-", "M-", 
} ;

static	Boolean	parseBackslashedChar	(const char**, int*, Char*) ;
static	int		parseKeyString	(Display*, const char**, int*, KeySym*, unsigned int*, int) ;
static	int		parseKeySymbol	(Display*, const char**, int*, KeySym*, unsigned int*, int) ;

/*	İʾΥ줿ˤ 100% ݾڤϤǤʤ
 */
static inline Boolean
ascii2keysym (
	register Display*		pDisplay,
	register char			ch,
	register KeySym*		pKeySymRet,
	register unsigned int*	puStateRet)
{
	register KeySym			keysym ;
	register unsigned int	uState	= 0 ;
	register unsigned int	keycode ;
	register int			nIndex ;

	assert (pDisplay   != NULL) ;
	assert (pKeySymRet != NULL) ;
	assert (puStateRet != NULL) ;

	if (ch == 0x7F) {
		/* Ȥϸ¤ʤɤʤɡ*/
		keysym	= XK_Delete ;
		uState	= 0 ;
	} else {
		KeySym	keysymL, keysymU ;

		if (ch < 0) {
			ch		&= 0x7F ;
			uState	|= Mod1Mask ;
		}
		if (ch < 0x20) {
			ch		= "@abcdefghijklmnopqrstuvwxyz[\\]^_"[(int)ch] ;
			uState	|= ControlMask ;
		}
		keysym	= ch & 0x7F ;

		XConvertCase (keysym, &keysymL, &keysymU) ;
		if (keysymL != keysymU && (keysym == keysymL || keysym == keysymU)) {
			if (keysymU == keysym) {
				uState	|= ShiftMask ;
				keysym	= keysymU ;
			}
		}
		keycode	= XKeysymToKeycode (pDisplay, keysym) ;
		for (nIndex = ShiftMapIndex + 1 ; nIndex <= Mod5MapIndex ; nIndex ++) {
			if (keysym == XKeycodeToKeysym (pDisplay, keycode, nIndex)) {
				uState	|= (1 << nIndex) ;
				break ;
			}
		}
	}
	*pKeySymRet	= keysym ;
	*puStateRet	= uState ;
	return	True ;
}

static inline void
skipSpace (
	register const char**	ppString, 
	register int*			pnString)
{
	register const char*	pString	= *ppString ;
	register int			nString	= *pnString ;

	while (nString > 0 && (*pString == ' ' || *pString == '\t')) {
		pString	++ ;
		nString	-- ;
	}
	*ppString	= pString ;
	*pnString	= nString ;
	return ;
}

Boolean
keysym2string (
	register char*			pDest,
	register int			nDest,
	register KeySym			keySym,
	register unsigned int	uState)
{
	register int			i, nString ;
	register unsigned int	uMask ;
	register const char*	pString ;

	uMask	= 1 ;
	for (i = 0 ; i < NELEMENTS (lispy_mask_strings) ; i ++, uMask <<= 1) {
		if ((uMask & uState) != 0) {
			if (nDest > 2) {
				memcpy (pDest, lispy_mask_strings [i], sizeof (char) * 2) ;
				pDest += 2 ;
				nDest -= 2 ;
			}
		}
	}
	if (nDest <= 0)
		return	False ;

	pString	= XKeysymToString (keySym) ;
	if (pString == NULL) {
		if (0x0400 <= keySym && keySym <= 0x0500) {
			pString	= lispy_kana_keys [keySym - 0x0400] ;
		} else if (0xFE00 <= keySym && keySym < 0xFF00) {
			pString	= iso_lispy_function_keys [keySym - 0xFE00] ;
		} else if (0xFF00 <= keySym && keySym <= 0xFFFF) {
			pString	= lispy_function_keys [keySym - 0xFF00] ;
		}
		if (pString == NULL)
			return	False ;
	}
	nString	= strlen (pString) ;
	if (nDest <= nString)
		return	False ;

	memcpy (pDest, pString, (nString + 1) * sizeof (char)) ;
	return	True ;
}

Boolean
string2keysym (
	register const char*	pString,
	register KeySym*		pKeySymReturn,
	register unsigned int*	puState)
{
	register const struct StringKeySymPair*	pTarget ;
	register KeySym			ksym ;
	register int			i, nTarget, nMin, nMax, nCmp ;
	register unsigned int	uState ;

	/*	ޤ state 򸫤ơ
	 */
	uState	= 0 ;
	while (*(pString + 0) != '\0' &&
		   *(pString + 1) == '-') {
		for (i = 0 ; i < NELEMENTS (lispy_mask_strings) && strncmp (pString, lispy_mask_strings [i], 2) ; i ++)
			;
		if (i < NELEMENTS (lispy_mask_strings)) {
			uState	|= 1 << i ;
			pString	+= 2 ;
		} else {
			break ;
		}
	}
	if (*pString == '\0')
		return	False ;

	/*	 symbol 򸫤롣
	 */
	ksym	= XStringToKeysym (pString) ;
	if (ksym != NoSymbol) 
		goto	found ;

	nMin	= 0 ;
	nMax	= NELEMENTS (reverse_lispy_function_keys) - 1 ;
	while (nMin <= nMax) {
		nTarget	= (nMin + nMax) / 2 ;
		pTarget	= &reverse_lispy_function_keys [nTarget] ;
		nCmp	= strcasecmp (pString, pTarget->m_pName) ;
		if (nCmp == 0) {
			ksym	= pTarget->m_nSymbol ;
			goto	found ;
		}
		if (nCmp < 0) {
			nMax	= nTarget - 1 ;
		} else {
			nMin	= nTarget + 1 ;
		}
	}

	if (0x20 <= *pString && *pString <= 0x7E && *(pString + 1) == '\0') {
		ksym	= (KeySym) tolower (*pString) ;
		if (isupper (*pString))
			uState	|= ShiftMask ;
		goto	found ;
	}

	return	False ;

  found:
	*puState		= uState ;
	*pKeySymReturn	= ksym ;
	return	True ;
}

Boolean
keysym2Cstring (
	register Char*			pDest,
	register int			nDest,
	register KeySym			keySym,
	register unsigned int	uState)
{
	register int			i, nString ;
	register unsigned int	uMask ;
	register const char*	pString ;

	uMask	= 1 ;
	for (i = 0 ; i < NELEMENTS (lispy_mask_strings) ; i ++, uMask <<= 1) {
		if ((uMask & uState) != 0) {
			if (nDest > 2) {
				strtocstr (pDest, lispy_mask_strings [i], 2) ;
				pDest += 2 ;
				nDest -= 2 ;
			}
		}
	}
	if (nDest <= 0)
		return	False ;

	pString	= XKeysymToString (keySym) ;
	if (pString == NULL) {
		if (0x0400 <= keySym && keySym <= 0x0500) {
			pString	= lispy_kana_keys [keySym - 0x0400] ;
		} else if (0xFE00 <= keySym && keySym < 0xFF00) {
			pString	= iso_lispy_function_keys [keySym - 0xFE00] ;
		} else if (0xFF00 <= keySym && keySym <= 0xFFFF) {
			pString	= lispy_function_keys [keySym - 0xFF00] ;
		}
		if (pString == NULL)
			return	False ;
	}
	nString	= strlen (pString) ;
	if (nDest <= nString)
		return	False ;

	strtocstr (pDest, pString, (nString + 1)) ;
	return	True ;
}

Boolean
cstring2keysym (
	register const Char*	pString,
	register int			nString,
	register KeySym*		pKeySymReturn,
	register unsigned int*	puState)
{
	char	buf [256] ;

	if (nString >= NELEMENTS (buf))
		return	False ;

	cstrtostr (buf, pString, nString) ;
	buf [nString]	= '\0' ;
	return	string2keysym (buf, pKeySymReturn, puState) ;
}

Boolean
cchar2keycode (
	register Display*		pDisplay,
	register Char			cc,
	register unsigned int*	puKeycodeRet,
	register unsigned int*	puStateRet)
{
	register int			nCharset ;
	register unsigned int	uCode, uKeyCode ;
	unsigned int	uState ;

	nCharset	= Char_Charset (cc) ;
	uCode		= Char_Code (cc) ;
	uState		= 0 ;
	switch (nCharset) {
		char	achBuffer [2] ;
		KeySym	keysymGuess, keysymLower, keysymUpper ;

	case	KCHARSET_XKEYSYM:
		uState		= (uCode >> 16) & 0x00FF ;
		keysymGuess	= uCode & 0xFFFF ;
		uKeyCode	= XKeysymToKeycode (pDisplay, keysymGuess) ;
		break ;

	case	KCHARSET_XCHAR:
		uState		= uCode >> 16 ;
		uCode		= uCode & 0xFFFF ;
		/*	fall down */
	case	KCHARSET_ASCII:
		if (uCode < 0x7E) {
			ascii2keysym (pDisplay, uCode, &keysymGuess, &uState) ;
		} else {
			achBuffer [0]		= uCode & 0x7F ;
			achBuffer [1]		= '\0' ;
			keysymGuess			= XStringToKeysym (achBuffer) ;
			if (keysymGuess == 0) 
				return	False ;
			XConvertCase (keysymGuess, &keysymLower, &keysymUpper) ;
			if (keysymLower != keysymUpper && keysymGuess == keysymUpper) 
				uState	|= ShiftMask ;
		}
		uKeyCode			= XKeysymToKeycode (pDisplay, keysymGuess) ;
#if defined (DEBUG)
		fprintf (stderr, "KeySym(%lx), KeyCode(%x), State(%x)\n", keysymGuess, uKeyCode, uState) ;
#endif
		break ;
	default:
		return	False ;
	}
	*puKeycodeRet	= uKeyCode ;
	*puStateRet		= uState ;
	return	True ;
}

Boolean
cchar2keysym (
	register Display*		pDisplay,
	register Char			cc,
	register KeySym*		pKeysymRet,
	register unsigned int*	puStateRet)
{
	register int			nCharset, nIndex ;
	register unsigned int	uCode ;
	KeySym					keysym ;
	unsigned int			uState ;

	nCharset	= Char_Charset (cc) ;
	uCode		= Char_Code (cc) ;
	uState		= 0 ;
	switch (nCharset) {
		char	achBuffer [2] ;
		KeySym	keysymLower, keysymUpper ;

	case	KCHARSET_XKEYSYM:
		uState	= (uCode >> 16) & 0x00FF ;
		keysym	= uCode & 0xFFFF ;
		break ;

	case	KCHARSET_XCHAR:
		uState		= uCode >> 16 ;
		uCode		= uCode & 0xFFFF ;
		for (nIndex = ShiftMapIndex ; nIndex <= Mod5MapIndex ; nIndex ++) {
			if (uState & (1 << nIndex)) {
				keysym	= XKeycodeToKeysym (pDisplay, uCode, nIndex) ;
				if (keysym != NoSymbol)
					break ;
			}
		}
		if (nIndex > Mod5MapIndex)
			return	False ;
		break ;

	case	KCHARSET_ASCII:
		if (uCode < 0x7E) {
			ascii2keysym (pDisplay, uCode, &keysym, &uState) ;
		} else {
			achBuffer [0]		= uCode & 0x7F ;
			achBuffer [1]		= '\0' ;
			keysym				= XStringToKeysym (achBuffer) ;
			if (keysym == 0) 
				return	False ;
			XConvertCase (keysym, &keysymLower, &keysymUpper) ;
			if (keysymLower != keysymUpper && keysym == keysymUpper) 
				uState	|= ShiftMask ;
		}
		break ;
	default:
		return	False ;
	}
	*pKeysymRet	= keysym ;
	*puStateRet	= uState ;
	return	True ;
}

int
string2keysymlist (
	register Display*		pDisplay,
	register const char**	ppString,
	register int*			pnString,
	register KeySym*		pKeysym,
	register unsigned int*	puState,
	register int			nKeysym)
{
	register int	nRetval ;
	const char*		pString ;
	int				nString ;

	pString	= *ppString ;
	nString	= *pnString ;

	skipSpace (&pString, &nString) ;
	if (nString <= 0)
		return	-1 ;
	if (*pString == 0x22) {
		nRetval	= parseKeyString (pDisplay, &pString, &nString, pKeysym, puState, nKeysym) ;
	} else if (*pString == '[') {
		nRetval	= parseKeySymbol (pDisplay, &pString, &nString, pKeysym, puState, nKeysym) ;
	} else {
		return	-1 ;
	}
	skipSpace (&pString, &nString) ;

	*ppString	= pString ;
	*pnString	= nString ;
	return	nRetval ;
}

/*
 *	³ʸ STRING Ǥ롣
 */
int
parseKeyString (
	register Display*		pDisplay,
	register const char**	ppString,
	register int*			pnString,
	register KeySym*		pKeysym,
	register unsigned int*	puState,
	register int			nKeysym)
{
	const char*		pString ;
	int				nString ;
	Char			cc ;
	register int	nDest	= nKeysym, nLength = 0 ;

	pString	= *ppString ;
	nString	= *pnString ;

	if (nString <= 0 || *pString != '\"') 
		return	-1 ;

	pString	++ ;
	nString	-- ;
	while (nString > 0) {
		cc	= *pString ++ ;
		nString	-- ;
		if (cc == 0x22)
			break ;
		if (cc == '\\') {
			if (TFAILED (parseBackslashedChar (&pString, &nString, &cc)))
				return	-1 ;
		}
		if (nDest > 0) {
			if (TFAILED (cchar2keysym (pDisplay, cc, pKeysym, puState)))
				return	-1 ;
			pKeysym	++ ;
			puState	++ ;
			nDest	-- ;
		}
		nLength	++ ;
	}
	if (cc != 0x22) 
		return	-1 ;

	pString	++ ;
	nString	-- ;
	*ppString	= pString ;
	*pnString	= nString ;

	/*	ܥȤä֤*/
	return	(nKeysym == 0)? nLength : (nKeysym - nDest) ;
}

int
parseKeySymbol (
	register Display*		pDisplay,
	register const char**	ppString,
	register int*			pnString,
	register KeySym*		pKeysym,
	register unsigned int*	puState,
	register int			nKeysym)
{
	char			buf [256] ;
	const char*		pString ;
	int				nString ;
	register int	nUsage, nDest, nLength = 0 ;

	pString	= *ppString ;
	nString	= *pnString ;
	nDest	= nKeysym ;

	if (nString <= 0 || *pString != '[')
		return	-1 ;

	pString	++ ;
	nString	-- ;
	skipSpace (&pString, &nString) ;

	nUsage	= 0 ;
	while (nString > 0) {
		if (*pString == ']' || *pString == ' ' || *pString == '\t') {
			buf [nUsage]	= '\0' ;
			if (nUsage > 0) {
				if (nDest > 0) {
					if (TFAILED (string2keysym (buf, pKeysym, puState)))
						return	-1 ;
					pKeysym	++ ;
					puState	++ ;
					nDest   -- ;
				}
				nLength	++ ;
			}
			if (*pString == ']')
				break ;

			nUsage	= 0 ;
			skipSpace (&pString, &nString) ;
			continue ;
		}
		if (nUsage >= NELEMENTS (buf))
			return	-1 ;
		buf [nUsage ++]	= *pString ;
		pString	++ ;
		nString	-- ;
	}
	if (nString <= 0 || *pString != ']')
		return	-1 ;

	pString	++ ;
	nString	-- ;
	*ppString	= pString ;
	*pnString	= nString ;
	return	(nKeysym == 0)? nLength : (nKeysym - nDest) ;
}

Boolean
parseBackslashedChar (
	register const char**	ppString,
	register int*			pnString,
	register Char*			pchRetval)
{
	const char*		pString ;
	int				nString ;
	Char			cc ;
	Char			chRet ;
	unsigned long	uAndMask, uOrMask ;

	pString	= *ppString ;
	nString	= *pnString ;
	if (nString <= 0)
		return	False ;

	/*	λ '\\' ɤ߽ξ֤ˤ롣*/
	cc	= *pString ++ ;
	nString	-- ;
	switch (cc) {
	case	'C':
		uAndMask	= 0x1F ;
		uOrMask		= 0 ;
		goto	meta_control_common ;
	case	'M':
		uAndMask	= 0x7F ;
		uOrMask		= 0x80 ;
	meta_control_common:
		if (nString <= 0)
			return	False ;
		cc	= *pString ++ ;
		nString	-- ;
		if (cc != '-' || nString <= 0)
			return	False ;
		cc	= *pString ++ ;
		nString	-- ;
		if (cc == '\\') {
			if (TFAILED (parseBackslashedChar (&pString, &nString, &cc)))
				return	False ;
		}
		*pchRetval	= (cc & uAndMask) | uOrMask ;
		break ;

	case	't':
		*pchRetval	= Char_Make (KCHARSET_ASCII, '\t') ;
		break ;

	case	'n':
		*pchRetval	= Char_Make (KCHARSET_ASCII, '\n') ;
		break ;

	case	'r':
		*pchRetval	= Char_Make (KCHARSET_ASCII, '\r') ;
		break ;

	case	'x':
		chRet	= 0 ;
		while (nString > 0) {
			cc	= *pString ++ ;
			nString	-- ;
			if ('0' <= cc && cc <= '9') {
				chRet	= chRet * 0x10 + (cc - '0') ;
			} else if ('a' <= cc && cc <= 'f') {
				chRet	= chRet * 0x10 + (cc - 'a' + 10) ;
			} else if ('A' <= cc && cc <= 'F') {
				chRet	= chRet * 0x10 + (cc - 'A' + 10) ;
			} else {
				pString	-- ;
				nString	++ ;
				break ;
			}
		}
		*pchRetval	= Char_Make (KCHARSET_ASCII, chRet) ;
		break ;

	default:
		if ('0' <= cc && cc <= '7') {
			register int	i ;
			chRet	= cc - '0' ;
			for (i = 0 ; i < 2 && nString > 0 ; i ++) {
				cc		= *pString ++ ;
				nString	-- ;
				if (cc < '0' || cc > '7') {
					pString	-- ;
					nString	++ ;
					break ;
				}
				chRet	= chRet * 8 + (cc - '0') ;				
			}
			*pchRetval	= Char_Make (KCHARSET_ASCII, chRet) ;
			break ;
		}
		*pchRetval	= cc ;
		break ;
	}
	*ppString	= pString ;
	*pnString	= nString ;
	return	True ;
}

