/******************************************************************************/
/* SlunkCrypt, by LoRd_MuldeR <MuldeR2@GMX.de>                                */
/* This work has been released under the CC0 1.0 Universal license!           */
/******************************************************************************/

#ifndef INC_SLUNKCRYPT_PLUSPLUS
#define INC_SLUNKCRYPT_PLUSPLUS

/*
 * Compiler check
 */
#if (!defined(__cplusplus) || (__cplusplus < 201103L)) && (!defined(_MSVC_LANG) || (_MSVC_LANG < 201103L))
#error This file requires compiler and library support for the ISO C++11 standard.
#endif

/*
 * Dependencies
 */
#include "slunkcrypt.h"
#include <string>
#include <vector>
#include <stdexcept>

/*
 * Namespace
 */
namespace slunkcrypt
{
	/*
	 * Base class for SlunkCrypt
	 */
	class SlunkBase
	{
	public:
		virtual bool process(const uint8_t *const input, uint8_t *const output, size_t length) = 0;
		virtual bool process(const std::vector<uint8_t> &input, std::vector<uint8_t> &output) = 0;
		virtual bool inplace(uint8_t *const buffer, size_t length) = 0;
		virtual bool inplace(std::vector<uint8_t> &buffer) = 0;
	protected:
		::slunkcrypt_t m_instance;
	};

	/*
	 * Class for encryption
	 */
	class Encryptor : public SlunkBase
	{
	public:
		Encryptor(const std::string &passwd)
		{
			if (::slunkcrypt_generate_nonce(&m_nonce) != SLUNKCRYPT_SUCCESS)
			{
				throw std::runtime_error("SlunkCryptEncr: Failed to generate the seed value!");
			}
			if ((m_instance = ::slunkcrypt_alloc(m_nonce, (const uint8_t*)passwd.c_str(), passwd.length(), SLUNKCRYPT_ENCRYPT)) == SLUNKCRYPT_NULL)
			{
				throw std::runtime_error("SlunkCryptEncr: Failed to create encoder instance!");
			}
		}

		Encryptor(Encryptor &&other) noexcept
		{
			this->m_instance = other.m_instance;
			this->m_nonce = other.m_nonce;
			other.m_instance = SLUNKCRYPT_NULL;
			other.m_nonce = (uint64_t)(-1);
		}

		~Encryptor(void)
		{
			if (m_instance != SLUNKCRYPT_NULL)
			{
				::slunkcrypt_free(m_instance);
			}
		}

		bool process(const uint8_t *const input, uint8_t *const output, size_t length)
		{
			return (::slunkcrypt_process(m_instance, input, output, length) == SLUNKCRYPT_SUCCESS);
		}

		bool process(const std::vector<uint8_t> &input, std::vector<uint8_t> &output)
		{
			if (output.size() >= input.size())
			{
				return (::slunkcrypt_process(m_instance, input.data(), output.data(), input.size()) == SLUNKCRYPT_SUCCESS);
			}
			return false;
		}

		bool inplace(uint8_t *const buffer, size_t length)
		{
			return (::slunkcrypt_inplace(m_instance, buffer, length) == SLUNKCRYPT_SUCCESS);
		}

		bool inplace(std::vector<uint8_t> &buffer)
		{
			return (::slunkcrypt_inplace(m_instance, buffer.data(), buffer.size()) == SLUNKCRYPT_SUCCESS);
		}

		uint64_t get_nonce(void) const
		{
			return m_nonce;
		}

	private:
		Encryptor(const Encryptor&) = delete;
		Encryptor& operator=(const Encryptor&) = delete;
		uint64_t m_nonce;
	};

	/*
	 * Class for decryption
	 */
	class Decryptor : public SlunkBase
	{
	public:
		Decryptor(const uint64_t nonce, const std::string &passwd)
		{
			if ((m_instance = ::slunkcrypt_alloc(nonce, (const uint8_t*)passwd.c_str(), passwd.length(), SLUNKCRYPT_DECRYPT)) == SLUNKCRYPT_NULL)
			{
				throw std::runtime_error("SlunkCryptDecr: Failed to create decoder instance!");
			}
		}

		Decryptor(Decryptor &&other) noexcept
		{
			this->m_instance = other.m_instance;
			other.m_instance = SLUNKCRYPT_NULL;
		}

		~Decryptor(void)
		{
			if (m_instance != SLUNKCRYPT_NULL)
			{
				::slunkcrypt_free(m_instance);
			}
		}

		bool process(const uint8_t *const input, uint8_t *const output, size_t length)
		{
			return (::slunkcrypt_process(m_instance, input, output, length) == SLUNKCRYPT_SUCCESS);
		}

		bool process(const std::vector<uint8_t> &input, std::vector<uint8_t> &output)
		{
			if (output.size() >= input.size())
			{
				return (::slunkcrypt_process(m_instance, input.data(), output.data(), input.size()) == SLUNKCRYPT_SUCCESS);
			}
			return false;
		}

		bool inplace(uint8_t *const buffer, size_t length)
		{
			return (::slunkcrypt_inplace(m_instance, buffer, length) == SLUNKCRYPT_SUCCESS);
		}

		bool inplace(std::vector<uint8_t> &buffer)
		{
			return (::slunkcrypt_inplace(m_instance, buffer.data(), buffer.size()) == SLUNKCRYPT_SUCCESS);
		}

	private:
		Decryptor(const Decryptor&) = delete;
		Decryptor& operator=(const Decryptor&) = delete;
	};
}
#endif
