/* $Id: ExtendedHTMLEditorKit.java 125 2011-03-09 09:49:51Z ohura $ */
/*
GNU Lesser General Public License

ExtendedHTMLEditorKit
Copyright (C) 2001  Frits Jalvingh, Jerry Pommer & Howard Kistler

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

package com.hexidec.ekit.component;

import java.awt.event.ActionEvent;
import java.io.IOException;
import java.io.Reader;
import java.io.Writer;

import javax.swing.Action;
import javax.swing.text.BadLocationException;
import javax.swing.text.Document;
import javax.swing.text.Element;
import javax.swing.text.StyleConstants;
import javax.swing.text.StyledDocument;
import javax.swing.text.TextAction;
import javax.swing.text.View;
import javax.swing.text.ViewFactory;
import javax.swing.text.html.HTML;
import javax.swing.text.html.HTMLDocument;
import javax.swing.text.html.HTMLEditorKit;
import javax.swing.text.html.MinimalHTMLWriter;
import javax.swing.text.html.StyleSheet;

/**
  * This class extends HTMLEditorKit so that it can provide other renderer classes
  * instead of the defaults. Most important is the part which renders relative
  * image paths.
  *
  * @author <a href="mailto:jal@grimor.com">Frits Jalvingh</a>
  * @version 1.0
  */

public class ExtendedHTMLEditorKit extends HTMLEditorKit
{
	
    public Action[] getActions() {
    	return TextAction.augmentList(super.getActions(), defaultActions);
    }
    
    private static final Action[] defaultActions = {
    	new FontFamilyAction("font-family-SansSerif", "SansSerif"),
    	new FontFamilyAction("font-family-Monospaced", "Monospaced"),
    	new FontFamilyAction("font-family-Serif", "Serif"),
    	new FontSizeAction("font-size-8", 8),
    	new FontSizeAction("font-size-10", 10),
    	new FontSizeAction("font-size-12", 12),
    	new FontSizeAction("font-size-14", 14),
    	new FontSizeAction("font-size-16", 16),
    	new FontSizeAction("font-size-18", 18),
    	new FontSizeAction("font-size-24", 24),
    	new FontSizeAction("font-size-36", 36),
    	new FontSizeAction("font-size-48", 48),
    	new AlignmentAction("left-justify", StyleConstants.ALIGN_LEFT),
    	new AlignmentAction("center-justify", StyleConstants.ALIGN_CENTER),
    	new AlignmentAction("right-justify", StyleConstants.ALIGN_RIGHT),
    	new BoldAction(),
    	new ItalicAction(),
    	//new InsertBreakAction(),
    	new ExtendedHTMLInsertBreakAction(),
    	new UnderlineAction()
        };	

    static class ExtendedHTMLInsertBreakAction extends StyledTextAction {
        ExtendedHTMLInsertBreakAction() {
            super(insertBreakAction);
        }

        public void actionPerformed(ActionEvent e) {
        	//System.out.println("break ignored");
        }
    }
    
	/** Constructor
	  */
	public ExtendedHTMLEditorKit()
	{
	}

	/** Method for returning a ViewFactory which handles the image rendering.
	  */
	public ViewFactory getViewFactory()
	{
		return new HTMLFactoryExtended();
	}

	public Document createDefaultDocument()
	{
		StyleSheet styles = getStyleSheet();
		StyleSheet ss = new StyleSheet();
		ss.addStyleSheet(styles);
		ExtendedHTMLDocument doc = new ExtendedHTMLDocument(ss);
		doc.setParser(getParser());
		doc.setAsynchronousLoadPriority(4);
		doc.setTokenThreshold(100);
		return doc;
	}
	
	

/* Inner Classes --------------------------------------------- */

	/** Class that replaces the default ViewFactory and supports
	  * the proper rendering of both URL-based and local images.
	  */
	public static class HTMLFactoryExtended extends HTMLFactory implements ViewFactory
	{
		/** Constructor
		  */
		public HTMLFactoryExtended()
		{
		}

		/** Method to handle IMG tags and
		  * invoke the image loader.
		  */
		public View create(Element elem)
		{
			Object obj = elem.getAttributes().getAttribute(StyleConstants.NameAttribute);
			if(obj instanceof HTML.Tag)
			{
				HTML.Tag tagType = (HTML.Tag)obj;
				if(tagType == HTML.Tag.IMG)
				{
					return new RelativeImageView(elem);
				}
			}
			return super.create(elem);
		}
	}

	boolean readWithoutHeader = false;
	
	public void setReadWithoutHeader(boolean f){
		readWithoutHeader = f;
	}
	
	public void read(Reader in, Document doc, int pos) throws IOException, BadLocationException{
		super.read(new ExtendedHTMLReader(in,readWithoutHeader), doc, pos);
		//super.read(in, doc, pos);
	}

	
    public void write(Writer out, Document doc, int pos, int len) 
    	throws IOException, BadLocationException {

		if (doc instanceof HTMLDocument) {
		    ExtendedHTMLWriter w = new ExtendedHTMLWriter(out, (HTMLDocument)doc, pos, len);
		    w.write();
		} else if (doc instanceof StyledDocument) {
		    MinimalHTMLWriter w = new MinimalHTMLWriter(out, (StyledDocument)doc, pos, len);
		    w.write();
		} else {
		    super.write(out, doc, pos, len);
		}
    }
}
