#ifndef INCLUDED_CREATE_LINK_H
#define INCLUDED_CREATE_LINK_H

#define _GNU_SOURCE

#include "ch_time_and_mod.h"
#include "struct_CPDD.h"
#include "struct_CPInfo.h"
#include "print_error.h"

#include <stdbool.h>

#include <linux/limits.h>

// 関数プロトタイプ
static inline _Bool create_symboliclink(const CPInfo *cpinfo, CPDD *cpdd) __attribute__((always_inline));

/*******************************************************************************
シンボリックリンクを作成
*******************************************************************************/
static inline _Bool create_symboliclink(const CPInfo *cpinfo, CPDD *cpdd)
{
	int readlink_len;
	char link_buf[PATH_MAX];
	_Bool check;
	errno = 0;

	if((readlink_len = readlink(cpinfo->from, link_buf, PATH_MAX)) == -1)
	{
		check = false;
		print_error("readlink", __FILE__, __LINE__, cpdd);
		fprintf(stderr, "シンボリックリンク読み込みエラーです (%s) \n", cpinfo->from);
	}
	else
	{
		/*
		 * readlinkは終端にヌル文字を付加しない、
		 * という仕様の所為でlink_bufの文字列がおかしくなるバグに悩まされた。
		 * ローカル変数はスコープから抜けると解放されるが、
		 * 即座に解放されるわけではなく（パフォーマンスが落ちるし）、
		 * 領域を使い回すこともしばしばある。
		 * この関数が呼び出されるとlink_bufにシンボリックリンクの内容がコピーされるが、
		 * 関数から抜けてもlink_bufの内容は保持されたままになる。
		 * 次回呼び出されたときにlink_bufの内容が上書きされ、
		 * それが前回よりも短い文字列だった場合、ヌル文字がないので、
		 * 文字列が混ざり合っておかしな事になっていた。
		*/
		link_buf[readlink_len] = '\0';

		if(symlink(link_buf, cpinfo->to) == -1)
		{
			check = false;
			print_error("symlink", __FILE__, __LINE__, cpdd);
			fprintf(stderr, "シンボリックリンク作成エラーです (%s) \n", cpinfo->to);
		}
		else
		{
			check = true;
			cpdd->mk_link_count++;
			struct stat stat_buf;

			if(lstat(cpinfo->from, &stat_buf) == -1)
			{
				print_error("lstat", __FILE__, __LINE__, cpdd);
				fprintf(stderr, "ファイル情報取得エラーです (%s) \n", cpinfo->from);
			}
			else
			{
				ch_time_and_mod(&stat_buf, cpinfo, cpdd);
			}
		}
	}

	return check;
}

#endif
