/*
 * ファイルのコピーを行う。
 * 一部、copybench-1.0（New BSD License）を参考にした。
*/
#define _GNU_SOURCE

#include "struct_CPDD.h"
#include "struct_CPInfo.h"
#include "struct_OPArg.h"

#include "ch_time_and_mod.h"
#include "compare_hash.h"
#include "compare_memcmp.h"
#include "print_error.h"

#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>

#include <errno.h>
#include <fcntl.h>
#include <pthread.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <unistd.h>

#define TO_OPEN \
{\
	errno = 0;\
	if((rwd.ito = open(cpinfo->to, O_CREAT | O_WRONLY, S_IRUSR | S_IWUSR)) == -1)\
	{\
		if(errno > 0)\
		{\
			print_error("open", __FILE__, __LINE__, cpdd);\
		}\
	}\
	else \
	{\
		to_open = true;\
		cpdd->mk_file_count++;\
	}\
}

typedef struct read_write_data
{
	void *buf[2];
	off_t from_size_local;
	off_t written_size[2];
	off_t next_write_size[2];
	off_t total_read_local;
	off_t total_write_local;
	off_t total_read_error; // スレッド用
	off_t total_write_error; // スレッド用
	off_t buffer_size_local;
	long psize;
	int ifrom;
	int ito;
	int rbuf_num;
	int wbuf_num;
} RWD;

// ※注意。グローバル変数。
char *write_now;

// 関数プロトタイプ
_Bool read_write(const OPArg *oparg, CPInfo *cpinfo, SDir *sdir, CPDD *cpdd);
static void * read_thread(void *read_t);
static void * write_thread(void *write_t);

/*******************************************************************************
*******************************************************************************/
_Bool read_write(const OPArg *oparg, CPInfo *cpinfo, SDir *sdir, CPDD *cpdd)
{
	_Bool from_stat = false;
	_Bool from_open = false;
	_Bool to_open = false;
	struct stat stat_buf;
	RWD rwd;
	rwd.from_size_local = 0;
	rwd.total_read_local = 0;
	rwd.total_write_local = 0;
	rwd.total_read_error = 0;
	rwd.total_write_error = 0;

	cpinfo->from_size = 0;
	cpinfo->to_size = 0;

	// signalハンドラ用
	write_now = cpinfo->to;

	errno = 0;
	if((rwd.ifrom = open(cpinfo->from, O_RDONLY | O_NOATIME)) == -1)
	{
		print_error("open", __FILE__, __LINE__, cpdd);
	}
	else
	{
		from_open = true;

		errno = 0;
		if(fstat(rwd.ifrom, &stat_buf) == -1)
		{
			print_error("fstat", __FILE__, __LINE__, cpdd);
		}
		else
		{
			from_stat = true;
			rwd.from_size_local = stat_buf.st_size;
		}

		if((cpinfo->file_test == false) || (oparg->write_mode == 3))
		{
			TO_OPEN
		}
		else if(from_stat == true)
		{
			struct stat stat_to;

			errno = 0;
			if(stat(cpinfo->to, &stat_to) == -1)
			{
				print_error("stat", __FILE__, __LINE__, cpdd);

				if(errno == ENOENT)
				{
					TO_OPEN
				}
			}
			else
			{
				switch(oparg->write_mode)
				{
				case 1:
					if((stat_buf.st_size != stat_to.st_size) || (stat_buf.st_mtime != stat_to.st_mtime))
					{
						TO_OPEN
					}
					else
					{
						cpinfo->skip = true;
					}
					break;

				case 2:
					if((stat_buf.st_mtime > stat_to.st_mtime))
					{
						TO_OPEN
					}
					else
					{
						cpinfo->skip = true;
					}
					break;
				}
			}
		}
		else
		{
			print_error("read_write", __FILE__, __LINE__, cpdd);
			fprintf(stderr, "%s をコピーできません\n", cpinfo->from);
		}
	}

	/*
	 * コピー前にファイルをトランケート。いわゆるsparse file化。
	 * Windowsだと断片化防止に効果があるのだが、Linuxだと？
	 * 
	 * 1.2TBほどコピーしてみたあと、fsckを実行したら、
	 * non-contiguousは3.2%だった。効果無し？
	 * 
	 * ↑と同じデータをトランケート無しでコピーしてみたら、
	 * non-contiguousの値は同じだった。
	 * non-contiguousって信頼していいのか？
	 * 
	 * NTFSやFATに書き込む前にftruncateすると、
	 * 断片化を押さえることが出来ることを確認。
	 * 
	 * ext4からfallocateシステムコールがサポートされた。
	 * sparse file化とは違うらしい？
	 * ext4以外の、それをサポートしていないファイルシステムだとかなり遅くなる様子。
	*/
	if((rwd.from_size_local > 0) && (to_open == true))
	{
		if(
		(cpinfo->tofs != NULL) &&
		(cpinfo->tofs[0] == 'e') &&
		(cpinfo->tofs[1] == 'x') &&
		(cpinfo->tofs[2] == 't') &&
		(cpinfo->tofs[3] == '4')
		)
		{
			errno = 0;
			if(posix_fallocate(rwd.ito, 0, rwd.from_size_local) != 0)
			{
				print_error("posix_fallocate", __FILE__, __LINE__, cpdd);

				errno = 0;
				if(ftruncate(rwd.ito, rwd.from_size_local) == -1)
				{
					print_error("ftruncate", __FILE__, __LINE__, cpdd);
				}
			}
		}
		else
		{
			errno = 0;
			if(pwrite(rwd.ito, "", 1, rwd.from_size_local - 1) == -1)
			{
				print_error("pwrite", __FILE__, __LINE__, cpdd);
			}
		}
	}

	// コピー開始
	if((rwd.from_size_local > 0) && (to_open == true))
	{
		if(oparg->V == VERBOS)
		{
			printf("%s -> %s\n", cpinfo->from, cpinfo->to);
		}

		if(oparg->buffer_size >= rwd.from_size_local)
		{
			errno = 0;
			rwd.buf[0] = mmap(NULL, rwd.from_size_local, PROT_READ, MAP_PRIVATE, rwd.ifrom, 0);

			if(rwd.buf[0] != MAP_FAILED)
			{
				rwd.total_read_local = rwd.from_size_local;
				errno = 0;
				if(madvise(rwd.buf[0], rwd.from_size_local, MADV_SEQUENTIAL) != 0)
				{
					print_error("madvise", __FILE__, __LINE__, cpdd);
				}

				char *tmp = rwd.buf[0];
				const char * const endp = tmp + rwd.from_size_local;

				while(tmp < endp)
				{
					errno = 0;
					if((rwd.written_size[0] = write(rwd.ito, tmp, endp - tmp)) == -1)
					{
						print_error("write", __FILE__, __LINE__, cpdd);
						break;
					}
					tmp += rwd.written_size[0];
					rwd.total_write_local += rwd.written_size[0];
				}

				errno = 0;
				if(munmap(rwd.buf[0], rwd.from_size_local) != 0)
				{
					print_error("munmap", __FILE__, __LINE__, cpdd);
				}
			}
			else
			{
				perror("mmap");
			}
		}
		else if(cpinfo->copy_mode == DIFFERENT)
		{
			rwd.psize = sysconf(_SC_PAGESIZE);

			// バッファを二つ使用するので
			rwd.buffer_size_local = oparg->buffer_size / 2;
			rwd.rbuf_num = 0;
			pthread_t t1;
			pthread_t t2;
			pthread_create(&t1, NULL, read_thread, (void *)&rwd);
			pthread_join(t1, NULL);

			for(;;)
			{
				rwd.rbuf_num = 1;
				rwd.wbuf_num = 0;
				pthread_create(&t1, NULL, read_thread, (void *)&rwd);
				pthread_create(&t2, NULL, write_thread, (void *)&rwd);
				pthread_join(t1, NULL);
				pthread_join(t2, NULL);

				if((rwd.from_size_local <= rwd.total_write_local) || (rwd.written_size[0] <= 0))
				{
					break;
				}

				rwd.rbuf_num = 0;
				rwd.wbuf_num = 1;
				pthread_create(&t1, NULL, read_thread, (void *)&rwd);
				pthread_create(&t2, NULL, write_thread, (void *)&rwd);
				pthread_join(t1, NULL);
				pthread_join(t2, NULL);

				if((rwd.from_size_local <= rwd.total_write_local) || (rwd.written_size[1] <= 0))
				{
					break;
				}
			}
		}
		else
		{
			rwd.psize = sysconf(_SC_PAGESIZE);

			for(;;)
			{
				off_t read_size = rwd.from_size_local - rwd.total_read_local;

				if(read_size >= oparg->buffer_size)
				{
					read_size = oparg->buffer_size;
				}
				else
				{
					off_t tmp = read_size % rwd.psize;

					if(tmp != read_size)
					{
						read_size = read_size - tmp;
					}
				}

				errno = 0;
				rwd.buf[0] = mmap(NULL, read_size, PROT_READ, MAP_PRIVATE, rwd.ifrom, rwd.total_read_local);

				if(rwd.buf[0] != MAP_FAILED)
				{
					rwd.total_read_local += read_size;

					errno = 0;
					if(madvise(rwd.buf[0], read_size, MADV_SEQUENTIAL) != 0)
					{
						print_error("madvise", __FILE__, __LINE__, cpdd);
					}

					char *tmp = rwd.buf[0];
					const char * const endp = tmp + read_size;

					while(tmp < endp)
					{
						errno = 0;
						if((rwd.written_size[0] = write(rwd.ito, tmp, endp - tmp)) == -1)
						{
							print_error("write", __FILE__, __LINE__, cpdd);

							errno = 0;
							if(munmap(rwd.buf[0], read_size) != 0)
							{
								print_error("munmap", __FILE__, __LINE__, cpdd);
							}

							goto LOOP_END;
						}
						tmp += rwd.written_size[0];
						rwd.total_write_local += rwd.written_size[0];
					}

					errno = 0;
					if(munmap(rwd.buf[0], read_size) != 0)
					{
						print_error("munmap", __FILE__, __LINE__, cpdd);
					}
				}
				else
				{
					perror("mmap");
					break;
				}

				if(rwd.total_read_local >= rwd.from_size_local)
				{
					break;
				}
			}
LOOP_END:
;
		}

		/*
		 * 余計に書き込んだ部分を切り詰める。
		 * または大きなファイルを小さなファイルで上書きした場合のリサイズ。
		 * open時にO_TRUNCしてない場合、重要な処理。
		 * ここを弄ってバージョン0.1.2でバグを出したので、うかつに弄らない。
		 * 
		 * 0.4.0から、
		 * openしたときにposix_fallocate、ftruncate、pwriteのいずれかを、
		 * 必ず行うようにしたが、念のためここの処理も残しておくべき。
		*/
		if(ftruncate(rwd.ito, rwd.from_size_local) == -1)
		{
			print_error("ftruncate", __FILE__, __LINE__, cpdd);
		}
	}

	if(from_open == true)
	{
		if(close(rwd.ifrom) == -1)
		{
			print_error("close", __FILE__, __LINE__, cpdd);
		}
	}

	if(to_open == true)
	{
		if(close(rwd.ito) == -1)
		{
			print_error("close", __FILE__, __LINE__, cpdd);
		}
	}

	if((from_stat == true) && (to_open == true))
	{
		ch_time_and_mod(&stat_buf, cpinfo, cpdd);
	}

	if(rwd.total_read_local > 0)
	{
		cpinfo->from_size = rwd.from_size_local;
		cpdd->total_read += rwd.total_read_local;
	}

	if(rwd.total_write_local > 0)
	{
		cpinfo->to_size = rwd.total_write_local;
		cpdd->total_write += rwd.total_write_local;
	}

	if(rwd.total_read_error > 0)
	{
		cpdd->total_error += rwd.total_read_error;
	}

	if(rwd.total_write_error > 0)
	{
		cpdd->total_error += rwd.total_write_error;
	}

	write_now = NULL;

	if(oparg->compare_mode == 1)
	{

		if((oparg->C == ONE) || (oparg->C == FIVE))
		{
			if((from_open == true) && (to_open == true))
			{
				compare_hash(oparg, cpinfo, sdir, cpdd);
			}
			else
			{
				fprintf(stderr, "ファイルのオープンに失敗しているため、コンペアをスキップします\n");
				fprintf(stderr, "from : %s\n", cpinfo->from);
				fprintf(stderr, "to   : %s\n", cpinfo->to);
			}
		}
		else if((oparg->C == COMPARE))
		{
			if((cpinfo->from_size > 0) && (cpinfo->to_size > 0))
			{
				compare_memcmp(oparg, cpinfo, sdir, cpdd);
			}
			else
			{
				fprintf(stderr, "ファイルサイズが0以下のため、コンペアをスキップします\n");
				fprintf(stderr, "from : %s\n", cpinfo->from);
				fprintf(stderr, "to   : %s\n", cpinfo->to);
			}
		}
		else
		{
			puts("コンペアオプションが不正です");
		}

		puts("");
	}

	if((from_open == true) && (to_open == true))
	{
		return true;
	}
	else
	{
		return false;
	}
}

/*******************************************************************************
 * 読み取り関数。pthreadで使うので仮引数はvoid *。
*******************************************************************************/
static void * read_thread(void *read_t)
{
	RWD *rwd = (RWD *)read_t;
	int i = rwd->rbuf_num;

	if(rwd->from_size_local <= rwd->total_read_local)
	{
		rwd->next_write_size[i] = 0;
		return NULL;
	}

	off_t read_size = rwd->from_size_local - rwd->total_read_local;

	if(read_size >= rwd->buffer_size_local)
	{
		read_size = rwd->buffer_size_local;
	}
	else
	{
		off_t tmp = read_size % rwd->psize;
		if(tmp != read_size)
		{
			read_size = read_size - tmp;
		}
	}

	errno = 0;
	rwd->buf[i] = mmap(NULL, read_size, PROT_READ, MAP_PRIVATE, rwd->ifrom, rwd->total_read_local);
	if(rwd->buf[i] != MAP_FAILED)
	{
		errno = 0;
		/*
		 * マルチスレッドの場合、
		 * MADV_SEQUENTIALよりもMADV_WILLNEEDの方が速い、ような気がする。
		*/
		if(madvise(rwd->buf[i], read_size, MADV_WILLNEED) != 0)
		{
			// このcpddに意味はない。print_errorの引数用。
			CPDD cpdd;
			cpdd.total_error = 0;
			print_error("madvise", __FILE__, __LINE__, &cpdd);
			rwd->total_read_error++;
		}

		rwd->total_read_local += read_size;
		rwd->next_write_size[i] = read_size;
	}
	else
	{
		rwd->next_write_size[i] = 0;
	}

	return NULL;
}

/*******************************************************************************
 * 書き込み関数。pthreadで使うので仮引数はvoid *。
*******************************************************************************/
static void * write_thread(void *write_t)
{
	RWD *rwd = (RWD *)write_t;
	int i = rwd->wbuf_num;
	char *tmp = (char *)rwd->buf[i];
	const char * const endp = tmp + rwd->next_write_size[i];

	if(rwd->next_write_size[i] > 0)
	{
		while(tmp < endp)
		{
			errno = 0;
			if((rwd->written_size[i] = write(rwd->ito, tmp, endp - tmp)) == -1)
			{
				// このcpddに意味はない。print_errorの引数用。
				CPDD cpdd;
				cpdd.total_error = 0;
				print_error("write", __FILE__, __LINE__, &cpdd);
				rwd->total_write_error++;
				break;
			}
			tmp += rwd->written_size[i];
			rwd->total_write_local += rwd->written_size[i];
		}

		errno = 0;
		if(munmap(rwd->buf[i], rwd->next_write_size[i]) == -1)
		{
			CPDD cpdd;
			cpdd.total_error = 0;
			print_error("munmap", __FILE__, __LINE__, &cpdd);
			rwd->total_write_error++;
		}
	}
	else
	{
		rwd->written_size[i] = 0;
	}

	return NULL;
}
