#include "analyze_argument.h"
#include "compare_hash.h"
#include "compare_memcmp.h"
#include "get_drive_list.h"
#include "get_file_list.h"
#include "get_settings_dir.h"
#include "help.h"
#include "print_error.h" // inline
#include "process_file_list.h"
#include "read_conf.h"
#include "set_signal.h"
#include "struct_CPDD.h"
#include "struct_DPath.h"
#include "struct_OPArg.h"
#include "struct_SData.h"
#include "write_conf.h"

#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>

// 関数プロトタイプ
static inline void do_compare_two(const OPArg *oparg, const VList *file_list, const SDir *sdir, CPDD *cpdd) __attribute__((always_inline));
static inline void delete_from(const OPArg *oparg, const VList *file_list, CPDD *cpdd) __attribute__((always_inline));

/*******************************************************************************
*******************************************************************************/
int main(int argc, char **argv)
{
	if(argc == 1)
	{
		help();
		return EXIT_SUCCESS;
	}

	SDir *sdir = get_settings_dir();

	// 引数が1つで、-oの場合、オプションを設定、保存し、終了する。
	if(argc == 2)
	{
		char *tmp = argv[1];
		if((tmp[0] == '-') && (tmp[1] == 'o'))
		{
			write_conf(sdir);
			return EXIT_SUCCESS;
		}
	}

	OPArg *oparg = read_conf(sdir);
	analyze_argument(argc, argv, oparg);
	VList *drive_list = get_drive_list();
	set_signal();

	puts("コピー対象の一覧を作成中です...");

	SData *sdata = get_file_list(oparg, drive_list);
	VList *file_list = sdata->file_list;
	CPDD *cpdd = sdata->cpdd;

	if(file_list != NULL)
	{
		puts("コピーを開始します...");

		process_file_list(oparg, file_list, sdir, cpdd);

		puts("コピーを完了しました...");
	}

	if((file_list != NULL) && (oparg->COMPARE_MODE == AFTER))
	{
		puts("コンペアを開始します...");

		do_compare_two(oparg, file_list, sdir, cpdd);

		puts("コンペアを完了しました...");
	}

	if(oparg->M == MOVE)
	{
		puts("コピー元を削除します...");

		delete_from(oparg, file_list, cpdd);

		puts("コピー元の削除を完了しました...");
	}

	puts("");

	printf("File   : %lld / %lld\n", cpdd->mk_file_count, cpdd->file_count);
	printf("Link   : %lld / %lld\n", cpdd->mk_link_count, cpdd->link_count);
	printf("Dir    : %lld / %lld\n", cpdd->mk_dir_count, cpdd->dir_count);
	printf("Source : %lld MB (%lld Byte)\n", cpdd->total_source / (1024 * 1024), cpdd->total_source);
	printf("Read   : %lld MB (%lld Byte)\n", cpdd->total_read / (1024 * 1024), cpdd->total_read);
	printf("Write  : %lld MB (%lld Byte)\n", cpdd->total_write / (1024 * 1024), cpdd->total_write);

	if(cpdd->total_error > 0)
	{
		printf("Error  : %lld\n", cpdd->total_error);
	}

	/*
	 * close()する必要があるもの
	 * sdir->current_fd
	*/

	/*
	 * 基本的に、mallocしたのはfreeしてない。
	*/

	puts("");

	return EXIT_SUCCESS;
}

/*******************************************************************************
*******************************************************************************/

#define HASH_CHECK \
{\
	if(cpinfo->write == true)\
	{\
		compare_hash(oparg, cpinfo, sdir, cpdd);\
	}\
	else if((cpinfo->write == false) && (cpinfo->skip == false))\
	{\
		fprintf(stderr, "ファイルの書き込みに失敗しているため、コンペアをスキップします\n");\
		fprintf(stderr, "from : %s\n", cpinfo->from);\
		fprintf(stderr, "to   : %s\n", cpinfo->to);\
	}\
}

static inline void do_compare_two(const OPArg *oparg, const VList *file_list, const SDir *sdir, CPDD *cpdd)
{
	/*
	* キャストしないと警告が出る
	* warning: initialization discards qualifiers from pointer target type
	*/
	VList *tmp = (VList *)file_list;

	for(;;)
	{
		CPInfo *cpinfo = (CPInfo *)tmp->data;

		if(cpinfo->from_type == REGULAR)
		{
			switch(oparg->C)
			{
			case ONE:
				HASH_CHECK
				break;

			case FIVE:
				HASH_CHECK
				break;

			case MEMCMP:
				if((cpinfo->from_size > 0) && (cpinfo->to_size > 0))
				{
					compare_memcmp(oparg, cpinfo, sdir, cpdd);
				}
				else
				{
					fprintf(stderr, "ファイルサイズが0以下のため、コンペアをスキップします\n");
					fprintf(stderr, "from : %s\n", cpinfo->from);
					fprintf(stderr, "to   : %s\n", cpinfo->to);
				}
				break;

			default:
				fprintf(stderr, "コンペアオプションが不正です");
				break;
			}
		}

		if(tmp->next == NULL)
		{
			break;
		}

		tmp = tmp->next;
	}
}

/*******************************************************************************
*******************************************************************************/
#define FILE_REMOVE \
{\
	errno = 0;\
	if(unlink(cpinfo->from) == -1)\
	{\
		if(errno != ENOENT)\
		{\
			print_error("unlink", __FILE__, __LINE__, cpdd);\
			fprintf(stderr, "%s の削除に失敗しました\n", cpinfo->from);\
		}\
	}\
}

#include <unistd.h>

static inline void delete_from(const OPArg *oparg, const VList *file_list, CPDD *cpdd)
{
	/*
	* キャストしないと警告が出る
	* warning: initialization discards qualifiers from pointer target type
	*/
	VList *tmp = (VList *)file_list;
	tmp = tmp->tail;

	for(;;)
	{
		CPInfo *cpinfo = (CPInfo *)tmp->data;

		switch(cpinfo->from_type)
		{
		case REGULAR:
			if(oparg->C != NOT)
			{
				if(cpinfo->compare == true)
				{
					FILE_REMOVE
				}
			}
			else
			{
				if((cpinfo->write == true) && (cpinfo->from_size <= cpinfo->to_size))
				{
					FILE_REMOVE
				}
			}
			break;

		case SYMBOLICLINK:
			if(cpinfo->write == true)
			{
				FILE_REMOVE
			}
			break;

		case DIRECTORY:
			errno = 0;
			if(rmdir(cpinfo->from) == -1)
			{
				if((errno != ENOTEMPTY) && (errno != ENOENT))
				{
					print_error("remove", __FILE__, __LINE__, cpdd);
					fprintf(stderr, "%s の削除に失敗しました\n", cpinfo->from);
				}
			}
			break;

		default:
			// ここは実行されないはず
			fprintf(stderr, "ファイルタイプが不正です\n");
			break;
		}

		if(tmp->prev == NULL)
		{
			break;
		}

		tmp = tmp->prev;
	}
}
