#include "struct_OPArg.h"
#include "struct_SDir.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>

#include <glib.h>
#include <sys/stat.h>
#include <unistd.h>

#define INPUT_AND_CLEAN \
{\
	if(fgets(stdin_buf, INPUT_LEN, stdin) == NULL)\
	{\
		fgets_is_null();\
	}\
\
	clean(stdin_buf, INPUT_LEN);\
}

#define MIN_BUF_SIZE 2

/* 関数プロトタイプ */
void write_conf(const SDir *sdir);
static _Bool is_enabled_option(char stdin_buf[], const int INPUT_LEN, const char option);
static char set_hash_check_mode(char stdin_buf[], const int INPUT_LEN);
static int change_buffer_size(char stdin_buf[], const int INPUT_LEN);
static char change_overwrite_mode(char stdin_buf[], const int INPUT_LEN);
static char change_thread_mode(char stdin_buf[], const int INPUT_LEN);
static void clean(const char stdin_buf[], const int INPUT_LEN);
static void fgets_is_null(void);
static void write_error(const char *mes);

/*******************************************************************************
 * 設定をファイルへ保存する
*******************************************************************************/
void write_conf(const SDir *sdir)
{
	const int INPUT_LEN = 256;
	char stdin_buf[INPUT_LEN];

	puts("設定の変更を行います");

	_Bool is_verify = is_enabled_option(stdin_buf, INPUT_LEN, 'r');

	char hash_check_mode = '\0';
	_Bool is_hash_save = false;

	if(is_verify == true)
	{
		hash_check_mode = set_hash_check_mode(stdin_buf, INPUT_LEN);
		is_hash_save = is_enabled_option(stdin_buf, INPUT_LEN, 'l');
	}

	_Bool is_buffer_size = is_enabled_option(stdin_buf, INPUT_LEN, 'b');
	int buffer_size = 0;

	if(is_buffer_size == true)
	{
		buffer_size = change_buffer_size(stdin_buf, INPUT_LEN);
	}

	_Bool is_info = is_enabled_option(stdin_buf, INPUT_LEN, 'i');
	_Bool is_overwrite = false;

	if(is_info != true)
	{
		is_overwrite = is_enabled_option(stdin_buf, INPUT_LEN, 'w');
	}

	char write_mode = '\0';

	if(is_overwrite == true)
	{
		write_mode = change_overwrite_mode(stdin_buf, INPUT_LEN);
	}

	_Bool is_delete = is_enabled_option(stdin_buf, INPUT_LEN, 'm');
	_Bool is_thread = is_enabled_option(stdin_buf, INPUT_LEN, 't');

	char thread_mode = '\0';

	if(is_thread == true)
	{
		thread_mode = change_thread_mode(stdin_buf, INPUT_LEN);
	}

	_Bool is_view = is_enabled_option(stdin_buf, INPUT_LEN, 'v');

	/* 設定をファイルに保存する */
	if(
		(true == is_verify) ||
		(MIN_BUF_SIZE <= buffer_size) ||
		(true == is_info) ||
		(true == is_overwrite) ||
		(true == is_delete) ||
		(true == is_thread) ||
		(true == is_view)
	)
	{
		/* フォルダが存在しない場合は作成 */
		if(g_file_test(sdir->config_dir, G_FILE_TEST_EXISTS) == FALSE)
		{
			mkdir(sdir->config_dir, 0777);
			chmod(sdir->config_dir, 0777);
		}
		if(g_file_test(sdir->user_settings_dir, G_FILE_TEST_EXISTS) == FALSE)
		{
			mkdir(sdir->user_settings_dir, 0777);
			chmod(sdir->user_settings_dir, 0777);
		}

		if(chdir(sdir->user_settings_dir) != 0)
		{
			fprintf(stderr, "設定フォルダにアクセスできませんでした\n");
			fprintf(stderr, "強制終了します\n");
			exit(EXIT_FAILURE);
		}

		/* ファイルに設定を書き込めない場合、Segmentation faultで落ちるかも */

		FILE *config;
		if((config = fopen(SNOWCP_CONF, "w")) != NULL)
		{
			if(fputs("!,", config) == EOF)
			{
				write_error("fputs");
			}
			else if(fputs(SNOWCP_CONF_VER, config) == EOF)
			{
				write_error("fputs");
			}
			else if(fputs("\n", config) == EOF)
			{
				write_error("fputs");
			}

			puts(" ");

			if(is_verify == true)
			{
				if(fputc('r', config) == EOF)
				{
					write_error("fputc");
				}
				else if(fputs(",", config) == EOF)
				{
					write_error("fputs");
				}
				else if(fputc(hash_check_mode, config) == EOF)
				{
					write_error("fputc");
				}
				else if(fputs("\n", config) == EOF)
				{
					write_error("fputs");
				}

				printf("ベリファイモード : %c\n", hash_check_mode);
			}

			if(is_hash_save == true)
			{
				if(
					(fputs("l", config) == EOF) ||
					(fputs(",", config) == EOF) ||
					(fputs("NULL", config) == EOF) ||
					(fputs("\n", config) == EOF)
				)
				{
					write_error("fputs");
				}

				puts("ベリファイ/コンペアのログを保存する");
			}

			if(MIN_BUF_SIZE <= buffer_size)
			{
				char tmp[INPUT_LEN];
				sprintf(tmp, "%d", buffer_size);

				if(
					(fputs("b", config) == EOF) ||
					(fputs(",", config) == EOF) ||
					(fputs(tmp, config) == EOF) ||
					(fputs("\n", config) == EOF)
				)
				{
					write_error("fputs");
				}

				printf("バッファサイズ : %d\n", buffer_size);
			}

			if(is_info == true)
			{
				if(
					(fputs("i", config) == EOF) ||
					(fputs(",", config) == EOF) ||
					(fputs("NULL", config) == EOF) ||
					(fputs("\n", config) == EOF)
				)
				{
					write_error("fputs");
				}

				puts("上書き確認する");
			}

			if(is_overwrite == true)
			{
				if(fputs("w", config) == EOF)
				{
					write_error("fputc");
				}
				else if(fputs(",", config) == EOF)
				{
					write_error("fputs");
				}
				else if(fputc(write_mode, config) == EOF)
				{
					write_error("fputc");
				}
				else if(fputs("\n", config) == EOF)
				{
					write_error("fputs");
				}

				printf("上書きモード   : %c\n", write_mode);
			}

			if(is_delete == true)
			{
				if(
					(fputs("m", config) == EOF) ||
					(fputs(",", config) == EOF) ||
					(fputs("NULL", config) == EOF) ||
					(fputs("\n", config) == EOF)
				)
				{
					write_error("fputs");
				}

				puts("移動モード");
			}

			if(is_thread == true)
			{
				if(fputs("t", config) == EOF)
				{
					write_error("fputc");
				}
				else if(fputs(",", config) == EOF)
				{
					write_error("fputs");
				}
				else if(fputc(thread_mode, config) == EOF)
				{
					write_error("fputc");
				}
				else if(fputs("\n", config) == EOF)
				{
					write_error("fputs");
				}

				printf("スレッドモード : %c\n", thread_mode);
			}

			if(is_view == true)
			{
				if(
					(fputs("v", config) == EOF) ||
					(fputs(",", config) == EOF) ||
					(fputs("NULL", config) == EOF) ||
					(fputs("\n", config) == EOF)
				)
				{
					write_error("fputs");
				}

				puts("コピー中のファイル名を表示する");
			}

			fclose(config);
			chmod(SNOWCP_CONF, 0666);
		}
		else
		{
			perror("fopen");
			fprintf(stderr, "設定の保存に失敗しました\n");
			fprintf(stderr, "設定ファイルを削除します\n");
			unlink(SNOWCP_CONF);
		}
	}
	else
	{
		if(chdir(sdir->user_settings_dir) != 0)
		{
			fprintf(stderr, "設定フォルダにアクセスできませんでした\n");
			fprintf(stderr, "強制終了します\n");
			exit(EXIT_FAILURE);
		}

		fprintf(stderr, "設定ファイルを削除します\n");
		unlink(SNOWCP_CONF);
	}
}

/*******************************************************************************
*******************************************************************************/
static _Bool is_enabled_option(char stdin_buf[], const int INPUT_LEN, const char option)
{
	puts("");

	switch(option)
	{
	case 'b':
		puts("バッファのサイズを変更しますか? (初期値は32MB)");
		break;

	case 'i':
		puts("ファイルを上書きする前に確認しますか?");
		break;

	case 'l':
		puts("ベリファイのログを保存しますか?");
		break;

	case 'm':
		puts("移動モード (ファイルコピー後、元ファイルを削除) に設定しますか?");
		break;

	case 'r':
		puts("ファイルコピー後にベリファイを行いますか?");
		break;

	case 't':
		puts("スレッドモードを指定しますか? (初期設定は自動判別)");
		break;

	case 'w':
		puts("上書きモードを変更しますか? (初期設定は、サイズ又は日付が異なる場合に上書き)");
		break;

	case 'v':
		puts("コピー中のファイル名を表示しますか?");
		break;

	default:
		puts("");
		break;
	}

	printf("(y / n) :");
	fflush(stdout);

	INPUT_AND_CLEAN

	return (stdin_buf[0] == 'y') ? true : false;
}

/*******************************************************************************
*******************************************************************************/
static char set_hash_check_mode(char stdin_buf[], const int INPUT_LEN)
{
	while(true)
	{
		puts("");
		puts("ベリファイモードを選んでください");
		puts("1 : コピー直後にベリファイ");
		puts("2 : 全コピー終了後に一括ベリファイ");
		printf("(1 / 2) :");
		fflush(stdout);

		INPUT_AND_CLEAN

		if(stdin_buf[0] == '1' || stdin_buf[0] == '2')
		{
			break;
		}
	}

	return stdin_buf[0];
}

/*******************************************************************************
*******************************************************************************/
static int change_buffer_size(char stdin_buf[], const int INPUT_LEN)
{
	puts("");
	puts("数値を入力してください");
	printf(" :");
	fflush(stdout);

	INPUT_AND_CLEAN

	char *endptr = NULL;
	int buffer_size = (int)strtol(stdin_buf, &endptr, 10);

	if(buffer_size < 2)
	{
		buffer_size = MIN_BUF_SIZE;
	}

	return buffer_size;
}

/*******************************************************************************
*******************************************************************************/
static char change_overwrite_mode(char stdin_buf[], const int INPUT_LEN)
{
	while(true)
	{
		puts("");
		puts("上書きモードを選んでください");
		puts("1 : 初期設定");
		puts("2 : 日付が新しい場合に上書き");
		puts("3 : 常に上書き");
		puts("0 : 上書きしない");
		printf("(1 / 2 / 3 / 0) :");
		fflush(stdout);

		INPUT_AND_CLEAN

		if((stdin_buf[0] == '1') || (stdin_buf[0] == '2') || (stdin_buf[0] == '3') || (stdin_buf[0] == '0'))
		{
			break;
		}
	}

	return stdin_buf[0];
}

/*******************************************************************************
*******************************************************************************/
static char change_thread_mode(char stdin_buf[], const int INPUT_LEN)
{
	while(true)
	{
		puts("");
		puts("モードを選んでください (初期設定は自動判別)");
		puts("1 : 常に同一ドライブモード");
		puts("2 : 常に別ドライブモード");
		puts("0 : 初期設定");
		printf("(1 / 2 / 0) :");
		fflush(stdout);

		INPUT_AND_CLEAN

		if((stdin_buf[0] == '1') || (stdin_buf[0] == '2') || (stdin_buf[0] == '0'))
		{
			break;
		}
	}

	return stdin_buf[0];
}

/*******************************************************************************
 * 入力バッファのクリア
*******************************************************************************/
static void clean(const char stdin_buf[], const int INPUT_LEN)
{
	/* バッファチェック */
	_Bool check = false;

	for(int i = 0; i < INPUT_LEN; i++)
	{
		if(stdin_buf[i] == '\n')
		{
			check = true;
			break;
		}
	}

	/* 入力バッファの掃除 */
	if(check == false)
	{
		while(getchar() != '\n');
	}
}

/*******************************************************************************
 * fgetsの戻り値がNULLだった場合
*******************************************************************************/
static void fgets_is_null(void)
{
	fprintf(stderr, "エラーが発生しました\n");
	fprintf(stderr, "fgetsの戻り値がNULLです\n");
	exit(EXIT_FAILURE);
}

/*******************************************************************************
 * 設定ファイルに書き込めなかった場合
*******************************************************************************/
static void write_error(const char *mes)
{
	perror(mes);
	fprintf(stderr, "設定ファイル書き込み時にエラーが発生しました\n");
	fprintf(stderr, "設定ファイルを削除します\n");
	unlink(SNOWCP_CONF);
	exit(EXIT_FAILURE);
}
