# -*- coding: utf-8 -*-
class Ticket < ActiveRecord::Base
  belongs_to :project
  belongs_to :state, class_name: 'State', foreign_key: 'state_id'
  belongs_to :priority, class_name: 'Priority', foreign_key: 'priority_id'
  belongs_to :workload, class_name: 'Workload', foreign_key: 'workload_id'
  belongs_to :assigned_user, class_name: 'User', foreign_key: 'assigned_user_id'
  belongs_to :created_by, class_name: 'User', foreign_key: 'updated_by_user_id'

  validates :title, presence: true
  validates :project_id, presence: true

  after_validation :set_assigned_if_private_project
  before_create :check_double_post
  after_create :create_ticket_log, :add_completed, :add_remaining
  before_update :add_ticket_log, :update_completed, :update_remaining

  private

  StateOpen = 1
  StateResolved = 2
  StateRejected = 3

  Workloads_weight = [ 0, 1, 3, 5 ]

  def set_assigned_if_private_project
    if not self.project.nil? and not self.project.is_group?
      self[:assigned_user_id] = self[:updated_by_user_id]
    end
  end

  def check_double_post
    # 全く同じ投稿が連続して行えないようにする
    last_ticket = Ticket.where(id: Ticket.maximum(:id)).first
    unless last_ticket
      # ID: 1 のチケットの場合は last_ticket が存在しないので、チェックしない
      return true
    end

    if last_ticket.title == self.title and
        last_ticket.comment == self.comment and
        last_ticket.state_id == self.state_id and
        last_ticket.priority_id == self.priority_id and
        last_ticket.workload_id == self.workload_id and
        last_ticket.assigned_user_id == self.assigned_user_id and
        last_ticket.updated_by_user_id == self.updated_by_user_id
      return false
    end

    return true
  end

  def create_ticket_log
    # 作成されたチケットのコピーを保存する
    TicketLogs.create(ticket_id: self.id,
                      title: self.title,
                      comment: self.comment,
                      state_id: self.state_id,
                      priority_id: self.priority_id,
                      workload_id: self.workload_id,
                      assigned_user_id: self.assigned_user_id,
                      updated_by_user_id: self.updated_by_user_id)
  end

  def add_ticket_log
    # 保存前の時点の Ticket 情報を取得する
    previous = Ticket.find(self.id)
    current = self

    # 保存する Ticket 情報と異なるフィールドを抽出し保存する
    ticket_log = {}

    ticket_log[:title] = current.title unless previous.title == current.title
    ticket_log[:comment] = current.comment unless
      current.comment.empty? or previous.comment == current.comment
    ticket_log[:state_id] = current.state_id unless
      previous.state_id == current.state_id
    ticket_log[:priority_id] = current.priority_id unless
      previous.priority_id == current.priority_id
    ticket_log[:workload_id] = current.workload_id unless
      previous.workload_id == current.workload_id
    ticket_log[:assigned_user_id] = current.assigned_user_id unless
      previous.assigned_user_id == current.assigned_user_id
    ticket_log[:project_id] = current.project_id unless
      previous.project_id == current.project_id

    if ticket_log.empty?
      # 変更がなければ、ログを保存しない
      return false
    end

    ticket_log[:ticket_id] = current.id
    ticket_log[:updated_by_user_id] = current.updated_by_user_id
    TicketLogs.create(ticket_log)
  end

  def add_completed
    current = self
    assign_completed(current)
    true
  end

  def update_completed
    previous = Ticket.find(self.id)
    current = self

    cleanup_completed(previous)
    assign_completed(current)
    true
  end

  def add_remaining
    current = self
    assign_remaining(current)
    true
  end

  def update_remaining
    previous = Ticket.find(self.id)
    current = self

    cleanup_remaining(previous)
    assign_remaining(current)
    true
  end

  def cleanup_completed(previous)
    workloads = workloads_weight(previous.workload_id)

    case previous.state_id
    when StateOpen
      # total の値を減らす
      add_to_total_completed(previous.project_id, previous.priority_id,
                             -workloads, 0)
    when StateResolved
      # total, completed の値を減らす
      add_to_total_completed(previous.project_id, previous.priority_id,
                             -workloads, -workloads)
    when StateRejected
      # 何もしない
    end
  end

  def assign_completed(current)
    workloads = workloads_weight(current.workload_id)

    case current.state_id
    when StateOpen
      # total の値を増やす
      add_to_total_completed(current.project_id, current.priority_id,
                             workloads, 0)

    when StateResolved
      # total, completed の値を増やす
      add_to_total_completed(current.project_id, current.priority_id,
                             workloads, workloads)

    when StateRejected
      # 何もしない
    end
  end

  def cleanup_remaining(previous)
    if previous.state_id == StateOpen
      workloads = workloads_weight(previous.workload_id)
      add_to_remaining(previous.project_id, previous.priority_id, -workloads)
    end
  end

  def assign_remaining(current)
    if current.state_id == StateOpen
      workloads = workloads_weight(current.workload_id)
      add_to_remaining(current.project_id, current.priority_id, workloads)
    end
  end

  def workloads_weight(workload_id)
    Workloads_weight[workload_id]
  end

  def add_to_total_completed(project_id, priority_id,
                             total_workloads, completed_workloads)
    today_completed = CompletedWorkloads.
      where(date: Date.today, project_id: project_id, priority_id: priority_id)
    if today_completed.count.zero?
      today_completed =
        CompletedWorkloads.new(date: Date.today, project_id: project_id,
                               priority_id: priority_id, total: 0, completed: 0)
    else
      today_completed = today_completed.first
    end

    today_completed.total += total_workloads
    today_completed.completed += completed_workloads
    today_completed.save()
  end

  def add_to_remaining(project_id, priority_id, workloads)
    today_remaining = RemainingWorkloads.
      where(date: Date.today, project_id: project_id, priority_id: priority_id)
    if today_remaining.count.zero?
      today_remaining =
        RemainingWorkloads.new(date: Date.today, project_id: project_id,
                               priority_id: priority_id, completed: 0)
    else
      today_remaining = today_remaining.first
    end

    today_remaining.completed += workloads
    today_remaining.save()
  end
end
