package org.dyndns.nuda.mapper;

import java.lang.reflect.Proxy;
import java.sql.Connection;
import java.sql.SQLException;

import javax.sql.DataSource;

import org.dyndns.nuda.mapper.annotation.JDBCQuery;
import org.dyndns.nuda.tools.util.StringUtil;

public class SQLInterfaceFactory {
	private Connection	con						= null;
	
	private boolean		useAutoCommit			= false;
	
	private boolean		useManualTransaction	= false;
	
	private ClassLoader	currentClassLoader		= null;
	
	private SQLInterfaceFactory() {
		// インスタンス化させない
	}
	
	/**
	 * このクラスのインスタンスを取得します
	 * 
	 * @return SQLInterfaceFactory
	 */
	public static SQLInterfaceFactory newInstance() {
		SQLInterfaceFactory builder = new SQLInterfaceFactory();
		
		return builder;
	}
	
	public <T> T create(final Class<T> interfaceClass) {
		
		T result = null;
		
		try {
			this.checkParameter(interfaceClass);
		} catch (Exception e) {
			throw new IllegalArgumentException(e.getMessage());
		}
		
		if (this.currentClassLoader == null) {
			this.currentClassLoader = interfaceClass.getClassLoader();
		}
		
		ClassLoader oldLoader = Thread.currentThread().getContextClassLoader();
		
		Thread.currentThread().setContextClassLoader(this.currentClassLoader);
		
		JDBCXMLInvocationHandler handler;
		try {
			handler =
				new JDBCXMLInvocationHandler(
					this.con,
					interfaceClass,
					this.currentClassLoader,
					this.useAutoCommit,
					this.useManualTransaction);
			
			Object intf =
				Proxy.newProxyInstance(
					Thread.currentThread().getContextClassLoader(),
					new Class<?>[] { interfaceClass },
					handler);
			
			result = interfaceClass.cast(intf);
			
			Thread.currentThread().setContextClassLoader(oldLoader);
			
			return result;
		} catch (Exception e) {
			e.printStackTrace();
		}
		
		return result;
	}
	
	private <T> void checkParameter(final Class<T> interfaceClass)
			throws Exception {
		if (interfaceClass == null) {
			String message =
				StringUtil.format("param [interfaceClass] is null");
			throw new Exception(message);
		}
		if (!interfaceClass.isInterface()) {
			String message =
				StringUtil.format(
					"param [interfaceClass({})] is not a interface",
					interfaceClass.getCanonicalName());
			throw new Exception(message);
		}
		if (!interfaceClass.isAnnotationPresent(JDBCQuery.class)) {
			String message =
				StringUtil
					.format(
						"param [interfaceClass({})] is not presented 'org.dyndns.nuda.mapper.JDBCQuery'",
						interfaceClass.getCanonicalName());
			throw new Exception(message);
		}
		
		if (this.con == null) {
			String message = StringUtil.format("connection is not set");
			throw new Exception(message);
		}
	}
	
	public SQLInterfaceFactory connection(final Connection con) {
		if (con != null) {
			this.con = con;
		}
		
		return this;
	}
	
	public SQLInterfaceFactory dataSource(final DataSource dataSource) {
		try {
			if (dataSource != null) {
				this.con = dataSource.getConnection();
			}
			
		} catch (SQLException e) {
			IllegalArgumentException e2 = new IllegalArgumentException(e);
			
			throw e2;
		}
		return this;
	}
	
	public SQLInterfaceFactory useAutoCommit(final boolean useAutoCommit) {
		this.useAutoCommit = useAutoCommit;
		return this;
	}
	
	public SQLInterfaceFactory useManualTransaction(
			final boolean useManualTransaction) {
		this.useManualTransaction = useManualTransaction;
		return this;
	}
	
	public SQLInterfaceFactory currentClassLoader(
			final ClassLoader currentClassLoader) {
		if (currentClassLoader != null) {
			this.currentClassLoader = currentClassLoader;
		}
		
		return this;
	}
}
