package orch;

import java.awt.Dimension;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.image.BufferedImage;
import java.awt.image.PixelGrabber;
import java.io.File;
import javax.imageio.ImageIO;
import sos.io.SOSImageIO;
import sos.util.ColorVector;

/*
 * Thumbnail.java
 *
 * Created on 2004/12/19, 22:48
 */

/**
 *
 * @author Scientific Open Source projects (Gaku Tanaka)
 * @version 1.0 (2005/02/15)
 */
public class ImagePiece{
	private File fileOfImage;
	private boolean hasImage;
	private boolean fixed = false;
	private int tableIndex;
	private Point tableLocation = new Point();
	
	private Rectangle bounds;
	private int[] pixelsData;
	private int[][] rgb;
	
	private Image thumbnail;
	private Dimension thumbnailSize = new Dimension();
	
	private int[][] histograms;
	private Rectangle partialBounds = new Rectangle();
	
	/** Creates a new instance of Thumbnail */
	public ImagePiece(int width, int height, File file) {
		fileOfImage = file;
		thumbnailSize.width = width;
		thumbnailSize.height = height;
		makeThumbnail();
	}
	
	public String getPath(){
		if (fileOfImage==null){
			return null;
		}
		return fileOfImage.getPath();
	}
	
	private int[] getPixels(Image img){
		int[] pixels = new int [bounds.width*bounds.height];
		PixelGrabber grabber = new PixelGrabber(img,0,0,bounds.width,bounds.height, pixels, 0,bounds.width);
		try{
			grabber.grabPixels();
		}catch(Exception e){
			e.printStackTrace();
		}
		return pixels;
	}
	
	private void makeThumbnail(){
		Dimension size = SOSImageIO.getImageSize(fileOfImage);
		bounds = new Rectangle(0,0,size.width,size.height);
		int stepX = (int)Math.ceil(1.0*bounds.width/thumbnailSize.width);
		int stepY = (int)Math.ceil(1.0*bounds.height/thumbnailSize.height);
		int step = Math.max(stepX,stepY);
		
		thumbnail = null;
		if (step<=1){
			thumbnail = SOSImageIO.readImage(fileOfImage);
		}else{
			thumbnail= SOSImageIO.getSubsampledImage(fileOfImage, step,step);
		}
		hasImage = (thumbnail==null)?false:true;
	}
	
	public Image getThumbnail(){
		if (thumbnail==null){
			makeThumbnail();
		}
		return thumbnail;
	}
	
	public void clearThumbnail(){
		thumbnail = null;
	}
	
	public boolean hasImage(){
		return hasImage;
	}
	
	public BufferedImage getImage(){
		BufferedImage img = null;
		try{
			img = ImageIO.read(fileOfImage);
		}catch(Exception e){
			e.printStackTrace();
		}
		return img;
	}
	
	public Rectangle getBounds(){
		return bounds;
	}
	
	public Dimension getSize(){
		return bounds.getSize();
	}
	
	public int[] getPixelsData(){
		if (pixelsData==null){
			Image img = getImage();
			pixelsData = ColorVector.intRGBtoL(getPixels(img));
		}
		
		return pixelsData;
	}
	
	public int[] getPixelsData(int step){
		if (pixelsData==null){
			Image img = SOSImageIO.readImage(fileOfImage);
			Image subsampledImg = img.getScaledInstance(bounds.width/step,bounds.height/step,Image.SCALE_AREA_AVERAGING);
			Image expandImg = subsampledImg.getScaledInstance(bounds.width,bounds.height, Image.SCALE_FAST);
			pixelsData = ColorVector.intRGBtoL(getPixels(expandImg));
		}
		
		return pixelsData;
	}
	
	public int[][] getRGB(int step){
		if (rgb==null){
			Image img = SOSImageIO.readImage(fileOfImage);
			Image subsampledImg = img.getScaledInstance(bounds.width/step,bounds.height/step,Image.SCALE_AREA_AVERAGING);
			//Image subsampledImg = SosIO.getSubsampledImage(fileOfImage, step,step);
			Image expandImg = subsampledImg.getScaledInstance(bounds.width,bounds.height, Image.SCALE_FAST);
			//new ImagePlus("step="+step, expandImg).show();
			rgb = new int[bounds.width*bounds.height][3];
			
			ColorVector.getRGB(getPixels(expandImg),rgb);
		}
		
		return rgb;
	}	
	
	public void clearRGB(){
		rgb = null;
	}
	
	public void clearPixelsData(){
		pixelsData = null;
	}
	
	public int getThumbnailOriginalRatio(){
		return bounds.width/thumbnailSize.width;
	}
	
	public boolean isFixed() {
		return fixed;
	}
	
	public void setFixed(boolean fixed) {
		this.fixed = fixed;
	}
	
	public int getIndex() {
		return tableIndex;
	}
	
	public void setIndex(int index) {
		this.tableIndex = index;
	}
	
	public Point getTableLocation() {
		return tableLocation;
	}
	
	public void setTableLocation(int x, int y) {
		tableLocation.setLocation(x,y);
	}
	
	private void makeReferenceHistogram(){
		if (histograms==null){
			histograms = new int[4][];
			
			partialBounds.setLocation(0,0);
			histograms[0] = makeHistogram(); // upper left
			
			partialBounds.setLocation(bounds.width-partialBounds.width,0);
			histograms[1] = makeHistogram();// upper right
			
			partialBounds.setLocation(bounds.width-partialBounds.width,bounds.height-partialBounds.height);
			histograms[2] = makeHistogram();	// lower right
			
			partialBounds.setLocation(0,bounds.height-partialBounds.height);
			histograms[3] = makeHistogram();	// lower left
		}
	}
	
	public int[][] getReferenceHistogram(int direction){
		makeReferenceHistogram();
		
		int[][] result = new int[2][];
		switch(direction){
			case 0:// top
				result[0] = histograms[0];
				result[1] = histograms[1];
				break;
			case 1:// left
				result[0] = histograms[0];
				result[1] = histograms[3];
				break;
			case 2:// bottom
				result[0] = histograms[3];
				result[1] = histograms[2];
				break;
			case 3:// right
				result[0] = histograms[1];
				result[1] = histograms[2];
				break;
		}
		return result;
	}
	
	private int[] makeHistogram(){
		if (rgb == null){
			Image img = getImage();
			int[] intRGB = getPixels(img);
			rgb = new int[intRGB.length][3];
			ColorVector.getRGB(intRGB,rgb);
		}
		
		int step = 64;
		int nColor = 256/step;
		int nColor2 = nColor*nColor;
		int[] histogram = new int[nColor*nColor2];
		int locationIndex;
		int colorIndex;
		for (int y=partialBounds.y; y<partialBounds.y+partialBounds.height; y++){
			for (int x=partialBounds.x; x<partialBounds.x+partialBounds.width; x++){
				locationIndex = x+y*bounds.width;
				colorIndex = nColor2*(rgb[locationIndex][0]/step)+nColor*(rgb[locationIndex][1]/step)+rgb[locationIndex][2]/step;
				histogram[colorIndex]++;
			}
		}
		
		return histogram;
	}
	
	public double collate(int[] referenceHistogram){
		int[] inputHistogram = makeHistogram();
		int sum = 0;
		for (int i=0; i<referenceHistogram.length; i++){
			sum += Math.min(referenceHistogram[i],inputHistogram[i]);
		}
		
		double result = 1.0*sum/(partialBounds.width*partialBounds.height);
		
		return result;
	}
	
	public double calcSimilarityUpperLimit(Rectangle rect, double similarity){
		Rectangle interRect = rect.intersection(partialBounds);
		int interRectSize = interRect.width*interRect.height;
		int partialBoundsSize = partialBounds.width*partialBounds.height;
		
		int diff = partialBoundsSize-interRectSize;
		diff = 0;
		return (Math.min(similarity*partialBoundsSize, interRectSize)+diff)/partialBoundsSize;
	}
	
	public boolean canSkip(Rectangle rect, double similarity, double maxCandidate){
		Rectangle interRect = rect.intersection(partialBounds);
		int interRectSize = interRect.width*interRect.height;
		int partialBoundsSize = partialBounds.width*partialBounds.height;
		
		return (partialBoundsSize-interRectSize<=partialBoundsSize*(maxCandidate-similarity))?true:false;
	}
	
	public Rectangle getPartialBounds(){
		return (Rectangle)partialBounds.clone();
	}
	
	public void setPartialLocation(int x, int y){
		partialBounds.setLocation(x,y);
	}
	
	public void setPartialSize(double ratioW, double ratioH){
		int refWidth = (int)(bounds.width*ratioW);
		int refHeight = (int)(bounds.height*ratioH);
		partialBounds.setSize(refWidth,refHeight);
	}
	
}
