/*
 * MatchImage.java
 *
 * Created on 2005/01/22, 14:00
 */

package orch.plugin;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.KeyEvent;
import java.awt.geom.AffineTransform;
import java.awt.image.VolatileImage;
import java.util.ArrayList;
import java.util.LinkedList;
import sos.util.MinMaxDouble;
import sos.util.MinMaxInteger;
import sos.util.SortDouble;
import orch.*;

/**
 *
 * @author Scientific Open Source projects (Gaku Tanaka)
 * @version 1.0 (2005/02/15)
 */
public abstract class Matching extends javax.swing.JFrame {
	static protected final int[] dX = new int[]{0,1,0,-1};
	static protected final int[] dY = new int[]{1,0,-1,0};
	static private Color transBlue = new Color(0,0,255,15);
	static private Color transOrange = new Color(255,0,128,64);
	
	protected volatile boolean pauseFlag;
	protected Rectangle searchBounds = new Rectangle();
	protected Point matchedPoint = new Point();
	protected double horizontalOverlapRatio;
	protected double verticalOverlapRatio;
	protected double minOverlapRatio;
		
	private AffineTransform scaleTransform = new AffineTransform();
	private VolatileImage fixedImages;
	
	private ImagePiece[][] ipArray;
	private int nRow;
	private int nColumn;
	private Point startCell;
	
	/** Creates new form MatchImage */
	public Matching() {

	}
	
	public void init(ImagePiece[][] array){
		initComponents();
		
		ipArray = array;
		nRow = ipArray.length;
		nColumn = ipArray[0].length;
		int count = 0;
		for (int y=0; y<nRow; y++){
			for (int x=0; x<nColumn; x++){
				if (ipArray[y][x]!=null){
					ipArray[y][x].setIndex(count);
					count++;
				}
			}
		}
		setupCanvas();
		
		pack();
		setVisible(true);
		
		fixedImages = createVolatileImage(jCanvasRects.getWidth(),jCanvasRects.getHeight());
		pauseFlag = false;		
	}
	
	private void setupCanvas(){
		int rowWidth = 0;
		int originalHeight = 0;
		MinMaxInteger mmiHeight = new MinMaxInteger();
		MinMaxInteger mmiWidth = new MinMaxInteger();
		mmiWidth.setInit(0);
		for (int y=0; y<nRow; y++){
			rowWidth = 0;
			mmiHeight.setInit(0);
			for (int x=0; x<nColumn; x++){
				if (ipArray[y][x]!=null){
					Dimension size = ipArray[y][x].getSize();
					mmiHeight.compareMax(size.height);
					if (x==0 || x==nColumn-1){
						rowWidth += (int)(size.width*1.5);
					}else{
						rowWidth += size.width;
					}
				}
			}
			if (y==0 || y==nRow-1){
				originalHeight += (int)(mmiHeight.getValue()*1.5);
			}else{
				originalHeight += mmiHeight.getValue();
			}
			mmiWidth.compareMax(rowWidth);
		}
		int originalWidth = mmiWidth.getValue();
		
		int canvasWidth = jCanvasRects.getWidth();
		int canvasHeight = (int)(1.0*originalHeight*canvasWidth/originalWidth);
		if (jCanvasRects.getHeight()<canvasHeight){
			canvasHeight = jCanvasRects.getHeight();
			canvasWidth = (int)(1.0*originalWidth*canvasHeight/originalHeight);
		}
		jCanvasRects.setSize(canvasWidth,canvasHeight);
		
		//calculate the mean of the cells which have image
		double meanX = (nColumn-1)/2.0;
		double meanY = (nRow-1)/2.0;
		MinMaxDouble mmd = new MinMaxDouble();
		mmd.setInit(Double.POSITIVE_INFINITY);
		for (int y=0; y<nRow; y++){
			for (int x=0; x<nColumn; x++){
				if (ipArray[y][x]!=null){
					double value = (meanX-x)*(meanX-x)+(meanY-y)*(meanY-y);
					mmd.compareMin(value, new Point(x,y));
				}
			}
		}
		startCell = (Point)mmd.getObject();
		
		int unitWidth = jCanvasRects.getWidth()/(nColumn+1); // +1 for the blank space
		int unitHeight = jCanvasRects.getHeight()/(nRow+1); // +1 for the blank space
		double startX = unitWidth*(startCell.x+0.5);
		double startY = unitHeight*(startCell.y+0.5);
		scaleTransform.setToTranslation(startX, startY);
		
		double ratioWidth = 1.0*jCanvasRects.getWidth()/originalWidth;
		double ratioHeight = 1.0*canvasHeight/originalHeight;
		scaleTransform.scale(ratioWidth,ratioHeight);
	}
	
	public void start(double horizontal, double vertical, double min){
		horizontalOverlapRatio = horizontal;
		verticalOverlapRatio = vertical;
		minOverlapRatio = min;
		
		LinkedList<Point> fifo = new LinkedList<Point>();
		fifo.add(startCell);
		ImagePiece ip1 = ipArray[startCell.y][startCell.x];
		ip1.setFixed(true);
		drawFixedImage(ip1);
		
		ImagePiece ip2 = null;
		ImagePiece ipSub = null;
		do{
			Point p = fifo.removeFirst();
			ip1 = ipArray[p.y][p.x];
			for (int i=0; i<dX.length; i++){
				int x2 = p.x+dX[i];
				int y2 = p.y+dY[i];
				if (!pauseFlag && 0<=x2 && x2<nColumn && 0<=y2 && y2<nRow){
					ip2 = ipArray[y2][x2];
					if (ip2!=null && !ip2.isFixed()){
						ipSub = getSubImagePiece(x2,y2,i);
						switch(i){
							case 0:
								setDirectionAndSearch(ip1,ip2,ipSub,false);
								break;
							case 1:
								setDirectionAndSearch(ip1,ip2,ipSub,true);
								break;
							case 2:
								setDirectionAndSearch(ip2,ip1,ipSub,false);
								break;
							case 3:
								setDirectionAndSearch(ip2,ip1,ipSub,true);
								break;
						}
						fifo.add(new Point(x2,y2));
					}
				}
			}
		}while(0<fifo.size());
	}
	
	private ImagePiece getSubImagePiece(int x, int y, int i){
		ImagePiece ip = null;
		
		int nx = x+dY[i];
		int ny = y+dX[i];
		if (0<=nx && nx<nColumn && 0<=ny && ny<nRow){
			ip = ipArray[ny][nx];
			if (ip!=null && ip.isFixed()){
				return ip;
			}
		}
		
		nx = x-dY[i];
		ny = y-dX[i];
		if (0<=nx && nx<nColumn && 0<=ny && ny<nRow){
			ip = ipArray[ny][nx];
			if (ip!=null && ip.isFixed()){
				return ip;
			}
		}
		
		return null;
	}
	
	private void setDirectionAndSearch(ImagePiece ip1, ImagePiece ip2, ImagePiece ipSub, boolean isHorizontal){
		if (ip1.isFixed()&ip2.isFixed()){
			return;
		}
		
		ImagePiece ipUnmove;
		ImagePiece ipMove;
		if (ip1.isFixed()){
			ipUnmove = ip1;
			ipMove = ip2;
		}else{
			ipUnmove = ip2;
			ipMove = ip1;
		}
		
		searchBounds = getSearchBounds(ipUnmove,ipMove,isHorizontal);
		matchedPoint = null;
		drawRects(ipUnmove,ipMove,ipSub,null,null);
		
		if (ipSub!=null){
			Rectangle anotherSearchBounds = getSearchBounds(ipSub,ipMove,!isHorizontal);
			searchBounds = searchBounds.intersection(anotherSearchBounds);
			
			matchThreeImage(ipUnmove,ipMove,ipSub,isHorizontal);
		}else{
			matchTwoImage(ipUnmove,ipMove,isHorizontal);
		}
		ipMove.getBounds().setLocation(matchedPoint);
		ipMove.setFixed(true);
		drawFixedImage(ipMove);
	}
	
	abstract protected Rectangle getSearchBounds(ImagePiece ipUnmove, ImagePiece ipMove, boolean isHorizontal);
	
	abstract protected void matchThreeImage(ImagePiece ipUnmove, ImagePiece ipMove, ImagePiece ipSub, boolean isHorizontal);
	
	abstract protected void matchTwoImage(ImagePiece ipUnmove, ImagePiece ipMove, boolean isHorizontal);
	
	private void updateFixedImage(ImagePiece ip){
		Graphics2D g = (Graphics2D)fixedImages.getGraphics();
		AffineTransform positionTransform = new AffineTransform(scaleTransform);
		Rectangle rect = ip.getBounds();
		positionTransform.translate(rect.x,rect.y);
		g.drawImage(ip.getImage(),positionTransform,jCanvasRects);
	}
	
	protected void drawRects(ImagePiece ipUnmove, ImagePiece ipMove, ImagePiece ipSub, SearchPoint sp1, SearchPoint sp2){
		//jCanvasRects.clearCanvas();
		Graphics2D g = (Graphics2D)jCanvasRects.getOffGraphics();
		g.drawImage(fixedImages,0,0,jCanvasRects);
		g.setTransform(scaleTransform);
		
		g.setColor(Color.yellow);
		g.draw(ipUnmove.getBounds());
		if (ipSub!=null){
			g.draw(ipSub.getBounds());
		}
		
		Rectangle rectMove = ipMove.getBounds();
		Rectangle moveBounds = new Rectangle(searchBounds);
		moveBounds.width += rectMove.width;
		moveBounds.height += rectMove.height;
		g.setColor(transBlue);
		g.fill(moveBounds);
		
		if (matchedPoint!=null){
			g.setColor(Color.orange);
			g.drawRect(matchedPoint.x,matchedPoint.y,rectMove.width,rectMove.height);
		}
		
		if (sp1!=null){
			g.setColor(transOrange);
			g.fill(sp1.getIntersectRect());
		}
		if (sp2!=null){
			g.setColor(transOrange);
			g.fill(sp2.getIntersectRect());
		}
		
		jCanvasRects.paintImmediately(0,0,jCanvasRects.getWidth(),jCanvasRects.getHeight());
	}
	
	private void drawFixedImage(ImagePiece ip){
		updateFixedImage(ip);
		Graphics2D g = (Graphics2D)jCanvasRects.getOffGraphics();
		g.drawImage(fixedImages,0,0,jCanvasRects);
		jCanvasRects.paintImmediately(0,0,jCanvasRects.getWidth(),jCanvasRects.getHeight());
	}
	
	/** This method is called from within the constructor to
	 * initialize the form.
	 * WARNING: Do NOT modify this code. The content of this method is
	 * always regenerated by the Form Editor.
	 */
    private void initComponents() {//GEN-BEGIN:initComponents
        jCanvasRects = new sos.awt.JCanvas();

        setDefaultCloseOperation(javax.swing.WindowConstants.EXIT_ON_CLOSE);
        setTitle("Matching");
        addKeyListener(new java.awt.event.KeyAdapter() {
            public void keyPressed(java.awt.event.KeyEvent evt) {
                formKeyPressed(evt);
            }
        });

        jCanvasRects.setPreferredSize(new java.awt.Dimension(640, 640));
        getContentPane().add(jCanvasRects, java.awt.BorderLayout.CENTER);

        pack();
    }//GEN-END:initComponents
	
	private void formKeyPressed(java.awt.event.KeyEvent evt) {//GEN-FIRST:event_formKeyPressed
		if (evt.getKeyCode()==KeyEvent.VK_ESCAPE){
			pauseFlag = true;
		}
	}//GEN-LAST:event_formKeyPressed
	
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private sos.awt.JCanvas jCanvasRects;
    // End of variables declaration//GEN-END:variables
	
}
