/*
 * GunmetryViewer.java
 *
 * Created on 2005/02/19, 13:24
 */

package gunmetry;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.KeyEventPostProcessor;
import java.awt.KeyboardFocusManager;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.Toolkit;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import javax.imageio.ImageIO;
import javax.swing.ImageIcon;
import javax.swing.JPanel;
import javax.swing.SpinnerNumberModel;
import javax.swing.table.DefaultTableColumnModel;
import javax.swing.table.DefaultTableModel;
import sos.io.Extension;
import sos.io.ExtensionFilter;
import sos.io.PluginClassLoader;
import sos.io.SOSFileIO;
import sos.io.SOSImageIO;
import sos.process.ImageInfo;
import sos.process.ImagePixelMediator;
import sos.process.Morphology;
import sos.process.NeighborFrame;
import sos.process.ROI;
import sos.process.Watershed;
import sos.util.ColorVector;
import sos.util.FirstInFirstOut;
import sos.util.MinMaxDouble;
import sos.util.PointRect;

/**
 * Created on 2005/02/19
 * @author Scientific Open Source projects (Gaku Tanaka)
 * @version 1.0 (2005/05/03)
 */
public class GunmetryViewer extends javax.swing.JFrame {
	
	static final private String	sep = System.getProperty("file.separator");
	static final private String pathExtractMarker = "extractMarker";
	static final private String pathAnalyze = "analyze";
	static final private ExtensionFilter filterImg;
	static final private ExtensionFilter filterGifPng;
	
	static private ImageIcon iconObject;
	static private ImageIcon iconBackground;
	
	static private Cursor cursorObject;
	static private Cursor cursorBackground;
	
	static private int BACKGROUND = 0;
	
	private File fileOriginal = null;
	private File fileCluster = null;
	private BufferedImage imgBackground;
	private Image imgResult;
	private BufferedImage imgOverlap;
	
	private ArrayList<Point> markersObject = new ArrayList<Point>();
	private ArrayList<Point> markersBackground = new ArrayList<Point>();
	private boolean[] analyzeMask;
	private ImageInfo imiAnalyze = new ImageInfo();
	private double[] gradient;
	private int[] contourRegion;
	private int[] contourSkiz;
	private ROI[] nuclei;
	private ROI[] skiz;
	
	private Point focusedMarker = new Point(0,0);
	private Point dragStartPoint = new Point();
	private Rectangle selectionRect;
	private HashSet<Point> selectedMarkers = new HashSet<Point>();
	private HashSet<Rectangle> dirtyRegion = new HashSet<Rectangle>();
	private PointRect pointRect = new PointRect(0,0,5);
	
	private int[] stepArray;
	private int[] orderArray;
	private int currentOrder;
	
	private SOSImageIO imageFileIO = new SOSImageIO();
	private SOSFileIO textFileIO = new SOSFileIO();
	
	private int preWidth;
	private int preHeight;
	
	private BufferedImage imgCursor = new BufferedImage(32,32,BufferedImage.TYPE_INT_ARGB);
	private boolean isObjectActive = true;
	
	private ExtractMarkerPlugin pluginMarkerObject;
	private ExtractMarkerPlugin pluginMarkerBackground;
	
	static{
		String[] exts = ImageIO.getReaderFormatNames();
		String description = "Image Files (ImageI/O readable)";
		filterImg = new ExtensionFilter(exts, description);
		filterGifPng = new ExtensionFilter(new String[]{"gif","png"}, "GIF or PNG");
	}
	
	/** Creates new form ClimberClimberViewer */
	public GunmetryViewer() {
		iconObject = new ImageIcon(getClass().getResource("/opBtn1.gif"));
		iconBackground = new ImageIcon(getClass().getResource("/opBtn2.gif"));
		
		setupKeyInput();
		
		initComponents();
		
		jCanvasColorMask.setColor(Color.black);
		jCanvasColorObj.setColor(Color.red);
		jCanvasColorBack.setColor(Color.yellow);
		jCanvasColorWatershed.setColor(Color.black);
		jCanvasColorSkiz.setColor(Color.black);
		jCanvasColorObjCent.setColor(Color.cyan);
		jCanvasColorSkizCent.setColor(Color.blue);
		jCanvasColorNearestNeighbor.setColor(Color.black);
		
		cursorObject = createCursor(jCanvasColorObj.getColor());
		cursorBackground = createCursor(jCanvasColorBack.getColor());
	
		SpinnerNumberModel modelMarkerSize = new SpinnerNumberModel(5,1,1000,1);
		jSpinnerMarkerSize.setModel(modelMarkerSize);		
		
		pack();
		
		// setting up for jDialogSelectColor
		loadExtractMarkerPlugin();
		jDialogMarker.pack();
		
		jDialogMask.pack();

		loadAnalyzePlugin();
		jDialogAnalyze.pack();
		DefaultTableColumnModel columnModel =(DefaultTableColumnModel)jTableAnalyze.getColumnModel();
		columnModel.getColumn(1).setPreferredWidth(50);
		
		setVisible(true);
	}
	
	private Cursor createCursor(Color color){
		int size = imgCursor.getWidth();
		int hSize = size/2;
		int fqSize = size/4;
		Graphics g = imgCursor.getGraphics();
		Color colorClear = new Color(0,0,0,0);
		g.setColor(colorClear);
		g.fillRect(0,0,size,size);
		
		g.setColor(Color.black);
		g.fillRect(fqSize,hSize-1, hSize+1,3);
		g.fillRect(hSize-1,fqSize, 3,hSize+1);
		
		g.setColor(color);
		g.drawLine(fqSize,hSize, size-fqSize,hSize);
		g.drawLine(hSize,fqSize, hSize,size-fqSize);
		
		return Toolkit.getDefaultToolkit().createCustomCursor(imgCursor, new Point(hSize,hSize),"object");
	}
	
	private void setupKeyInput(){
		// setup for key inputs
		KeyboardFocusManager keyboardManager = KeyboardFocusManager.getCurrentKeyboardFocusManager();
		keyboardManager.addKeyEventPostProcessor(new KeyEventPostProcessor(){
			public boolean postProcessKeyEvent(KeyEvent evt){
				switch(evt.getID()){
					case KeyEvent.KEY_PRESSED:
						int keyCode = evt.getKeyCode();
						if (keyCode==KeyEvent.VK_DELETE){
							removeSelectedMarkers();
						}else if (keyCode==KeyEvent.VK_X){
							switchEditableMarkerType();
						}else if (keyCode==KeyEvent.VK_TAB){
							changeOperationButton();
						}else if (KeyEvent.VK_LEFT<=keyCode && keyCode<=KeyEvent.VK_DOWN){
							moveMarker(keyCode);
						}
						break;
					case KeyEvent.KEY_RELEASED:
						break;
					case KeyEvent.KEY_TYPED:
						break;
				}
				return false;
			}
		});
	}
	
	private void moveMarker(int keyCode){
		if (0<selectedMarkers.size()){
			int dx = 0;
			int dy = 0;
			switch (keyCode){
				case KeyEvent.VK_LEFT:
					dx = -1;
					break;
				case KeyEvent.VK_UP:
					dy = -1;
					break;
				case KeyEvent.VK_RIGHT:
					dx = 1;
					break;
				case KeyEvent.VK_DOWN:
					dy = 1;
					break;
			}
			
			for (Point p: selectedMarkers){
				p.translate(dx,dy);
			}
			drawMarkers();
			updateFullCanvas();
		}
	}
	
	private void switchEditableMarkerType(){
		isObjectActive = !isObjectActive;
		if (isObjectActive){
			jToggleMarker.setIcon(iconObject);
			if (jToggleMarker.isSelected()){
				jImageMain.setCursor(cursorObject);
			}
		}else{
			jToggleMarker.setIcon(iconBackground);
			if (jToggleMarker.isSelected()){
				jImageMain.setCursor(cursorBackground);
			}
		}
	}
	
	private void changeOperationButton(){
		if (jToggleMarker.isSelected()){
			jToggleImage.setSelected(true);
		}else{
			jToggleMarker.setSelected(true);
		}
	}
	
	private void loadExtractMarkerPlugin(){
		File file = new File(pathExtractMarker);
		File[] fileList = file.listFiles();
		for (int i=0; i<fileList.length; i++){
			String algorithmName = Extension.remove(fileList[i].getName());
			jComboObjAlgorithm.addItem(algorithmName);
			jComboBackAlgorithm.addItem(algorithmName);
		}
		
		jComboObjAlgorithm.setSelectedItem("HDoom");
		jComboBackAlgorithm.setSelectedItem("HDoom");
	}
	
	private ExtractMarkerPlugin setMarkerOption(JPanel panel, String algorithmName){
		PluginClassLoader loader = new PluginClassLoader(pathExtractMarker);
		ExtractMarkerPlugin plugin = null;
		try{
			plugin = (ExtractMarkerPlugin)loader.loadClass(algorithmName).newInstance();
			JPanel optionPanel = plugin.getOptionPanel();
			if (2==panel.getComponentCount()){
				panel.remove(1);
			}
			if (optionPanel!=null){
				panel.add(optionPanel, BorderLayout.CENTER);
			}
			jDialogMarker.pack();
		}catch(Exception e){
			e.printStackTrace();
		}
		return plugin;
	}
	
	private void loadAnalyzePlugin(){
		File file = new File(pathAnalyze);
		File[] fileList = file.listFiles();
		DefaultTableModel model = (DefaultTableModel)jTableAnalyze.getModel();
		model.setRowCount(fileList.length);
		for (int i=0; i<fileList.length; i++){
			String algorithmName = Extension.remove(fileList[i].getName());
			model.setValueAt(algorithmName, i, 0);
			model.setValueAt(false,i,1);
		}
	}
	
	private void updateResultImage(){
		if (imgResult!=null){
			double value = jSliderAlpha.getValue()/100.0;
			imgResult = ImagePixelMediator.setAlpha(imgResult, value);
			overlapImage();
			drawMarkers();
			updateFullCanvas();
		}
	}
	
	private boolean checkBackgroundImageSize(Dimension size){
		if (imgBackground==null || (size.width!=imgBackground.getWidth() || size.height!=imgBackground.getHeight())){
			return false;
		}
		
		return true;
	}
	
	private void drawImageToImgBackground(Image img){
		if (img==null){
			return;
		}
		
		Graphics g = imgBackground.getGraphics();
		g.drawImage(img,0,0,null);
		overlapImage();
		jImageMain.setFullSrc();
		drawMarkers();
		updateFullCanvas();
	}
	
	private void overlapImage(){
		BufferedImage img = imgBackground;
		
		if(imgResult!=null){
			Graphics2D g = (Graphics2D)imgOverlap.getGraphics();
			g.drawImage(imgBackground,0,0,this);
			g.drawImage(imgResult,0,0,this);
			
			int x, y;
			// draw contour of object
			if (jToggleWatershed.isSelected()){
				g.setColor(jCanvasColorWatershed.getColor());
				drawContour(g,contourRegion);
			}
			// draw contour of SKIZ
			if (jToggleSkiz.isSelected()){
				g.setColor(jCanvasColorSkiz.getColor());
				drawContour(g,contourSkiz);
			}
			// draw center of object
			if (jToggleObjCent.isSelected()){
				g.setColor(jCanvasColorObjCent.getColor());
				drawCentroid(g,nuclei);
			}
			// draw center of skiz
			if (jToggleSkizCent.isSelected()){
				g.setColor(jCanvasColorSkizCent.getColor());
				drawCentroid(g,skiz);
			}
			
			// for test
			if (jToggleNearestNeighbor.isSelected()){
				g.setColor(jCanvasColorNearestNeighbor.getColor());
				for(int i=0; i<nuclei.length; i++){
					Point2D.Double p1 = nuclei[i].getCentroid();
					Point2D.Double p2 = nuclei[i].getNearestNeighbor().getCentroid();
					g.drawLine((int)p1.x,(int)p1.y,(int)p2.x,(int)p2.y);
				}
			}
			img = imgOverlap;
		}
		
		if (jCheckMask.isSelected()){
			Graphics g = img.getGraphics();
			g.setColor(jCanvasColorMask.getColor());
			int x,y;
			for (int i=0; i<analyzeMask.length; i++){
				if (analyzeMask[i]){
					x = i%img.getWidth();
					y = i/img.getWidth();
					g.drawLine(x,y, x, y);
				}
			}
		}
		jImageMain.setImage(img);
		
	}
	
	private void drawContour(Graphics g, int[] contourPixels){
		int x,y;
		for (int i=0; i<contourPixels.length; i++){
			x = contourPixels[i]%imgBackground.getWidth();
			y = contourPixels[i]/imgBackground.getWidth();
			g.drawLine(x,y, x,y);
		}
	}
	
	private void drawCentroid(Graphics2D g, ROI[] roiSet){
		for (ROI roi: roiSet){
			Point2D.Double p = roi.getCentroid();
			Rectangle2D.Double r = new Rectangle2D.Double(p.x,p.y,1,1);
			g.draw(r);
		}
	}
	
	private void drawMarkers(){
		jImageMain.drawImage();
		
		// draw unselected markers
		Graphics2D g = (Graphics2D)jImageMain.getOffGraphics();
		g.setTransform(jImageMain.getAffineTransform());
		if (jToggleObject.isSelected() && 0<markersObject.size()){
			g.setColor(jCanvasColorObj.getColor());
			for (Point p: markersObject){
				pointRect.setLocation(p.x,p.y);
				g.draw(pointRect.getBounds());
			}
		}
		if (jToggleBackground.isSelected() && 0<markersBackground.size()){
			g.setColor(jCanvasColorBack.getColor());
			for (Point p: markersBackground){
				pointRect.setLocation(p.x,p.y);
				g.draw(pointRect.getBounds());
			}
		}
		
		// draw selection
		g.setColor(Color.white);
		PointRect pointRect = new PointRect(0,0,(Integer)jSpinnerMarkerSize.getValue());
		for (Point p : selectedMarkers){
			pointRect.setLocation(p.x, p.y);
			g.fill(pointRect.getBounds());
		}
		
		// draw selection rectangle
		if (selectionRect!=null){
			g.draw(selectionRect);
		}
		
	}
	
	private void updateFullCanvas(){
		jImageMain.repaint();
		dirtyRegion.clear();
	}
	
	private void updatePartialCanvas(Rectangle rect){
		dirtyRegion.add(rect);
		repaintDirtyRegion();
		dirtyRegion.add((Rectangle)rect.clone());
	}
	
	private void repaintDirtyRegion(){
		int extra = (int)jImageMain.getMagnification();
		AffineTransform it =jImageMain.getAffineTransform();
		for (Rectangle r: dirtyRegion){
			Shape shape = it.createTransformedShape(r);
			if (shape!=null){
				Rectangle transR = shape.getBounds();
				transR.grow(extra,extra);
				jImageMain.paintImmediately(transR);
			}
		}
		dirtyRegion.clear();
	}
	
	private ArrayList<Point> extractMarker(ExtractMarkerPlugin plugin, boolean[] binaryMap, String algorithmName){
		NeighborFrame nf = new NeighborFrame(imgBackground.getWidth(),imgBackground.getHeight());
		Morphology mo = new Morphology(nf);
		
		return plugin.extractMarker(mo,binaryMap);
	}
	
	private Point contains(ArrayList<Point> markers, Point2D point){
		for (Point p: markers){
			pointRect.setLocation(p.x,p.y);
			if (pointRect.contains(point)){
				return p;
			}
		}
		return null;
	}
	
	private void findFocusedMarker(Point mousePoint){
		Point preFocus = focusedMarker;
		
		Point2D mousePoint2D = jImageMain.getImageTransform().inverseTransform(mousePoint);
		focusedMarker = contains(markersObject, mousePoint2D);
		if (focusedMarker==null){
			focusedMarker = contains(markersBackground, mousePoint2D);
		}
		
		if (preFocus!=focusedMarker){
			if (focusedMarker!=null){
				jImageMain.setCursor(new Cursor(Cursor.MOVE_CURSOR));
				pointRect.setLocation(focusedMarker.x, focusedMarker.y);
				dirtyRegion.add(pointRect.getRectangle());
			}else{
				if (isObjectActive){
					jImageMain.setCursor(cursorObject);
				}else{
					jImageMain.setCursor(cursorBackground);
				}
			}
		}
	}
	
	private void addToSelectedMarkers(ArrayList<Point> markers){
		for (Point p: markers){
			if (selectionRect.contains(p)){
				selectedMarkers.add(p);
			}
		}
	}
	
	private void removeSelectedMarkers(){
		if (0==selectedMarkers.size()){
			return;
		}
		
		readyToDrawSelectMarkers();
		markersObject.removeAll(selectedMarkers);
		markersBackground.removeAll(selectedMarkers);
		
		selectedMarkers.clear();
		drawMarkers();
		repaintDirtyRegion();
		updateMarkerNumber();
		checkReadyToAnalyze();
	}
	
	private void checkSelectionType(int modifier){
		if ((modifier&InputEvent.SHIFT_MASK)==0){
			readyToDrawSelectMarkers();
			selectedMarkers.clear();
		}
	}
	
	private void readyToDrawSelectMarkers(){
		for (Point p: selectedMarkers){
			pointRect.setLocation(p.x, p.y);
			dirtyRegion.add(pointRect.getRectangle());
		}
	}
	
	private ROI[] makeROI(){
		ROI[] result = new ROI[markersObject.size()];
		for (int n=0; n<result.length; n++){
			result[n] = new ROI();
		}
		int count = 0;
		for (Point p:markersObject){
			result[count].expandRegion(p.x, p.y);
			count++;
		}
		
		// make the region with adding points
		int offset,pn, index;
		int FIRST_LABEL = BACKGROUND+1;
		for (int y=0; y<imgBackground.getHeight(); y++){
			offset = y*imgBackground.getWidth();
			for (int x=0; x<imgBackground.getWidth(); x++){
				pn = x+offset;
				index = imiAnalyze.getPixelsLabel(pn)-FIRST_LABEL;
				if (0<=index){
					result[index].expandRegion(x,y);
				}
			}
		}
		
		// set mask within the region
		for (int n=0; n<result.length; n++){
			result[n].initMask();
			Rectangle bounds = result[n].getBounds();
			for (int y=0; y<bounds.height; y++){
				offset = (y+bounds.y)*imgBackground.getWidth();
				for (int x=0; x<bounds.width; x++){
					pn = x+bounds.x+offset;
					if (imiAnalyze.getPixelsLabel(pn)-FIRST_LABEL==n){
						result[n].setMask(x,y);
					}
				}
			}
		}
		
		return result;
	}
	
	private boolean[] getBinaryMap(boolean[] flag, int[] cluster){
		boolean[] result = new boolean[cluster.length];
		for (int i=0; i<cluster.length; i++){
			result[i] = flag[cluster[i]];
		}
		
		return result;
	}
	
	private void setMarkers(boolean[] flag, int[] colorCluster){
		boolean[] binaryMap = getBinaryMap(flag, colorCluster);
		markersObject = extractMarker(pluginMarkerObject, binaryMap, (String)jComboObjAlgorithm.getSelectedItem());
		for (int i=0; i<binaryMap.length; i++){
			binaryMap[i] = !binaryMap[i];
		}
		markersBackground = extractMarker(pluginMarkerBackground, binaryMap, (String)jComboBackAlgorithm.getSelectedItem());
		
		drawMarkers();
		updateFullCanvas();
		updateMarkerNumber();
		checkReadyToAnalyze();
	}
	
	private void checkReadyToAnalyze(){
		boolean isReady = false;
		if (0<markersObject.size() && 0<markersBackground.size()){
			isReady = true;
		}
		
		jMenuSaveMarkers.setEnabled(isReady);
		jCheckMask.setEnabled(isReady);
		jButtonAnalyze.setEnabled(isReady);
		if (isReady){
			jLabelInfo.setText("Seed setting is done.");
		}else{
			jLabelInfo.setText("Inproper seed setting");
		}
	}
	
	private void removeMaskedMarkers(ArrayList<Point> markers){
		HashSet<Point> removeSet = new HashSet<Point>();
		for (Point p: markers){
			if (analyzeMask[pointToOffset(p)]){
				removeSet.add(p);
			}
		}
		markers.removeAll(removeSet);
	}
	
	private int pointToOffset(Point p){
		return p.x+p.y*imiAnalyze.getWidth();
	}
	
	private void convertMaskToWatershed(int indexWatershed){
		if (jCheckMask.isSelected()){
			for (int pn=0; pn<imiAnalyze.getImageSize(); pn++){
				if (analyzeMask[pn]){
					imiAnalyze.setPixelsLabel(pn, indexWatershed);
				}
			}
		}
	}
	
	private void makeResultImage(){
		// make result image
		int[] result = new int[imiAnalyze.getImageSize()];
		int label;
		for (int i=0; i<result.length; i++){
			label = imiAnalyze.getPixelsLabel(i);
			if (label>BACKGROUND){
				result[i] = jCanvasColorObj.getColorInt();
			}else{
				if (label==BACKGROUND){
					result[i] = jCanvasColorBack.getColorInt();
				}else{
					result[i] = jCanvasColorMask.getColorInt();
				}
			}
		}
		imgResult = ImagePixelMediator.convertToImage(result, imiAnalyze.getWidth(),imiAnalyze.getHeight());
	}
	
	private int[] deleteMaskedContour(int[] contour){
		ArrayList<Integer> notMaskedContour = new ArrayList<Integer>();
		for (int i=0; i<contour.length; i++){
			if (!analyzeMask[contour[i]]){
				notMaskedContour.add(contour[i]);
			}
		}
		int[] result = new int[notMaskedContour.size()];
		for (int i=0; i<notMaskedContour.size(); i++){
			result[i] = notMaskedContour.get(i);
		}
		return result;
	}
	
	private void initializeAnalysis(){
		// properties
		imgResult = null;
		imgOverlap = null;
		fileCluster = null;
		markersObject.clear();
		markersBackground.clear();
		
		// controls
		jMenuFile.setEnabled(false);
		jMenuSaveMarkers.setEnabled(false);
		jMenuResult.setEnabled(false);
		
		jCheckMask.setEnabled(false);
		
		jRadioCluster.setEnabled(false);
		jButtonSetCluster.setEnabled(false);
		
		jButtonAnalyze.setEnabled(false);
	}
	
	private void setNearestNeighbor(int[] distanceMap){
		NeighborFrame nf = imiAnalyze.getNeighborFrame();
		nf.set8();
		for (int i=0; i<skiz.length; i++){
			skiz[i].setMinDistance(Integer.MAX_VALUE);
		}
		for (int i=0; i<contourSkiz.length; i++){
			int pn = contourSkiz[i];
			int[] neighbor = nf.getDir(pn);
			for (int n=0; n<neighbor.length; n++){
				if (neighbor[n]!=0){
					int label = imiAnalyze.getPixelsLabel(pn+neighbor[n])-BACKGROUND-1;
					if (0<=label && distanceMap[pn+neighbor[n]]<skiz[label].getMinDistance()){
						int contraNeighbor = pn+neighbor[(n+4)%neighbor.length];
						int contraLabel = imiAnalyze.getPixelsLabel(contraNeighbor)-BACKGROUND-1;
						if (pn!=contraNeighbor && 0<=contraLabel){
							skiz[label].setMinDistance(distanceMap[pn+neighbor[n]]);
							skiz[label].setNearestNeighbor(skiz[contraLabel]);
							nuclei[label].setNearestNeighbor(nuclei[contraLabel]);
						}
					}
				}
			}
		}
	}
	
	private void updateMarkerNumber(){
		jLabelNObject.setText(Integer.toString(markersObject.size()));
		jLabelNBackground.setText(Integer.toString(markersBackground.size()));
	}
	
	private Image createImageFromGradient(){
		MinMaxDouble mmd = new MinMaxDouble();
		mmd.compareMax(gradient);
		double maxGradient = mmd.getValue();
		int[] pixels = new int[gradient.length];
		int value;
		int alpha = 255<<24;
		for (int i=0; i<gradient.length; i++){
			value = (int)(255*gradient[i]/maxGradient);
			pixels[i] = alpha+(value<<16)+(value<<8)+value;
		}
		
		return ImagePixelMediator.convertToImage(pixels,imgBackground.getWidth(),imgBackground.getHeight());
	}
	
	/** This method is called from within the constructor to
	 * initialize the form.
	 * WARNING: Do NOT modify this code. The content of this method is
	 * always regenerated by the Form Editor.
	 */
    private void initComponents() {//GEN-BEGIN:initComponents
        java.awt.GridBagConstraints gridBagConstraints;

        buttonGroupImage = new javax.swing.ButtonGroup();
        buttonGroupOperation = new javax.swing.ButtonGroup();
        jDialogMarker = new javax.swing.JDialog();
        jPanelButtons = new javax.swing.JPanel();
        jButtonSetMarker = new javax.swing.JButton();
        jButtonCancelMarker = new javax.swing.JButton();
        jPanelObject = new javax.swing.JPanel();
        jPanel1 = new javax.swing.JPanel();
        jLabel4 = new javax.swing.JLabel();
        jComboObjAlgorithm = new javax.swing.JComboBox();
        jPanelBackground = new javax.swing.JPanel();
        jPanel2 = new javax.swing.JPanel();
        jLabel5 = new javax.swing.JLabel();
        jComboBackAlgorithm = new javax.swing.JComboBox();
        jScrollPaneMarker = new javax.swing.JScrollPane();
        imageColorPickerMarker = new gunmetry.ImageColorPicker();
        jDialogMask = new javax.swing.JDialog();
        jPanelButtons1 = new javax.swing.JPanel();
        jButtonSetMask = new javax.swing.JButton();
        jButtonCancelMask = new javax.swing.JButton();
        jScrollPaneMask = new javax.swing.JScrollPane();
        imageColorPickerMask = new gunmetry.ImageColorPicker();
        jDialogAnimation = new javax.swing.JDialog();
        jImageAnimation = new sos.awt.JImageCanvas();
        jPanel7 = new javax.swing.JPanel();
        jSliderFrame = new javax.swing.JSlider();
        jPanel8 = new javax.swing.JPanel();
        jLabel6 = new javax.swing.JLabel();
        jLabelFrame = new javax.swing.JLabel();
        jButtonSaveAnimation = new javax.swing.JButton();
        jDialogAnalyze = new javax.swing.JDialog();
        jScrollPane1 = new javax.swing.JScrollPane();
        jTableAnalyze = new javax.swing.JTable();
        jButtonOKResult = new javax.swing.JButton();
        jButtonCancelResult = new javax.swing.JButton();
        jPopupMenuDelete = new javax.swing.JPopupMenu();
        jMenuDeleteSelection = new javax.swing.JMenuItem();
        jImageMain = new sos.awt.JImageCanvas();
        jLabelInfo = new javax.swing.JLabel();
        jPanelMain = new javax.swing.JPanel();
        jPanelImage = new javax.swing.JPanel();
        jRadioCluster = new javax.swing.JRadioButton();
        jRadioOriginal = new javax.swing.JRadioButton();
        jButtonSetOriginal = new javax.swing.JButton();
        jButtonSetCluster = new javax.swing.JButton();
        jCheckMask = new javax.swing.JCheckBox();
        jCanvasColorMask = new sos.gui.JCanvasColor();
        jRadioGradient = new javax.swing.JRadioButton();
        jPanelMarkers = new javax.swing.JPanel();
        jButtonAssignColorsToMarkers = new javax.swing.JButton();
        jToggleObject = new javax.swing.JToggleButton();
        jToggleBackground = new javax.swing.JToggleButton();
        jLabel1 = new javax.swing.JLabel();
        jSpinnerMarkerSize = new javax.swing.JSpinner();
        jCanvasColorObj = new sos.gui.JCanvasColor();
        jCanvasColorBack = new sos.gui.JCanvasColor();
        jLabel7 = new javax.swing.JLabel();
        jLabelNObject = new javax.swing.JLabel();
        jLabel9 = new javax.swing.JLabel();
        jLabelNBackground = new javax.swing.JLabel();
        jPanelOeration = new javax.swing.JPanel();
        jButtonAnalyze = new javax.swing.JButton();
        jToggleMarker = new javax.swing.JToggleButton();
        jToggleImage = new javax.swing.JToggleButton();
        jButtonSwitchMarker = new javax.swing.JButton();
        jPanelResult = new javax.swing.JPanel();
        jLabel3 = new javax.swing.JLabel();
        jSliderAlpha = new javax.swing.JSlider();
        jToggleWatershed = new javax.swing.JToggleButton();
        jToggleSkiz = new javax.swing.JToggleButton();
        jToggleObjCent = new javax.swing.JToggleButton();
        jToggleSkizCent = new javax.swing.JToggleButton();
        jToggleNearestNeighbor = new javax.swing.JToggleButton();
        jCanvasColorWatershed = new sos.gui.JCanvasColor();
        jCanvasColorSkiz = new sos.gui.JCanvasColor();
        jCanvasColorObjCent = new sos.gui.JCanvasColor();
        jCanvasColorSkizCent = new sos.gui.JCanvasColor();
        jCanvasColorNearestNeighbor = new sos.gui.JCanvasColor();
        jButtonAnimation = new javax.swing.JButton();
        jMenuBarMain = new javax.swing.JMenuBar();
        jMenuFile = new javax.swing.JMenu();
        jMenuOpenMarkers = new javax.swing.JMenuItem();
        jMenuSaveMarkers = new javax.swing.JMenuItem();
        jMenuResult = new javax.swing.JMenu();
        jMenuSaveData = new javax.swing.JMenuItem();
        jMenuSaveImage = new javax.swing.JMenuItem();

        jDialogMarker.getContentPane().setLayout(new java.awt.GridBagLayout());

        jDialogMarker.setTitle("Set Maker");
        jDialogMarker.setModal(true);
        jButtonSetMarker.setText("SET");
        jButtonSetMarker.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonSetMarkerActionPerformed(evt);
            }
        });

        jPanelButtons.add(jButtonSetMarker);

        jButtonCancelMarker.setText("CANCEL");
        jButtonCancelMarker.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonCancelMarkerActionPerformed(evt);
            }
        });

        jPanelButtons.add(jButtonCancelMarker);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        jDialogMarker.getContentPane().add(jPanelButtons, gridBagConstraints);

        jPanelObject.setLayout(new java.awt.BorderLayout());

        jPanelObject.setBorder(new javax.swing.border.TitledBorder("Object"));
        jLabel4.setText("algorithm: ");
        jPanel1.add(jLabel4);

        jComboObjAlgorithm.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                jComboObjAlgorithmItemStateChanged(evt);
            }
        });

        jPanel1.add(jComboObjAlgorithm);

        jPanelObject.add(jPanel1, java.awt.BorderLayout.NORTH);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        jDialogMarker.getContentPane().add(jPanelObject, gridBagConstraints);

        jPanelBackground.setLayout(new java.awt.BorderLayout());

        jPanelBackground.setBorder(new javax.swing.border.TitledBorder("Background"));
        jLabel5.setText("algorithm: ");
        jPanel2.add(jLabel5);

        jComboBackAlgorithm.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                jComboBackAlgorithmItemStateChanged(evt);
            }
        });

        jPanel2.add(jComboBackAlgorithm);

        jPanelBackground.add(jPanel2, java.awt.BorderLayout.NORTH);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        jDialogMarker.getContentPane().add(jPanelBackground, gridBagConstraints);

        jScrollPaneMarker.setBorder(new javax.swing.border.TitledBorder("Select colors as OBJECT"));
        jScrollPaneMarker.setViewportView(imageColorPickerMarker);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        jDialogMarker.getContentPane().add(jScrollPaneMarker, gridBagConstraints);

        jDialogMask.getContentPane().setLayout(new java.awt.GridBagLayout());

        jDialogMask.setTitle("Set Mask");
        jDialogMask.setModal(true);
        jButtonSetMask.setText("SET");
        jButtonSetMask.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonSetMaskActionPerformed(evt);
            }
        });

        jPanelButtons1.add(jButtonSetMask);

        jButtonCancelMask.setText("CANCEL");
        jButtonCancelMask.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonCancelMaskActionPerformed(evt);
            }
        });

        jPanelButtons1.add(jButtonCancelMask);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        jDialogMask.getContentPane().add(jPanelButtons1, gridBagConstraints);

        jScrollPaneMask.setBorder(new javax.swing.border.TitledBorder("Select color as MASK"));
        jScrollPaneMask.setViewportView(imageColorPickerMask);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        jDialogMask.getContentPane().add(jScrollPaneMask, gridBagConstraints);

        jDialogAnimation.setTitle("Gunmetry Animation");
        jDialogAnimation.getContentPane().add(jImageAnimation, java.awt.BorderLayout.CENTER);

        jPanel7.setLayout(new java.awt.BorderLayout());

        jSliderFrame.addChangeListener(new javax.swing.event.ChangeListener() {
            public void stateChanged(javax.swing.event.ChangeEvent evt) {
                jSliderFrameStateChanged(evt);
            }
        });

        jPanel7.add(jSliderFrame, java.awt.BorderLayout.CENTER);

        jLabel6.setText("Level: ");
        jPanel8.add(jLabel6);

        jLabelFrame.setMinimumSize(new java.awt.Dimension(30, 16));
        jLabelFrame.setPreferredSize(new java.awt.Dimension(30, 16));
        jPanel8.add(jLabelFrame);

        jPanel7.add(jPanel8, java.awt.BorderLayout.WEST);

        jButtonSaveAnimation.setText("SAVE");
        jButtonSaveAnimation.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonSaveAnimationActionPerformed(evt);
            }
        });

        jPanel7.add(jButtonSaveAnimation, java.awt.BorderLayout.EAST);

        jDialogAnimation.getContentPane().add(jPanel7, java.awt.BorderLayout.SOUTH);

        jDialogAnalyze.getContentPane().setLayout(new java.awt.GridBagLayout());

        jDialogAnalyze.setTitle("RESULT");
        jScrollPane1.setMinimumSize(new java.awt.Dimension(300, 300));
        jScrollPane1.setPreferredSize(new java.awt.Dimension(300, 300));
        jTableAnalyze.setModel(new javax.swing.table.DefaultTableModel(
            new Object [][] {

            },
            new String [] {
                "Variable", "Output"
            }
        ) {
            Class[] types = new Class [] {
                java.lang.String.class, java.lang.Boolean.class
            };
            boolean[] canEdit = new boolean [] {
                false, true
            };

            public Class getColumnClass(int columnIndex) {
                return types [columnIndex];
            }

            public boolean isCellEditable(int rowIndex, int columnIndex) {
                return canEdit [columnIndex];
            }
        });
        jScrollPane1.setViewportView(jTableAnalyze);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 2;
        jDialogAnalyze.getContentPane().add(jScrollPane1, gridBagConstraints);

        jButtonOKResult.setText("OK");
        jButtonOKResult.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonOKResultActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 0, 5);
        jDialogAnalyze.getContentPane().add(jButtonOKResult, gridBagConstraints);

        jButtonCancelResult.setText("CANCEL");
        jButtonCancelResult.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonCancelResultActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 0, 0);
        jDialogAnalyze.getContentPane().add(jButtonCancelResult, gridBagConstraints);

        jMenuDeleteSelection.setText("Delete Selected");
        jMenuDeleteSelection.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jMenuDeleteSelectionActionPerformed(evt);
            }
        });

        jPopupMenuDelete.add(jMenuDeleteSelection);

        setDefaultCloseOperation(javax.swing.WindowConstants.EXIT_ON_CLOSE);
        setTitle("Gunmetry Ver. 2.0");
        setCursor(new java.awt.Cursor(java.awt.Cursor.DEFAULT_CURSOR));
        jImageMain.setComponentPopupMenu(jPopupMenuDelete);
        jImageMain.setPreferredSize(new java.awt.Dimension(640, 480));
        jImageMain.addComponentListener(new java.awt.event.ComponentAdapter() {
            public void componentResized(java.awt.event.ComponentEvent evt) {
                jImageMainComponentResized(evt);
            }
        });
        jImageMain.addMouseMotionListener(new java.awt.event.MouseMotionAdapter() {
            public void mouseDragged(java.awt.event.MouseEvent evt) {
                jImageMainMouseDragged(evt);
            }
            public void mouseMoved(java.awt.event.MouseEvent evt) {
                jImageMainMouseMoved(evt);
            }
        });
        jImageMain.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                jImageMainMouseClicked(evt);
            }
            public void mousePressed(java.awt.event.MouseEvent evt) {
                jImageMainMousePressed(evt);
            }
            public void mouseReleased(java.awt.event.MouseEvent evt) {
                jImageMainMouseReleased(evt);
            }
        });

        getContentPane().add(jImageMain, java.awt.BorderLayout.CENTER);

        jLabelInfo.setText(" ");
        getContentPane().add(jLabelInfo, java.awt.BorderLayout.SOUTH);

        jPanelMain.setLayout(new java.awt.GridBagLayout());

        jPanelImage.setLayout(new java.awt.GridBagLayout());

        jPanelImage.setBorder(new javax.swing.border.TitledBorder("Image"));
        buttonGroupImage.add(jRadioCluster);
        jRadioCluster.setText("cluster");
        jRadioCluster.setEnabled(false);
        jRadioCluster.setFocusable(false);
        jRadioCluster.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                jRadioClusterItemStateChanged(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        jPanelImage.add(jRadioCluster, gridBagConstraints);

        buttonGroupImage.add(jRadioOriginal);
        jRadioOriginal.setSelected(true);
        jRadioOriginal.setText("original");
        jRadioOriginal.setFocusable(false);
        jRadioOriginal.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                jRadioOriginalItemStateChanged(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        jPanelImage.add(jRadioOriginal, gridBagConstraints);

        jButtonSetOriginal.setText("Load");
        jButtonSetOriginal.setFocusCycleRoot(true);
        jButtonSetOriginal.setFocusPainted(false);
        jButtonSetOriginal.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonSetOriginalActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        jPanelImage.add(jButtonSetOriginal, gridBagConstraints);

        jButtonSetCluster.setText("Load");
        jButtonSetCluster.setEnabled(false);
        jButtonSetCluster.setFocusable(false);
        jButtonSetCluster.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonSetClusterActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        jPanelImage.add(jButtonSetCluster, gridBagConstraints);

        jCheckMask.setText("mask");
        jCheckMask.setEnabled(false);
        jCheckMask.setFocusable(false);
        jCheckMask.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                jCheckMaskItemStateChanged(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        jPanelImage.add(jCheckMask, gridBagConstraints);

        jCanvasColorMask.setEnabled(false);
        jCanvasColorMask.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mousePressed(java.awt.event.MouseEvent evt) {
                jCanvasColorMaskMousePressed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(0, 37, 0, 0);
        jPanelImage.add(jCanvasColorMask, gridBagConstraints);

        buttonGroupImage.add(jRadioGradient);
        jRadioGradient.setText("gradient");
        jRadioGradient.setEnabled(false);
        jRadioGradient.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                jRadioGradientItemStateChanged(evt);
            }
        });
        jRadioGradient.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jRadioGradientActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        jPanelImage.add(jRadioGradient, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.ipadx = 5;
        jPanelMain.add(jPanelImage, gridBagConstraints);

        jPanelMarkers.setLayout(new java.awt.GridBagLayout());

        jPanelMarkers.setBorder(new javax.swing.border.TitledBorder("Markers"));
        jButtonAssignColorsToMarkers.setText("RESETTING");
        jButtonAssignColorsToMarkers.setEnabled(false);
        jButtonAssignColorsToMarkers.setFocusable(false);
        jButtonAssignColorsToMarkers.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonAssignColorsToMarkersActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.gridwidth = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(5, 0, 0, 0);
        jPanelMarkers.add(jButtonAssignColorsToMarkers, gridBagConstraints);

        jToggleObject.setIcon(new javax.swing.ImageIcon(getClass().getResource("/mkBtn1.gif")));
        jToggleObject.setSelected(true);
        jToggleObject.setFocusable(false);
        jToggleObject.setMargin(new java.awt.Insets(1, 1, 1, 1));
        jToggleObject.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jToggleObjectActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridheight = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(5, 0, 0, 0);
        jPanelMarkers.add(jToggleObject, gridBagConstraints);

        jToggleBackground.setIcon(new javax.swing.ImageIcon(getClass().getResource("/mkBtn2.gif")));
        jToggleBackground.setSelected(true);
        jToggleBackground.setFocusable(false);
        jToggleBackground.setMargin(new java.awt.Insets(1, 1, 1, 1));
        jToggleBackground.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jToggleBackgroundActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.gridheight = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(2, 0, 0, 0);
        jPanelMarkers.add(jToggleBackground, gridBagConstraints);

        jLabel1.setText("size: ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        jPanelMarkers.add(jLabel1, gridBagConstraints);

        jSpinnerMarkerSize.setFocusable(false);
        jSpinnerMarkerSize.setRequestFocusEnabled(false);
        jSpinnerMarkerSize.setVerifyInputWhenFocusTarget(false);
        jSpinnerMarkerSize.addChangeListener(new javax.swing.event.ChangeListener() {
            public void stateChanged(javax.swing.event.ChangeEvent evt) {
                jSpinnerMarkerSizeStateChanged(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        jPanelMarkers.add(jSpinnerMarkerSize, gridBagConstraints);

        jCanvasColorObj.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mousePressed(java.awt.event.MouseEvent evt) {
                jCanvasColorObjMousePressed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 0, 0);
        jPanelMarkers.add(jCanvasColorObj, gridBagConstraints);

        jCanvasColorBack.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mousePressed(java.awt.event.MouseEvent evt) {
                jCanvasColorBackMousePressed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(2, 5, 0, 0);
        jPanelMarkers.add(jCanvasColorBack, gridBagConstraints);

        jLabel7.setText("n =");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 30, 0, 0);
        jPanelMarkers.add(jLabel7, gridBagConstraints);

        jLabelNObject.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        jLabelNObject.setText("0");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        jPanelMarkers.add(jLabelNObject, gridBagConstraints);

        jLabel9.setText("n =");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 30, 0, 0);
        jPanelMarkers.add(jLabel9, gridBagConstraints);

        jLabelNBackground.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        jLabelNBackground.setText("0");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        jPanelMarkers.add(jLabelNBackground, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.ipadx = 6;
        jPanelMain.add(jPanelMarkers, gridBagConstraints);

        jPanelOeration.setLayout(new java.awt.GridBagLayout());

        jPanelOeration.setBorder(new javax.swing.border.TitledBorder("Operation"));
        jButtonAnalyze.setText("ANALYZE");
        jButtonAnalyze.setEnabled(false);
        jButtonAnalyze.setFocusable(false);
        jButtonAnalyze.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonAnalyzeActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        jPanelOeration.add(jButtonAnalyze, gridBagConstraints);

        buttonGroupOperation.add(jToggleMarker);
        jToggleMarker.setIcon(new javax.swing.ImageIcon(getClass().getResource("/opBtn1.gif")));
        jToggleMarker.setSelected(true);
        jToggleMarker.setFocusPainted(false);
        jToggleMarker.setFocusable(false);
        jToggleMarker.setMargin(new java.awt.Insets(1, 1, 1, 1));
        jToggleMarker.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                jToggleMarkerItemStateChanged(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        jPanelOeration.add(jToggleMarker, gridBagConstraints);

        buttonGroupOperation.add(jToggleImage);
        jToggleImage.setIcon(new javax.swing.ImageIcon(getClass().getResource("/opBtn4.gif")));
        jToggleImage.setFocusPainted(false);
        jToggleImage.setFocusable(false);
        jToggleImage.setMargin(new java.awt.Insets(1, 1, 1, 1));
        jToggleImage.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                jToggleImageItemStateChanged(evt);
            }
        });
        jToggleImage.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jToggleImageActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        jPanelOeration.add(jToggleImage, gridBagConstraints);

        jButtonSwitchMarker.setText("SWITCH MARKER");
        jButtonSwitchMarker.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonSwitchMarkerActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        jPanelOeration.add(jButtonSwitchMarker, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        jPanelMain.add(jPanelOeration, gridBagConstraints);

        jPanelResult.setLayout(new java.awt.GridBagLayout());

        jPanelResult.setBorder(new javax.swing.border.TitledBorder("Result"));
        jLabel3.setText("alpha: ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        jPanelResult.add(jLabel3, gridBagConstraints);

        jSliderAlpha.setMajorTickSpacing(50);
        jSliderAlpha.setMinorTickSpacing(10);
        jSliderAlpha.setPaintTicks(true);
        jSliderAlpha.setEnabled(false);
        jSliderAlpha.setPreferredSize(new java.awt.Dimension(100, 24));
        jSliderAlpha.addChangeListener(new javax.swing.event.ChangeListener() {
            public void stateChanged(javax.swing.event.ChangeEvent evt) {
                jSliderAlphaStateChanged(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        jPanelResult.add(jSliderAlpha, gridBagConstraints);

        jToggleWatershed.setIcon(new javax.swing.ImageIcon(getClass().getResource("/reBtn1.gif")));
        jToggleWatershed.setSelected(true);
        jToggleWatershed.setFocusable(false);
        jToggleWatershed.setMargin(new java.awt.Insets(1, 1, 1, 1));
        jToggleWatershed.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                jToggleWatershedItemStateChanged(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        jPanelResult.add(jToggleWatershed, gridBagConstraints);

        jToggleSkiz.setIcon(new javax.swing.ImageIcon(getClass().getResource("/reBtn2.gif")));
        jToggleSkiz.setSelected(true);
        jToggleSkiz.setFocusable(false);
        jToggleSkiz.setMargin(new java.awt.Insets(1, 1, 1, 1));
        jToggleSkiz.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                jToggleSkizItemStateChanged(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        jPanelResult.add(jToggleSkiz, gridBagConstraints);

        jToggleObjCent.setIcon(new javax.swing.ImageIcon(getClass().getResource("/reBtn3.gif")));
        jToggleObjCent.setFocusable(false);
        jToggleObjCent.setMargin(new java.awt.Insets(1, 1, 1, 1));
        jToggleObjCent.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                jToggleObjCentItemStateChanged(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        jPanelResult.add(jToggleObjCent, gridBagConstraints);

        jToggleSkizCent.setIcon(new javax.swing.ImageIcon(getClass().getResource("/reBtn4.gif")));
        jToggleSkizCent.setFocusable(false);
        jToggleSkizCent.setMargin(new java.awt.Insets(1, 1, 1, 1));
        jToggleSkizCent.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                jToggleSkizCentItemStateChanged(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 4;
        jPanelResult.add(jToggleSkizCent, gridBagConstraints);

        jToggleNearestNeighbor.setIcon(new javax.swing.ImageIcon(getClass().getResource("/reBtn5.gif")));
        jToggleNearestNeighbor.setFocusable(false);
        jToggleNearestNeighbor.setMargin(new java.awt.Insets(1, 1, 1, 1));
        jToggleNearestNeighbor.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                jToggleNearestNeighborItemStateChanged(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 5;
        jPanelResult.add(jToggleNearestNeighbor, gridBagConstraints);

        jCanvasColorWatershed.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mousePressed(java.awt.event.MouseEvent evt) {
                jCanvasColorWatershedMousePressed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 0, 0);
        jPanelResult.add(jCanvasColorWatershed, gridBagConstraints);

        jCanvasColorSkiz.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mousePressed(java.awt.event.MouseEvent evt) {
                jCanvasColorSkizMousePressed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 0, 0);
        jPanelResult.add(jCanvasColorSkiz, gridBagConstraints);

        jCanvasColorObjCent.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mousePressed(java.awt.event.MouseEvent evt) {
                jCanvasColorObjCentMousePressed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 0, 0);
        jPanelResult.add(jCanvasColorObjCent, gridBagConstraints);

        jCanvasColorSkizCent.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mousePressed(java.awt.event.MouseEvent evt) {
                jCanvasColorSkizCentMousePressed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 0, 0);
        jPanelResult.add(jCanvasColorSkizCent, gridBagConstraints);

        jCanvasColorNearestNeighbor.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mousePressed(java.awt.event.MouseEvent evt) {
                jCanvasColorNearestNeighborMousePressed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 0, 0);
        jPanelResult.add(jCanvasColorNearestNeighbor, gridBagConstraints);

        jButtonAnimation.setText("ANIMATION");
        jButtonAnimation.setEnabled(false);
        jButtonAnimation.setFocusable(false);
        jButtonAnimation.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButtonAnimationActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 6;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(3, 0, 0, 0);
        jPanelResult.add(jButtonAnimation, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        jPanelMain.add(jPanelResult, gridBagConstraints);

        getContentPane().add(jPanelMain, java.awt.BorderLayout.WEST);

        jMenuFile.setText("File");
        jMenuFile.setEnabled(false);
        jMenuOpenMarkers.setText("Open Markers");
        jMenuOpenMarkers.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jMenuOpenMarkersActionPerformed(evt);
            }
        });

        jMenuFile.add(jMenuOpenMarkers);

        jMenuSaveMarkers.setText("Save Markers");
        jMenuSaveMarkers.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jMenuSaveMarkersActionPerformed(evt);
            }
        });

        jMenuFile.add(jMenuSaveMarkers);

        jMenuBarMain.add(jMenuFile);

        jMenuResult.setText("Result");
        jMenuResult.setEnabled(false);
        jMenuSaveData.setText("Save Data");
        jMenuSaveData.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jMenuSaveDataActionPerformed(evt);
            }
        });

        jMenuResult.add(jMenuSaveData);

        jMenuSaveImage.setText("Save Image");
        jMenuSaveImage.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jMenuSaveImageActionPerformed(evt);
            }
        });

        jMenuResult.add(jMenuSaveImage);

        jMenuBarMain.add(jMenuResult);

        setJMenuBar(jMenuBarMain);

        pack();
    }//GEN-END:initComponents

	private void jButtonSwitchMarkerActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonSwitchMarkerActionPerformed
		switchEditableMarkerType();
	}//GEN-LAST:event_jButtonSwitchMarkerActionPerformed
	
	private void jComboBackAlgorithmItemStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_jComboBackAlgorithmItemStateChanged
		pluginMarkerBackground = setMarkerOption(jPanelBackground, (String)jComboBackAlgorithm.getSelectedItem());
	}//GEN-LAST:event_jComboBackAlgorithmItemStateChanged
	
	private void jComboObjAlgorithmItemStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_jComboObjAlgorithmItemStateChanged
		pluginMarkerObject = setMarkerOption(jPanelObject, (String)jComboObjAlgorithm.getSelectedItem());
	}//GEN-LAST:event_jComboObjAlgorithmItemStateChanged
	
	private void jButtonCancelMaskActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonCancelMaskActionPerformed
		jDialogMask.setVisible(false);
	}//GEN-LAST:event_jButtonCancelMaskActionPerformed
	
	private void jButtonSetMaskActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonSetMaskActionPerformed
		int nChecked = imageColorPickerMask.getCheckedNumber();
		if (nChecked==0 || nChecked==imageColorPickerMask.getColorNumber()){
			jLabelInfo.setText("Invalid color selections");
			return;
		}
		
		jDialogMask.setVisible(false);
	}//GEN-LAST:event_jButtonSetMaskActionPerformed
	
	private void jRadioGradientActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jRadioGradientActionPerformed
		// ɏR[hǉ:
	}//GEN-LAST:event_jRadioGradientActionPerformed
	
	private void jRadioGradientItemStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_jRadioGradientItemStateChanged
		if (jRadioGradient.isSelected() && gradient!=null){
			Image img = createImageFromGradient();
			drawImageToImgBackground(img);
		}
	}//GEN-LAST:event_jRadioGradientItemStateChanged
	
	private void jCanvasColorNearestNeighborMousePressed(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_jCanvasColorNearestNeighborMousePressed
		overlapImage();
		drawMarkers();
		updateFullCanvas();
	}//GEN-LAST:event_jCanvasColorNearestNeighborMousePressed
	
	private void jToggleImageItemStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_jToggleImageItemStateChanged
		if (jToggleImage.isSelected()){
			jImageMain.setCursor(new Cursor(Cursor.HAND_CURSOR));
			jImageMain.setComponentPopupMenu(null);
		}
	}//GEN-LAST:event_jToggleImageItemStateChanged
	
	private void jToggleMarkerItemStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_jToggleMarkerItemStateChanged
		if (jToggleMarker.isSelected()){
			jImageMain.setCursor(cursorObject);
			jImageMain.setComponentPopupMenu(jPopupMenuDelete);
		}
	}//GEN-LAST:event_jToggleMarkerItemStateChanged
	
	private void jImageMainComponentResized(java.awt.event.ComponentEvent evt) {//GEN-FIRST:event_jImageMainComponentResized
		drawMarkers();
		updateFullCanvas();
	}//GEN-LAST:event_jImageMainComponentResized
	
	private void jMenuDeleteSelectionActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jMenuDeleteSelectionActionPerformed
		removeSelectedMarkers();
	}//GEN-LAST:event_jMenuDeleteSelectionActionPerformed
	
	private void jToggleNearestNeighborItemStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_jToggleNearestNeighborItemStateChanged
		overlapImage();
		drawMarkers();
		updateFullCanvas();
	}//GEN-LAST:event_jToggleNearestNeighborItemStateChanged
	
	private void jMenuSaveImageActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jMenuSaveImageActionPerformed
		BufferedImage img = jImageMain.getImage();
		imageFileIO.writeImage(img);
	}//GEN-LAST:event_jMenuSaveImageActionPerformed
	
	private void jCanvasColorMaskMousePressed(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_jCanvasColorMaskMousePressed
		overlapImage();
		drawMarkers();
		updateFullCanvas();
	}//GEN-LAST:event_jCanvasColorMaskMousePressed
	
	private void jCanvasColorSkizCentMousePressed(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_jCanvasColorSkizCentMousePressed
		overlapImage();
		drawMarkers();
		updateFullCanvas();
	}//GEN-LAST:event_jCanvasColorSkizCentMousePressed
	
	private void jCanvasColorObjCentMousePressed(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_jCanvasColorObjCentMousePressed
		overlapImage();
		drawMarkers();
		updateFullCanvas();
	}//GEN-LAST:event_jCanvasColorObjCentMousePressed
	
	private void jCanvasColorSkizMousePressed(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_jCanvasColorSkizMousePressed
		overlapImage();
		drawMarkers();
		updateFullCanvas();
	}//GEN-LAST:event_jCanvasColorSkizMousePressed
	
	private void jCanvasColorWatershedMousePressed(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_jCanvasColorWatershedMousePressed
		overlapImage();
		drawMarkers();
		updateFullCanvas();
	}//GEN-LAST:event_jCanvasColorWatershedMousePressed
	
	private void jToggleSkizCentItemStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_jToggleSkizCentItemStateChanged
		overlapImage();
		drawMarkers();
		updateFullCanvas();
	}//GEN-LAST:event_jToggleSkizCentItemStateChanged
	
	private void jToggleObjCentItemStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_jToggleObjCentItemStateChanged
		overlapImage();
		drawMarkers();
		updateFullCanvas();
	}//GEN-LAST:event_jToggleObjCentItemStateChanged
	
	private void jToggleSkizItemStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_jToggleSkizItemStateChanged
		overlapImage();
		drawMarkers();
		updateFullCanvas();
	}//GEN-LAST:event_jToggleSkizItemStateChanged
	
	private void jToggleWatershedItemStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_jToggleWatershedItemStateChanged
		overlapImage();
		drawMarkers();
		updateFullCanvas();
	}//GEN-LAST:event_jToggleWatershedItemStateChanged
	
	private void jCheckMaskItemStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_jCheckMaskItemStateChanged
		if (jCheckMask.isSelected()){
			imageFileIO.resetChoosableFileFilters();
			imageFileIO.setFileFilter(filterGifPng);
			File fileMask = imageFileIO.getOpenFile("Select a GIF file for mask...");
			if (fileMask==null){
				jCheckMask.setSelected(false);
				return;
			}
			
			BufferedImage img = imageFileIO.readImage(fileMask);
			ImageInfo imiMask = new ImageInfo(img);
			int[] maskCluster = imiMask.getPixelsLabel();
			imageColorPickerMask.setColorTable(imiMask.getPixels(), imiMask.getPixelsLabel());
			jDialogMask.setVisible(true);
			
			// after dialog operation
			boolean[] flag = imageColorPickerMask.getFlag();
			analyzeMask = getBinaryMap(flag, maskCluster);
			overlapImage();
			drawMarkers();
			updateFullCanvas();
		}else{
			analyzeMask = null;
		}
		
		jCanvasColorMask.setEnabled(jCheckMask.isSelected());
	}//GEN-LAST:event_jCheckMaskItemStateChanged
	
	private void jButtonSaveAnimationActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonSaveAnimationActionPerformed
		BufferedImage img = jImageAnimation.getImage();
		imageFileIO.writeImage(img);
	}//GEN-LAST:event_jButtonSaveAnimationActionPerformed
	
	private void jButtonOKResultActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonOKResultActionPerformed
		
		ArrayList<ArrayList<Double>> result = new ArrayList<ArrayList<Double>>();
		ArrayList<String> variableName = new ArrayList<String>();
		PluginClassLoader loader = new PluginClassLoader(pathAnalyze);
		for (int i=0; i<jTableAnalyze.getRowCount(); i++){
			if ((Boolean)jTableAnalyze.getValueAt(i,1)){
				String algorithmName = (String)jTableAnalyze.getValueAt(i,0);
				try{
					AnalyzePlugin plugin = (AnalyzePlugin)loader.loadClass(algorithmName).newInstance();
					result.add(plugin.analyze(nuclei,skiz));
				}catch(Exception e){
					e.printStackTrace();
					return;
				}
				variableName.add(algorithmName);
			}
		}
		
		String[] strResult = new String[markersObject.size()+1];
		strResult[0] = "";
		for (int j=0; j<variableName.size(); j++){
			strResult[0] += variableName.get(j)+",";
		}
		for (int i=1; i<strResult.length; i++){
			strResult[i] = "";
			for (int j=0; j<result.size(); j++){
				strResult[i] += result.get(j).get(i-1)+",";
			}
		}
		
		textFileIO.resetChoosableFileFilters();
		textFileIO.writeStrings(strResult);
		
		jDialogAnalyze.setVisible(false);
	}//GEN-LAST:event_jButtonOKResultActionPerformed
	
	private void jButtonCancelResultActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonCancelResultActionPerformed
		jDialogAnalyze.setVisible(false);
	}//GEN-LAST:event_jButtonCancelResultActionPerformed
	
	private void jMenuSaveDataActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jMenuSaveDataActionPerformed
		jDialogAnalyze.setVisible(true);
	}//GEN-LAST:event_jMenuSaveDataActionPerformed
	
	private void jSliderFrameStateChanged(javax.swing.event.ChangeEvent evt) {//GEN-FIRST:event_jSliderFrameStateChanged
		if (!jDialogAnimation.isVisible()){
			return;
		}
		
		int distFrame = jSliderFrame.getValue();
		int currentFrame = stepArray[orderArray[currentOrder]];
		
		Graphics g = jImageAnimation.getOffGraphics();
		int x, y;
		if (currentFrame<distFrame){
			while(stepArray[orderArray[currentOrder]]<distFrame){
				int pn = orderArray[currentOrder];
				if (imiAnalyze.getPixelsLabel(pn)>BACKGROUND){
					g.setColor(jCanvasColorObj.getColor());
				}else{
					g.setColor(jCanvasColorBack.getColor());
				}
				x = pn%imgBackground.getWidth();
				y = pn/imgBackground.getWidth();
				g.drawLine(x,y, x,y);
				currentOrder++;
			}
		}else{
			g.setColor(Color.black);
			while(distFrame<stepArray[orderArray[currentOrder]]){
				int pn = orderArray[currentOrder];
				x = pn%imgBackground.getWidth();
				y = pn/imgBackground.getWidth();
				g.drawLine(x,y, x,y);
				currentOrder--;
			}
		}
		jImageAnimation.repaint();
		
		jLabelFrame.setText(Integer.toString(distFrame));
	}//GEN-LAST:event_jSliderFrameStateChanged
	
	private void jToggleImageActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jToggleImageActionPerformed
		
	}//GEN-LAST:event_jToggleImageActionPerformed
	
	private void jButtonAnimationActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonAnimationActionPerformed
		jImageAnimation.setPreferredSize(new Dimension(imgBackground.getWidth(),imgBackground.getHeight()));
		jDialogAnimation.pack();
		jDialogAnimation.setVisible(true);
		
		jImageAnimation.setFullSrc();
		jImageAnimation.drawImage();
		jImageAnimation.repaint();
		
	}//GEN-LAST:event_jButtonAnimationActionPerformed
	
	private void jCanvasColorBackMousePressed(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_jCanvasColorBackMousePressed
		drawMarkers();
		updateFullCanvas();
		cursorBackground = createCursor(jCanvasColorBack.getColor());
		if (jToggleMarker.isSelected() && !isObjectActive){
			jImageMain.setCursor(cursorBackground);
		}
	}//GEN-LAST:event_jCanvasColorBackMousePressed
	
	private void jCanvasColorObjMousePressed(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_jCanvasColorObjMousePressed
		drawMarkers();
		updateFullCanvas();
		cursorObject = createCursor(jCanvasColorObj.getColor());
		if (jToggleMarker.isSelected() && isObjectActive){
			jImageMain.setCursor(cursorObject);
		}
	}//GEN-LAST:event_jCanvasColorObjMousePressed
	
	private void jMenuOpenMarkersActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jMenuOpenMarkersActionPerformed
		ArrayList<String> data = textFileIO.readStrings();
		if (data==null){
			return;
		}
		
		markersObject.clear();
		String line;
		String[] location;
		int x,y;
		line = data.get(0);
		location = line.split(",");
		int width = Integer.valueOf(location[0]);
		int height = Integer.valueOf(location[1]);
		int nObject = Integer.valueOf(data.get(1));
		int size = (Integer)jSpinnerMarkerSize.getValue();
		for (int i=2; i<nObject; i++){
			line = data.get(i);
			location = line.split(",");
			x = Integer.valueOf(location[0])*imgBackground.getWidth()/width;
			y = Integer.valueOf(location[1])*imgBackground.getHeight()/height;
			Point p= new Point(x,y);
			markersObject.add(p);
		}
		
		for (int i=nObject; i<data.size(); i++){
			line = data.get(i);
			location = line.split(",");
			x = Integer.valueOf(location[0])*imgBackground.getWidth()/width;
			y = Integer.valueOf(location[1])*imgBackground.getHeight()/height;
			Point p= new Point(x,y);
			markersBackground.add(p);
		}
		
		jRadioCluster.setEnabled(false);
		imiAnalyze = new ImageInfo(imgBackground);
		drawMarkers();
		updateFullCanvas();
		updateMarkerNumber();
		checkReadyToAnalyze();
	}//GEN-LAST:event_jMenuOpenMarkersActionPerformed
	
	private void jMenuSaveMarkersActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jMenuSaveMarkersActionPerformed
		ArrayList<String> data = new ArrayList<String>();
		data.add(imgBackground.getWidth()+","+imgBackground.getHeight());
		data.add(Integer.toString(markersObject.size()+2));
		for (Point p: markersObject){
			String arg = p.x+","+p.y;
			data.add(arg);
		}
		for (Point p: markersBackground){
			String arg = p.x+","+p.y;
			data.add(arg);
		}
		textFileIO.resetChoosableFileFilters();
		textFileIO.writeStrings(data);
	}//GEN-LAST:event_jMenuSaveMarkersActionPerformed
	
	private void jButtonCancelMarkerActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonCancelMarkerActionPerformed
		jDialogMarker.setVisible(false);
	}//GEN-LAST:event_jButtonCancelMarkerActionPerformed
	
	private void jSliderAlphaStateChanged(javax.swing.event.ChangeEvent evt) {//GEN-FIRST:event_jSliderAlphaStateChanged
		updateResultImage();
	}//GEN-LAST:event_jSliderAlphaStateChanged
	
	private void jImageMainMouseReleased(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_jImageMainMouseReleased
		if (selectionRect!=null){
			//checkSelectionType();
			// add PointRects which is inside of selectionRect to selectedMarkers
			addToSelectedMarkers(markersObject);
			addToSelectedMarkers(markersBackground);
			selectionRect = null;
			
			readyToDrawSelectMarkers();
			drawMarkers();
			repaintDirtyRegion();
		}
	}//GEN-LAST:event_jImageMainMouseReleased
	
	private void jImageMainMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_jImageMainMouseClicked
		if (jToggleImage.isSelected()){
			jImageMain.mouseClicked(evt);
			jImageMain.clearCanvas();
			jImageMain.drawImage();
			drawMarkers();
			updateFullCanvas();
			return;
		}
		
		Point2D p2D = jImageMain.getImageTransform().inverseTransform(evt.getPoint());
		if (focusedMarker==null){
			Point p = new Point((int)p2D.getX(),(int)p2D.getY());
			if(jToggleMarker.isSelected()){
				if (isObjectActive){	// add a marker to markers as object
					markersObject.add(p);
				}else{	// add a marker to markers as background
					markersBackground.add(p);
				}
				if (!jMenuSaveMarkers.isEnabled()){
					jMenuSaveMarkers.setEnabled(true);
				}
			}
			drawMarkers();
			pointRect.setLocation(p.x,p.y);
			updatePartialCanvas(pointRect.getBounds());
			checkReadyToAnalyze();
			updateMarkerNumber();
		}else{
			selectedMarkers.add(focusedMarker);
			drawMarkers();
			pointRect.setLocation(focusedMarker.x,focusedMarker.y);
			updatePartialCanvas(pointRect.getBounds());
		}
		
	}//GEN-LAST:event_jImageMainMouseClicked
	
	private void jButtonAnalyzeActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonAnalyzeActionPerformed
		imgOverlap = new BufferedImage(imgBackground.getWidth(),imgBackground.getHeight(),BufferedImage.TYPE_INT_RGB);
		
		jImageMain.drawImage();
		
		if (jCheckMask.isSelected()){
			removeMaskedMarkers(markersObject);
			removeMaskedMarkers(markersBackground);
		}
		
		Watershed watershed = new Watershed(imiAnalyze, BACKGROUND);
		int INIT = watershed.getIndexInitial();
		Arrays.fill(imiAnalyze.getPixelsLabel(),INIT);
		int offset;
		int count = BACKGROUND+1;
		for (Point p: markersObject){
			imiAnalyze.setPixelsLabel(pointToOffset(p), count);
			count++;
		}
		for (Point p: markersBackground){
			imiAnalyze.setPixelsLabel(pointToOffset(p), BACKGROUND);
		}
		Morphology mo = new Morphology(imiAnalyze.getNeighborFrame());
		watershed.setOrder(gradient);
		
		FirstInFirstOut fifo = watershed.getFIFO();
		stepArray = new int[imiAnalyze.getImageSize()];
		orderArray = new int[imiAnalyze.getImageSize()];
		Arrays.fill(stepArray,-1);
		Arrays.fill(orderArray,-1);
		
		int size;
		int step = 0;
		currentOrder =-1;
		do{
			watershed.setFIFOMarker();
			count = watershed.flood4();
			if (0<count){
				fifo.rewindToMarker();
				size = fifo.getSize();
				for (int i=0; i<size; i++){
					int pn = fifo.get();
					if (stepArray[pn]==-1){
						stepArray[pn] = step;
						
						currentOrder ++;
						orderArray[currentOrder] = pn;
					}
				}
				step ++;
			}
		}while(!watershed.isConvergence());
		watershed.undeterminedToWatershed();
		contourRegion = watershed.getWatershed();
		convertMaskToWatershed(watershed.getIndexWatershed());
		nuclei = makeROI();
		makeResultImage();
		
		// make skiz
		boolean[] backgroundMap = new boolean[imiAnalyze.getImageSize()];
		for (int pn=0; pn<imiAnalyze.getImageSize(); pn++){
			if (imiAnalyze.getPixelsLabel(pn)>BACKGROUND){
				backgroundMap[pn] = false;
			}else{
				imiAnalyze.setPixelsLabel(pn,INIT);
				backgroundMap[pn] = true;
			}
		}
		watershed = new Watershed(imiAnalyze, BACKGROUND+1);
		int[] dist = mo.geodesicDistance4(backgroundMap);
		watershed.setOrder(dist);
		do{
			watershed.flood4();
		}while(!watershed.isConvergence());
		watershed.undeterminedToWatershed();
		if (jCheckMask.isSelected()){
			contourSkiz = deleteMaskedContour(watershed.getWatershed());
		}else{
			contourSkiz = watershed.getWatershed();
		}
		convertMaskToWatershed(watershed.getIndexWatershed());
		skiz = makeROI();
		
		setNearestNeighbor(dist);
		
		// show result image
		jImageAnimation.setImage(imgResult);
		jSliderAlpha.setEnabled(true);
		updateResultImage();
		
		// set up the conditions of controls
		jSliderFrame.setMaximum(step-1);
		jSliderFrame.setValue(step-1);
		jButtonAnimation.setEnabled(true);
		jMenuResult.setEnabled(true);
	}//GEN-LAST:event_jButtonAnalyzeActionPerformed
	
	private void jImageMainMouseDragged(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_jImageMainMouseDragged
		if (jToggleImage.isSelected()){
			jImageMain.mouseDragged(evt);
			jImageMain.clearCanvas();
			jImageMain.drawImage();
			drawMarkers();
			updateFullCanvas();
			return;
		}
		
		Point2D p = jImageMain.getImageTransform().inverseTransform(evt.getPoint());
		if (focusedMarker==null){
			if (selectionRect==null){
				selectionRect = new Rectangle((int)p.getX(),(int)p.getY(),1,1);
			}else{
				selectionRect.setLocation(dragStartPoint);
				selectionRect.setSize(1,1);
				selectionRect.add(p);
			}
			drawMarkers();
			updatePartialCanvas(selectionRect);
		}else{
			focusedMarker.setLocation((int)p.getX(),(int)p.getY());
			drawMarkers();
			pointRect.setLocation(focusedMarker.x,focusedMarker.y);
			updatePartialCanvas(pointRect.getBounds());
		}
		
		jLabelInfo.setText("mouse(x,y)="+evt.getX()+","+evt.getY());
		jLabelInfo.setText("mouse(x,y)="+(int)p.getX()+","+(int)p.getY());
	}//GEN-LAST:event_jImageMainMouseDragged
	
	private void jImageMainMousePressed(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_jImageMainMousePressed
		if (jToggleImage.isSelected()){
			jImageMain.mousePressed(evt);
			return;
		}
		
		if (evt.getModifiers()!=4){
			checkSelectionType(evt.getModifiers());
			selectionRect = null;
			
			Point2D p = jImageMain.getImageTransform().inverseTransform(evt.getPoint());
			dragStartPoint.setLocation(p);
		}
		
	}//GEN-LAST:event_jImageMainMousePressed
	
	private void jSpinnerMarkerSizeStateChanged(javax.swing.event.ChangeEvent evt) {//GEN-FIRST:event_jSpinnerMarkerSizeStateChanged
		int value = (Integer)jSpinnerMarkerSize.getValue();
		pointRect.setSize(value);
		drawMarkers();
		updateFullCanvas();
	}//GEN-LAST:event_jSpinnerMarkerSizeStateChanged
	
	private void jImageMainMouseMoved(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_jImageMainMouseMoved
		if (!jToggleImage.isSelected()){
			findFocusedMarker(evt.getPoint());
		}
		jLabelInfo.setText("mouse(x,y)="+evt.getX()+","+evt.getY());
	}//GEN-LAST:event_jImageMainMouseMoved
	
	private void jToggleBackgroundActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jToggleBackgroundActionPerformed
		drawMarkers();
		updateFullCanvas();
	}//GEN-LAST:event_jToggleBackgroundActionPerformed
	
	private void jToggleObjectActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jToggleObjectActionPerformed
		drawMarkers();
		updateFullCanvas();
	}//GEN-LAST:event_jToggleObjectActionPerformed
	
	private void jRadioClusterItemStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_jRadioClusterItemStateChanged
		if (jRadioCluster.isSelected() && fileCluster!=null){
			Image img = imageFileIO.readImage(fileCluster);
			drawImageToImgBackground(img);
		}
	}//GEN-LAST:event_jRadioClusterItemStateChanged
	
	private void jRadioOriginalItemStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_jRadioOriginalItemStateChanged
		if (jRadioOriginal.isSelected() && fileOriginal!=null){
			Image img = imageFileIO.readImage(fileOriginal);
			drawImageToImgBackground(img);
		}
	}//GEN-LAST:event_jRadioOriginalItemStateChanged
	
	private void jButtonSetMarkerActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonSetMarkerActionPerformed
		// check if any color is selected
		int nChecked = imageColorPickerMarker.getCheckedNumber();
		if (nChecked==0 || nChecked==imageColorPickerMarker.getColorNumber()){
			jLabelInfo.setText("Invalid selections for seed");
			return;
		}
		
		jDialogMarker.setVisible(false);
		
	}//GEN-LAST:event_jButtonSetMarkerActionPerformed
	
	private void jButtonAssignColorsToMarkersActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonAssignColorsToMarkersActionPerformed
		jDialogMarker.setVisible(true);
	}//GEN-LAST:event_jButtonAssignColorsToMarkersActionPerformed
	
	private void jButtonSetClusterActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonSetClusterActionPerformed
		imageFileIO.resetChoosableFileFilters();
		imageFileIO.setFileFilter(filterGifPng);
		fileCluster = imageFileIO.getOpenFile("Select a file for cluster image...");
		if (fileCluster==null){
			return;
		}
		
		Dimension size = imageFileIO.getImageSize(fileCluster);
		if (!checkBackgroundImageSize(size)){
			return;
		}
		
		ImageInfo imiCluster = new ImageInfo(imageFileIO.readImage(fileCluster));
		int[] colorCluster = imiCluster.getPixelsLabel();
		imageColorPickerMarker.setColorTable(imiCluster.getPixels(), imiCluster.getPixelsLabel());
		jDialogMarker.setVisible(true);
		
		// after dialog operation
		boolean[] flag = imageColorPickerMarker.getFlag();
		setMarkers(flag, colorCluster);
		
		jRadioCluster.setEnabled(true);
		jButtonAssignColorsToMarkers.setEnabled(true);
	}//GEN-LAST:event_jButtonSetClusterActionPerformed
	
	private void jButtonSetOriginalActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButtonSetOriginalActionPerformed
		initializeAnalysis();
		
		imageFileIO.resetChoosableFileFilters();
		imageFileIO.setFileFilter(filterImg);
		fileOriginal = imageFileIO.getOpenFile("Select the original image...");
		if (fileOriginal==null){
			return;
		}
		
		Dimension size = imageFileIO.getImageSize(fileOriginal);
		if (!checkBackgroundImageSize(size)){
			imgBackground = new BufferedImage(size.width,size.height, BufferedImage.TYPE_INT_RGB);
			jImageMain.setSize(size.width,size.height);
			pack();
		}
		if (jRadioOriginal.isSelected()){
			Image img = imageFileIO.readImage(fileOriginal);
			drawImageToImgBackground(img);
		}else{
			jRadioOriginal.setSelected(true);
		}
		
		imiAnalyze = new ImageInfo(imgBackground);
		Morphology mo = new Morphology(imiAnalyze.getNeighborFrame());
		double[][] LAB = new double[imiAnalyze.getImageSize()][3];
		ColorVector.getLAB(imiAnalyze.getPixels(),LAB);
		gradient = mo.calcSobel(LAB);
		
		jMenuFile.setEnabled(true);
		jRadioGradient.setEnabled(true);
		jButtonSetCluster.setEnabled(true);
		
		setTitle("Gunmetry Ver2.0 ("+fileOriginal.getName()+")");
	}//GEN-LAST:event_jButtonSetOriginalActionPerformed
	
	/**
	 * @param args the command line arguments
	 */
	public static void main(String args[]) {
		java.awt.EventQueue.invokeLater(new Runnable() {
			public void run() {
				new GunmetryViewer().setVisible(true);
			}
		});
	}
	
    // ϐ錾 - ҏWs//GEN-BEGIN:variables
    private javax.swing.ButtonGroup buttonGroupImage;
    private javax.swing.ButtonGroup buttonGroupOperation;
    private gunmetry.ImageColorPicker imageColorPickerMarker;
    private gunmetry.ImageColorPicker imageColorPickerMask;
    private javax.swing.JButton jButtonAnalyze;
    private javax.swing.JButton jButtonAnimation;
    private javax.swing.JButton jButtonAssignColorsToMarkers;
    private javax.swing.JButton jButtonCancelMarker;
    private javax.swing.JButton jButtonCancelMask;
    private javax.swing.JButton jButtonCancelResult;
    private javax.swing.JButton jButtonOKResult;
    private javax.swing.JButton jButtonSaveAnimation;
    private javax.swing.JButton jButtonSetCluster;
    private javax.swing.JButton jButtonSetMarker;
    private javax.swing.JButton jButtonSetMask;
    private javax.swing.JButton jButtonSetOriginal;
    private javax.swing.JButton jButtonSwitchMarker;
    private sos.gui.JCanvasColor jCanvasColorBack;
    private sos.gui.JCanvasColor jCanvasColorMask;
    private sos.gui.JCanvasColor jCanvasColorNearestNeighbor;
    private sos.gui.JCanvasColor jCanvasColorObj;
    private sos.gui.JCanvasColor jCanvasColorObjCent;
    private sos.gui.JCanvasColor jCanvasColorSkiz;
    private sos.gui.JCanvasColor jCanvasColorSkizCent;
    private sos.gui.JCanvasColor jCanvasColorWatershed;
    private javax.swing.JCheckBox jCheckMask;
    private javax.swing.JComboBox jComboBackAlgorithm;
    private javax.swing.JComboBox jComboObjAlgorithm;
    private javax.swing.JDialog jDialogAnalyze;
    private javax.swing.JDialog jDialogAnimation;
    private javax.swing.JDialog jDialogMarker;
    private javax.swing.JDialog jDialogMask;
    private sos.awt.JImageCanvas jImageAnimation;
    private sos.awt.JImageCanvas jImageMain;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel3;
    private javax.swing.JLabel jLabel4;
    private javax.swing.JLabel jLabel5;
    private javax.swing.JLabel jLabel6;
    private javax.swing.JLabel jLabel7;
    private javax.swing.JLabel jLabel9;
    private javax.swing.JLabel jLabelFrame;
    private javax.swing.JLabel jLabelInfo;
    private javax.swing.JLabel jLabelNBackground;
    private javax.swing.JLabel jLabelNObject;
    private javax.swing.JMenuBar jMenuBarMain;
    private javax.swing.JMenuItem jMenuDeleteSelection;
    private javax.swing.JMenu jMenuFile;
    private javax.swing.JMenuItem jMenuOpenMarkers;
    private javax.swing.JMenu jMenuResult;
    private javax.swing.JMenuItem jMenuSaveData;
    private javax.swing.JMenuItem jMenuSaveImage;
    private javax.swing.JMenuItem jMenuSaveMarkers;
    private javax.swing.JPanel jPanel1;
    private javax.swing.JPanel jPanel2;
    private javax.swing.JPanel jPanel7;
    private javax.swing.JPanel jPanel8;
    private javax.swing.JPanel jPanelBackground;
    private javax.swing.JPanel jPanelButtons;
    private javax.swing.JPanel jPanelButtons1;
    private javax.swing.JPanel jPanelImage;
    private javax.swing.JPanel jPanelMain;
    private javax.swing.JPanel jPanelMarkers;
    private javax.swing.JPanel jPanelObject;
    private javax.swing.JPanel jPanelOeration;
    private javax.swing.JPanel jPanelResult;
    private javax.swing.JPopupMenu jPopupMenuDelete;
    private javax.swing.JRadioButton jRadioCluster;
    private javax.swing.JRadioButton jRadioGradient;
    private javax.swing.JRadioButton jRadioOriginal;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JScrollPane jScrollPaneMarker;
    private javax.swing.JScrollPane jScrollPaneMask;
    private javax.swing.JSlider jSliderAlpha;
    private javax.swing.JSlider jSliderFrame;
    private javax.swing.JSpinner jSpinnerMarkerSize;
    private javax.swing.JTable jTableAnalyze;
    private javax.swing.JToggleButton jToggleBackground;
    private javax.swing.JToggleButton jToggleImage;
    private javax.swing.JToggleButton jToggleMarker;
    private javax.swing.JToggleButton jToggleNearestNeighbor;
    private javax.swing.JToggleButton jToggleObjCent;
    private javax.swing.JToggleButton jToggleObject;
    private javax.swing.JToggleButton jToggleSkiz;
    private javax.swing.JToggleButton jToggleSkizCent;
    private javax.swing.JToggleButton jToggleWatershed;
    // ϐ錾̏I//GEN-END:variables
	
}
