package orch;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.image.BufferedImage;
import java.awt.image.PixelGrabber;
import java.io.File;
import javax.imageio.ImageIO;
import sos.graphics.SOSGraphics;
import sos.io.SOSImageIO;
import sos.util.ColorVector;

/*
 * ImagePiece.java
 *
 * Created on 2004/12/19, 22:48
 */

/**
 * This class has information about the image on the Viewer to handle template matching.
 * @author Scientific Open Source projects (Gaku Tanaka)
 * @version 1.0 (2005/02/15)
 */
public class ImagePiece implements SOSGraphics{
	private File fileOfImage;
	private boolean hasImage;
	private boolean fixed = false;
	private int tableIndex;
	private Point tableLocation = new Point();
	
	private Rectangle bounds;
	private int[] pixelsData;
	private int[][] rgb;
	
	private Image thumbnail;
	private Dimension thumbnailSize = new Dimension();

	/**
	 * Creates a new instance of Thumbnail
	 * @param width the image width on the viewer.
	 * @param height the image height on the viewer.
	 * @param file the file of the image to show on the viewer.
	 */
	public ImagePiece(int width, int height, File file) {
		fileOfImage = file;
		thumbnailSize.width = width;
		thumbnailSize.height = height;
		makeThumbnail();
	}
	
	/**
	 * Return the path to the image file as String.
	 * @return path to the image file.
	 */
	public String getPath(){
		if (fileOfImage==null){
			return null;
		}
		return fileOfImage.getPath();
	}
	
	private int[] getPixels(Image img){
		int[] pixels = new int [bounds.width*bounds.height];
		PixelGrabber grabber = new PixelGrabber(img,0,0,bounds.width,bounds.height, pixels, 0,bounds.width);
		try{
			grabber.grabPixels();
		}catch(Exception e){
			e.printStackTrace();
		}
		return pixels;
	}
	
	private void makeThumbnail(){
		Dimension size = SOSImageIO.getImageSize(fileOfImage);
		bounds = new Rectangle(0,0,size.width,size.height);
		int stepX = (int)Math.ceil(1.0*bounds.width/thumbnailSize.width);
		int stepY = (int)Math.ceil(1.0*bounds.height/thumbnailSize.height);
		int step = Math.max(stepX,stepY);
		
		thumbnail = null;
		if (step<=1){
			thumbnail = SOSImageIO.readImage(fileOfImage);
		}else{
			thumbnail= SOSImageIO.getSubsampledImage(fileOfImage, step,step);
		}
		hasImage = (thumbnail==null)?false:true;
	}
	
	/**
	 * Return a thumbnail (small image) of original image.
	 * If the thumbnail was not exist, it will be created and keeped.
	 * @return small image to show on the viewer.
	 */
	public Image getThumbnail(){
		if (thumbnail==null){
			makeThumbnail();
		}
		return thumbnail;
	}
	
	public void clearThumbnail(){
		thumbnail = null;
	}
	
	public boolean hasImage(){
		return hasImage;
	}
	
	public BufferedImage getImage(){
		BufferedImage img = null;
		try{
			img = ImageIO.read(fileOfImage);
		}catch(Exception e){
			e.printStackTrace();
		}
		return img;
	}
	
	public Rectangle getBounds(){
		return bounds;
	}
	
	public Dimension getSize(){
		return bounds.getSize();
	}
	
	public int[] getPixelsData(){
		if (pixelsData==null){
			Image img = getImage();
			pixelsData = ColorVector.intRGBtoL(getPixels(img));
		}
		
		return pixelsData;
	}
	
	public int[] getPixelsData(int step){
		if (pixelsData==null){
			Image img = SOSImageIO.readImage(fileOfImage);
			Image subsampledImg = img.getScaledInstance(bounds.width/step,bounds.height/step,Image.SCALE_AREA_AVERAGING);
			Image expandImg = subsampledImg.getScaledInstance(bounds.width,bounds.height, Image.SCALE_FAST);
			pixelsData = ColorVector.intRGBtoL(getPixels(expandImg));
		}
		
		return pixelsData;
	}
	
	public int[][] getRGB(int step){
		if (rgb==null){
			Image img = SOSImageIO.readImage(fileOfImage);
			Image subsampledImg = img.getScaledInstance(bounds.width/step,bounds.height/step,Image.SCALE_AREA_AVERAGING);
			//Image subsampledImg = SosIO.getSubsampledImage(fileOfImage, step,step);
			Image expandImg = subsampledImg.getScaledInstance(bounds.width,bounds.height, Image.SCALE_FAST);
			//new ImagePlus("step="+step, expandImg).show();
			rgb = new int[bounds.width*bounds.height][3];
			
			ColorVector.getRGB(getPixels(expandImg),rgb);
		}
		
		return rgb;
	}	
	
	public void clearRGB(){
		rgb = null;
	}
	
	public void clearPixelsData(){
		pixelsData = null;
	}
	
	public int getThumbnailOriginalRatio(){
		return bounds.width/thumbnailSize.width;
	}
	
	public boolean isFixed() {
		return fixed;
	}
	
	public void setFixed(boolean fixed) {
		this.fixed = fixed;
	}
	
	public int getIndex() {
		return tableIndex;
	}
	
	public void setIndex(int index) {
		this.tableIndex = index;
	}
	
	public Point getTableLocation() {
		return tableLocation;
	}
	
	public void setTableLocation(int x, int y) {
		tableLocation.setLocation(x,y);
	}

	public void draw(java.awt.Graphics g) {
		Graphics2D g2d = (Graphics2D)g;
		g2d.setColor(Color.yellow);
		g2d.draw(bounds);
	}
	
}
