package jp.sourceforge.sos.gunmetry.main;

import java.awt.AlphaComposite;
import java.awt.Color;
import java.awt.image.BufferedImage;
import java.io.File;
import java.util.ArrayList;

import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableModel;

import jp.sourceforge.sos.framework.IAbstraction;
import jp.sourceforge.sos.gunmetry.common.AnalysisPlugin;
import jp.sourceforge.sos.lib.canvas.JObjectCanvas;
import jp.sourceforge.sos.lib.canvas.JTransformCanvas;
import jp.sourceforge.sos.lib.colorspace.IntRGB;
import jp.sourceforge.sos.lib.graphics.GraphicsComposite;
import jp.sourceforge.sos.lib.graphics.GraphicsImage;
import jp.sourceforge.sos.lib.graphics.GraphicsOffscreen;
import jp.sourceforge.sos.lib.image.ImageInfo;
import jp.sourceforge.sos.lib.image.ImagePixelMediator;
import jp.sourceforge.sos.lib.io.Extension;
import jp.sourceforge.sos.lib.io.PluginClassLoader;
import jp.sourceforge.sos.lib.io.SOSFileIO;
import jp.sourceforge.sos.lib.io.SOSImageIO;
import jp.sourceforge.sos.lib.morphology.connection.Neighbor;

public class ModelMain implements IAbstraction {

	static final private String	pathAnalyze	= "plugins/analysis";

	static private SOSFileIO	textFileIO	= new SOSFileIO();

	private SOSImageIO			imageFileIO	= new SOSImageIO();

	private RegionSet			nuclei		= new RegionSet("nuclei");

	private RegionSet			skiz		= new RegionSet("skiz");

	private DefaultTableModel	tableModel;

	private JTransformCanvas	canvas;

	private GraphicsOffscreen	graphicsOffscreen;

	private GraphicsImage		graphicsImage;

	private GraphicsComposite	graphicComposite;

	@SuppressWarnings("serial")
	public ModelMain() {
		super();

		setupCanvas();

		tableModel = new DefaultTableModel(new Object[][]{}, new String[]{ "Variable", "Output" }) {
			Class[]		types	= new Class[]{ String.class, Boolean.class };

			boolean[]	canEdit	= new boolean[]{ false, true };

			@SuppressWarnings("unchecked")
			public Class getColumnClass(int columnIndex) {
				return types[columnIndex];
			}

			public boolean isCellEditable(int rowIndex, int columnIndex) {
				return canEdit[columnIndex];
			}
		};
		loadAnalysisPlugin();
	}

	/**
	 * 
	 */
	private void setupCanvas() {
		canvas = new JTransformCanvas();
		graphicsOffscreen = new GraphicsOffscreen();
		graphicsImage = new GraphicsImage();

		graphicComposite = new GraphicsComposite();
		graphicComposite.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, 0.5F));
		graphicsOffscreen.addGraphicObject(graphicComposite);
		graphicsOffscreen.addGraphicObject(graphicsImage);
		graphicsOffscreen.addGraphicObject(new GraphicsComposite(AlphaComposite.Src));

		canvas.addGraphicObject(graphicsOffscreen);
	}

	private void loadAnalysisPlugin() {
		File file = new File(pathAnalyze);
		File[] fileList = file.listFiles();
		if (fileList!=null) {
			tableModel.setRowCount(fileList.length);
			for (int i = 0; i<fileList.length; i++) {
				String algorithmName = Extension.remove(fileList[i].getName());
				tableModel.setValueAt(algorithmName, i, 0);
				tableModel.setValueAt(false, i, 1);
			}
		}
	}

	public RegionSet getNuclei() {
		return nuclei;
	}

	public RegionSet getSkiz() {
		return skiz;
	}

	void saveImage() {
		BufferedImage img = canvas.getScreenImage();
		imageFileIO.writeImage(img);
	}

	void saveResult() {
		ArrayList<ArrayList<Double>> result = new ArrayList<ArrayList<Double>>();
		ArrayList<String> variableName = new ArrayList<String>();
		PluginClassLoader loader = new PluginClassLoader(pathAnalyze);
		for (int i = 0; i<tableModel.getRowCount(); i++) {
			if ((Boolean)tableModel.getValueAt(i, 1)) {
				String algorithmName = (String)tableModel.getValueAt(i, 0);
				try {
					AnalysisPlugin plugin = (AnalysisPlugin)loader.loadClass(algorithmName).newInstance();
					result.add(plugin.analyze(nuclei.getRegion(), skiz.getRegion()));
				} catch (Exception e) {
					e.printStackTrace();
					return;
				}
				variableName.add(algorithmName);
			}
		}

		String[] strResult = new String[nuclei.size()+1];
		strResult[0] = "";
		for (int j = 0; j<variableName.size(); j++) {
			strResult[0] += variableName.get(j)+",";
		}
		for (int i = 1; i<strResult.length; i++) {
			strResult[i] = "";
			for (int j = 0; j<result.size(); j++) {
				strResult[i] += result.get(j).get(i-1)+",";
			}
		}

		textFileIO.resetChoosableFileFilters();
		textFileIO.writeStrings(strResult);
	}

	TableModel getTableModel() {
		return tableModel;
	}

	public JObjectCanvas getCanvas() {
		return canvas;
	}

	public void setNearestNeighbor(int[] distanceMap, ImageInfo imageInfo, int BACKGROUND) {
		Neighbor nf = imageInfo.getNeighbor();
		nf.set8();
		skiz.initMinDistance();
		int[] contourSkiz = skiz.getContour();
		for (int i = 0; i<contourSkiz.length; i++) {
			int pn = contourSkiz[i];
			int[] neighbor = nf.getDir(pn);
			findNeighbor(distanceMap, BACKGROUND+1, imageInfo.getPixelsLabel(), pn, neighbor);
		}
	}

	/**
	 * @param distanceMap
	 * @param BACKGROUND
	 * @param labels
	 * @param pn
	 * @param neighbor
	 */
	private void findNeighbor(int[] distanceMap, int FIRST_LABEL, int[] labels, int pn, int[] neighbor) {
		int half = neighbor.length/2;
		for (int n = 0; n<neighbor.length; n++) {
			if (neighbor[n]!=0) {
				int label = labels[pn+neighbor[n]]-FIRST_LABEL;
				if (0<=label&&distanceMap[pn+neighbor[n]]<skiz.getMinDistance(label)) {
					int contraNeighbor = pn+neighbor[ (n+half)%neighbor.length];
					int neighborLabel = labels[contraNeighbor]-FIRST_LABEL;
					if (pn!=contraNeighbor&&0<=neighborLabel) {
						skiz.setMinDistance(label, distanceMap[pn+neighbor[n]]);
						skiz.setNearestNeighbor(label, neighborLabel);
						nuclei.setNearestNeighbor(label, neighborLabel);
					}
				}
			}
		}
	}

	public void createImage(Color colorObject, Color colorBackground, ImageInfo imageInfo, int BACKGROUND) {
		IntRGB rgb = new IntRGB();
		int[] rgbObject = rgb.toRGB(colorObject.getRGB());
		int[] rgbBackground = rgb.toRGB(colorBackground.getRGB());

		int[] label = imageInfo.getPixelsLabel();
		int[][] result = new int[label.length][];
		for (int i = 0; i<result.length; i++) {
			if (label[i]>BACKGROUND) {
				result[i] = rgbObject;
			} else {
				result[i] = rgbBackground;
			}
		}
		BufferedImage img = ImagePixelMediator.toImage(result, imageInfo.getSize());
		graphicsImage.setImage(img);
		repaint();
	}

	void setAlpha(float value) {
		AlphaComposite composite = AlphaComposite.getInstance(AlphaComposite.SRC_OVER, value);
		graphicComposite.setComposite(composite);
		repaint();
	}

	private void repaint() {
		((GraphicsOffscreen)canvas.getGraphicObject(1)).update();
		canvas.repaint();
	}

	public void init() {
		nuclei.clear();
		skiz.clear();
	}

}
