package jp.sourceforge.sos.gunmetry.marker;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.KeyEventPostProcessor;
import java.awt.KeyboardFocusManager;
import java.awt.Point;
import java.awt.Shape;
import java.awt.event.KeyEvent;
import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.LinkedList;

import jp.sourceforge.sos.framework.IAbstraction;
import jp.sourceforge.sos.framework.IModel;
import jp.sourceforge.sos.gunmetry.common.Mediator;
import jp.sourceforge.sos.gunmetry.main.ModelMain;
import jp.sourceforge.sos.gunmetry.main.RegionSet;
import jp.sourceforge.sos.lib.canvas.JObjectCanvas;
import jp.sourceforge.sos.lib.color.ColorModel;
import jp.sourceforge.sos.lib.graphics.GraphicsRestoreTransform;
import jp.sourceforge.sos.lib.graphics.PointRect;
import jp.sourceforge.sos.lib.image.ImageInfo;
import jp.sourceforge.sos.lib.image.ROIRectangle;
import jp.sourceforge.sos.lib.io.SOSFileIO;
import jp.sourceforge.sos.lib.morphology.connection.Neighbor;
import jp.sourceforge.sos.lib.morphology.data.DataBinary;
import jp.sourceforge.sos.lib.morphology.data.Elements;
import jp.sourceforge.sos.lib.morphology.extra.GeodesicDistance;
import jp.sourceforge.sos.lib.morphology.extra.Watershed;
import jp.sourceforge.sos.lib.util.SOSArrays;

public class ModelMarker implements IModel {

	static final private int	BACKGROUND		= 0;

	static private SOSFileIO	textFileIO		= new SOSFileIO();

	private MarkerSet			object			= new MarkerSet("object");

	private MarkerSet			background		= new MarkerSet("background");

	private FocusedMarker		focusedMarker;

	private PointRect			pointRect		= new PointRect(0, 0, 5);

	private ROIRectangle		selectionRect	= new ROIRectangle();

	private MarkerColorPicker	colorPicker		= new MarkerColorPicker();

	private JObjectCanvas		canvas;

	private ModelMain			modelMain;

	private ImageInfo			imageInfo;

	private double[]			gradient;

	private LinkedList<Memento>	mementos		= new LinkedList<Memento>();

	public ModelMarker() {
		super();

		pointRect.setVisible(true);
		pointRect.setDrawBounds(true);

		object.setPointRect(pointRect);
		object.setColor(Color.red);
		background.setPointRect(pointRect);
		background.setColor(Color.yellow);

		focusedMarker = new FocusedMarker(object, background);

		setupKeyInput();

	}

	private void setupKeyInput() {
		// setup for key inputs
		KeyboardFocusManager keyboardManager = KeyboardFocusManager.getCurrentKeyboardFocusManager();
		keyboardManager.addKeyEventPostProcessor(new KeyEventPostProcessor() {
			public boolean postProcessKeyEvent(KeyEvent evt) {
				if (evt.getID()==KeyEvent.KEY_PRESSED) {
					int keyCode = evt.getKeyCode();
					if (keyCode==KeyEvent.VK_DELETE) {
						object.removeSelected();
						background.removeSelected();
						updateNumber();
					} else if (KeyEvent.VK_LEFT<=keyCode&&keyCode<=KeyEvent.VK_DOWN) {
						translateSelected(keyCode);
						repaint();
					}
				}
				return false;
			}
		});
	}

	/**
	 * @param dx
	 */
	private void translateSelected(int keyCode) {
		int dx = 0;
		int dy = 0;
		switch (keyCode) {
		case KeyEvent.VK_LEFT:
			dx = -1;
		break;
		case KeyEvent.VK_UP:
			dy = -1;
		break;
		case KeyEvent.VK_RIGHT:
			dx = 1;
		break;
		case KeyEvent.VK_DOWN:
			dy = 1;
		break;
		}
		object.translateSelected(dx, dy);
		background.translateSelected(dx, dy);

		setMemento();
	}

	public void setAbstraction(IAbstraction abstraction) {
		modelMain = (ModelMain)abstraction;

		canvas = modelMain.getCanvas();
		canvas.addGraphicObject(object);
		canvas.addGraphicObject(background);
		canvas.addGraphicObject(new GraphicsRestoreTransform());
		canvas.addGraphicObject(selectionRect);
	}

	void clearSelection() {
		object.clearSelection();
		background.clearSelection();
	}

	void selectFromFocus() {
		object.select(focusedMarker.get());
		background.select(focusedMarker.get());
		repaint();
	}

	int getBackgroundNumber() {
		return background.size();
	}

	int getObjectNumber() {
		return object.size();
	}

	/**
	 * @param rect
	 */
	void select(Shape rect) {
		object.select(rect);
		background.select(rect);
		selectionRect.setVisible(false);
		repaint();
	}

	void setBackgroundColor(Color color) {
		background.setColor(color);
		repaint();
	}

	void setObjectColor(Color color) {
		object.setColor(color);
		repaint();
	}

	MarkerSet getObject() {
		return object;
	}

	boolean isFocusActive() {
		return focusedMarker.isActive();
	}

	void setFocusPoint(Point2D p) {
		int x = (int)p.getX();
		int y = (int)p.getY();
		if (x<0){
			x = 0;
		}else if (imageInfo.getWidth()<=x){
			x = imageInfo.getWidth()-1;
		}
		if (y<0){
			y = 0;
		}else if (imageInfo.getHeight()<=y){
			y = imageInfo.getHeight()-1;
		}
		
		focusedMarker.setLocation(x, y);
		focusedMarker.addToMarkers();
		repaint();
		setMemento();
	}

	void removeFocus() {
		focusedMarker.removeFromMarkers();
		repaint();
	}

	boolean isSameFocus(Point2D mousePoint2D) {
		Point preFocus = focusedMarker.get();
		focusedMarker.set(mousePoint2D);
		return focusedMarker.equals(preFocus);
	}

	void repaint() {
		canvas.repaint();
	}

	private void updateNumber() {
		checkReadyToAnalyze();
		setMemento();
		repaint();
	}

	ColorModel getObjColorModel() {
		return object.getColorModel();
	}

	ColorModel getBackColorModel() {
		return background.getColorModel();
	}

	public void setColorPicker(ImageInfo imiCluster) {
		int[] colorCluster = imiCluster.getPixelsLabel();
		colorPicker.setColorTable(imiCluster.getPixels(), colorCluster);

		// after dialog closed
		boolean[] flag = colorPicker.getFlag();
		setMarkers(flag, colorCluster);
	}

	private void setMarkers(boolean[] flag, int[] colorCluster) {
		boolean[] binaryMap = SOSArrays.toBoolean(flag, colorCluster);
		object.setAll(colorPicker.extractMarkerObj(imageInfo.getWidth(), imageInfo.getHeight(), binaryMap));

		// reverse binaryMap to extract background markers
		for (int i = 0; i<binaryMap.length; i++) {
			binaryMap[i] = !binaryMap[i];
		}
		background.setAll(colorPicker.extractMarkerBack(imageInfo.getWidth(), imageInfo.getHeight(), binaryMap));

		updateNumber();
	}

	private void checkReadyToAnalyze() {
		boolean isReady = (0<getObjectNumber()&&0<getBackgroundNumber());
		Mediator.inform(Mediator.Subjects.READY_TO_ANALYSIS, isReady);
	}

	public JObjectCanvas getCanvas() {
		return canvas;
	}

	void setPointRectSize(int value) {
		pointRect.setSize(value);
		repaint();
	}

	void saveMarkers() {
		ArrayList<String> data = new ArrayList<String>();
		data.add(imageInfo.getWidth()+","+imageInfo.getHeight());
		data.add(Integer.toString(getObjectNumber()+2));
		addToData(data, object);
		addToData(data, background);
		textFileIO.resetChoosableFileFilters();
		textFileIO.writeStrings(data);
	}

	private void addToData(ArrayList<String> data, MarkerSet markers) {
		for (int i = 0; i<markers.size(); i++) {
			Point p = markers.get(i);
			String arg = p.x+","+p.y;
			data.add(arg);
		}
	}

	void openMarker() {
		ArrayList<String> data = textFileIO.readStrings();
		if (data==null || 0==data.size()) {
			return;
		}

		// get the image size at when save the data
		String[] size = data.get(0).split(",");
		Dimension dim = new Dimension();
		double w = 1.0*imageInfo.getWidth()/Integer.valueOf(size[0]);
		double h = 1.0*imageInfo.getHeight()/Integer.valueOf(size[1]);
		dim.setSize(w, h);

		int nObject = Integer.valueOf(data.get(1));
		add(object, data, 2, nObject, dim);
		add(background, data, nObject, data.size(), dim);

		updateNumber();
	}

	private void add(MarkerSet marker, ArrayList<String> data, int start, int end, Dimension d) {
		marker.clear();
		for (int i = start; i<end; i++) {
			String[] location = data.get(i).split(",");
			int x = (int) (Integer.valueOf(location[0])*d.getWidth());
			int y = (int) (Integer.valueOf(location[1])*d.getHeight());
			marker.add(new Point(x, y));
		}
	}

	MarkerSet getBackground() {
		return background;
	}

	ROIRectangle getSelectionRect() {
		return selectionRect;
	}

	void doWatershed() {
		int INIT = BACKGROUND-4;
		setLabel(INIT);

		Watershed watershed = new Watershed(imageInfo, BACKGROUND);
		watershed.setOrder(gradient);
		flood(watershed);
		setRegion(modelMain.getNuclei(), watershed);

		modelMain.createImage(object.getColor(), background.getColor(), imageInfo, BACKGROUND);

		// make skiz
		int[] dist = makeBackgroungMap(INIT);

		watershed = new Watershed(imageInfo, BACKGROUND+1);
		watershed.setOrder(dist);
		flood(watershed);
		setRegion(modelMain.getSkiz(), watershed);

		modelMain.setNearestNeighbor(dist, imageInfo, BACKGROUND);

		Mediator.inform(Mediator.Subjects.POST_ANALYSIS, Boolean.TRUE);
	}

	private void setLabel(int init) {
		imageInfo.fillLabel(init);
		int count = BACKGROUND+1;
		boolean[] mask = imageInfo.getNeighbor().getMask();
		if (mask==null) {
			setLabelSimple(count);
		} else {
			setLabelMasked(count, mask);
		}
	}

	/**
	 * @param count
	 * @param mask
	 */
	private void setLabelMasked(int count, boolean[] mask) {
		for (int i = 0; i<object.size(); i++) {
			int pn = imageInfo.pointToOffset(object.get(i));
			if (0<=pn && !mask[pn]) {
				imageInfo.setPixelsLabel(pn, count);
				count++;
			}
		}
		for (int i = 0; i<background.size(); i++) {
			int pn = imageInfo.pointToOffset(background.get(i));
			if (0<=pn && !mask[pn]) {
				imageInfo.setPixelsLabel(pn, BACKGROUND);
			}
		}
	}

	/**
	 * @param count
	 */
	private void setLabelSimple(int count) {
		for (int i = 0; i<object.size(); i++) {
			imageInfo.setPixelsLabel(object.get(i), count);
			count++;
		}
		for (int i = 0; i<background.size(); i++) {
			imageInfo.setPixelsLabel(background.get(i), BACKGROUND);
		}
	}

	/**
	 * @param watershed
	 */
	private void setRegion(RegionSet regions, Watershed watershed) {
		int[] contour = watershed.getWatershed();
		regions.setContour(contour);
		regions.setContourPoints(convertToPoints(contour));
		regions.setRegion(object, imageInfo, BACKGROUND);
	}

	private Point[] convertToPoints(int[] contour) {
		Point[] result = new Point[contour.length];
		int w = imageInfo.getWidth();
		for (int i = 0; i<result.length; i++) {
			result[i] = new Point(contour[i]%w, contour[i]/w);
		}
		return result;
	}

	/**
	 * @param INIT
	 * @return
	 */
	private int[] makeBackgroungMap(int INIT) {
		int[] labels = imageInfo.getPixelsLabel();
		boolean[] backgroundMap = new boolean[labels.length];
		for (int pn = 0; pn<labels.length; pn++) {
			if (labels[pn]>BACKGROUND) {
				backgroundMap[pn] = false;
			} else {
				labels[pn] = INIT;
				backgroundMap[pn] = true;
			}
		}
		Neighbor neighbor = imageInfo.getNeighbor();
		neighbor.set4();
		GeodesicDistance distance = new GeodesicDistance();
		Elements elements = new Elements(neighbor, new DataBinary(backgroundMap));
		distance.operate(elements);
		return (int[])elements.getResultData();
	}

	/**
	 * @param watershed
	 */
	private void flood(Watershed watershed) {
		do {
			watershed.flood4();
		} while (!watershed.isConvergence());
		// watershed.undeterminedToWatershed();
	}

	public void setImageInfo(ImageInfo imageInfo) {
		this.imageInfo = imageInfo;
	}

	public void setGradient(double[] gradient) {
		this.gradient = gradient;
	}

	public int getImageWidth() {
		return imageInfo.getWidth();
	}

	public int getImageHeight() {
		return imageInfo.getHeight();
	}

	private void setMemento() {
		mementos.add(new Memento(object, background));
	}

	void restoreMemento() {
		if (!mementos.isEmpty()) {
			mementos.removeLast();// the Last memento is the same as the
			// current state
			Memento memento = mementos.getLast();
			memento.restore(object, background);
			repaint();
			checkReadyToAnalyze();
		}
	}

	void addPoint(MarkerSet currentMarker, Point2D p2D) {
		Point p = new Point((int)p2D.getX(), (int)p2D.getY());
		if (p.x<0 || imageInfo.getWidth()<=p.x || p.y<0 || imageInfo.getHeight()<=p.y){
			return;
		}
		currentMarker.add(p);
		updateNumber();
	}

	public ImageInfo getImageInfo() {
		return imageInfo;
	}

	public void init() {
		object.clear();
		background.clear();
		modelMain.init();
	}

}
