/************************************************************************
*									*
*    Adaptive Mel Cepstral Analysis					*
*									*
*					1990    T.Fukada		*
*					1996.1  K.Koishida		*
*									*
*	usage:								*
*		amcep [ options ] [ pefile ] < stdin > stdout		*
*	options:							*
*		-m m     :  order of mel cepstrum		[25]	*
*		-a a     :  alpha				[0.35]	*
*		-l l     :  leakage factor			[0.98]	*
*		-k k     :  step size				[0.1]	*
*		-t t     :  momentum constant			[0.9]	*
*		-p p     :  output period of mel cepstrum	[1]	*
*		-e e     :  minimum value for epsilon		[FALSE]	*
*		-s       :  smoothed (average) mel cepstrum	[FALSE]	*
*		-P P     :  order of pade approximation		[4]	*
*	infile:								*
*		data sequence						*
*		    , x(0), x(1), ...					*
*	stdout:								*
*		mel cepstrum						*
*		    , c~(0), c~(1), ..., c~(M),				*
*	output:								*
*		prediction error (if pefile is specified)		*
*		    , e(0), e(1), ...					*
*	note:								*
*		P = 4 or 5						*
*	require:							*
*		mlsadf()						*
*									*
************************************************************************/

static char *rcs_id = "$Id: amcep.c,v 1.1.1.1 2000/03/01 13:58:28 yossie Exp $";


/*  Standard C Libraries  */
#include <stdio.h>
#include <string.h>
#include <SPTK.h>


typedef enum _Boolean {FA, TR} Boolean;
char *BOOL[] = {"FALSE", "TRUE"};


/*  Required Functions  */
double	mlsadf(), log();


/*  Default Values  */
#define	ALPHA		0.35
#define ORDER		25
#define LAMBDA		0.98
#define TAU		0.9
#define STEP		0.1
#define PERIOD		1
#define AVEFLAG		FA
#define PEFLAG		FA
#define EPS		0.0
#define PADEORDER	4


/*  Command Name  */
char	*cmnd;


void usage(int status)
{
    fprintf(stderr, "\n");
    fprintf(stderr, " %s - adaptive mel cepstral analysis\n",cmnd);
    fprintf(stderr, "\n");
    fprintf(stderr, "  usage:\n");
    fprintf(stderr, "       %s [ options ] [ pefile ] < stdin > stdout\n", cmnd);
    fprintf(stderr, "  options:\n");
    fprintf(stderr, "       -m m  : order of mel cepstrum         [%d]\n", ORDER);
    fprintf(stderr, "       -a a  : all-pass constant             [%g]\n", ALPHA);
    fprintf(stderr, "       -l l  : leakage factor                [%g]\n", LAMBDA);
    fprintf(stderr, "       -t t  : momentum constant             [%g]\n", TAU);
    fprintf(stderr, "       -k k  : step size                     [%g]\n", STEP);
    fprintf(stderr, "       -p p  : output period of mel cepstrum [%d]\n", PERIOD);
    fprintf(stderr, "       -s    : output smoothed mel cepstrum  [%s]\n", BOOL[AVEFLAG]);
    fprintf(stderr, "       -e e  : minimum value for epsilon     [%g]\n", EPS);
    fprintf(stderr, "       -P P  : order of pade approximation   [%d]\n", PADEORDER);
    fprintf(stderr, "       -h    : print this message\n");
    fprintf(stderr, "  stdin:\n");
    fprintf(stderr, "       data sequence (float)\n");
    fprintf(stderr, "  stdout:\n");
    fprintf(stderr, "       mel-cepstrum (float)\n");
    fprintf(stderr, "  pefile:\n");
    fprintf(stderr, "       prediction error (float)\n");
    fprintf(stderr, "  note:\n");
    fprintf(stderr, "       P = 4 or 5\n");
    fprintf(stderr, "\n");
    exit(status);
}


void main(int argc, char **argv)
{
    int	     m = ORDER, period = PERIOD, pd = PADEORDER, i, j;
    FILE     *fp = stdin, *fpe = NULL;
    Boolean  aveflag = AVEFLAG;
    double   alpha = ALPHA, lambda = LAMBDA, tau = TAU, step = STEP,eps = EPS,
             *mc, *b, *ep, *e, *bb, *d, *avemc, 
             x, xx, ll, tt, gg, mu, ttx, atof();

    if ((cmnd = strrchr(argv[0], '/')) == NULL)
	cmnd = argv[0];
    else
	cmnd++;
    while (--argc)
	if (**++argv == '-') {
	    switch (*(*argv+1)) {
	        case 'a':
		    alpha = atof(*++argv);
		    --argc;
		    break;
	        case 'l':
		    lambda = atof(*++argv);
		    --argc;
		    break;
	        case 't':
		    tau = atof(*++argv);
		    --argc;
		    break;
	        case 'k':
		    step = atof(*++argv);
		    --argc;
		    break;
		case 'm':
		    m = atoi(*++argv);
		    --argc;
		    break;
		case 'p':
		    period = atoi(*++argv);
		    --argc;
		    break;
		case 'P':
		    pd = atoi(*++argv);
		    --argc;
		    break;
		case 's':
		    aveflag = 1 - aveflag;
		    break;
		case 'e':
		    eps = atof(*++argv);
		    --argc;
		    break;
		case 'h':
		    usage(0);
		default:
		    fprintf(stderr, "%s : Invalid option '%c' !\n", cmnd, *(*argv+1));
		    usage(1);
		}
	}
	else 
	    fpe = getfp(*argv, "w");

    if ((pd < 4) || (pd > 5)){
	fprintf(stderr,"%s : Order of pade approximation is 4 or 5!\n",cmnd);
	exit(1);
    }

    mc = dgetmem(6*(m+1)+3*(pd+1)+pd*(m+2));
    b  = mc + m + 1;
    bb = b  + m + 1;
    e  = bb + m + 1;
    ep = e  + m + 1;
    avemc = ep + m + 1;
    d = avemc  + m + 1;

    j  = period;
    ll = 1.0 - lambda;
    gg = 1.0;
    tt = 2 * (1.0 - tau);
    step /= (double) m;
    xx = 0.0;
    
    while (freadf(&x, sizeof(x), 1, fp) == 1){
	for(i=1; i<=m; i++) 
	    bb[i] = -b[i];

	x = mlsadf(x, bb, m, alpha, pd, d);
	phidf(xx, m, alpha, e);
	xx = x;

	gg = gg * lambda + ll * x * x;
	gg = ( gg < eps )? eps : gg;
	b[0] = 0.5 * log(gg);

	mu = step / gg;
	ttx = tt * x;

	for(i=1; i<=m; i++){
	    ep[i] = tau * ep[i] - ttx * e[i];
	    b[i] -= mu * ep[i];
	}
	
	b2mc(b, mc, m, alpha);

	if(aveflag)
	    for(i=0; i<=m; i++)
		avemc[i] += mc[i];

	if (fpe != NULL)
	    fwritef(&x, sizeof(x), 1, fpe);
	
	if (--j == 0){
	    j = period;
	    if (aveflag){
		for(i=0; i<=m; i++)
		    avemc[i] /= period;
		fwritef(avemc, sizeof(*avemc), m+1, stdout);
		fillz(avemc, sizeof(*avemc), m+1);
	    }
	    else
		fwritef(mc, sizeof(*mc), m+1, stdout);
	}
    }
    exit(0);
}

