package com.ozacc.springframework.mail.javamail;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Date;

import javax.activation.DataHandler;
import javax.activation.DataSource;
import javax.mail.BodyPart;
import javax.mail.MessagingException;
import javax.mail.Part;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;

import org.springframework.mail.javamail.MimeMessageHelper;

/**
 * 
 * @author Tomohiro Otsuka
 * @version $Id: IntlMimeMessageHelper.java,v 1.1 2004/02/24 10:30:00 otsuka Exp $
 */
public class IntlMimeMessageHelper extends MimeMessageHelper {

	protected final static String DEFAULT_CHARSET = "iso-2022-jp";

	protected String charset = DEFAULT_CHARSET;

	protected boolean multipart = false;

	/**
	 * 
	 * @param mimeMessage
	 * @throws MessagingException
	 */
	public IntlMimeMessageHelper(MimeMessage mimeMessage) throws MessagingException {
		super(mimeMessage);

		setHeader();
	}

	/**
	 * 
	 * @param mimeMessage
	 * @param charset
	 * @throws MessagingException
	 */
	public IntlMimeMessageHelper(MimeMessage mimeMessage, String charset)
		throws MessagingException {
		super(mimeMessage);
		this.charset = charset;

		setHeader();
	}

	/**
	 * @param mimeMessage
	 * @param multipart
	 * @throws MessagingException 
	 */
	public IntlMimeMessageHelper(MimeMessage mimeMessage, boolean multipart)
		throws MessagingException {
		super(mimeMessage, multipart);
		this.multipart = multipart;

		setHeader();
	}

	/**
	 * @param mimeMessage
	 * @param multipart
	 * @throws MessagingException 
	 */
	public IntlMimeMessageHelper(MimeMessage mimeMessage, boolean multipart, String charset)
		throws MessagingException {
		super(mimeMessage, multipart);
		this.multipart = multipart;
		this.charset = charset;

		setHeader();
	}

	/**
	 * Content-Transfer-Encoding򥻥åȤޤ
	 * @throws MessagingException
	 */
	private void setHeader() throws MessagingException {
		getMimeMessage().setSentDate(new Date());

		if (charset.equalsIgnoreCase("iso-2022-jp"))
			getMimeMessage().setHeader("Content-Transfer-Encoding", "7bit");
	}

	/**
	 * ʸɤꤷƷ̾򥻥åȤޤ
	 * @see org.springframework.mail.javamail.MimeMessageHelper#setSubject(java.lang.String)
	 */
	public void setSubject(String subject) throws MessagingException {
		getMimeMessage().setSubject(subject, charset);
	}

	/**
	 * Sets the given text directly as content in non-multipart mode
	 * respectively as default body part in multipart mode.
	 * @param text text to set
	 * @param html whether to apply content type "text/html" for an
	 * HTML mail, using default content type ("text/plain") else
	 */
	public void setText(final String text, boolean html) throws MessagingException {
		Part partToUse = null;

		MimeMultipart mimeMultipart = null;
		if (multipart) {
			try {
				mimeMultipart = (MimeMultipart)getMimeMessage().getContent();
			} catch (IOException e) {
				throw new MessagingException(e.getMessage());
			}
		}

		if (mimeMultipart != null) {
			BodyPart bodyPart = null;
			for (int i = 0; i < mimeMultipart.getCount(); i++) {
				BodyPart bp = mimeMultipart.getBodyPart(i);
				if (bp.getFileName() == null) {
					bodyPart = bp;
				}
			}
			if (bodyPart == null) {
				MimeBodyPart mimeBodyPart = new MimeBodyPart();
				mimeMultipart.addBodyPart(mimeBodyPart);
				bodyPart = mimeBodyPart;
			}
			partToUse = bodyPart;
		} else {
			partToUse = getMimeMessage();
		}
		if (html) {
			// need to use a javax.activation.DataSource (!) to set a text
			// with content type "text/html"
			partToUse.setDataHandler(new DataHandler(new DataSource() {
				public InputStream getInputStream() throws IOException {
					return new ByteArrayInputStream(text.getBytes());
				}
				public OutputStream getOutputStream() throws IOException {
					throw new UnsupportedOperationException("Read-only javax.activation.DataSource");
				}
				public String getContentType() {
					return "text/html";
				}
				public String getName() {
					return "text";
				}
			}));
		} else {
			getMimeMessage().setText(text, charset);
		}
	}

}
