package com.ozacc.mail.impl;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;
import org.apache.velocity.exception.MethodInvocationException;
import org.apache.velocity.exception.ParseErrorException;
import org.apache.velocity.exception.ResourceNotFoundException;
import org.apache.velocity.runtime.log.LogSystem;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.output.XMLOutputter;

import com.ozacc.mail.Mail;
import com.ozacc.mail.MailBuildException;
import com.ozacc.mail.VelocityMailBuilder;

/**
 * <a href="http://www.jdom.org/">JDOM</a>ѤXMLե뤫Mail󥹥󥹤륯饹
 * <p>
 * XMLɤ߹ݤˡDTDХǡ󤬼¹ԤޤΤXMLǡ(Valid XML Document)ǤʤФޤ
 * 
 * @since 1.0
 * 
 * @author Tomohiro Otsuka
 * @version $Id: JDomXMLMailBuilder.java,v 1.10.2.3 2005/01/17 15:44:21 otsuka Exp $
 */
public class JDomXMLMailBuilder implements VelocityMailBuilder {

	private static Log log = LogFactory.getLog(JDomXMLMailBuilder.class);

	protected LogSystem velocityLogSystem = new VelocityLogSystem();

	private boolean cacheEnabled = false;

	protected Map templateCache = new HashMap();

	/**
	 * 󥹥ȥ饯
	 */
	public JDomXMLMailBuilder() {}

	/**
	 * ꤵ줿饹ѥXMLե뤫Mail󥹥󥹤ޤ
	 * 
	 * @param classPath ᡼Ƥ򵭽ҤXMLեΥѥ
	 * @return 줿Mail󥹥
	 * @throws MailBuildException Mail󥹥󥹤˼Ԥ
	 */
	public Mail buildMail(String classPath) throws MailBuildException {
		Document doc = getDocumentFromClassPath(classPath);
		return build(doc);
	}

	/**
	 * ꤵ줿饹ѥXMLե뤫Mail󥹥󥹤ޤ
	 * ꤵ줿VelocityContextȤäơXMLեƤưŪǤޤ
	 * 
	 * @param classPath ᡼Ƥ򵭽ҤXMLեΥѥ
	 * @param context VelocityContext
	 * @return 줿Mail󥹥
	 * @throws MailBuildException Mail󥹥󥹤˼Ԥ
	 */
	public Mail buildMail(String classPath, VelocityContext context) throws MailBuildException {
		String templateXmlText;
		if (!hasTemplateCache(classPath)) {
			Document doc = getDocumentFromClassPath(classPath);
			XMLOutputter output = new XMLOutputter();
			templateXmlText = output.outputString(doc);
			putTemplateCache(classPath, templateXmlText);
		} else {
			templateXmlText = getTemplateCache(classPath);
		}

		try {
			return build(templateXmlText, context);
		} catch (Exception e) {
			throw new MailBuildException("᡼˼Ԥޤ", e);
		}
	}

	/**
	 * ꤵ줿XMLե뤫Mail󥹥󥹤ޤ
	 * 
	 * @param file ᡼Ƥ򵭽ҤXMLե
	 * @return 줿Mail󥹥
	 * @throws MailBuildException Mail󥹥󥹤˼Ԥ
	 */
	public Mail buildMail(File file) throws MailBuildException {
		Document doc = getDocumentFromFile(file);
		return build(doc);
	}

	/**
	 * ꤵ줿XMLե뤫Mail󥹥󥹤ޤ
	 * ꤵ줿VelocityContextȤäơXMLեƤưŪǤޤ
	 * 
	 * @param file ᡼Ƥ򵭽ҤXMLե
	 * @param context VelocityContext
	 * @return 줿Mail󥹥
	 * @throws MailBuildException Mail󥹥󥹤˼Ԥ
	 */
	public Mail buildMail(File file, VelocityContext context) throws MailBuildException {
		String templateXmlText;
		if (!hasTemplateCache(file.getAbsolutePath())) {
			Document doc = getDocumentFromFile(file);
			XMLOutputter output = new XMLOutputter();
			templateXmlText = output.outputString(doc);
			putTemplateCache(file.getAbsolutePath(), templateXmlText);
		} else {
			templateXmlText = getTemplateCache(file.getAbsolutePath());
		}

		try {
			return build(templateXmlText, context);
		} catch (Exception e) {
			throw new MailBuildException("᡼˼Ԥޤ", e);
		}
	}

	/**
	 * ꤵ줿饹ѥΥեɤ߹ǡXMLɥȤޤ
	 * 
	 * @param classPath
	 * @return JDOM Document
	 */
	protected Document getDocumentFromClassPath(String classPath) throws MailBuildException {
		InputStream is = getClass().getResourceAsStream(classPath);
		SAXBuilder builder = new SAXBuilder(true);
		builder.setEntityResolver(new DTDEntityResolver());
		Document doc;
		try {
			doc = builder.build(is);
			is.close();
		} catch (JDOMException e) {
			throw new MailBuildException("XMLΥѡ˼Ԥޤ" + e.getMessage(), e);
		} catch (IOException e) {
			throw new MailBuildException("XMLեɤ߹ߤ˼Ԥޤ", e);
		}
		return doc;
	}

	/**
	 * ꤵ줿եɤ߹ǡXMLɥȤޤ
	 * 
	 * @param file
	 * @return JDOM Document
	 */
	protected Document getDocumentFromFile(File file) {
		SAXBuilder builder = new SAXBuilder(true);
		builder.setEntityResolver(new DTDEntityResolver());
		Document doc;
		try {
			doc = builder.build(file);
		} catch (JDOMException e) {
			throw new MailBuildException("XMLΥѡ˼Ԥޤ" + e.getMessage(), e);
		} catch (IOException e) {
			throw new MailBuildException("XMLեɤ߹ߤ˼Ԥޤ", e);
		}
		return doc;
	}

	/**
	 * XMLɥȤMail󥹥󥹤ޤ
	 * 
	 * @param doc
	 * @return Mail
	 */
	protected Mail build(Document doc) {
		Element root = doc.getRootElement();

		Mail mail = new Mail();
		setFrom(root, mail);
		setRecipients(root, mail);
		setSubject(root, mail);
		setBody(root, mail);
		setReplyTo(root, mail);
		setReturnPath(root, mail);

		setHtml(root, mail);

		return mail;
	}

	/**
	 * VelocityContextXMLƥץ졼ȤޡMail󥹥󥹤ޤ
	 * 
	 * @param templateText ޡXMLƥץ졼Ȥʸ
	 * @param context ޡVelocityContext
	 * @return Mail
	 * 
	 * @throws Exception
	 * @throws ParseErrorException
	 * @throws MethodInvocationException
	 * @throws ResourceNotFoundException
	 * @throws IOException
	 * @throws JDOMException 
	 */
	protected Mail build(String templateText, VelocityContext context) throws Exception,
																		ParseErrorException,
																		MethodInvocationException,
																		ResourceNotFoundException,
																		IOException, JDOMException {
		if (log.isDebugEnabled()) {
			log.debug("Source XML Mail Data\n" + templateText);
		}

		Velocity.setProperty(Velocity.RUNTIME_LOG_LOGSYSTEM, velocityLogSystem);
		Velocity.init();
		StringWriter w = new StringWriter();
		Velocity.evaluate(context, w, "XML Mail Data", templateText);

		if (log.isDebugEnabled()) {
			String newXmlContent = w.toString();
			log.debug("VelocityContext-merged XML Mail Data\n" + newXmlContent);
		}

		StringReader reader = new StringReader(w.toString());
		SAXBuilder builder = new SAXBuilder(true);
		builder.setEntityResolver(new DTDEntityResolver());
		Document doc2 = builder.build(reader);

		return build(doc2);
	}

	/**
	 * @param root
	 * @param mail 
	 */
	protected void setReturnPath(Element root, Mail mail) {
		Element returnPathElem = root.getChild("returnPath");
		if (returnPathElem != null && returnPathElem.getAttributeValue("email") != null) {
			mail.setReturnPath(returnPathElem.getAttributeValue("email"));
		}
	}

	/**
	 * @param root
	 * @param mail 
	 */
	protected void setReplyTo(Element root, Mail mail) {
		Element replyToElem = root.getChild("replyTo");
		if (replyToElem != null && replyToElem.getAttributeValue("email") != null) {
			mail.setReplyTo(replyToElem.getAttributeValue("email"));
		}
	}

	/**
	 * @param root
	 * @param mail 
	 */
	protected void setBody(Element root, Mail mail) {
		Element bodyElem = root.getChild("body");
		if (bodyElem != null) {
			mail.setText(bodyElem.getTextTrim());
		}
	}

	/**
	 * @param root
	 * @param mail
	 */
	protected void setHtml(Element root, Mail mail) {
		Element htmlElem = root.getChild("html");
		if (htmlElem != null) {
			mail.setHtmlText(htmlElem.getTextTrim());
		}
	}

	/**
	 * @param root
	 * @param mail 
	 */
	protected void setSubject(Element root, Mail mail) {
		Element subjectElem = root.getChild("subject");
		if (subjectElem != null) {
			mail.setSubject(subjectElem.getTextTrim());
		}
	}

	/**
	 * @param root
	 * @param mail 
	 */
	protected void setRecipients(Element root, Mail mail) {
		Element recipientsElem = root.getChild("recipients");
		if (recipientsElem == null) {
			return;
		}

		List recipientElemList = recipientsElem.getChildren();
		for (int i = 0, max = recipientElemList.size(); i < max; i++) {
			Element e = (Element)recipientElemList.get(i);
			if ("to".equals(e.getName())) { // to
				if (e.getAttributeValue("email") != null) {
					if (e.getAttributeValue("name") != null) {
						mail.addTo(e.getAttributeValue("email"), e.getAttributeValue("name"));
					} else {
						mail.addTo(e.getAttributeValue("email"));
					}
				}
			} else if ("cc".equals(e.getName())) { // cc
				if (e.getAttributeValue("email") != null) {
					if (e.getAttributeValue("name") != null) {
						mail.addCc(e.getAttributeValue("email"), e.getAttributeValue("name"));
					} else {
						mail.addCc(e.getAttributeValue("email"));
					}
				}
			} else {
				if (e.getAttributeValue("email") != null) { // bcc
					mail.addBcc(e.getAttributeValue("email"));
				}
			}
		}
	}

	/**
	 * @param root
	 * @param mail 
	 */
	protected void setFrom(Element root, Mail mail) {
		Element fromElem = root.getChild("from");
		if (fromElem != null && fromElem.getAttributeValue("email") != null) {
			if (fromElem.getAttributeValue("name") != null) {
				mail.setFrom(fromElem.getAttributeValue("email"), fromElem
						.getAttributeValue("name"));
			} else {
				mail.setFrom(fromElem.getAttributeValue("email"));
			}
		}
	}

	/**
	 * @see com.ozacc.mail.VelocityMailBuilder#clearCache()
	 */
	public synchronized void clearCache() {
		log.debug("ƥץ졼ȥå򥯥ꥢޤ");
		templateCache.clear();
	}

	/**
	 * @see com.ozacc.mail.VelocityMailBuilder#isCacheEnabled()
	 */
	public boolean isCacheEnabled() {
		return cacheEnabled;
	}

	/**
	 * @see com.ozacc.mail.VelocityMailBuilder#setCacheEnabled(boolean)
	 */
	public void setCacheEnabled(boolean cacheEnabled) {
		if (!cacheEnabled) {
			clearCache();
		}
		this.cacheEnabled = cacheEnabled;
	}

	protected boolean hasTemplateCache(String key) {
		if (cacheEnabled) {
			return templateCache.containsKey(key);
		}
		return false;
	}

	protected void putTemplateCache(String key, String templateXmlText) {
		if (cacheEnabled) {
			log.debug("ƥץ졼Ȥ򥭥å夷ޤ[key='" + key + "']");
			templateCache.put(key, templateXmlText);
		}
	}

	protected String getTemplateCache(String key) {
		if (hasTemplateCache(key)) {
			log.debug("ƥץ졼ȥå֤ޤ[key='" + key + "']");
			return (String)templateCache.get(key);
		}
		return null;
	}

}