package com.ozacc.blog.feed.impl;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import javax.swing.text.MutableAttributeSet;
import javax.swing.text.html.HTML;
import javax.swing.text.html.HTMLEditorKit;
import javax.swing.text.html.HTML.Tag;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.HttpRecoverableException;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.HttpConnection.ConnectionTimeoutException;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.ozacc.blog.feed.FeedAutoDiscovery;
import com.ozacc.blog.feed.ParseException;
import com.ozacc.blog.feed.ParseTimeoutException;
import com.ozacc.blog.util.CommonsHttpClientUtils;
import com.ozacc.blog.util.DefaultHTMLParser;

/**
 * FeedAutoDiscovery インターフェースの実装クラス。
 * 
 * @since 1.0
 * @author Tomohiro Otsuka
 * @version $Id: FeedAutoDiscoveryImpl.java 197 2005-08-10 01:43:23Z otsuka $
 */
public class FeedAutoDiscoveryImpl implements FeedAutoDiscovery {

	private static Log log = LogFactory.getLog(FeedAutoDiscoveryImpl.class);

	public static final int DEFAULT_CONNECTION_TIMEOUT = 5000;

	public static final int DEFAULT_READ_TIMEOUT = 5000;

	private int connectionTimeout = DEFAULT_CONNECTION_TIMEOUT;

	private int readTimeout = DEFAULT_READ_TIMEOUT;

	private boolean followRedirect = true;

	/**
	 * @see com.ozacc.blog.feed.FeedAutoDiscovery#discoverFeedUrls(java.net.URL)
	 */
	public URL[] discoverFeedUrls(URL url) throws ParseException {
		return discoverFeedUrls(url.toString());
	}

	/**
	 * @see com.ozacc.blog.feed.FeedAutoDiscovery#discoverFeedUrls(java.lang.String)
	 */
	public URL[] discoverFeedUrls(String url) throws ParseException {
		HttpClient client = new HttpClient();
		log.debug("接続タイムアウトを設定します。[" + connectionTimeout + "]");
		client.setConnectionTimeout(connectionTimeout);
		log.debug("読込タイムアウトを設定します。[" + readTimeout + "]");
		client.setTimeout(readTimeout);

		HttpMethod method = new GetMethod(url);
		method.setFollowRedirects(followRedirect);
		try {
			log.debug("HTTP接続を行います。[" + url + "]");
			int statusCode = client.executeMethod(method);
			if (!CommonsHttpClientUtils.isSuccessfulResponse(statusCode, followRedirect)) {
				throw new ParseException("HTTP接続ができませんでした。[HTTP_STATUS='"
						+ HttpStatus.getStatusText(statusCode) + "', url='" + url + "']");
			}
			InputStream is = method.getResponseBodyAsStream();
			return parse(is);
		} catch (ConnectionTimeoutException e) {
			throw new ParseTimeoutException("HTTP接続タイムアウト。[url='" + url + "']", e);
		} catch (HttpRecoverableException e) {
			throw new ParseTimeoutException("HTTP読込タイムアウト。[url='" + url + "']", e);
		} catch (IOException e) {
			throw new ParseException("RSSオートディスカバリーに失敗しました。[url='" + url + "']", e);
		} finally {
			method.releaseConnection();
		}
	}

	/**
	 * @param is
	 * @return
	 * @throws IOException
	 * @throws MalformedURLException 
	 */
	private URL[] parse(InputStream is) throws IOException, MalformedURLException {
		InputStreamReader reader = new InputStreamReader(is);
		try {
			HTMLEditorKit.Parser parser = new DefaultHTMLParser().getParser();
			HTMLParserCallback callback = new HTMLParserCallback();
			parser.parse(reader, callback, true);

			String[] urls = callback.getRSSUrls();
			URL[] result = new URL[urls.length];
			for (int i = 0; i < urls.length; i++) {
				result[i] = new URL(urls[i]);
			}
			return result;
		} finally {
			if (reader != null) {
				reader.close();
			}
		}
	}

	private class HTMLParserCallback extends HTMLEditorKit.ParserCallback {

		private List rssUrls;

		public HTMLParserCallback() {
			rssUrls = new ArrayList();
		}

		/**
		 * @see javax.swing.text.html.HTMLEditorKit.ParserCallback#handleSimpleTag(javax.swing.text.html.HTML.Tag, javax.swing.text.MutableAttributeSet, int)
		 */
		public void handleSimpleTag(Tag t, MutableAttributeSet a, int pos) {
			if (t == HTML.Tag.LINK) {
				String rel = (String)a.getAttribute(HTML.Attribute.REL);
				String type = (String)a.getAttribute(HTML.Attribute.TYPE);
				if ("alternate".equalsIgnoreCase(rel)
						&& "application/rss+xml".equalsIgnoreCase(type)) {
					String rssUrl = (String)a.getAttribute(HTML.Attribute.HREF);
					if (rssUrl != null) {
						rssUrls.add(rssUrl);
					}
				}
			}
		}

		public String[] getRSSUrls() {
			return (String[])rssUrls.toArray(new String[rssUrls.size()]);
		}
	}

	/**
	 * 接続タイムアウト時間をセットします。単位はミリ秒。
	 * デフォルトは5,000ミリ秒(5秒)です。
	 * 
	 * @param connectionTimeout 接続タイムアウト (ms)
	 */
	public void setConnectionTimeout(int connectionTimeout) {
		this.connectionTimeout = connectionTimeout;
	}

	/**
	 * 接続後の読込タイムアウト時間をセットします。単位はミリ秒。
	 * デフォルトは5,000ミリ秒(5秒)です。
	 * 
	 * @param timeout 読込タイムアウト (ms)
	 */
	public void setReadTimeout(int timeout) {
		this.readTimeout = timeout;
	}

	/**
	 * アクセスしたURLがリダイレクトレスポンス(HTTP Status Code 3xx)を返してきた場合に、
	 * リダイレクト先にアクセスするかどうかを設定します。デフォルトでは、リダイレクト先にアクセスします。
	 * 
	 * @param followRedirect リダイレクト先にアクセスする場合 true。デフォルトはtrue。
	 */
	public void setFollowRedirect(boolean followRedirect) {
		this.followRedirect = followRedirect;
	}
}