package com.ozacc.blog.trackback.impl;

import java.io.UnsupportedEncodingException;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.ozacc.blog.trackback.TrackBackExtractor;
import com.ozacc.blog.trackback.TrackBackPing;
import com.ozacc.blog.trackback.TrackBackUtils;
import com.ozacc.blog.trackback.support.CharsetResolver;
import com.ozacc.blog.trackback.support.impl.DefaultCharsetResolver;

/**
 * TrackBackExtractorインターフェースの実装クラス。<br>
 * CharsetResolverを使用してトラックバックの文字コードを検出し、自動的に文字コードを変換します。
 * <p>
 * トラックバックのcharsetプロパティに値がセットされている場合、検出処理は行わず、そのcharsetが示す文字コードに変換します。
 * また、指定されたHttpServletRequestにcharacterEncodingがセットされている場合は、文字コードの検出、変換処理は行いません。
 * <p>
 * CharsetResolverが一つもセットされていない場合と、セットされているCharsetResolverでは文字コードを検出できなかった場合、
 * DefaultCharsetResolverが使用されます。
 * 
 * @see DefaultCharsetResolver
 * @since 1.2.3
 * @author Tomohiro Otsuka
 * @version $Id: TrackBackExtractorImpl.java 202 2005-12-08 14:32:51Z otsuka $
 */
public class TrackBackExtractorImpl implements TrackBackExtractor {

	private static Log log = LogFactory.getLog(TrackBackExtractorImpl.class);

	private List charsetResolvers;

	/**
	 * コンストラクタ。
	 */
	public TrackBackExtractorImpl() {
		super();
	}

	/**
	 * コンストラクタ。
	 * 
	 * @param charsetResolvers
	 */
	public TrackBackExtractorImpl(List charsetResolvers) {
		this();
		setCharsetResolvers(charsetResolvers);
	}

	/**
	 * CharsetResolver実装インスタンスのListをセットします。
	 * Listの順番にCharsetResolverが呼び出され、最初に検出された文字コードが適用されます。
	 * 
	 * @param charsetResolvers CharsetResolver実装インスタンスのList
	 */
	public void setCharsetResolvers(List charsetResolvers) {
		this.charsetResolvers = charsetResolvers;
	}

	/**
	 * @see com.ozacc.blog.trackback.TrackBackExtractor#getTrackBackPingFromRequest(javax.servlet.http.HttpServletRequest)
	 */
	public TrackBackPing getTrackBackPingFromRequest(HttpServletRequest request) throws UnsupportedEncodingException {
		if (log.isDebugEnabled()) {
			logReceivedRequest(request);
		}

		String encoding = request.getCharacterEncoding();
		log.debug("characterEncoding='" + encoding + "'");

		TrackBackPing ping = TrackBackUtils.getTrackBackPingFromRequest(request);

		if (isEmpty(encoding)) {
			// charsetがセットされていない場合に文字コードを判定
			if (isCharsetEmpty(ping)) {
				resolveCharset(ping, request);
			}
			//	文字コード変換
			try {
				ping.setExcerpt(TrackBackUtils.convertCharset(ping.getCharset(), ping.getExcerpt()));
				ping.setBlogName(TrackBackUtils.convertCharset(ping.getCharset(), ping.getBlogName()));
				ping.setTitle(TrackBackUtils.convertCharset(ping.getCharset(), ping.getTitle()));
				ping.setUrl(TrackBackUtils.convertCharset(ping.getCharset(), ping.getUrl()));
			} catch (UnsupportedEncodingException e) {
				log.warn("トラックバックPingの文字コード変換に失敗しました。" + ping, e);
				throw e;
			}
		} else {
			if (isCharsetEmpty(ping)) {
				ping.setCharset(encoding);
			}
		}
		return ping;
	}

	private boolean isCharsetEmpty(TrackBackPing ping) {
		return ping.getCharset() == null || ping.getCharset().length() == 0;
	}

	private boolean isEmpty(String str) {
		return str == null || str.length() == 0;
	}

	/**
	 * 指定されたTrackBackPingとHttpServletRequestの情報から、トラックバックに使用されている
	 * 文字コードを検出し、指定されたTrackBackPingのcharsetプロパティにセットします。
	 * 
	 * @param ping 
	 * @param request
	 */
	private void resolveCharset(TrackBackPing ping, HttpServletRequest request) {
		if (charsetResolvers != null) {
			for (Iterator itr = charsetResolvers.iterator(); itr.hasNext();) {
				CharsetResolver resolver = (CharsetResolver)itr.next();
				String charset = resolver.resolveCharset((TrackBackPing)ping.clone(), request);
				if (!isEmpty(charset)) {
					ping.setCharset(charset);
					return;
				}
			}
		}
		CharsetResolver resolver = new DefaultCharsetResolver();
		String charset = resolver.resolveCharset(ping, request);
		ping.setCharset(charset);
	}

	/**
	 * 受信したリクエストのヘッダとパラメータの一覧をDEBUGログ出力します。
	 * 
	 * @param request 受信したリクエスト
	 */
	private static void logReceivedRequest(HttpServletRequest request) {
		StringBuffer buf = new StringBuffer();
		buf.append("受信したリクエストのヘッダとパラメータの一覧を出力します。");
		buf.append("----------------------------------------\n");
		buf.append("リクエストヘッダ\n");
		buf.append("----------------------------------------\n");
		Enumeration headers = request.getHeaderNames();
		while (headers.hasMoreElements()) {
			String headerName = (String)headers.nextElement();
			String headerValue = request.getHeader(headerName);
			buf.append(headerName).append("='").append(headerValue).append("'\n");
		}
		buf.append("----------------------------------------\n");
		buf.append("リクエストパラメータ\n");
		buf.append("----------------------------------------\n");
		Enumeration params = request.getParameterNames();
		while (params.hasMoreElements()) {
			String paramName = (String)params.nextElement();
			String paramValue = request.getParameter(paramName);
			buf.append(paramName).append("='").append(paramValue).append("'\n");
		}
		buf.append("----------------------------------------\n");
		log.debug(buf.toString());
	}

}