/* File : sqlite3jni.i */
%module SQLite3
%{
#include <assert.h>
#include "sqlite3.h"

#ifdef __cplusplus
extern "C" {
#endif

/* org.sqlite.udf.Function class */
static jclass clsFunction = 0;

/* Field ID of org.sqlite.udf.Function member fields  */
static jfieldID fid_function_name = 0;
static jfieldID fid_function_argc = 0;
static jfieldID fid_function_this = 0;

/* org.sqlite.udf.AggregateFunction class */
static jclass clsAggregateFunction = 0;

/* org.sqlite.text.Collator class */
static jclass clsCollator = 0;

/* Field ID of org.sqlite.text.Collator member fields  */
static jfieldID fid_collator_name = 0;
static jfieldID fid_collator_this = 0;

JNIEXPORT jint JNICALL JNI_OnLoad(JavaVM *vm, void *reserved) {
    JNIEnv *jenv = NULL;
    // require JNI ver 1.2 or later
    if ((*vm)->GetEnv(vm, (void **)&jenv, JNI_VERSION_1_2) != JNI_OK) {
        return JNI_ERR;
    }
    return JNI_VERSION_1_2;
}

/* delete clsFunction and clsAggregateFunction */
JNIEXPORT void JNICALL JNI_OnUnload(JavaVM *vm, void *reserved) {
    JNIEnv *jenv = NULL;
    if ((*vm)->AttachCurrentThread(vm, (void **)&jenv, 0) == JNI_OK) {
        if (clsFunction) {
            (*jenv)->DeleteGlobalRef(jenv, clsFunction);
            clsFunction = 0;
        }
        if (clsAggregateFunction) {
            (*jenv)->DeleteGlobalRef(jenv, clsAggregateFunction);
            clsAggregateFunction = 0;
        }
        if (clsCollator) {
            (*jenv)->DeleteGlobalRef(jenv, clsCollator);
            clsCollator = 0;
        }
    }
}

/* throw new Exception */
static void JavaThrowException(JNIEnv *jenv, const char *clazz, const char *message) {
    if (jenv) {
        jclass ex;
        char *cls = (char *)clazz;
        char *msg = (char *)message;
        (*jenv)->ExceptionClear(jenv);
        do {
            ex = (*jenv)->FindClass(jenv, cls);
            if (ex) {
                (*jenv)->ThrowNew(jenv, ex, msg);
                (*jenv)->DeleteLocalRef(jenv, ex);
                return;
            }
            // exception class not found
            msg = cls;
            cls = "java/lang/ClassNotFoundException";
        } while (cls != msg);
    }
}

/* sqlite3_column_blob by byte[] */
JNIEXPORT jbyteArray JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1column_1blob_1by_1bytes(JNIEnv *jenv, jclass jcls, jlong stmt, jint col) {
    sqlite3_stmt *pStmt = *(sqlite3_stmt **)&stmt;
    jsize len;
    jbyteArray result;
    jbyte *pSrc;
    const void *blob = sqlite3_column_blob(pStmt, col);

    if (!blob) {
        return NULL;
    }

    len = sqlite3_column_bytes(pStmt, col);
    result = (*jenv)->NewByteArray(jenv, len);
    if (!result) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return NULL;
    }

    pSrc = (*jenv)->GetPrimitiveArrayCritical(jenv, result, 0);
    memcpy(pSrc, blob, len);
    (*jenv)->ReleasePrimitiveArrayCritical(jenv, result, pSrc, 0);

    return result;
}

/* sqlite3_value_blob by byte[] */
JNIEXPORT jbyteArray JNICALL Java_org_sqlite_swig_SQLite3JNI_sqlite3_1value_1blob_1by_1bytes(JNIEnv *jenv, jclass jcls, jlong value) {
    sqlite3_value *pVal = *(sqlite3_value **)&value;
    jsize len;
    jbyteArray result;
    jbyte *pSrc;
    const void *blob = sqlite3_value_blob(pVal);

    if (!blob) {
        return NULL;
    }

    len = sqlite3_value_bytes(pVal);
    result = (*jenv)->NewByteArray(jenv, len);
    if (!result) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return NULL;
    }

    pSrc = (*jenv)->GetPrimitiveArrayCritical(jenv, result, 0);
    memcpy(pSrc, blob, len);
    (*jenv)->ReleasePrimitiveArrayCritical(jenv, result, pSrc, 0);

    return result;
}

/* load 'org.sqlite.udf.Function' class */
static int LoadFunctionClass(JNIEnv *jenv) {
    if (clsFunction) {
        // alrealy loaded
        return 1;
    }

    clsFunction = (*jenv)->FindClass(jenv, "org/sqlite/udf/Function");
    if (!clsFunction) {
        JavaThrowException(jenv, "java/lang/ClassNotFoundException", "org.sqlite.udf.Function");
        return 0;
    }
    clsFunction = (*jenv)->NewGlobalRef(jenv, clsFunction);
    
    if (!fid_function_name) {
        fid_function_name = (*jenv)->GetFieldID(jenv, clsFunction, "name", "Ljava/lang/String;");
    }
    if (!fid_function_argc) {
        fid_function_argc = (*jenv)->GetFieldID(jenv, clsFunction, "argc", "I");
    }
    if (!fid_function_this) {
        fid_function_this = (*jenv)->GetFieldID(jenv, clsFunction, "_this", "J");
    }
    
    return 1;
}

/* load 'org.sqlite.udf.AggregateFunction' class */
static int LoadAggregateFunctionClass(JNIEnv *jenv) {
    if (clsAggregateFunction) {
        // alrealy loaded
        return 1;
    }

    clsAggregateFunction = (*jenv)->FindClass(jenv, "org/sqlite/udf/AggregateFunction");
    if (!clsAggregateFunction) {
        JavaThrowException(jenv, "java/lang/ClassNotFoundException", "org.sqlite.udf.AggregateFunction");
        return 0;
    }
    clsAggregateFunction = (*jenv)->NewGlobalRef(jenv, clsAggregateFunction);
    return 1;
}

/* load 'org.sqlite.text.Collator' class */
static int LoadCollatorClass(JNIEnv *jenv) {
    if (clsCollator) {
        // alrealy loaded
        return 1;
    }

    clsCollator = (*jenv)->FindClass(jenv, "org/sqlite/text/Collator");
    if (!clsCollator) {
        JavaThrowException(jenv, "java/lang/ClassNotFoundException", "org.sqlite.text.Collator");
        return 0;
    }
    clsCollator = (*jenv)->NewGlobalRef(jenv, clsCollator);

    if (!fid_collator_name) {
        fid_collator_name = (*jenv)->GetFieldID(jenv, clsCollator, "name", "Ljava/lang/String;");
    }
    if (!fid_collator_this) {
        fid_collator_this = (*jenv)->GetFieldID(jenv, clsCollator, "_this", "J");
    }

    return 1;
}

/* convert pointer to jlong */
static jlong _jlong(void *p) {
    jvalue val;
    val.l = p;
    return val.j;
}

/* convert jlong to pointer */
static void * _voidp(jlong j) {
    jvalue val;
    val.j = j;
    return (void *)val.l;
}

/* structure for user-defined function and user-defined collating sequences */
typedef struct _JAVA_OBJECT {
    JavaVM *jvm;
    jobject obj;
} JAVA_OBJECT, *PJAVA_OBJECT;

static void xFunc(sqlite3_context* ctx, int argc, sqlite3_value** value) {
    static jmethodID mid_xFunc = 0;
    JNIEnv *jenv = NULL;
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)sqlite3_user_data(ctx);
    if ((*jobj->jvm)->AttachCurrentThread(jobj->jvm, (void **)&jenv, 0) != JNI_OK) {
        // AttachCurrentThread() is not supported
        JavaThrowException(jenv, "java/lang/InternalError", "AttachCurrentThread() failed");
        return;
    }

    if (!mid_xFunc) {
        mid_xFunc = (*jenv)->GetMethodID(jenv, clsFunction, "xFunc", "(JIJ)V");
    }

    (*jenv)->CallVoidMethod(jenv, jobj->obj, mid_xFunc, _jlong(ctx), argc, _jlong(value));
    (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
}

static void xFinal(sqlite3_context* ctx) {
    static jmethodID mid_xFinal = 0;
    JNIEnv *jenv = NULL;
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)sqlite3_user_data(ctx);
    if ((*jobj->jvm)->AttachCurrentThread(jobj->jvm, (void **)&jenv, 0) != JNI_OK) {
        // AttachCurrentThread() is not supported
        JavaThrowException(jenv, "java/lang/InternalError", "AttachCurrentThread() failed");
        return;
    }

    if (!mid_xFinal) {
        mid_xFinal = (*jenv)->GetMethodID(jenv, clsAggregateFunction, "xFinal", "(J)V");
    }

    (*jenv)->CallVoidMethod(jenv, jobj->obj, mid_xFinal, _jlong(ctx));
    (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
}

/* sqlite3_create_function by org.sqlite.udf.Function */
JNIEXPORT jint JNICALL Java_org_sqlite_swig_SQLite3JNI_register_1function(JNIEnv *jenv, jclass jcls, jlong db, jobject func) {
    PJAVA_OBJECT jobj = NULL;
    jstring jname = NULL;
    char *name = NULL;
    jint argc = 0;
    int isAgFunc = 0;
    jint jresult = SQLITE_ERROR;
    int result;

    if (!clsFunction) {
        if (!LoadFunctionClass(jenv)) {
            // not found 'org.sqlite.udf.Function' class
            return jresult;
        }
        assert(clsFunction);
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, func, clsFunction)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "function is NOT instanceof 'org.sqlite.udf.Function'.");
        return jresult;
    }

    // get org.sqlite.udf.Function._this value
    if ((*jenv)->GetLongField(jenv, func, fid_function_this)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "function is already registered.");
        return jresult;
    }

    // get org.sqlite.udf.Function.name value
    jname = (jstring)(*jenv)->GetObjectField(jenv, func, fid_function_name);
    if (jname) {
        name = (char *)(*jenv)->GetStringUTFChars(jenv, jname, 0);
        if (!name) {
            SWIG_JavaThrowException(jenv, SWIG_JavaRuntimeException, "Failed JNIEnv#GetStringUTFChars().");
            return jresult;
        }
    } else {
        SWIG_JavaThrowException(jenv, SWIG_JavaNullPointerException, "name is null.");
        return jresult;
    }

    // get org.sqlite.udf.Function.argc value
    argc = (*jenv)->GetIntField(jenv, func, fid_function_argc);

    // allocate user data
    jobj = (PJAVA_OBJECT)calloc(1, sizeof(JAVA_OBJECT));
    if (!jobj) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return jresult;
    }
    jobj->obj = (*jenv)->NewGlobalRef(jenv, func);
    (*jenv)->GetJavaVM(jenv, &jobj->jvm);
    
    if (!clsAggregateFunction) {
        if (!LoadAggregateFunctionClass(jenv)) {
            // not found 'org.sqlite.udf.AggregateFunction' class
            return jresult;
        }
        assert(clsAggregateFunction);
    }
    
    // func insetanceof org.sqlite.udf.AggregateFunction
    isAgFunc = (*jenv)->IsInstanceOf(jenv, func, clsAggregateFunction);
    
    // register function
    result = (jint)sqlite3_create_function(*(sqlite3 **)&db, name, (int)argc, SQLITE_UTF8, jobj, (isAgFunc ? NULL : &xFunc), (isAgFunc ? &xFunc : NULL), (isAgFunc ? &xFinal : NULL));
    if (result == SQLITE_OK) {
        // set org.sqlite.udf.Function.pUserData value
        (*jenv)->SetLongField(jenv, func, fid_function_this, _jlong(jobj));

    } else {
        // clean up user data
        (*jenv)->DeleteGlobalRef(jenv, jobj->obj);
        free(jobj);
    }

    if (name) {
        (*jenv)->ReleaseStringUTFChars(jenv, jname, (const char *)name);
    }

    jresult = (jint)result;
    return jresult;
}

/* sqlite3_create_function by org.sqlite.udf.Function */
JNIEXPORT jint JNICALL Java_org_sqlite_swig_SQLite3JNI_unregister_1function(JNIEnv *jenv, jclass jcls, jlong db, jobject func) {
    jlong _this;
    jstring jname = NULL;
    char *name = NULL;
    jint jresult = SQLITE_ERROR;
    int result;

    if (!clsFunction) {
        if (!LoadFunctionClass(jenv)) {
            // not found 'org.sqlite.udf.Function' class
            return jresult;
        }
        assert(clsFunction);
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, func, clsFunction)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "function is NOT instanceof 'org.sqlite.udf.Function'.");
        return jresult;
    }

    // get org.sqlite.udf.Function._this value
    _this = (*jenv)->GetLongField(jenv, func, fid_function_this);
    if (!_this) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "function is not registered.");
        return jresult;
    }

    // get org.sqlite.udf.Function.name value
    jname = (jstring)(*jenv)->GetObjectField(jenv, func, fid_function_name);
    if (jname) {
        name = (char *)(*jenv)->GetStringUTFChars(jenv, jname, 0);
        if (!name) {
            SWIG_JavaThrowException(jenv, SWIG_JavaRuntimeException, "Failed JNIEnv#GetStringUTFChars().");
            return jresult;
        }
    } else {
        SWIG_JavaThrowException(jenv, SWIG_JavaNullPointerException, "name is null.");
        return jresult;
    }

    // unregister function
    result = (jint)sqlite3_create_function(*(sqlite3 **)&db, name, -1, SQLITE_UTF8, NULL, NULL, NULL, NULL);
    if (result == SQLITE_OK) {
        // clean up user data
        PJAVA_OBJECT jobj = _voidp(_this);
        (*jenv)->DeleteGlobalRef(jenv, jobj->obj);
        free(jobj);
        (*jenv)->SetLongField(jenv, func, fid_function_this, 0);
    }

    if (name) {
        (*jenv)->ReleaseStringUTFChars(jenv, jname, (const char *)name);
    }

    jresult = (jint)result;
    return jresult;
}

static int xCompare(void *p, int len1, const void *str1, int len2, const void *str2) {
    static jmethodID mid_xCompare = 0;
    PJAVA_OBJECT jobj = (PJAVA_OBJECT)p;
    JNIEnv *jenv = NULL;
    jstring jstr1 = NULL;
    jstring jstr2 = NULL;
    jint jresult = 0;
    
    if ((*jobj->jvm)->AttachCurrentThread(jobj->jvm, (void **)&jenv, 0) != JNI_OK) {
        // AttachCurrentThread() is not supported
        JavaThrowException(jenv, "java/lang/InternalError", "AttachCurrentThread() failed");
        return jresult;
    }

    if (!mid_xCompare) {
        mid_xCompare = (*jenv)->GetMethodID(jenv, clsCollator, "xCompare", "(Ljava/lang/String;Ljava/lang/String;)I");
    }

    if (str1) {
        if (strlen((const char *)str1) == len1) {
			jstr1 = (*jenv)->NewStringUTF(jenv, (const char *)str1);        
		} else {
		    char *str = (char *)calloc(len1 + 1, sizeof(char));
		    strncpy(str, str1, len1);
		    str[len1] = '\0';
			jstr1 = (*jenv)->NewStringUTF(jenv, (const char *)str);
			free(str);
		}
    }

    if (str2) {
        if (strlen((const char *)str2) == len2) {
			jstr2 = (*jenv)->NewStringUTF(jenv, (const char *)str2);        
		} else {
		    char *str = (char *)calloc(len2 + 1, sizeof(char));
		    strncpy(str, str2, len2);
		    str[len2] = '\0';
			jstr2 = (*jenv)->NewStringUTF(jenv, (const char *)str);
			free(str);
		}
    }
    
    jresult = (jint)(*jenv)->CallIntMethod(jenv, jobj->obj, mid_xCompare, jstr1, jstr2);
    
    (*jobj->jvm)->DetachCurrentThread(jobj->jvm);
    
    return jresult;
}

/* sqlite3_create_collation by org.sqlite.text.Collator */
JNIEXPORT jint JNICALL Java_org_sqlite_swig_SQLite3JNI_register_1collation(JNIEnv *jenv, jclass jcls, jlong db, jobject col) {
    PJAVA_OBJECT jobj = NULL;
    jstring jname = NULL;
    char *name = NULL;
    jint jresult = SQLITE_ERROR;
    int result;

    if (!clsCollator) {
        if (!LoadCollatorClass(jenv)) {
            // not found 'org.sqlite.text.Collator' class
            return jresult;
        }
        assert(clsCollator);
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, col, clsCollator)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "function is NOT instanceof 'org.sqlite.text.Collator'.");
        return jresult;
    }

    // get org.sqlite.text.Collator._this value
    if ((*jenv)->GetLongField(jenv, col, fid_collator_this)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "collation is already registered.");
        return jresult;
    }

    // get org.sqlite.text.Collator.name value
    jname = (jstring)(*jenv)->GetObjectField(jenv, col, fid_collator_name);
    if (jname) {
        name = (char *)(*jenv)->GetStringUTFChars(jenv, jname, 0);
        if (!name) {
            SWIG_JavaThrowException(jenv, SWIG_JavaRuntimeException, "Failed JNIEnv#GetStringUTFChars().");
            return jresult;
        }
    } else {
        SWIG_JavaThrowException(jenv, SWIG_JavaNullPointerException, "name is null.");
        return jresult;
    }

    // allocate user data
    jobj = (PJAVA_OBJECT)calloc(1, sizeof(JAVA_OBJECT));
    if (!jobj) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return jresult;
    }
    jobj->obj = (*jenv)->NewGlobalRef(jenv, col);
    (*jenv)->GetJavaVM(jenv, &jobj->jvm);
    
    // register collation
    result = (jint)sqlite3_create_collation(*(sqlite3 **)&db, name, SQLITE_UTF8, jobj, &xCompare);
    if (result == SQLITE_OK) {
        // set org.sqlite.text.Collator.pThis value
        (*jenv)->SetLongField(jenv, col, fid_collator_this, _jlong(jobj));

    } else {
        // clean up user data
        (*jenv)->DeleteGlobalRef(jenv, jobj->obj);
        free(jobj);
    }

    if (name) {
        (*jenv)->ReleaseStringUTFChars(jenv, jname, (const char *)name);
    }

    jresult = (jint)result;
    return jresult;
}

/* sqlite3_create_collation by org.sqlite.text.Collator */
JNIEXPORT jint JNICALL Java_org_sqlite_swig_SQLite3JNI_unregister_1collation(JNIEnv *jenv, jclass jcls, jlong db, jobject col) {
    jlong _this;
    jstring jname = NULL;
    char *name = NULL;
    jint jresult = SQLITE_ERROR;
    int result;

    if (!clsCollator) {
        if (!LoadCollatorClass(jenv)) {
            // not found 'org.sqlite.text.Collator' class
            return jresult;
        }
        assert(clsCollator);
    }

    // validate instance
    if (!(*jenv)->IsInstanceOf(jenv, col, clsCollator)) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "function is NOT instanceof 'org.sqlite.text.Collator'.");
        return jresult;
    }

    // get org.sqlite.text.Collator.pThis value
    _this = (*jenv)->GetLongField(jenv, col, fid_collator_this);
    if (!_this) {
        SWIG_JavaThrowException(jenv, SWIG_JavaIllegalArgumentException, "collation is not registered.");
        return jresult;
    }

    // get org.sqlite.text.Collator.name value
    jname = (jstring)(*jenv)->GetObjectField(jenv, col, fid_collator_name);
    if (jname) {
        name = (char *)(*jenv)->GetStringUTFChars(jenv, jname, 0);
        if (!name) {
            SWIG_JavaThrowException(jenv, SWIG_JavaRuntimeException, "Failed JNIEnv#GetStringUTFChars().");
            return jresult;
        }
    } else {
        SWIG_JavaThrowException(jenv, SWIG_JavaNullPointerException, "name is null.");
        return jresult;
    }

    // unregister collation
    result = (jint)sqlite3_create_collation(*(sqlite3 **)&db, name, SQLITE_UTF8, NULL, NULL);
    if (result == SQLITE_OK) {
        // clean up user data
        PJAVA_OBJECT jobj = _voidp(_this);
        (*jenv)->DeleteGlobalRef(jenv, jobj->obj);
        free(jobj);
        (*jenv)->SetLongField(jenv, col, fid_collator_this, 0);
    }

    if (name) {
        (*jenv)->ReleaseStringUTFChars(jenv, jname, (const char *)name);
    }

    jresult = (jint)result;
    return jresult;
}

#ifdef __cplusplus
}
#endif
%}

/* typemap for 'const unsigned char *' to 'java.lang.String' */
%typemap(jni) const unsigned char *    "jstring"
%typemap(jtype) const unsigned char *  "String"
%typemap(jstype) const unsigned char * "String"
%typemap(out, noblock=1) const unsigned char * { if($1) $result = JCALL1(NewStringUTF, jenv, (const char *)$1); }
%typemap(javaout) const unsigned char * {
    return $jnicall;
  }

/*
%pragma(java) jniclassclassmodifiers="public final class"
%pragma(java) moduleimports="import org.sqlite.udf.Function;\n";
*/
%javamethodmodifiers sqlite3_bind_blob_by_bytes "protected";
%javamethodmodifiers register_function "protected";
%javamethodmodifiers unregister_function "protected";
%pragma(java) modulecode="
  public static final int SQLITE_TRANSIENT = -1;

  /**
   * Returns in-memory filename.
   * @return \":memory:\"
   * @see <a href=\"http://sqlite.org/c3ref/open.html\">Opening A New Database Connection</a>
   */
  public static String getInMemoryFileName() {
    return \":memory:\";
  }

  /**
   * Returns date format pattern.
   * @return \"yyyy-MM-dd\"
   * @see <a href=\"http://sqlite.org/lang_createtable.html\">CREATE TABLE</a>
   */
  public static String getDateFormatPattern() {
    return \"yyyy-MM-dd\";
  }

  /**
   * Returns time format pattern.
   * @return \"HH:mm:ss\"
   * @see <a href=\"http://sqlite.org/lang_createtable.html\">CREATE TABLE</a>
   */
  public static String getTimeFormatPattern() {
    return \"HH:mm:ss\";
  }

  /**
   * Returns timestamp format pattern.
   * @return \"yyyy-MM-dd HH:mm:ss\"
   * @see <a href=\"http://sqlite.org/lang_createtable.html\">CREATE TABLE</a>
   */
  public static String getTimestampFormatPattern() {
    return \"yyyy-MM-dd HH:mm:ss\";
  }
  
  public static String format(String pattern, java.util.Date x) {
    return new java.text.SimpleDateFormat(pattern).format(x);
  }
  
  public static String format(java.sql.Date x) {
    return format(getDateFormatPattern(), x);
  }
  
  public static String format(java.sql.Time x) {
    return format(getTimeFormatPattern(), x);
  }
  
  public static String format(java.sql.Timestamp x) {
    return format(getTimestampFormatPattern(), x);
  }

  public static long parse(String pattern, String x) throws java.sql.SQLException {
    final java.text.DateFormat formatter = new java.text.SimpleDateFormat(pattern);
    formatter.setLenient(false);
    final java.text.ParsePosition position = new java.text.ParsePosition(0);
    final java.util.Date date = formatter.parse(x, position);
    if (position.getErrorIndex() != -1 || position.getIndex() != x.length()) {
        // parse failed
        throw new java.sql.SQLException(\"Format error.\", \"90J09\");
    }
    return date.getTime();
  }
  
  public static long parseDate(String x) throws java.sql.SQLException {
    return parse(getDateFormatPattern(), x);
  }
  
  public static long parseTime(String x) throws java.sql.SQLException {
    return parse(getTimeFormatPattern(), x);
  }
  
  public static long parseTimestamp(String x) throws java.sql.SQLException {
    return parse(getTimestampFormatPattern(), x);
  }

  public static long addressOf(SWIGTYPE_p_sqlite3_stmt stmt) {
    return SWIGTYPE_p_sqlite3_stmt.getCPtr(stmt);
  }

  public static int sqlite3_exec(SWIGTYPE_p_sqlite3 db, String sql) {
    return SQLite3JNI.sqlite3_exec(SWIGTYPE_p_sqlite3.getCPtr(db), sql, 0, 0, 0);
  }

  public static int sqlite3_bind_blob(SWIGTYPE_p_sqlite3_stmt stmt, int parameterIndex, byte[] val, int len) {
    return SQLite3JNI.sqlite3_bind_blob_by_bytes(SWIGTYPE_p_sqlite3_stmt.getCPtr(stmt), parameterIndex, val, len, SQLITE_TRANSIENT);
  }

  public static int sqlite3_bind_text(SWIGTYPE_p_sqlite3_stmt stmt, int parameterIndex, String val) {
    return SQLite3JNI.sqlite3_bind_text(SWIGTYPE_p_sqlite3_stmt.getCPtr(stmt), parameterIndex, val, -1, SQLITE_TRANSIENT);
  }

  public static int register_function(SWIGTYPE_p_sqlite3 db, org.sqlite.udf.Function func) {
    return SQLite3JNI.register_function(SWIGTYPE_p_sqlite3.getCPtr(db), func);
  }

  public static int unregister_function(SWIGTYPE_p_sqlite3 db, org.sqlite.udf.Function func) {
    return SQLite3JNI.unregister_function(SWIGTYPE_p_sqlite3.getCPtr(db), func);
  }

  public static void sqlite3_result_blob(SWIGTYPE_p_sqlite3_context ctx, SWIGTYPE_p_void blob, int len) {
    SQLite3JNI.sqlite3_result_blob(SWIGTYPE_p_sqlite3_context.getCPtr(ctx), SWIGTYPE_p_void.getCPtr(blob), len, SQLITE_TRANSIENT);
  }

  public static void sqlite3_result_blob(SWIGTYPE_p_sqlite3_context ctx, byte[] val, int len) {
    SQLite3JNI.sqlite3_result_blob_by_bytes(SWIGTYPE_p_sqlite3_context.getCPtr(ctx), val, len, SQLITE_TRANSIENT);
  }

  public static void sqlite3_result_error(SWIGTYPE_p_sqlite3_context ctx, String message) {
    SQLite3JNI.sqlite3_result_error(SWIGTYPE_p_sqlite3_context.getCPtr(ctx), message, -1);
  }

  public static void sqlite3_result_text(SWIGTYPE_p_sqlite3_context ctx, String val) {
    SQLite3JNI.sqlite3_result_text(SWIGTYPE_p_sqlite3_context.getCPtr(ctx), val, -1, SQLITE_TRANSIENT);
  }

  public static int sqlite3_clear_bindings(SWIGTYPE_p_sqlite3_stmt stmt) {
    final int max = sqlite3_bind_parameter_count(stmt) + 1;
    int ret = SQLITE_OK;
    for (int i = 1; ret == SQLITE_OK && i < max; ++i) {
      ret = sqlite3_bind_null(stmt, i);
    }
    return ret;
  }
"

%javaconst(1);
%javaconst(0) SQLITE_VERSION;
%javaconst(0) SQLITE_VERSION_NUMBER;
%include "sqlite3.h"
%include "various.i"
%include "cpointer.i"

/*******************************************************************/
%exception new_p_p_sqlite3 {
    $action
    if (!result) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return $null;
    }
}

%exception new_p_p_sqlite3_stmt {
    $action
    if (!result) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return $null;
    }
}

%exception new_p_p_char {
    $action
    if (!result) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return $null;
    }
}

%exception new_p_int {
    $action
    if (!result) {
        SWIG_JavaThrowException(jenv, SWIG_JavaOutOfMemoryError, "Not enough memory.");
        return $null;
    }
}

/* Create some functions for working with "sqlite3**" */
%pointer_functions(sqlite3*, p_p_sqlite3);

/* Create some functions for working with "sqlite3_stmt**" */
%pointer_functions(sqlite3_stmt*, p_p_sqlite3_stmt);

/* Create some functions for working with "char const*" */
%pointer_functions(char const*, p_p_char);

/* Create some functions for working with "int*" */
%pointer_functions(int, p_int);

%inline %{
/* sqlite3_bind_blob by byte[] */
static int sqlite3_bind_blob_by_bytes(sqlite3_stmt *pStmt, int parameterIndex, char *BYTE, int length, void (*xDel)(void*)) {
    return sqlite3_bind_blob(pStmt, parameterIndex, BYTE, length, xDel);
}

/* sqlite3_column_blob by java.sql.Blob */
static void read_blob(const void *blob, long long pos, char *BYTE, int offset, int len) {
    memcpy(&BYTE[offset], &((const char *)blob)[pos], len);
}

/* get (sqlite3_value*)sqlite3_value**[i] */
static sqlite3_value* get_p_sqlite3_value(sqlite3_value** value, int i) {
    return value[i];
}

/* sqlite3_result_blob by byte[] */
static void sqlite3_result_blob_by_bytes(sqlite3_context *pCtx, char *BYTE, int length, void (*xDel)(void*)) {
    sqlite3_result_blob(pCtx, BYTE, length, xDel);
}

%}

/* sqlite3_column_blob by byte[] */
%native(sqlite3_column_blob_by_bytes) jbyteArray sqlite3_column_blob_by_bytes(sqlite3_stmt*, int);

/* sqlite3_value_blob by byte[] */
%native(sqlite3_value_blob_by_bytes) jbyteArray sqlite3_value_blob_by_bytes(sqlite3_value*);

/* register_function */
%native(register_function) jint register_function(sqlite3*, jobject);

/* unregister_function */
%native(unregister_function) jint unregister_function(sqlite3*, jobject);

/* register_collation */
%native(register_collation) jint register_collation(sqlite3*, jobject);

/* unregister_collation */
%native(unregister_collation) jint unregister_collation(sqlite3*, jobject);

/*******************************************************************/
