package org.sqlite.jdbc;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import static org.junit.Assert.*;

/**
 *
 * @author calico
 */
public class JdbcStatementTest {

    public JdbcStatementTest() {
    }

    @BeforeClass
    public static void setUpClass() throws Exception {
    }

    @AfterClass
    public static void tearDownClass() throws Exception {
    }

    @Before
    public void setUp() {
    }

    @After
    public void tearDown() {
    }

    private static final String DRIVER_CLASS = "org.sqlite.Driver";
    private static final String DATABASE = System.getProperty("user.dir") + "/test/unittest.db";
    
    private static Connection newConnection() throws ClassNotFoundException, SQLException {
        Class.forName(DRIVER_CLASS);
        return DriverManager.getConnection("jdbc:sqlite:file:" + DATABASE);
    }
    
    @Test
    public void getGeneratedKeys() throws ClassNotFoundException, SQLException {
        final Connection conn = newConnection();
        try {
            final Statement stmt = conn.createStatement();
            String sql
                    = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                        + "  ROWID INTEGER PRIMARY KEY AUTOINCREMENT"
                        + ", ID INTEGER NOT NULL"
                        + ", tVALUE TEXT DEFAULT '_blank'"
                        + ", rVALUE REAL"
                        + ", bVALUE BLOB"
                        + ", nVALUE NULL"
                        + ", tVALUE2 text DEFAULT ''"
                        + ", tVALUE3 VarChar DEFAULT NULL"
                    + ")";
            stmt.executeUpdate(sql);
            
            ResultSet rs = stmt.getGeneratedKeys();
            assertTrue(rs.next());
            assertEquals(0, rs.getInt(1));
            rs.close();
            
            sql = "INSERT INTO temp_tbl_1("
                    + "ID"
                    + ", rVALUE"
                    + ", bVALUE"
                    + ", nVALUE"
                + ") VALUES(1, 2.345, X'E38182E38184E38186E38188E3818A00', 10)";
            stmt.executeUpdate(sql);
            
            rs = stmt.getGeneratedKeys();
            assertTrue(rs.next());
            assertEquals(1, rs.getInt(1));
            rs.close();

            stmt.executeUpdate(sql);
            rs = stmt.getGeneratedKeys();
            assertTrue(rs.next());
            assertEquals(2, rs.getInt(1));
            rs.close();

            stmt.executeUpdate(sql);
            stmt.executeUpdate(sql);
            stmt.executeUpdate(sql);
            rs = stmt.getGeneratedKeys();
            assertTrue(rs.next());
            assertEquals(5, rs.getInt(1));
            rs.close();
            
            stmt.close();
            
        } finally {
            conn.close();
        }
    }
    
    @Test(expected = java.sql.SQLException.class)
    public void setQueryTimeout() throws ClassNotFoundException, SQLException {
        final Connection conn = newConnection();
        try {
            final Statement stmt = conn.createStatement();
            String sql
                    = "CREATE TABLE IF NOT EXISTS timeout_tbl_1("
                        + "  ROWID INTEGER PRIMARY KEY AUTOINCREMENT"
                        + ", ID INTEGER NOT NULL"
                        + ", tVALUE TEXT DEFAULT '_blank'"
                        + ", rVALUE REAL"
                        + ", bVALUE BLOB"
                        + ", nVALUE NULL"
                        + ", tVALUE2 text DEFAULT ''"
                        + ", tVALUE3 VarChar DEFAULT NULL"
                    + ")";
            stmt.executeUpdate(sql);
            
//            stmt.setQueryTimeout(1);      
            ((JdbcConnection) conn).setBusyTimeout(1);
            
            sql = "INSERT INTO timeout_tbl_1("
                    + "ID"
                    + ", rVALUE"
                    + ", bVALUE"
                    + ", nVALUE"
                + ") VALUES(1, 2.345, X'E38182E38184E38186E38188E3818A00', 10)";

            for (int i = 0; i < 1000; ++i) {
                assertEquals(1, stmt.executeUpdate(sql));
                Thread worker
                        = new Thread() {
                            @Override
                            public void run() {
                                Connection conn2 = null;
                                try {
                                    conn2 = newConnection();
                                    final Statement stmt2 = conn2.createStatement();
                                    final String sql = "SELECT ROWID FROM timeout_tbl_1";
                                    final ResultSet rs2 = stmt2.executeQuery(sql);
                                    while (rs2.next()) {
                                        System.out.printf("ROWID:[%d]\n", rs2.getInt(1));
                                    }
                                    rs2.close();
                                    stmt2.close();

                                } catch (Exception ex) {
                                    Logger.getLogger(JdbcStatementTest.class.getName()).severe(ex.toString());
                                } finally {
                                    if (conn2 != null) {
                                        try {
                                            conn2.close();
                                        } catch (SQLException ex) {
                                            Logger.getLogger(JdbcStatementTest.class.getName()).log(Level.SEVERE, null, ex);
                                        }
                                    }
                                }
                            }
                        };
                worker.start();
            }
            
        } catch (SQLException ex) {
            ex.printStackTrace();
            throw ex;
            
        } finally {
            conn.createStatement().executeUpdate("DROP TABLE IF EXISTS timeout_tbl_1");
            conn.close();
        }
    }
    
    @Test
    public void getMoreResults() throws ClassNotFoundException, SQLException {
        final Connection conn = newConnection();
        try {
            final Statement stmt = conn.createStatement();
            ResultSet rs = null;
            String sql
                    = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                        + "  ROWID INTEGER PRIMARY KEY AUTOINCREMENT"
                        + ", ID INTEGER NOT NULL"
                        + ", tVALUE TEXT DEFAULT '_blank'"
                        + ", rVALUE REAL"
                        + ", bVALUE BLOB"
                        + ", nVALUE NULL"
                        + ", tVALUE2 text DEFAULT ''"
                        + ", tVALUE3 VarChar DEFAULT NULL"
                    + ")";
            stmt.executeUpdate(sql);
            
            sql = "INSERT INTO temp_tbl_1("
                    + "ID"
                    + ", rVALUE"
                    + ", bVALUE"
                    + ", nVALUE"
                + ") VALUES(1, 2.345, X'E38182E38184E38186E38188E3818A00', 10)";
            stmt.executeUpdate(sql);

            sql = "SELECT * FROM temp_tbl_1;"
                + "SELECT * FROM sqlite_temp_master";
            assertTrue(stmt.execute(sql));
            rs = stmt.getResultSet();
            assertNotNull(rs);
            assertTrue(rs.next());
            assertEquals("temp_tbl_1", rs.getMetaData().getTableName(1));
            assertTrue(stmt.getMoreResults());
            rs.close();
            
            rs = stmt.getResultSet();
            assertNotNull(rs);
            assertTrue(rs.next());
            assertEquals("sqlite_temp_master", rs.getMetaData().getTableName(1));
            assertFalse(stmt.getMoreResults());
            rs.close();

            sql = "INSERT INTO temp_tbl_1("
                    + "ID"
                    + ", rVALUE"
                    + ", bVALUE"
                    + ", nVALUE"
                + ") VALUES(2, 6.789, X'E38182E38184E38186E38188E3818A00', 20);"
                + "SELECT * FROM temp_tbl_1;"
                + "SELECT * FROM sqlite_temp_master";                
            assertFalse(stmt.execute(sql));
            rs = stmt.getResultSet();
            assertNull(rs);
            assertEquals(1, stmt.getUpdateCount());
            
            assertTrue(stmt.getMoreResults());
            rs = stmt.getResultSet();
            assertNotNull(rs);
            assertTrue(rs.next());
            assertEquals("temp_tbl_1", rs.getMetaData().getTableName(1));
            ((JdbcStatement) stmt).detach(rs);
            
            assertTrue(stmt.getMoreResults());
            ResultSet rs2 = stmt.getResultSet();
            assertNotNull(rs2);
            assertTrue(rs2.next());
            assertEquals("sqlite_temp_master", rs2.getMetaData().getTableName(1));
            ((JdbcStatement) stmt).detach(rs2);

            stmt.close();

            assertEquals("temp_tbl_1", rs.getMetaData().getTableName(1));
            rs.close();

            assertEquals("sqlite_temp_master", rs2.getMetaData().getTableName(1));
            rs2.close();
            
        } finally {
            conn.close();
        }        
    }
    
    @Test
    public void detach() throws ClassNotFoundException, SQLException {
        final Connection conn = newConnection();
        try {
            JdbcStatement stmt = (JdbcStatement) conn.createStatement();
            ResultSet rs = null;
            String sql
                    = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                        + "  ROWID INTEGER PRIMARY KEY AUTOINCREMENT"
                        + ", ID INTEGER NOT NULL"
                        + ", tVALUE TEXT DEFAULT '_blank'"
                        + ", rVALUE REAL"
                        + ", bVALUE BLOB"
                        + ", nVALUE NULL"
                        + ", tVALUE2 text DEFAULT ''"
                        + ", tVALUE3 VarChar DEFAULT NULL"
                    + ")";
            stmt.executeUpdate(sql);
            
            sql = "INSERT INTO temp_tbl_1("
                    + "ID"
                    + ", rVALUE"
                    + ", bVALUE"
                    + ", nVALUE"
                + ") VALUES(1, 2.345, X'E38182E38184E38186E38188E3818A00', 10)";
            stmt.executeUpdate(sql);

            sql = "SELECT * FROM temp_tbl_1;"
                + "SELECT * FROM sqlite_temp_master";
            assertTrue(stmt.execute(sql));
            rs = stmt.getResultSet();
            assertNotNull(rs);
            assertTrue(rs.next());
            assertEquals("temp_tbl_1", rs.getMetaData().getTableName(1));
            // detach ResultSet
            stmt.detach(rs);
            assertNull(stmt.getResultSet());
            
            assertTrue(stmt.getMoreResults());
            ResultSet rs2 = stmt.getResultSet();
            assertNotNull(rs2);
            assertTrue(rs2.next());
            assertEquals("sqlite_temp_master", rs2.getMetaData().getTableName(1));
            assertFalse(stmt.getMoreResults());
            rs2.close();
            stmt.close();
            
            assertEquals(1, rs.getInt(1));
            rs.close();
            
            sql = "INSERT INTO temp_tbl_1("
                    + "ID"
                    + ", rVALUE"
                    + ", bVALUE"
                    + ", nVALUE"
                + ") VALUES(2, 6.789, X'E38182E38184E38186E38188E3818A00', 20);"
                + "SELECT * FROM temp_tbl_1;"
                + "SELECT * FROM sqlite_temp_master";
            stmt = (JdbcStatement) conn.createStatement();
            assertFalse(stmt.execute(sql));
            rs = stmt.getResultSet();
            assertNull(rs);
            assertEquals(1, stmt.getUpdateCount());
            
            assertTrue(stmt.getMoreResults());
            rs = stmt.getResultSet();
            assertNotNull(rs);
            assertTrue(rs.next());
            assertEquals("temp_tbl_1", rs.getMetaData().getTableName(1));
            ((JdbcStatement) stmt).detach(rs);
            
            assertTrue(stmt.getMoreResults());
            rs2 = stmt.getResultSet();
            assertNotNull(rs2);
            assertTrue(rs2.next());
            assertEquals("sqlite_temp_master", rs2.getMetaData().getTableName(1));
            ((JdbcStatement) stmt).detach(rs2);

            stmt.close();

            assertEquals("temp_tbl_1", rs.getMetaData().getTableName(1));
            rs.close();

            assertEquals("sqlite_temp_master", rs2.getMetaData().getTableName(1));
            rs2.close();
            
        } finally {
            conn.close();
        }        
    }
    
    @Test
    public void execute() throws ClassNotFoundException, SQLException {
        final Connection conn = newConnection();
        try {
            final Statement stmt = conn.createStatement();
            ResultSet rs = null;
            String sql
                    = "CREATE TEMPORARY TABLE IF NOT EXISTS temp_tbl_1("
                        + "  ROWID INTEGER PRIMARY KEY AUTOINCREMENT"
                        + ", ID INTEGER NOT NULL"
                        + ", tVALUE TEXT DEFAULT '_blank'"
                        + ", rVALUE REAL"
                        + ", bVALUE BLOB"
                        + ", nVALUE NULL"
                        + ", tVALUE2 text DEFAULT ''"
                        + ", tVALUE3 VarChar DEFAULT NULL"
                    + ")";
            assertFalse(stmt.execute(sql));
            
            sql = "PRAGMA table_info(temp_tbl_1)";
            assertTrue(stmt.execute(sql));
            rs = stmt.getResultSet();
            assertNotNull(rs);
            assertTrue(rs.next());
            rs.close();

            sql = "PRAGMA table_info(temp_tbl_123)";
            assertFalse(stmt.execute(sql));
            rs = stmt.getResultSet();
            assertNull(rs);
            
            stmt.close();
            
        } finally {
            conn.close();
        }
    }
    
    @Test
    public void tempDirectory() throws ClassNotFoundException, SQLException {
        Connection conn = newConnection();
        String defaultTempDir = null;
        try {
            final Statement stmt = conn.createStatement();
            final boolean ret = stmt.execute("PRAGMA temp_store_directory");
            if (ret) {
                final ResultSet rs = stmt.getResultSet();
                assertTrue(rs.next());
                defaultTempDir = rs.getString(1);
                assertNotNull(defaultTempDir);
                rs.close();
            }
            stmt.close();
            
        } finally {
            conn.close();
        }
        
        Class.forName(DRIVER_CLASS);
        final String TEMP_DIR = "TEMP_DIR=C:\\tmp";
        conn = DriverManager.getConnection("jdbc:sqlite:file:" + DATABASE + ";" + TEMP_DIR);
        try {
            final Statement stmt = conn.createStatement();
            assertTrue(stmt.execute("PRAGMA temp_store_directory"));
            final ResultSet rs = stmt.getResultSet();
            assertTrue(rs.next());
            final String newTempDir = rs.getString(1);
            assertNotNull(newTempDir);
            
            assertFalse(newTempDir.equalsIgnoreCase(defaultTempDir));
            
            rs.close();
            stmt.close();
            
        } finally {
            conn.close();
        }
    }
}