package org.sqlite;

import org.sqlite.jdbc.TransactionType;
import org.sqlite.schema.ColumnMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import org.sqlite.auth.Authorizer;
import org.sqlite.callback.ExecCallback;
import org.sqlite.event.BusyHandler;
import org.sqlite.event.CommitHook;
import org.sqlite.event.ProgressHandler;
import org.sqlite.event.RollbackHook;
import org.sqlite.event.UpdateHook;
import org.sqlite.jdbc.JdbcSQLException;
import org.sqlite.swig.SWIGTYPE_p_int;
import org.sqlite.swig.SWIGTYPE_p_p_char;
import org.sqlite.swig.SWIGTYPE_p_sqlite3;
import org.sqlite.event.CollationNeededHandler;
import org.sqlite.profiler.Profiler;
import org.sqlite.profiler.Tracer;
import org.sqlite.text.Collator;
import org.sqlite.udf.Function;
import static org.sqlite.swig.SQLite3.*;

/**
 * sqlite3 wrapper class.<br/>
 * NOTE: SQLite 3.3.17 based.
 * @author calico
 */
public class Database {
    /**
     * database properties
     */
    protected final Map<String, String> info;
    
    private final boolean isInMemory;
    private final SQLite3PtrPtr ppDb;
    private List<Statement> statements;
    private List<Function> functions;
    private List<Collator> collators;
    private Authorizer authorizer;
    private BusyHandler busyHandler;
    private CollationNeededHandler collNeeded;
    private ProgressHandler progressHandler;
    
    /** timeout(ms) : sqlite3_busy_timeout */
    private int timeout;
    
    /**
     * open database.
     * @param filename database file path
     * @param info database properties. Changes the temporary directory by "TEMP_DIR" property.
     * @throws java.sql.SQLException When the return value of the sqlite3_open() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/temp_directory.html">Name Of The Folder Holding Temporary Files</a>
     */
    public Database(String filename, Map<String, String> info) throws SQLException {
        this.info = info;
        this.isInMemory
                = (filename == null || getInMemoryFileName().equals(filename));
        this.ppDb = new SQLite3PtrPtr();

        // set temporary directory
        if (info != null && info.containsKey("TEMP_DIR")) {
            set_sqlite3_temp_directory(info.get("TEMP_DIR"));
        }
        
        // open database
        open(filename);
        
        // execute PRAGMA commands
        if (info != null && !info.isEmpty()) {
            final StringBuilder pragmas = new StringBuilder();
            for (final String key : info.keySet()) {
                if (key.startsWith("PRAGMA")) {
                    pragmas.append(key);
                    final String value = info.get(key);
                    if (value.length() != 0) {
                        pragmas.append("=").append(value);
                    }
                    pragmas.append(";");
                }
            }
            if (pragmas.length() != 0) {
                execute(pragmas.toString());
            }
        }
    }
    
    /**
     * It always returns "SQLite".
     * @return "SQLite"
     */
    public String getProductName() {
        return "SQLite";
    }
    
    /**
     * true is returned for the In-Memory mode.
     * @return true is returned for the In-Memory mode.
     */
    public boolean isInMemoryMode() {
        return isInMemory;
    }
    
    /**
     * Returns the database handle.
     * @return the database handle.
     * @see <a href="http://sqlite.org/c3ref/sqlite3.html">Database Connection Handle</a>
     */
    private SWIGTYPE_p_sqlite3 getHandle() {
        return ppDb.getSQLite3Ptr();
    }
    
    /**
     * invoke sqlite3_open() function.
     * @param filename database file path
     * @throws java.sql.SQLException When the return value of the sqlite3_open() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/open.html">Opening A New Database Connection</a>
     */
    protected void open(String filename) throws SQLException {
        int ret = sqlite3_open(filename, ppDb);
        ppDb.allocateHandle();
        if (ret != SQLITE_OK) {
            SWIGTYPE_p_sqlite3 db = getHandle();
            SQLException ex = new JdbcSQLException(ret, db);
            ppDb.delete();
            throw ex;
        }
    }
    
    /**
     * It always returns false.<br/>
     * NOTE: The sqlite3_open_v2() function is not supported in this version.
     * @return false.
     */
    public boolean isReadOnly() {
        return false;
    }
    
    /**
     * Retrieves whether this Database object has been closed.
     * @return true if this Database object is closed. false if it is still open.
     */
    public boolean isClosed() {
        return (ppDb.isDeleted());
    }
    
    /**
     * invoke sqlite3_close() function.
     * @throws java.sql.SQLException When the return value of the sqlite3_close() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/close.html">Closing A Database Connection</a>
     */
    public void close() throws SQLException {
        if (!isClosed()) {
            // close all statements
            closeStatements();
            // unregister all functions
            dropFunctions();
            // unregister all collators
            dropCollators();
            // clear authorizer
            internalClearAuthorizer();
            // clear busy handler
            internalClearBusyHandler();
            // clear collation needed handler
            internalClearCollationNeededHandler();
            // clear progress handler
            clearProgressHandler();
            // clear commit hook
            clearCommitHook();
            // clear rollback hook
            clearRollbackHook();
            // clear update hook
            clearUpdateHook();
            // clear profiler
            clearProfiler();
            // clear tracer
            clearTracer();
            
            // SQLITE_ENABLE_MEMORY_MANAGEMENTが定義された状態でビルドされたSQLiteを使用している場合、
            // sqlite3_open()を呼び出したスレッドとは異なるスレッドからsqlite3_close()を呼び出すと
            // ACCESS_VIOLATION例外が発生してJVMがクラッシュするので要注意！
            // 尚、SQLITE_ENABLE_MEMORY_MANAGEMENTを定義してビルドする必要がある環境は組み込みデバイスなどのメモリが極小の環境のみ。
            // @see http://readlist.com/lists/sqlite.org/sqlite-users/0/572.html
            final SWIGTYPE_p_sqlite3 db = getHandle();
            int ret = sqlite3_close(db);
            if (ret != SQLITE_OK) {
                throw new JdbcSQLException(ret, db);
            }
            ppDb.delete();
        }
    }
    
    /**
     * invoke sqlite3_get_autocommit() function.
     * @return  true if auto commit mode.
     * @see <a href="http://sqlite.org/c3ref/get_autocommit.html">Test To See If The Database Is In Auto-Commit Mode</a>
     */
    public boolean getAutoCommit() {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        return (sqlite3_get_autocommit(db) != 0);
    }
    
    /**
     * invoke sqlite3_busy_timeout() function.
     * @param ms milliseconds
     * @throws java.sql.SQLException When the return value of the sqlite3_busy_timeout() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/busy_timeout.html">Set A Busy Timeout</a>
     */
    public void setBusyTimeout(int ms) throws SQLException {
        clearBusyHandler();

        final SWIGTYPE_p_sqlite3 db = getHandle();
        int ret = sqlite3_busy_timeout(db, ms);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(ret, db);
        }
        timeout = (ms < 1 ? 0 : ms);
    }

    /**
     * Returns the value of timeout(ms).
     * @return  timeout(ms) value.
     */
    public int getBusyTimeout() {
        return timeout;
    }
    
    /**
     * invoke sqlite3_exec() function.
     * @param sql SQL to be evaluated
     * @throws java.sql.SQLException When the return value of the sqlite3_exec() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/exec.html">One-Step Query Execution Interface</a>
     * @see #execute(String, ExecCallback, SWIGTYPE_p_p_char)
     */
    public void execute(String sql) throws SQLException {
        execute(sql, null, null);
    }
    
    /**
     * invoke sqlite3_exec() function.
     * @param sql SQL to be evaluated
     * @param callback callback object
     * @param errmsg Error message written here
     * @throws java.sql.SQLException When the return value of the sqlite3_exec() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/exec.html">One-Step Query Execution Interface</a>
     */
    public void execute(String sql, ExecCallback callback, SWIGTYPE_p_p_char errmsg) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        int ret = 0;
        while (((ret = sqlite3_exec(db, sql, callback, errmsg)) == SQLITE_BUSY)
                && (timeout == 0)) {
            // waiting...
        }
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(ret, db);
        }
    }

    /**
     * execute PRAGMA commands by sqlite3_exec() finction.
     * @param commands the command list <em>without semicolon</em>
     * @throws java.sql.SQLException When the return value of the sqlite3_exec() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/pragma.html">Pragma statements supported by SQLite</a>
     * @see #execute(String)
     */
    public void pragma(String[] commands) throws SQLException {
        final StringBuilder sql = new StringBuilder();
        for (final String cmd : commands) {
            sql.append("PRAGMA ").append(cmd).append(";");
        }
        execute(sql.toString());
    }
    
    /**
     * begin transaction.
     * @param type transaction type.
     * @throws java.sql.SQLException When the return value of the sqlite3_exec() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/lang_transaction.html">BEGIN TRANSACTION</a>
     */
    public void beginTransaction(TransactionType type) throws SQLException {
        closeStatements();
        if (type == null) {
            execute("BEGIN");
        } else {
            execute("BEGIN " + type);
        }
    }
    
    /**
     * commit toransaction.
     * @throws java.sql.SQLException When the return value of the sqlite3_exec() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/lang_transaction.html">BEGIN TRANSACTION</a>
     */
    public void commitTransaction() throws SQLException {
        closeStatements();
        execute("COMMIT");
    }
    
    /**
     * rollback transaction.
     * @throws java.sql.SQLException When the return value of the sqlite3_exec() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/lang_transaction.html">BEGIN TRANSACTION</a>
     */
    public void rollbackTransaction() throws SQLException {
        closeStatements();
        execute("ROLLBACK");
    }
    
    /**
     * create MANAGED Statement instance.
     * @param sql SQL to be evaluated
     * @param ppStmt SQLite3StmtPtrPtr object
     * @return Statement object
     * @throws java.sql.SQLException When the return value of the sqlite3_prepare() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/prepare.html">Compiling An SQL Statement</a>
     */
    public Statement prepare(String sql, SQLite3StmtPtrPtr ppStmt) throws SQLException {
        if (sql == null) {
            throw new NullPointerException("sql is null.");
        }
        if (ppStmt == null) {
            throw new NullPointerException("ppStmt is null.");
        }
        
        final SWIGTYPE_p_sqlite3 db = getHandle();
        int ret = sqlite3_prepare(db, sql, -1, ppStmt, null);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(ret, db);
        }
        return new Statement(this, ppStmt.getSQLite3StmtPtr());
    }
    
    /**
     * create UNMANAGED Statement instance.
     * @param sql SQL to be evaluated
     * @return Statement object
     * @throws java.sql.SQLException When the return value of the sqlite3_prepare() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/prepare.html">Compiling An SQL Statement</a>
     */
    public Statement prepare(String sql) throws SQLException {
        if (sql == null) {
            throw new NullPointerException("sql is null.");
        }

        final SWIGTYPE_p_sqlite3 db = getHandle();
        final SQLite3StmtPtrPtr ppStmt = new SQLite3StmtPtrPtr();
        int ret = sqlite3_prepare(db, sql, -1, ppStmt, null);
        if (ret != SQLITE_OK) {
            ppStmt.delete();
            throw new JdbcSQLException(ret, db);
        }
        return new Statement(this, ppStmt);
    }
    
    /**
     * create multiple UNMANAGED Statement instance.
     * @param sql SQL to be evaluated
     * @return array of Statement
     * @throws java.sql.SQLException When the return value of the sqlite3_prepare() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/prepare.html">Compiling An SQL Statement</a>
     */
    public List<Statement> prepareMultiple(String sql) throws SQLException {
        if (sql == null) {
            throw new NullPointerException("sql is null.");
        }

        final SWIGTYPE_p_sqlite3 db = getHandle();
        final List<SQLite3StmtPtrPtr> stmts = new ArrayList<SQLite3StmtPtrPtr>();
        final String[] tail = new String[1];
        do {
            final SQLite3StmtPtrPtr ppStmt = new SQLite3StmtPtrPtr();
            stmts.add(ppStmt);
            final int ret = sqlite3_prepare(db, sql, -1, ppStmt, tail);
            if (ret != SQLITE_OK) {
                for (final SQLite3StmtPtrPtr stmt : stmts) {
                    stmt.delete();
                }
                throw new JdbcSQLException(ret, db);
            }
            sql = tail[0].trim();
        } while (sql.length() > 0);
        
        final List<Statement> ret = new  ArrayList<Statement>(stmts.size());
        for (final SQLite3StmtPtrPtr stmt : stmts) {
            ret.add(new Statement(this, stmt));
        }
        return ret;
    }
    
    /**
     * invoke sqlite3_interrupt() function.
     * @see <a href="http://sqlite.org/c3ref/interrupt.html">Interrupt A Long-Running Query</a>
     */
    public void interrupt() {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        sqlite3_interrupt(db);
    }
    
    /**
     * invoke sqlite3_changes() function.
     * @return the number of database rows that were changed or inserted or deleted by the most recently completed SQL statement on the connection specified.
     * @see <a href="http://sqlite.org/c3ref/changes.html">Count The Number Of Rows Modified</a>
     * @see #totalChanges()
     */
    public int changes() {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        return sqlite3_changes(db);
    }
    
    /**
     * invoke sqlite3_total_changes() function.
     * @return the number of row changes caused by INSERT, UPDATE or DELETE statements since the database handle was opened.
     * @see <a href="http://sqlite.org/c3ref/total_changes.html">Total Number Of Rows Modified</a>
     * @see #changes()
     */
    public int totalChanges() {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        return sqlite3_total_changes(db);
    }
    
    /**
     * invoke sqlite3_last_insert_rowid() function.
     * @return the rowid of the most recent successful INSERT into the database from the database connection.
     * @see <a href="http://sqlite.org/c3ref/last_insert_rowid.html">Last Insert Rowid</a>
     */
    public long lastInsertRowId() {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        return sqlite3_last_insert_rowid(db);
    }
    
    /**
     * invoke sqlite3_table_column_metadata() function.
     * @param dbName database name
     * @param tableName table name
     * @param columnName column name
     * @return the meta-data about a specific column of a specific database table accessible using the connection.
     * @throws java.sql.SQLException When the return value of the sqlite3_table_column_metadata() function is not SQLITE_OK.
     * @see <a href="http://www.sqlite.org/c3ref/table_column_metadata.html">Extract Metadata About A Column Of A Table</a>
     */
    public ColumnMetaData getColumnMetaData(String dbName, String tableName, String columnName) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        SWIGTYPE_p_p_char dataType = null;
        SWIGTYPE_p_p_char collSeq = null;
        SWIGTYPE_p_int notNull = null;
        SWIGTYPE_p_int primaryKey = null;
        SWIGTYPE_p_int autoInc = null;
        try {
            dataType = new_p_p_char();
            collSeq = new_p_p_char();
            notNull = new_p_int();
            primaryKey = new_p_int();
            autoInc = new_p_int();
        
            int ret = sqlite3_table_column_metadata(
                            db, dbName, tableName, columnName,
                            dataType, collSeq, notNull, primaryKey, autoInc
                        );
            if (ret != SQLITE_OK) {
                throw new JdbcSQLException(ret, db);
            }
            
            return new ColumnMetaData(
                            get_p_char(dataType),
                            get_p_char(collSeq),
                            get_int(notNull),
                            get_int(primaryKey),
                            get_int(autoInc)
                        );
        } finally {
            if (dataType != null) {
                delete_p_p_char(dataType);
            }
            if (collSeq != null) {
                delete_p_p_char(collSeq);
            }
            if (notNull != null) {
                delete_p_int(notNull);
            }
            if (primaryKey != null) {
                delete_p_int(primaryKey);
            }
            if (autoInc != null) {
                delete_p_int(autoInc);
            }
        }
    }
    
    /**
     * invoke from Statement#&lt;init&gt;().
     * @param stmt
     * @throws java.sql.SQLException
     */
    void addStatement(Statement stmt) throws SQLException {
        if (statements == null) {
            statements = new ArrayList<Statement>();
        }
        if (statements.contains(stmt)) {
            throw new SQLException("Duplicate sqlite3_stmt handle error.", "90J31");
        }
        statements.add(stmt);
    }
    
    /**
     * invoke from Statement#close().
     * @param stmt
     * @throws java.sql.SQLException
     */
    void removeStatement(Statement stmt) throws SQLException {
        if (statements != null) {
            if (!statements.remove(stmt)) {
                throw new SQLException("Unmanaged sqlite3_stmt handle error.", "90J32");
            }
        }
    }
    
    private void closeStatements() {
        if (statements != null) {
            final List<Statement> list = statements;
            statements = null;
            for (final Statement stmt : list) {
                try {
                    stmt.close();
                } catch (SQLException ex) {
                    Logger.getLogger(Database.class.getName()).info(ex.toString());
                }
            }
        }
    }
    
    private void addFunction(Function func) {
        if (functions == null) {
            functions = new ArrayList<Function>();
        }
        functions.add(func);
    }
    
    private void removeFunction(Function func) {
        if (functions != null) {
            functions.remove(func);
        }
    }
    
    private void dropFunctions() {
        if (functions != null) {
            final List<Function> list = functions;
            functions = null;
            for (final Function func : list) {
                if (func.isRegistered()) {
                    try {
                        dropFunction(func);
                    } catch (SQLException ex) {
                        Logger.getLogger(Database.class.getName()).info(ex.toString());
                    } finally {
                        if (func.isRegistered()) {
                            delete_callback(func);
                        }
                    }
                }
            }
        }
    }
    
    /**
     * invoke sqlite3_create_function() function.
     * @param func User-Defined function
     * @throws java.sql.SQLException When the return value of the sqlite3_create_function() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/create_function.html">Create Or Redefine SQL Functions</a>
     */
    public void createFunction(Function func) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        int ret = sqlite3_create_function(db, func);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(ret, db);
        }
        addFunction(func);
    }
    
    /**
     * invoke sqlite3_create_function() function.
     * @param func User-Defined function
     * @throws java.sql.SQLException When the return value of the sqlite3_create_function() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/create_function.html">Create Or Redefine SQL Functions</a>
     */
    public void dropFunction(Function func) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        int ret = sqlite3_drop_function(db, func);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(ret, db);
        }
        removeFunction(func);
    }
    
    /**
     * invoke sqlite3_create_function() function.
     * @param name the function name
     * @throws java.sql.SQLException When the return value of the sqlite3_create_function() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/create_function.html">Create Or Redefine SQL Functions</a>
     */
    public void dropFunction(String name) throws SQLException {
        if (functions != null) {
            final List<Function> drops = new ArrayList<Function>();
            for (final Function func : functions) {
                if (func.getName().equalsIgnoreCase(name)) {
                    drops.add(func);
                }
            }
            for (final Function func : drops) {
                dropFunction(func);
            }
        }
    }
    
    private void addCollator(Collator col) {
        if (collators == null) {
            collators = new ArrayList<Collator>();
        }
        collators.add(col);
    }
    
    private void removeCollator(Collator col) {
        if (collators != null) {
            collators.remove(col);
        }
    }
    
    private void dropCollators() {
        if (collators != null) {
            final List<Collator> list = collators;
            collators = null;
            for (final Collator col : list) {
                if (col.isRegistered()) {
                    try {
                        dropCollationSequence(col);
                    } catch (SQLException ex) {
                        Logger.getLogger(Database.class.getName()).info(ex.toString());
                    } finally {
                        if (col.isRegistered()) {
                            delete_callback(col);
                        }
                    }
                }
            }
        }
    }
    
    /**
     * invoke sqlite3_create_collation() function.
     * @param col User-Defined Collating Sequences
     * @throws java.sql.SQLException When the return value of the sqlite3_create_collation() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/create_collation.html">Define New Collating Sequences</a>
     */
    public void createCollationSequence(Collator col) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        int ret = sqlite3_create_collation(db, col);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(ret, db);
        }
        addCollator(col);
    }
    
    /**
     * invoke sqlite3_create_function() function.
     * @param col User-Defined Collating Sequences
     * @throws java.sql.SQLException When the return value of the sqlite3_create_collation() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/create_collation.html">Define New Collating Sequences</a>
     */
    public void dropCollationSequence(Collator col) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        int ret = sqlite3_drop_collation(db, col);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(ret, db);
        }
        removeCollator(col);
    }
    
    /**
     * invoke sqlite3_create_function() function.
     * @param name the collation sequence name
     * @throws java.sql.SQLException When the return value of the sqlite3_create_collation() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/create_collation.html">Define New Collating Sequences</a>
     */
    public void dropCollationSequence(String name) throws SQLException {
        if (collators != null) {
            final List<Collator> drops = new ArrayList<Collator>();
            for (final Collator col : collators) {
                if (col.getName().equalsIgnoreCase(name)) {
                    drops.add(col);
                }
            }
            for (final Collator col : drops) {
                dropCollationSequence(col);
            }
        }
    }
    
    /**
     * invoke sqlite3_set_authorizer() function.
     * @param auth authorizer
     * @throws java.sql.SQLException When the return value of the sqlite3_set_authorizer() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/set_authorizer.html">Compile-Time Authorization Callbacks</a>
     */
    public void setAuthorizer(Authorizer auth) throws SQLException {
        clearAuthorizer();
        
        final SWIGTYPE_p_sqlite3 db = getHandle();
        int ret = sqlite3_set_authorizer(db, auth);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(ret, db);
        }
        authorizer = auth;
    }
    
    /**
     * invoke sqlite3_set_authorizer() function.
     * @throws java.sql.SQLException When the return value of the sqlite3_set_authorizer() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/set_authorizer.html">Compile-Time Authorization Callbacks</a>
     */
    public void clearAuthorizer() throws SQLException {
        if (authorizer != null) {
            final SWIGTYPE_p_sqlite3 db = getHandle();
            int ret = sqlite3_clear_authorizer(db, authorizer);
            if (ret != SQLITE_OK) {
                throw new JdbcSQLException(ret, db);
            }
            authorizer = null;
        }
    }

    private void internalClearAuthorizer() {
        try {
            clearAuthorizer();
        } catch (SQLException ex) {
            Logger.getLogger(Database.class.getName()).info(ex.toString());
        } finally {
            if (authorizer != null && authorizer.isRegistered()) {
                delete_callback(authorizer);
                authorizer = null;
            }
        }
    }
    
    /**
     * invoke sqlite3_busy_handler() function.
     * @param busy busy handler
     * @throws java.sql.SQLException When the return value of the sqlite3_busy_handler() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/busy_handler.html">Register A Callback To Handle SQLITE_BUSY Errors</a>
     */
    public void setBusyHandler(BusyHandler busy) throws SQLException {
        clearBusyHandler();
        
        final SWIGTYPE_p_sqlite3 db = getHandle();
        int ret = sqlite3_busy_handler(db, busy);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(ret, db);
        }
        busyHandler = busy;
        timeout = -1;
    }
    
    /**
     * invoke sqlite3_set_authorizer() function.
     * @throws java.sql.SQLException When the return value of the sqlite3_busy_handler() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/busy_handler.html">Register A Callback To Handle SQLITE_BUSY Errors</a>
     */
    public void clearBusyHandler() throws SQLException {
        if (busyHandler != null) {
            final SWIGTYPE_p_sqlite3 db = getHandle();
            int ret = sqlite3_clear_busy_handler(db, busyHandler);
            if (ret != SQLITE_OK) {
                throw new JdbcSQLException(ret, db);
            }
            busyHandler = null;
        }
    }

    private void internalClearBusyHandler() {
        try {
            clearBusyHandler();
        } catch (SQLException ex) {
            Logger.getLogger(Database.class.getName()).info(ex.toString());
        } finally {
            if (busyHandler != null && busyHandler.isRegistered()) {
                delete_callback(busyHandler);
                busyHandler = null;
            }
        }
    }
    
    /**
     * invoke sqlite3_collation_needed() function.
     * @param needed the CollationNeededHandler object
     * @throws java.sql.SQLException When the return value of the sqlite3_collation_needed() function is not SQLITE_OK.
     * @see <a href="http://www.sqlite.org/c3ref/collation_needed.html">Collation Needed Callbacks</a>
     */
    public void setCollationNeededHandler(CollationNeededHandler needed) throws SQLException {
        clearCollationNeededHandler();
        
        final SWIGTYPE_p_sqlite3 db = getHandle();
        int ret = sqlite3_collation_needed(db, needed);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(ret, db);
        }
        collNeeded = needed;        
        collNeeded.setDatabase(this);
    }
    
    /**
     * invoke sqlite3_collation_needed() function.
     * @throws java.sql.SQLException When the return value of the sqlite3_collation_needed() function is not SQLITE_OK.
     * @see <a href="http://www.sqlite.org/c3ref/collation_needed.html">Collation Needed Callbacks</a>
     */
    public void clearCollationNeededHandler() throws SQLException {
        if (collNeeded != null) {
            final SWIGTYPE_p_sqlite3 db = getHandle();
            int ret = sqlite3_clear_collation_needed(db, collNeeded);
            if (ret != SQLITE_OK) {
                throw new JdbcSQLException(ret, db);
            }
            collNeeded.setDatabase(null);
            collNeeded = null;
        }
    }

    private void internalClearCollationNeededHandler() {
        try {
            clearCollationNeededHandler();
        } catch (SQLException ex) {
            Logger.getLogger(Database.class.getName()).info(ex.toString());
        } finally {
            if (collNeeded != null && collNeeded.isRegistered()) {
                delete_callback(collNeeded);
                collNeeded.setDatabase(null);
                collNeeded = null;
            }
        }
    }
    
    /**
     * invoke sqlite3_progress_handler() function.
     * @param prog progress handler
     * @see <a href="http://sqlite.org/c3ref/progress_handler.html">Query Progress Callbacks</a>
     */
    public void setProgressHandler(ProgressHandler prog) {
        clearProgressHandler();
        
        final SWIGTYPE_p_sqlite3 db = getHandle();
        sqlite3_progress_handler(db, prog);
        progressHandler = prog;
    }
    
    /**
     * invoke sqlite3_progress_handler() function.
     * @see <a href="http://sqlite.org/c3ref/progress_handler.html">Query Progress Callbacks</a>
     */
    public void clearProgressHandler() {
        if (progressHandler != null) {
            final SWIGTYPE_p_sqlite3 db = getHandle();
            sqlite3_clear_progress_handler(db, progressHandler);
            if (progressHandler.isRegistered()) {
                delete_callback(progressHandler);
            }
            progressHandler = null;
        }
    }
    
    /**
     * invoke sqlite3_commit_hook() function.
     * @param hook commit hoot
     * @see <a href="http://sqlite.org/c3ref/commit_hook.html">Commit And Rollback Notification Callbacks</a>
     */
    public void setCommitHook(CommitHook hook) {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        sqlite3_commit_hook(db, hook);
    }
    
    /**
     * invoke sqlite3_commit_hook() function.
     * @see <a href="http://sqlite.org/c3ref/commit_hook.html">Commit And Rollback Notification Callbacks</a>
     */
    public void clearCommitHook() {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        sqlite3_clear_commit_hook(db);
    }
    
    /**
     * invoke sqlite3_rollback_hook() function.
     * @param hook rollback hoot
     * @see <a href="http://sqlite.org/c3ref/commit_hook.html">Commit And Rollback Notification Callbacks</a>
     */
    public void setRollbackHook(RollbackHook hook) {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        sqlite3_rollback_hook(db, hook);
    }
    
    /**
     * invoke sqlite3_rollback_hook() function.
     * @see <a href="http://sqlite.org/c3ref/commit_hook.html">Commit And Rollback Notification Callbacks</a>
     */
    public void clearRollbackHook() {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        sqlite3_clear_rollback_hook(db);
    }
    
    /**
     * invoke sqlite3_update_hook() function.
     * @param hook update hoot
     * @see <a href="http://sqlite.org/c3ref/update_hook.html">Data Change Notification Callbacks</a>
     */
    public void setUpdateHook(UpdateHook hook) {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        sqlite3_update_hook(db, hook);
    }
    
    /**
     * invoke sqlite3_update_hook() function.
     * @see <a href="http://sqlite.org/c3ref/update_hook.html">Data Change Notification Callbacks</a>
     */
    public void clearUpdateHook() {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        sqlite3_clear_update_hook(db);
    }
    
    /**
     * invoke sqlite3_profile() function.
     * @param profiler profiler
     * @see <a href="http://sqlite.org/c3ref/profile.html">Tracing And Profiling Functions</a>
     */
    public void setProfiler(Profiler profiler) {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        sqlite3_profile(db, profiler);
    }
    
    /**
     * invoke sqlite3_profile() function.
     * @see <a href="http://sqlite.org/c3ref/profile.html">Tracing And Profiling Functions</a>
     */
    public void clearProfiler() {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        sqlite3_clear_profile(db);
    }
    
    /**
     * invoke sqlite3_trace() function.
     * @param tracer tracer
     * @see <a href="http://sqlite.org/c3ref/profile.html">Tracing And Profiling Functions</a>
     */
    public void setTracer(Tracer tracer) {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        sqlite3_trace(db, tracer);
    }
    
    /**
     * invoke sqlite3_trace() function.
     * @see <a href="http://sqlite.org/c3ref/profile.html">Tracing And Profiling Functions</a>
     */
    public void clearTracer() {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        sqlite3_clear_trace(db);
    }
    
    /**
     * invoke sqlite3_enable_shared_cache(on) function.
     * @throws java.sql.SQLException When the return value of the sqlite3_enable_shared_cache() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/enable_shared_cache.html">Enable Or Disable Shared Pager Cache</a>
     */
    public static void enableSharedCache() throws SQLException {
        final int ret = sqlite3_enable_shared_cache(1);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(ret);
        }
    }
    
    /**
     * invoke sqlite3_enable_shared_cache(off) function.
     * @throws java.sql.SQLException When the return value of the sqlite3_enable_shared_cache() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/enable_shared_cache.html">Enable Or Disable Shared Pager Cache</a>
     */
    public static void disableSharedCache() throws SQLException {
        final int ret = sqlite3_enable_shared_cache(0);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(ret);
        }
    }
    
    /**
     * invoke sqlite3_errcode() function.
     * @return the numeric result code or extended result code for the most recent failed sqlite3_* API call associated with sqlite3 handle 'db'.
     * @see <a href="http://sqlite.org/c3ref/errcode.html">Error Codes And Messages</a>
     * @see <a href="http://sqlite.org/c3ref/c_abort.html">Result Codes</a>
     * @see <a href="http://sqlite.org/c3ref/c_ioerr_blocked.html">Extended Result Codes</a>
     * @see #enableExtendedResultCodes()
     * @see #disableExtendedResultCodes()
     */
    public int getLastError() {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        return sqlite3_errcode(db);
    }
    
    /**
     * invoke sqlite3_errmsg() function.
     * @return the numeric result code or extended result code for the most recent failed sqlite3_* API call associated with sqlite3 handle 'db'.
     * @see <a href="http://sqlite.org/c3ref/errcode.html">Error Codes And Messages</a>
     * @see #getLastError()
     * @see #enableExtendedResultCodes()
     * @see #disableExtendedResultCodes()
     */
    public String getLastErrorMessage() {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        return sqlite3_errmsg(db);
    }
    
    /**
     * invoke sqlite3_get_table() function.
     * @param sql SQL to be evaluated
     * @param errmsg Error message written here
     * @return Results of the query
     * @throws java.sql.SQLException When the return value of the sqlite3_get_table() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/free_table.html">Convenience Routines For Running Queries</a>
     */
    public List<String[]> getTable(String sql, SWIGTYPE_p_p_char errmsg) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        final List<String[]> result = new ArrayList<String[]>();
        int ret = 0;
        while (((ret = sqlite3_get_table(db, sql, result, errmsg)) == SQLITE_BUSY)
                && (timeout == 0)) {
            // waiting...
        }
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(ret, db);
        }
        return result;
    }
    
    /**
     * invoke sqlite3_extended_result_codes(on) function.
     * @throws java.sql.SQLException When the return value of the sqlite3_extended_result_codes() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/extended_result_codes.html">Enable Or Disable Extended Result Codes</a>
     */
    public void enableExtendedResultCodes() throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        final int ret = sqlite3_extended_result_codes(db, 1);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(ret, db);
        }
    }
    
    /**
     * invoke sqlite3_extended_result_codes(off) function.
     * @throws java.sql.SQLException When the return value of the sqlite3_extended_result_codes() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/extended_result_codes.html">Enable Or Disable Extended Result Codes</a>
     */
    public void disableExtendedResultCodes() throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        final int ret = sqlite3_extended_result_codes(db, 0);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(ret, db);
        }
    }
    
    /**
     * invoke sqlite3_enable_load_extension(on) function.
     * @throws java.sql.SQLException When the return value of the sqlite3_enable_load_extension() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/enable_load_extension.html">Enable Or Disable Extension Loading</a>
     */
    public void enableLoadExtention() throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        final int ret = sqlite3_enable_load_extension(db, 1);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(ret, db);
        }
    }
    
    /**
     * invoke sqlite3_enable_load_extension(off) function.
     * @throws java.sql.SQLException When the return value of the sqlite3_enable_load_extension() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/enable_load_extension.html">Enable Or Disable Extension Loading</a>
     */
    public void disableLoadExtention() throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        final int ret = sqlite3_enable_load_extension(db, 0);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(ret, db);
        }
    }
    
    /**
     * invoke sqlite3_load_extension() function.
     * @param filename the Name of the shared library containing extension
     * @param entryPoint the Entry point. Use "sqlite3_extension_init" if null.
     * @param errmsg Error message written here
     * @throws java.sql.SQLException When the return value of the sqlite3_load_extension() function is not SQLITE_OK.
     * @see <a href="http://sqlite.org/c3ref/load_extension.html">Load An Extension</a>
     */
    public void loadExtention(String filename, String entryPoint, SWIGTYPE_p_p_char errmsg) throws SQLException {
        final SWIGTYPE_p_sqlite3 db = getHandle();
        final int ret = sqlite3_load_extension(db, filename, entryPoint, errmsg);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(ret, db);
        }
    }
    
    /**
     * invoke sqlite3_reset_auto_extension() function.
     * @see <a href="http://www.sqlite.org/c3ref/reset_auto_extension.html">Reset Automatic Extension Loading</a>
     */
    public static void resetAutoExtention() {
        sqlite3_reset_auto_extension();
    }
    
    /**
     * Close database if database is not closed yet.
     * @throws java.lang.Throwable
     * @see #close()
     */
    @Override
    protected void finalize() throws Throwable {
        if (!isClosed()) {
            Logger.getLogger(Database.class.getName()).severe("Database connection has leaked!");
            close();
        }
        super.finalize();
    }
    
}