/*

 QuestionnaireStructure.java
 
 Copyright 2004 KUBO Hiroya (hiroya@sfc.keio.ac.jp).
 
 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at
 
 http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 
 Created on 2004/07/03

 */
package net.sf.sqs_xml.reader.logic;

import net.sf.sqs_xml.translator.logic.PDFtoSQMTranslator;
import net.sf.sqs_xml.translator.logic.TranslatorException;
import net.sf.sqs_xml.translator.logic.TranslatorResource;
import net.sf.sqs_xml.util.DOMUtil;
import net.sf.sqs_xml.xmlns.SQSNamespaces;

import org.apache.avalon.framework.logger.ConsoleLogger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.FactoryConfigurationError;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;


import java.awt.Point;
import java.awt.Rectangle;
import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.Map.Entry;

/**
 * @author hiroya
 *
 */
public class QuestionnaireSchema {
    Document document;
    LinkedList columnNames = null;
    HashMap pageToIDListMap = new LinkedHashMap();
    HashMap qidToIDListMap = new LinkedHashMap();
    HashMap idToPageMap = new HashMap();
    HashMap idToElementMap = new HashMap();
    HashMap idToIDIndexMap = new HashMap();
    
    static final int TYPE_MAP_VALUE_IS_FORM = 0;
    static final int TYPE_MAP_VALUE_IS_RECT = 1;
    
    public QuestionnaireSchema(File pdffile, String baseURI) throws TranslatorException {
        try {
            document = createDocument(pdffile, baseURI);
            initIDMap();
            initQIDMap();
            this.columnNames = createColumnNames();
        } catch (IOException ex) {
            throw new TranslatorException(ex);
        } catch (TranslatorException e) {
            e.printStackTrace();
        } catch (SAXException e) {
            e.printStackTrace();
        } catch (TransformerException e) {
            e.printStackTrace();
        } catch (ParserConfigurationException e) {
            e.printStackTrace();
        } catch (FactoryConfigurationError e) {
            e.printStackTrace();
        }
    }
    
    public int getIDIndex(String id){
        return ((Integer)idToIDIndexMap.get(id)).intValue();
    }
    
    /**
     * @param pdffile
     * @throws TranslatorException
     * @throws FileNotFoundException
     * @throws FactoryConfigurationError
     * @throws SAXException
     * @throws IOException
     * @throws ParserConfigurationException
     */
    private Document createDocument(File pdffile, String baseURI) throws TranslatorException, FileNotFoundException, FactoryConfigurationError, SAXException, IOException, ParserConfigurationException {
        PDFtoSQMTranslator translator = new PDFtoSQMTranslator();
        String[] xsltFileNames = PDFtoSQMTranslator.BOOKMARK2SQM.split(" ");
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        translator.translate(new BufferedInputStream(new FileInputStream(pdffile)),
                new TranslatorResource(new ConsoleLogger(),
                xsltFileNames), outputStream, baseURI);
        ByteArrayInputStream inputStream = new ByteArrayInputStream(
                outputStream.toByteArray());

        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        factory.setNamespaceAware(true);
        return factory.newDocumentBuilder().parse(inputStream);
    }

    /**
     * @throws JaxenException
     */
    private void initQIDMap() {
        for(Iterator it = pageToIDListMap.entrySet().iterator(); it.hasNext(); ){
            Entry entry = (Entry)it.next();
            Integer page = (Integer)entry.getKey();
            List idList = (List)entry.getValue();
            for(int i=0; i<idList.size(); i++){
                String id = (String)idList.get(i);
                int p = id.lastIndexOf("/v");
                if(0 < p){
                    String qid = id.substring(0, p);
                    List qidList = (List)qidToIDListMap.get(qid);
                    if(qidList == null){
                        qidList = new LinkedList();
                        qidToIDListMap.put(qid, qidList);
                    }
                    qidList.add(id);
                    idToPageMap.put(qid, page);
                    //System.err.println("page "+page+" * "+qid);
                }else{
                    qidToIDListMap.put(id, null);
                }
            }
        }
    }
    
    private void initIDMap() throws TransformerException{
        int num = getNumberOfPages();
        NodeList gList = DOMUtil.getElementList(document.getDocumentElement(), SQSNamespaces.SVG_URI, "g");
        int i = 0;
        for(int pageNumber = 1; pageNumber <= num; pageNumber++){
            LinkedList idList = new LinkedList();
            pageToIDListMap.put(new Integer(pageNumber), idList);
            for(int j = 0; i < gList.getLength(); i++){
                Element gElem = (Element)gList.item(i);
                Element e2 = DOMUtil.getElement(gElem, SQSNamespaces.SVG_URI, "metadata");
                Element e3 = DOMUtil.getElement(e2, SQSNamespaces.SQS2004READER_URI, "page");
                if(Integer.parseInt(e3.getAttribute("number")) == pageNumber){
                    String id = gElem.getAttribute("id");
                    idList.add(id);
                    idToElementMap.put(id, gElem);
                    idToIDIndexMap.put(id, new Integer(j));
                    idToPageMap.put(id, new Integer(pageNumber));
                    j++;
                }else{
                    break;
                }
            }
        }
    }
    
    LinkedHashMap getFormElements(int pageNumber){
        return getArea(pageNumber, TYPE_MAP_VALUE_IS_FORM);
    }

    LinkedHashMap getRectangle(int pageNumber){
        return getArea(pageNumber, TYPE_MAP_VALUE_IS_RECT);
    }

    private LinkedHashMap getArea(int pageNumber, int type){
        try {
            LinkedHashMap ret = new LinkedHashMap();
            List list = null;
            List idList = (List)pageToIDListMap.get(new Integer(pageNumber));
            for(int i=0; i<idList.size(); i++){
                String id = (String)idList.get(i);
                if(type == TYPE_MAP_VALUE_IS_FORM){
                    Element gElem = (Element)idToElementMap.get(id);
                    Element mElem = DOMUtil.getElement(DOMUtil.getElement(gElem, SQSNamespaces.SVG_URI, "rect"), SQSNamespaces.SVG_URI, "metadata");
                    ret.put(id, mElem.getFirstChild());
                }else{
                    Element gElem = (Element)idToElementMap.get(id);
                    Element rElem = DOMUtil.getElement(gElem, SQSNamespaces.SVG_URI, "rect");
                    Rectangle rect = createRectangle(rElem);
                    ret.put(id, rect);
                }
            }
            return ret;
        } catch (NullPointerException e) {
            e.printStackTrace();
            return null;
        } 
    }
    
    Point[] getMasterPoints(){
        try{
            Element e1 = DOMUtil.getElement(document.getDocumentElement(), SQSNamespaces.SVG_URI, "metadata");
            Element elem = DOMUtil.getElement(e1, SQSNamespaces.SQS2004READER_URI, "page");
            Point[] ret = new Point[4];
            for(int i = 0; i < 4; i++){
                ret[i] = createPoint(elem, i+1);
            }
            return ret;
        }catch(NullPointerException e){
            e.printStackTrace();
            return null;
        }
    }
    
    private Point createPoint(Element elem, int n){
        return new Point(Integer.parseInt(elem.getAttribute("x"+n)),
                		Integer.parseInt(elem.getAttribute("y"+n)));        
    }
    
    private static Rectangle createRectangle(Element rElem){
        int x = Integer.parseInt(rElem.getAttribute("x"));
        int y = Integer.parseInt(rElem.getAttribute("y"));
        int width = Integer.parseInt(rElem.getAttribute("width"));
        int height = Integer.parseInt(rElem.getAttribute("height"));
        return new Rectangle(x, y, width, height);
    }

    int getNumberOfPages(){
        try {
            Element e1 = DOMUtil.getElement(document.getDocumentElement(), SQSNamespaces.SVG_URI, "metadata");
            Element elem = DOMUtil.getElement(e1, SQSNamespaces.SQS2004READER_URI, "page");
            String value = elem.getAttribute("total");
            return Integer.parseInt(value);
        } catch (NumberFormatException e) {
            e.printStackTrace();
        } catch (NullPointerException e) {
            e.printStackTrace();
        }
        return -1;
    }
    
    Set getQIDSet(){
        // this method may be used by select1-schema validator
        return (Set)qidToIDListMap.keySet();
    }

    List getIDList(String qid){
        // this method may be used by select1-schema validator
        return (List)qidToIDListMap.get(qid);
    }

    Element getElement(String id){
        // this method may be used by select1-schema validator
        return (Element)idToElementMap.get(id);
        //return (Element)document.getElementById(id);
    }

    /*
    int getPage(Element elem)throws TransformerException{
        Element e1 = DOMUtil.getElement(elem, SQSNamespaces.SVG_URI, "metadata");
        Element e2 = DOMUtil.getElement(e1, SQSNamespaces.SQS2004READER_URI, "page");
        return Integer.parseInt(e2.getAttribute("number"));
    }
    */
    
    int getPage(String id)throws TransformerException{
        //System.err.println(qid+" : "+getElement(qid));
        return ((Integer)idToPageMap.get(id)).intValue();
    }
    
    Element getFormElement(Element elem)throws TransformerException{
        Element e1 = DOMUtil.getElement(elem, SQSNamespaces.SVG_URI, "rect");
        Element e2 = DOMUtil.getElement(e1, SQSNamespaces.SVG_URI, "metadata");
        return (Element)e2.getFirstChild();
    }

    private LinkedList createColumnNames()throws TransformerException{
        LinkedList columnNames = new LinkedList();
        for (Iterator it = getQIDSet().iterator(); it.hasNext();) {
            String qid = (String) it.next();
            CellValue cellValue = null;
            List list = getIDList(qid);
            if(list != null){
                for (int itemIndex = 0; itemIndex < list.size(); itemIndex++) {
                    String id = (String) list.get(itemIndex);
                    cellValue = createCellOptionValue(qid, id, itemIndex);
                }
                addColumnNames(columnNames, qid, cellValue, list);
            }else{
                columnNames.add(qid);
            }
        }
        return columnNames;
    }
    
    private void addColumnNames(LinkedList columnNames, String qid, CellValue cellValue, List list) {
        if(cellValue.type.equals("select1")){
            columnNames.add(qid);
        }else if(cellValue.type.equals("select")){
            for (int itemIndex = 0; itemIndex < list.size(); itemIndex++) {
                columnNames.add(list.get(itemIndex));
            }
        }
    }

    int getNumberOfColumns(){
        return columnNames.size();
    }
    
    CellValue createCellOptionValue(String qid, String id, int index) throws TransformerException {
        Element elem = getElement(id);
        int page = getPage(id);
        Element formElem = getFormElement(elem);
        String type = formElem.getLocalName();
        String value = id.substring(qid.length()+2);
        return new CellValueOption(page, qid, type, id, index, value);
    }
}
