/*

 ReaderBaseTask.java
 
 Copyright 2004 KUBO Hiroya (hiroya@sfc.keio.ac.jp).
 
 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at
 
 http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 
 Created on 2004/10/13

 */
package net.sf.sqs_xml.reader.logic;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.util.Date;
import java.util.List;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;

import net.sf.sqs_xml.httpd.MarkReaderHttpd;
import net.sf.sqs_xml.reader.swing.ProgressLoggerConsoleFrame;
import net.sf.sqs_xml.translator.logic.TranslatorException;
import net.sf.sqs_xml.util.FileUtil;

abstract class ReaderBaseTask extends ReaderTask{    
    public abstract void processErrorImages()throws ThreadSuspendingException;
    public abstract void showResult();
    public abstract void save() throws TranslatorException, TransformerException, IOException, ParserConfigurationException, TransformerException, ThreadSuspendingException ;
    public abstract void processCoreTasks() throws ThreadSuspendingException, TranslatorException, TransformerException;
    public abstract void checkImageFiles() throws ThreadSuspendingException, TranslatorException;
    
    /*
    public static void main(String[] args)throws Exception{
        File imagedir = new File(args[0]);
        File srcfile = null;
        File tgtfile = (1 < args.length)? new File(args[1]):null;
        
        List srcFileList = FileUtil.find(imagedir, ".txt");
        if(srcFileList.size() == 1){
            srcfile = (File)srcFileList.get(0);
            tgtfile = (tgtfile == null)? srcfile : tgtfile;
        }
        BaseTask task = new BaseTask();
        reader.run(imagedir, srcfile, tgtfile);
    }
    */
    
    String textareaImageDirPath;
    String errorImageDirPath;

    InitQuestionnaireStructureTaskTarget initQuestionnaireStructureTaskTarget = new InitQuestionnaireStructureTaskTarget();
    FindImageFilesTaskTarget findImageFilesTaskTarget = new FindImageFilesTaskTarget();
    
    class InitQuestionnaireStructureTaskSource extends ReaderTaskSource{
        File pdffile;
        InitQuestionnaireStructureTaskSource(File pdffile){
            this.pdffile = pdffile;
        }
    }
    
    class InitQuestionnaireStructureTaskTarget extends ReaderTaskTarget{
        QuestionnaireSchema qSchema;
        ReaderResultBuilder resultBuilder = new ReaderResultBuilder();
    }
    
    class InitQuestionnaireStructureTask extends ReaderTask{
        InitQuestionnaireStructureTask(InitQuestionnaireStructureTaskSource source, InitQuestionnaireStructureTaskTarget target, ReaderProcessView view){
            super((ReaderTaskSource)source, (ReaderTaskTarget)target, view);
        }
        
        public InitQuestionnaireStructureTaskSource getSource(){
            return (InitQuestionnaireStructureTaskSource)source;
        }
        public InitQuestionnaireStructureTaskTarget getTarget(){
            return (InitQuestionnaireStructureTaskTarget)target;
        }
        
        void run() throws Exception{
            view.setProgressState(1, ProgressLoggerConsoleFrame.PROCESSING);
            //List pdffiles = FileUtil.find(getSource().imagedir, ".pdf");
            //for(int i = 0; i< pdffiles.size(); i++){
            //    File pdffile = (File)pdffiles.get(i);           
            //}
            getTarget().qSchema = new QuestionnaireSchema(getSource().pdffile, MarkReaderHttpd.getSingleton().getBase());
            getTarget().resultBuilder.setQuestionnaireSchema(getTarget().qSchema);
            view.setPDFFileInfo(getSource().pdffile, getTarget().qSchema);

            view.setProgressState(1, ProgressLoggerConsoleFrame.SUCCEED);
        }
    }
    
    class FindImageFilesTask extends ReaderTask{
        FindImageFilesTask(FindImageFilesTaskSource source, FindImageFilesTaskTarget target, ReaderProcessView view){
            super((ReaderTaskSource)source, (ReaderTaskTarget)target, view);
        }
        
        public FindImageFilesTaskSource getSource(){
            return (FindImageFilesTaskSource)source;
        }
        public FindImageFilesTaskTarget getTarget(){
            return (FindImageFilesTaskTarget)target;
        }
        
        void run() throws TranslatorException{
            view.setProgressState(2, ProgressLoggerConsoleFrame.PROCESSING);
            getTarget().fileList = FileUtil.find(getSource().imagedir, 
                    new FileFilter(){
                		public boolean accept(File file){
                		    String filename = file.getName().toLowerCase();
                		    return filename.endsWith(".tif") || filename.endsWith(".png") || filename.endsWith(".jpg") || filename.endsWith(".gif") ;
                		}
        	});
            int numFiles = getTarget().fileList.size();
            int numPages = getSource().qSchema.getNumberOfPages();
            view.initProgressBar(numFiles, numPages);
            if(numFiles % numPages == 0){
                view.setFiles(numFiles);
            }else{
                //engineState.getState().setNumErrorPages(-1);
                view.setProgressState(2, ProgressLoggerConsoleFrame.FAIL);
                view.setPageSetError(numFiles, numPages);
                view.throwFileNumberError(numFiles, numPages);
            }
            view.setProgressState(2, ProgressLoggerConsoleFrame.SUCCEED);
        }
    }

    class FindImageFilesTaskSource extends ReaderTaskSource{
        File imagedir;
        QuestionnaireSchema qSchema;
        FindImageFilesTaskSource(File imagedir, QuestionnaireSchema qSchema){
            this.imagedir = imagedir;
            this.qSchema = qSchema;
        }
    }
    
    class FindImageFilesTaskTarget extends ReaderTaskTarget{
        List fileList = null;
    }
    
    public ReaderBaseTask(BaseTaskSource source, BaseTaskTarget target, ReaderProcessView view){
        super((ReaderTaskSource)source, (ReaderTaskTarget)target, view);
    }
    
    public BaseTaskSource getSource(){
        return (BaseTaskSource)source;
    }
    public BaseTaskTarget getTarget(){
        return (BaseTaskTarget)target;
    }
    
    void run()throws Exception{
        //Calendar now = null;
        //now = Date.getInstance();
        //long startTime = now.getTimeInMillis();
        System.err.println(new Date());
        try{
            process();
        }catch(TranslatorException ex){
            processException(ex);
            processErrorImages();
            throw ex;
        }catch(TransformerException ex){
            processException(ex);
        }catch(ParserConfigurationException ex){
            processException(ex);
        }catch(IOException ex){
            processException(ex);
        }catch(ThreadSuspendingException ex){
            throw ex;
        }catch(Exception ex){
            ex.printStackTrace();
            throw ex;
        }finally{
            //processTestImageFiles();
            removeTextareaImageDirWhenEmpty();
            removeErrorImageDirWhenEmpty();
            showResult();
            System.err.println(new Date());
        }
    }
    
    private void process() throws TranslatorException, TransformerException, ThreadSuspendingException, TransformerException, ParserConfigurationException, IOException {
        init();
        findImageFiles();
        checkImageFiles();
        processCoreTasks();
        processErrorImages();
        save();
        view.setSucceed();
    }

    private void init() throws TranslatorException {
        view.setImagedir(getSource().srcdir);
        if(getSource().pdffileList.size() == 0){
            view.throwNoPDFException();
        }
        File pdffile = null;
        for (int i = 0; i < getSource().pdffileList.size(); i++) {
            try {
                pdffile = (File) getSource().pdffileList.get(i);
                System.err.println(i+"/"+getSource().pdffileList.size()+" "+pdffile);
                view.setPDFFile(pdffile);
                initQuestionnaireStructure(pdffile);
                initTextareaImageDir();
                initErrorImageDir();
                return;
            } catch (Exception e) {
                try{
                    view.throwIncorrectPDFException(pdffile);
                }catch(Exception ignore){}
                pdffile = null;
            }
        }
        if(pdffile == null){
            view.throwNoPDFException();
            return;
        }
    }

    private void processException(Exception ex) throws Exception {
        ex.printStackTrace();
        view.errorDialog(ex, false);
    }


    private void initQuestionnaireStructure(File pdffile) throws Exception {
        new InitQuestionnaireStructureTask(new InitQuestionnaireStructureTaskSource(pdffile),
        		   initQuestionnaireStructureTaskTarget, view).run();
    }

    private void findImageFiles() throws TranslatorException {
        new FindImageFilesTask(new FindImageFilesTaskSource(getSource().srcdir, initQuestionnaireStructureTaskTarget.qSchema),                    
                findImageFilesTaskTarget, view).run();
    }

    private void initTextareaImageDir() {
        this.textareaImageDirPath = getSource().tgtdir.getAbsolutePath()+File.separator+"TEXTAREA";
        FileUtil.createDirectory(textareaImageDirPath);
        FileUtil.clearDirectory(textareaImageDirPath);
    }

    private void initErrorImageDir() {
        this.errorImageDirPath = getSource().tgtdir.getAbsolutePath()+File.separator+"ERROR";
        FileUtil.createDirectory(errorImageDirPath);
        FileUtil.clearDirectory(errorImageDirPath);
    }
    
    private void removeTextareaImageDirWhenEmpty(){
        try{
            new File(textareaImageDirPath).delete();
        }catch(Exception ignore){}
    }

    private void removeErrorImageDirWhenEmpty(){
        try{
            new File(errorImageDirPath).delete();
        }catch(Exception ignore){}
    }

    void checkTaskStopRequest()throws ThreadSuspendingException{
        if(getTarget().state.taskStopRequested()){
            //prevReadData = readData;
            throw new ThreadSuspendingException(); 
        }
        Thread.yield();
    }
}