/*

 ReaderCoreTask.java
 
 Copyright 2004 KUBO Hiroya (hiroya@sfc.keio.ac.jp).
 
 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at
 
 http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 
 Created on 2004/11/01

 */
package net.sf.sqs_xml.reader.logic;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.geom.Point2D;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.sql.Timestamp;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.xml.transform.TransformerException;

import net.sf.sqs_xml.image.RasterFactory;
import net.sf.sqs_xml.image.RasterSource;
import net.sf.sqs_xml.reader.model.FormElementsMarkarea;
import net.sf.sqs_xml.reader.model.FormElementsTextarea;
import net.sf.sqs_xml.reader.model.IFormElementsChoice;
import net.sf.sqs_xml.reader.model.ImageFiles;
import net.sf.sqs_xml.reader.model.ImageFilesImagefile;
import net.sf.sqs_xml.reader.model.Page;
import net.sf.sqs_xml.reader.model.ScanFrame;
import net.sf.sqs_xml.reader.model.TextareaImage;
import net.sf.sqs_xml.reader.swing.ProgressLoggerConsoleFrame;
import net.sf.sqs_xml.translator.logic.TranslatorException;
import net.sf.sqs_xml.util.FileUtil;
import net.sf.sqs_xml.util.StringUtil;
import net.sf.sqs_xml.xmlns.SQSNamespaces;

import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * @author hiroya
 *
 */

public abstract class ReaderCoreTask extends FilteredReaderBaseTask{
    public static double MARK_RECOGNIZE_DENSITY = 0.25; // 0〜0.25:not marked / 0.25〜1.0:marked
    public static double DOUBLE_MARK_RECOGNIZE_DENSITY = 0.15; // warn when ((max - second max) < DOUBLE_MARK_RECOGNIZE_DENSITY)
    public static boolean GUIDE_SQUARE_SHAPE_CHECKER_ENABLED = false;
    
    static final String FORMVALUE_NON_SELECTED = "";
    static final String FORMVALUE_SELECTED = "1";
    static final String FORMTYPE_SELECT = "select";
    static final String FORMTYPE_SELECT1 = "select1";
    static final String FORMTYPE_INPUT = "input";
    static final String FORMTYPE_TEXTAREA = "textarea";
    public static int BLACK_THRESHOLD = 190; // 0〜190:black / 191〜255:white

    ProcessScanFramesTaskTarget processScanFramesTaskTarget = new ProcessScanFramesTaskTarget();
    ProcessAreaTaskTarget processAreaTaskTarget = new ProcessAreaTaskTarget(); 
    ProcessValuesTaskTarget processValuesTaskTarget = new ProcessValuesTaskTarget(); 
    ProcessErrorImagesTaskTarget processErrorImagesTaskTarget = new ProcessErrorImagesTaskTarget();

    public ReaderCoreTask(BaseTaskSource source, BaseTaskTarget target, ReaderProcessView view){
        super(source, target, view);
    }

    public void processErrorImages() throws ThreadSuspendingException {
        new ProcessErrorImagesTask(new ProcessErrorImagesTaskSource(getSource().srcdir, initQuestionnaireStructureTaskTarget.qSchema,
                checkImageFilesTaskTarget.imageFiles,
                processScanFramesTaskTarget.mapper, initQuestionnaireStructureTaskTarget.resultBuilder), processErrorImagesTaskTarget, view).run();
    }

    private void processValues() throws ThreadSuspendingException, TransformerException {
        new ProcessValuesTask(new ProcessValuesTaskSource(checkImageFilesTaskTarget.imageFiles.getImageFilesImagefile().length,
                processScanFramesTaskTarget.mapper,
                initQuestionnaireStructureTaskTarget.qSchema,
                checkImageFilesTaskTarget.imageFiles,
                initQuestionnaireStructureTaskTarget.resultBuilder),
                processValuesTaskTarget, view).run();
    }

    private void processArea() throws ThreadSuspendingException {
        //processScanFramesTaskTarget.mapper.setStart(new Timestamp(startTime));
        new ProcessAreaTask(new ProcessAreaTaskSource(getSource().srcdir, initQuestionnaireStructureTaskTarget.qSchema,
                checkImageFilesTaskTarget.imageFiles, processScanFramesTaskTarget.mapper,
                initQuestionnaireStructureTaskTarget.qSchema.getMasterPoints(),
                initQuestionnaireStructureTaskTarget.resultBuilder),
                processAreaTaskTarget, view).run();
    }

    private void processScanFrames() throws ThreadSuspendingException, TranslatorException {
        new ProcessScanFramesTask(new ProcessScanFramesTaskSource(getSource().srcdir,
                initQuestionnaireStructureTaskTarget.qSchema.getMasterPoints(),
                checkImageFilesTaskTarget.imageFiles), processScanFramesTaskTarget, view).run();
    }

    public void showResult(){
        int numFiles = checkImageFilesTaskTarget.imageFiles.getImageFilesImagefile().length;
        int numValidFiles = processScanFramesTaskTarget.mapper.getValidFileIndexSize();
        int numPages = initQuestionnaireStructureTaskTarget.qSchema.getNumberOfPages();
        int numSamples = numValidFiles / numPages;
        int errors = processScanFramesTaskTarget.mapper.getNumErrorPages();
        int warns = processScanFramesTaskTarget.mapper.getNumErrorMarkSets(); 
        view.showResult(errors, warns, numFiles, numValidFiles, numPages, numSamples);
    }
    
    class ProcessScanFramesTask extends ReaderTask{
        ProcessScanFramesTask(ProcessScanFramesTaskSource source, ProcessScanFramesTaskTarget target, ReaderProcessView view){
            super(source, target, view);
        }
        
        ProcessScanFramesTaskSource getSource(){
            return (ProcessScanFramesTaskSource)source;
        }
        ProcessScanFramesTaskTarget getTarget(){
            return (ProcessScanFramesTaskTarget)target;
        }
        
        void run() throws ThreadSuspendingException, TranslatorException {
            view.setProgressState(4, ProgressLoggerConsoleFrame.PROCESSING);
            int errors = 0;
            int numImageFiles = getSource().imageFiles.sizeImageFilesImagefile();
            for (int i = 0; i < numImageFiles; i++) {
                ImageFilesImagefile target = getSource().imageFiles.getImageFilesImagefile(i);
                try {
                    checkTaskStopRequest();
                    view.setProgressState(4, i + 1, numImageFiles);
                    Timestamp prevMTime = FileFunctions.getPrevMTime(target.getFilename());
                    if (prevMTime == null || target.getMtime().after(prevMTime)) {
                        view.logScanFrame(i + 1, target.getFilename());
                        processScanFrame(i, target);
                    }
                    getTarget().mapper.addRealValidFileIndex(i);
                } catch (ReaderSourceException ex) {
                    ex.printStackTrace();
                    errors++;
                    continue;
                } catch (IOException ex) {
                    ex.printStackTrace();
                    errors++;
                    continue;
                }
                view.setProcessedFile(i + 1, numImageFiles);
            }
            
            if (0 < errors) {
                view.setProgressState(4, ProgressLoggerConsoleFrame.FAIL);
                view.setNumGuideErrorFiles(errors, numImageFiles);
                view.throwPageError(errors);
            }
            view.setProcessedFile(numImageFiles, numImageFiles);
            view.setProgressState(4, ProgressLoggerConsoleFrame.SUCCEED);
        }


        private void processScanFrame(final int fileIndex, final ImageFilesImagefile target) throws ReaderSourceException,IOException,ThreadSuspendingException{
            ScanFrame scanFrame = new ScanFrame(); 
            target.setScanFrame(scanFrame);
            String filename = target.getFilename();
            final ReaderSource source = new ReaderSource(RasterFactory.create(filename));
            Point2D[] scanFrameCorner = null;
            try{
                scanFrameCorner = source.createScanFrameCorner();
            }catch(ReaderSourceException ex){
                FileFunctions.saveErrorImage(getSource().imagedir, errorImageDirPath,
                        "PageError", fileIndex, target, source, Color.red);
                view.logPageError(ex, filename);
                getTarget().mapper.incrementNumErrorPages();
                throw new ReaderSourceException("InvalidPage:"+filename+":"+ex.getMessage());
            }
            view.logGuide(scanFrameCorner);
            setScanFrameCorners(scanFrame, scanFrameCorner);
            if(GUIDE_SQUARE_SHAPE_CHECKER_ENABLED && ! source.checkGuideSquareShape(getSource().masterConerPoints)){                
                view.showGuideError(filename);
                getTarget().mapper.incrementNumErrorPages();
                FileFunctions.saveErrorImage(getSource().imagedir, errorImageDirPath, "GuideError", fileIndex, target, source, Color.magenta);
            }
        }

        private void setScanFrameCorners(ScanFrame scanFrame, Point2D[] scanFrameCorner) {
            scanFrame.setX1((int)scanFrameCorner[0].getX());
            scanFrame.setY1((int)scanFrameCorner[0].getY());
            scanFrame.setX2((int)scanFrameCorner[1].getX());
            scanFrame.setY2((int)scanFrameCorner[1].getY());
            scanFrame.setX3((int)scanFrameCorner[2].getX());
            scanFrame.setY3((int)scanFrameCorner[2].getY());
            scanFrame.setX4((int)scanFrameCorner[3].getX());
            scanFrame.setY4((int)scanFrameCorner[3].getY());
        }
    }
    
    class ProcessScanFramesTaskSource extends ReaderTaskSource{
        File imagedir;
        ImageFiles imageFiles;
        Point2D[] masterConerPoints;
        ProcessScanFramesTaskSource(File imagedir, Point2D[] masterConerPoints, ImageFiles imageFiles){
            this.imagedir = imagedir;
            this.masterConerPoints = masterConerPoints;
            this.imageFiles = imageFiles;
        }
        
    }

    class ProcessScanFramesTaskTarget extends ReaderTaskTarget{
        ReaderIndexMapper mapper = new ReaderIndexMapper();
    }

    class ProcessAreaTask extends ReaderTask{
                        
        ProcessAreaTask(ProcessAreaTaskSource source, ProcessAreaTaskTarget target, ReaderProcessView view){
            super(source, target, view);
        }
        
        ProcessAreaTaskSource getSource(){
            return (ProcessAreaTaskSource)source;
        }

        ProcessAreaTaskTarget getTarget(){
            return (ProcessAreaTaskTarget)target;
        }

        void run() throws ThreadSuspendingException {
            ImageFiles imageFiles = getSource().imageFiles;
            view.setProgressState(5, ProgressLoggerConsoleFrame.PROCESSING);
            view.initProgressBar(0, imageFiles.sizeImageFilesImagefile());
            processImageFiles(imageFiles);
        }

        private void processImageFiles(ImageFiles imageFiles) throws ThreadSuspendingException {
            for(int i = 0; i < imageFiles.sizeImageFilesImagefile(); i++){
                checkTaskStopRequest();
                if(! getSource().mapper.isValidFileIndex(i)){
                    continue;
                }
                ImageFilesImagefile target = imageFiles.getImageFilesImagefile(i);
                Timestamp prevMTime = FileFunctions.getPrevMTime(target.getFilename());
                if(prevMTime == null || target.getMtime().after(prevMTime)){
                    try {
                        view.setProgressBarValue(i+1);                           
                        processArea(i, target);
                        view.setProgressState(5, ProgressLoggerConsoleFrame.SUCCEED);
                    } catch(ReaderSourceException ex){
                        view.error(ex);
                        view.setProgressState(5, ProgressLoggerConsoleFrame.FAIL);
                    } catch(IOException ex){
                        view.error(ex);
                        view.setProgressState(5, ProgressLoggerConsoleFrame.FAIL);
                    }
                }
            }
        }

        private void processArea(int fileIndex, ImageFilesImagefile target)throws IOException,ThreadSuspendingException,ReaderSourceException{
            String filename = target.getFilename();
            ReaderSource source = new ReaderSource(RasterFactory.create(filename));
            int validFileIndex = getSource().mapper.getValidFileIndex(fileIndex);
            int sampleID = 1 + validFileIndex / getSource().numPages;
            int pageNumber = 1 + validFileIndex % getSource().numPages;
            
            Page page = new Page();
            page.setSampleID(sampleID);
            page.setNumber(pageNumber);
            target.getScanFrame().setPage(page);
            if(pageNumber == 1){
                getSource().resultBuilder.addRowFileList(new LinkedList());
             
            }
            getSource().resultBuilder.addRowFile(new File(target.getFilename()));
            view.logProcessArea(pageNumber, sampleID, target.getFilename());
            
            //if(depth_first_search_level == PAGE_LEVEL){
            IFormElementsChoice[] formStates = processFormElements(target, source, sampleID, pageNumber);
            target.getScanFrame().getPage().setFormElements(formStates);
            //}
        }
        
        private IFormElementsChoice[] processFormElements(ImageFilesImagefile target, RasterSource source, int sampleID, int pageNumber) throws IOException,ThreadSuspendingException,ReaderSourceException {
            ReaderTranslatedSource tSource = new ReaderTranslatedSource(source.getRaster(), getSource().masterPoints,
                    ReaderImageProcessor.getPointArray(target.getScanFrame()), BLACK_THRESHOLD);
            HashMap formMap = getSource().qSchema.getFormElements(pageNumber);
            HashMap rectMap = getSource().qSchema.getRectangle(pageNumber);
            IFormElementsChoice[] formStates = new IFormElementsChoice[formMap.keySet().size()];
            int i = 0;
            int formElementSize = formMap.size();
            int validFileIndexSize = getSource().mapper.getValidFileIndexSize();
            for(Iterator it = formMap.keySet().iterator(); it.hasNext(); i++){
                checkTaskStopRequest();
                String id = (String)it.next();
                int validFileIndex = (pageNumber+(sampleID-1)*getSource().qSchema.getNumberOfPages());
                view.setProgressState(5, i+1, formElementSize, validFileIndex, validFileIndexSize);
                Rectangle rect = (Rectangle) rectMap.get(id);
                Element formElem = (Element)formMap.get(id);
                String localName = formElem.getLocalName();
                if(localName.equals(FORMTYPE_SELECT1) || localName.equals(FORMTYPE_SELECT)){
                    formStates[i] = processFormElementMarkarea(sampleID, id, rect, formElem, tSource);
                }else if(localName.equals(FORMTYPE_TEXTAREA) || localName.equals(FORMTYPE_INPUT)){
                    formStates[i] = processFormElementTextarea(sampleID - 1, id, rect, formElem, tSource,
                            target.getFilename());
                    getSource().resultBuilder.addAreaImage(sampleID - 1, id, new File(target.getFilename()));
                }else{
                    throw new ReaderSourceException("UNSUPPORTED ELEMENT:"+localName);
                }
                view.setProcessedElements(validFileIndex, validFileIndexSize);
            }
            return formStates;
        }
        
        private FormElementsMarkarea processFormElementMarkarea(int sampleID, String id, Rectangle rect, Element select1Element, ReaderTranslatedSource source){
            FormElementsMarkarea ret = new FormElementsMarkarea();
            ret.setRef(id);
            double density = source.getDensity(rect);
            ret.setMarkDensity(density);
            view.logRectInfo(id, rect, source, MARK_RECOGNIZE_DENSITY < density, density);
            return ret;
        }

        private FormElementsTextarea processFormElementTextarea(int sampleID, String id, Rectangle rect, Element select1Element, ReaderTranslatedSource source,
                String filename)throws ThreadSuspendingException{
            File imageFile = FileFunctions.createTextareaImageFile(getSource().imagedir, textareaImageDirPath, sampleID, id, filename);
            getSource().resultBuilder.putAreaFile(sampleID, id, imageFile);
            while(true){
                try{
                    ReaderImageProcessor.createTextareaImage(rect, source, imageFile);
                    break;
                }catch(IOException ex){
                    FileFunctions.confirmRetryPageSaving(view, ex, imageFile);
                }
            }
            view.logRectInfo(id, rect, source, imageFile);
            FormElementsTextarea ret = new FormElementsTextarea(); 
            ret.setRef(id);
            TextareaImage image = new TextareaImage(); 
            image.setFilename(imageFile.getAbsolutePath());
            image.setMtime(new Timestamp(imageFile.lastModified()));
            ret.setTextareaImage(image);        
            return ret;
        }
    }
    
    class ProcessAreaTaskSource extends ReaderTaskSource{
        File imagedir; 
        QuestionnaireSchema qSchema;
        ImageFiles imageFiles;
        ReaderIndexMapper mapper;
        int numPages;
        Point2D[] masterPoints;
        ReaderResultBuilder resultBuilder;
        ProcessAreaTaskSource(File imagedir, QuestionnaireSchema qSchema, ImageFiles imageFiles, ReaderIndexMapper mapper, Point2D[] masterPoints,
                ReaderResultBuilder resultBuilder){
            this.imagedir = imagedir;
            this.qSchema = qSchema;
            this.numPages = qSchema.getNumberOfPages();
            this.imageFiles = imageFiles;
            this.mapper = mapper;
            this.masterPoints = masterPoints;
            this.resultBuilder = resultBuilder;
        }
    }
    
    class ProcessAreaTaskTarget extends ReaderTaskTarget{
    }
    
    class ProcessValuesTask extends ReaderTask{
        /*
        class ProcessValuesState{
            int previousPageNumber = -1;
            int idIndex = -1;
            public int nextIdIndex(int pageNumber){
                if(previousPageNumber == pageNumber){
                    idIndex++;
                }else{
                    idIndex = 0;
                }
                System.err.println(""+previousPageNumber+" -> "+pageNumber+", "+idIndex);
                previousPageNumber = pageNumber;                
                return idIndex;
            }
        }
*/
        ProcessValuesTask(ProcessValuesTaskSource source, ProcessValuesTaskTarget target, ReaderProcessView view){
            super(source, target, view);
        }
        ProcessValuesTaskSource getSource(){
            return (ProcessValuesTaskSource)source;
        }
        ProcessValuesTaskTarget getTarget(){
            return (ProcessValuesTaskTarget)target;
        }
        
        void run() throws ThreadSuspendingException,NumberFormatException,TransformerException{
            //data.columnNames = createColumnNames();
            int numFiles = getSource().numFiles;
            int numValidFiles = getSource().mapper.getValidFileIndexSize();
            int numPages = getSource().qSchema.getNumberOfPages();
            int numSamples = numValidFiles / numPages;
            getSource().resultBuilder.setRowSize(numSamples);
            int numColumns = getSource().qSchema.getNumberOfColumns();
            view.setProgressState(6, ProgressLoggerConsoleFrame.PROCESSING);
            
            for (int rowIndex = 0; rowIndex < numSamples; rowIndex++) {
                checkTaskStopRequest();
                view.setProgressState(6, rowIndex+1, numSamples);
                view.setProcessedValues(rowIndex+1, numSamples);
                getSource().resultBuilder.setMatrix(rowIndex, processValuesParSample(numPages, numColumns, rowIndex));
            }
            view.setProgressState(6, ProgressLoggerConsoleFrame.SUCCEED);
        }
       
        private Object[] processValuesParSample(int numPages, int numColumns, int rowIndex) throws ThreadSuspendingException, TransformerException {
            Object[] sampleData = new Object[numColumns];
            Iterator it = getSource().qSchema.getQIDSet().iterator();
            for (int columnIndex = 0; columnIndex < numColumns; ) {
                checkTaskStopRequest();
                String qid = (String) it.next();
                List idList = getSource().qSchema.getIDList(qid);
                if (idList != null) {
                    columnIndex += processValuesParMarkareas(sampleData, numPages, rowIndex, columnIndex, qid, idList);
                } else {
                    processValuesParTextarea(sampleData, rowIndex, columnIndex, qid);
                    columnIndex++;
                }
            }
            return sampleData;
        }

        private void processValuesParTextarea(Object[] sampleData, int rowIndex, int columnIndex, String qid) throws TransformerException {
            //sampleData[columnIndex] = null;
            ImageFilesImagefile imagefile = getSource().imageFiles.getImageFilesImagefile(getSource().mapper.getRealFileIndex(rowIndex * getSource().qSchema.getNumberOfPages()));
            Page page = imagefile.getScanFrame().getPage();
            //System.err.println("  @@@@@@@@@@@@@@@   ("+columnIndex+") "+state.idIndex+"/"+page.getFormElements().length);
        }

        private CellValue processValue(String qid) throws TransformerException {
            Element elem = getSource().qSchema.getElement(qid);
            int page = getSource().qSchema.getPage(qid);
            Element formElement = getSource().qSchema.getFormElement(elem);
            String type = formElement.getLocalName();
            NodeList nlValue = formElement.getElementsByTagNameNS("value", SQSNamespaces.SQS2004READER_URI);
            NodeList fileElem = formElement.getElementsByTagNameNS("file", SQSNamespaces.SQS2004READER_URI);
            String value = (nlValue == null)? null : (nlValue.item(0) == null )? null : nlValue.item(0).getNodeValue();
            String filename = (fileElem == null)? null : (fileElem.item(0) == null )? null : ((Element)fileElem.item(0)).getAttributeNodeNS("filename", SQSNamespaces.SQS2004READER_URI).getValue();
            //logger.info(" page:"+page+" qid:"+qid+" type:"+type+" value:"+value);
            value = (value == null && filename != null)? filename: value;
            return new CellValue(page, qid, type, value);
        }

        private int processValuesParMarkareas(Object[] sampleData, int numPages, int rowIndex, int columnIndex, String qid, List idList) throws TransformerException {
            double max = 0.0;
            double secondMax = 0.0;
            int maxItemIndex = -1;
            List itemList = new LinkedList();
            CellValue cellValue = null;
            ImageFilesImagefile imagefile = null;
            for (int itemIndex = 0; itemIndex < idList.size(); itemIndex++) {
                String id = (String) idList.get(itemIndex);
                cellValue = getSource().qSchema.createCellOptionValue(qid, id, itemIndex);
                int validFileIndex = numPages * rowIndex + cellValue.page - 1;
                int realFileIndex = getSource().mapper.getRealFileIndex(validFileIndex);
                imagefile = getSource().imageFiles.getImageFilesImagefile(realFileIndex);
                Page pageObject = imagefile.getScanFrame().getPage();
                try{
                    int page = pageObject.getNumber();
                    int idIndex = getSource().qSchema.getIDIndex(id);
                    //System.err.println("  ############### "+itemIndex+"/"+idList.size()+"  ("+columnIndex+") "+state.idIndex+"/"+page.getFormElements().length);
                    double density = ((FormElementsMarkarea) pageObject.getFormElements(idIndex)).getMarkDensity();
                        //((FormElementsMarkarea) page.getFormElements(state.idIndex)).getMarkDensity();
                    if (MARK_RECOGNIZE_DENSITY < density) {
                        if (max <= density) {
                            secondMax = max;
                            max = density;
                            maxItemIndex = itemIndex;
                        }else if(secondMax <= density){
                            secondMax = density;                    
                        }
                        itemList.add(cellValue);
                   }
                }catch(Exception ignore){
                    ignore.printStackTrace();
                }
            }
            return processValuesParMarkareas(sampleData, rowIndex,
                    columnIndex, qid, idList, max, secondMax, maxItemIndex, itemList, cellValue, imagefile);
        }
        
        private int processValuesParMarkareas(Object[] sampleData, int rowIndex, int columnIndex, String qid, List idList, double max, double secondMax, int maxItemIndex, List itemList, CellValue cellValue, ImageFilesImagefile imagefile) {
            if (cellValue.type.equals(FORMTYPE_SELECT1)) {
                if(1 < itemList.size() &&  (max - secondMax) < DOUBLE_MARK_RECOGNIZE_DENSITY ){
                    view.logInvalidMark(rowIndex+1, imagefile.getFilename(), qid, itemList.size());
                    sampleData[columnIndex] = "ERROR:"+StringUtil.join(itemList, ",");
                    getSource().mapper.incrementNumErrorMarkSets();
                    getSource().mapper.addErrorArea(rowIndex, cellValue.page, qid);//rowIndex
                }
                if(0 <= maxItemIndex){
                    sampleData[columnIndex] = ((String)idList.get(maxItemIndex)).substring(qid.length()+2);
                }
                return 1;
            } else if (cellValue.type.equals(FORMTYPE_SELECT)) {
                return processSelectValues(sampleData, columnIndex, idList, itemList);
            }else{
                throw new RuntimeException("invalidType:"+cellValue.type);
            }
        }
        
        private int processSelectValues(Object[] sampleData, int columnIndex, List idList, List itemList) {
            for (int itemIndex = 0; itemIndex < idList.size(); itemIndex++) {
                String selectId = (String)idList.get(itemIndex);
                sampleData[columnIndex+itemIndex] = FORMVALUE_NON_SELECTED;
            }
            for (int i = 0; i < itemList.size(); i++) {
                CellValueOption cv = (CellValueOption)itemList.get(i);
                //System.err.println("cv.index:"+i+":"+cv.index);
                sampleData[columnIndex+cv.index] = FORMVALUE_SELECTED;
            }
            return idList.size();
        }
    }
    
    class ProcessValuesTaskSource extends ReaderTaskSource{
        int numFiles;
        ReaderIndexMapper mapper;
        QuestionnaireSchema qSchema;
        ImageFiles imageFiles;
        ReaderResultBuilder resultBuilder;
        ProcessValuesTaskSource(int numFiles, ReaderIndexMapper mapper, QuestionnaireSchema qSchema, ImageFiles imageFiles, ReaderResultBuilder resultBuilder){
            this.numFiles = numFiles;
            this.mapper = mapper;
            this.qSchema = qSchema;
            this.imageFiles = imageFiles;
            this.resultBuilder = resultBuilder;
        }
    }
    
    class ProcessValuesTaskTarget extends ReaderTaskTarget{
    }

    
    class ProcessErrorImagesTask extends ReaderTask{
        ProcessErrorImagesTask(ProcessErrorImagesTaskSource source, ProcessErrorImagesTaskTarget target, ReaderProcessView view){
            super(source, target, view);
        }
        ProcessErrorImagesTaskSource getSource(){
            return (ProcessErrorImagesTaskSource)source;
        }
        ProcessErrorImagesTaskTarget getTarget(){
            return (ProcessErrorImagesTaskTarget)target;
        }

        void run() throws ThreadSuspendingException {
            ImageFiles imageFiles = getSource().imageFiles;
            Set set = getSource().mapper.getErrorAreaEntrySet();
            if (0 < set.size()) {
                int errorFileIndex = 0;
                view.setWaitCursor();
                for (Iterator it = set.iterator(); it.hasNext(); errorFileIndex++) {
                    checkTaskStopRequest();
                    view.setProgressState(7, errorFileIndex + 1, set.size());
                    processErrorImage(imageFiles, it);
                }
                view.setDefaultCursor();
            }else{
                return;
            }
        }
        
        private void processErrorImage(ImageFiles imageFiles, Iterator it) throws ThreadSuspendingException {
            Map.Entry entry = (Map.Entry) it.next();
            int rowIndex = ((Integer) entry.getKey()).intValue();
            int realFileIndex = getSource().mapper.getRealFileIndex(rowIndex * getSource().qSchema.getNumberOfPages());
            ImageFilesImagefile target = imageFiles.getImageFilesImagefile()[realFileIndex];
            List list = (List) entry.getValue();
            saveErrorImages(errorImageDirPath, view, imageFiles, rowIndex, list, BLACK_THRESHOLD);
        }
        
        void saveErrorImages(String errorImageDirPath, ReaderProcessView view, ImageFiles imagefiles, int rowIndex, List list, int threshold) throws ThreadSuspendingException{
            File imagedir = getSource().imagedir; 
            QuestionnaireSchema qSchema = getSource().qSchema;
            while(true){
                ScanFrame prevScanFrame = null;
                ReaderTranslatedSource tSource = null;
                BufferedImage image = null;
                ImageFilesImagefile target = null;
                ScanFrame scanFrame;
                try{
                    File errorFile = null;
                    Graphics2D g = null;
                    for(int i = 0; i < list.size(); i++){
                        Object[] arr = (Object[])list.get(i);
                        int page = ((Integer)arr[0]).intValue();
                        String qid = (String)arr[1];
                        target = imagefiles.getImageFilesImagefile(rowIndex*qSchema.getNumberOfPages()+page-1);
                        scanFrame = target.getScanFrame();
                        if(prevScanFrame != scanFrame){
                            if(image != null){
                                saveErrorImage(errorFile, image);
                            }
                            prevScanFrame = scanFrame;
                            ReaderSource source = new ReaderSource(RasterFactory.create(target.getFilename()));
                            tSource = new ReaderTranslatedSource(source.getRaster(), qSchema.getMasterPoints(),
                                    ReaderImageProcessor.getPointArray(target.getScanFrame()), threshold);
                            
                            image = FileFunctions.createErrorImage(target, source, Color.green);
                            errorFile = FileFunctions.createErrorImageFile(imagedir, errorImageDirPath, "MarkError", rowIndex, target.getFilename());
    						g = (Graphics2D)image.getGraphics();
    		                g.setColor(Color.red);
                        }
                        System.err.println(rowIndex+" "+qid+" "+errorFile);
                        ReaderImageProcessor.drawErrorMarks(qSchema, g, page, qid, tSource);
                        getSource().resultBuilder.addErrorImage(rowIndex, qid, errorFile);
                    }
                    if(image != null){
                        saveErrorImage(errorFile, image);
                    }
                    break;
                }catch(IOException ex){
                    if(target != null){
                        FileFunctions.confirmRetryPageSaving(view, ex, new File(target.getFilename()));
                    }else{
                        FileFunctions.confirmRetryPageSaving(view, ex, null);
                    }
                }
            }
        }

        void saveErrorImage(File path, BufferedImage image) throws IOException {
            FileUtil.writeImageFile(path, image);
        }
    }
    
    class ProcessErrorImagesTaskSource extends ReaderTaskSource{
        File imagedir;
        QuestionnaireSchema qSchema;
        ImageFiles imageFiles;
        ReaderIndexMapper mapper;
        ReaderResultBuilder resultBuilder;
        ProcessErrorImagesTaskSource(File imagedir, QuestionnaireSchema qSchema, ImageFiles imageFiles, ReaderIndexMapper mapper, ReaderResultBuilder resultBuilder){
            this.imagedir = imagedir;
            this.qSchema = qSchema;
            this.imageFiles = imageFiles;
            this.mapper = mapper;
            this.resultBuilder = resultBuilder;
        }
    }
    
    class ProcessErrorImagesTaskTarget extends ReaderTaskTarget{
    }     

    public void processCoreTasks() throws ThreadSuspendingException, TranslatorException, TransformerException{
        processScanFrames();
        processArea();
        processValues();
    }
}

