/*

 ReaderProcessView.java
 
 Copyright 2004 KUBO Hiroya (hiroya@sfc.keio.ac.jp).
 
 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at
 
 http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 
 Created on 2004/10/13

 */
package net.sf.sqs_xml.reader.logic;

import java.awt.Component;
import java.awt.Cursor;
import java.awt.Rectangle;
import java.awt.geom.Point2D;
import java.io.File;
import java.io.IOException;

import javax.swing.JOptionPane;
import javax.swing.JProgressBar;

import net.sf.sqs_xml.reader.swing.ProgressLoggerConsoleFrame;
import net.sf.sqs_xml.swing.LoggerConsoleFrame;
import net.sf.sqs_xml.translator.logic.TranslatorException;

import org.apache.avalon.framework.logger.Logger;


/**
 * @author hiroya
 *
 */
public class ReaderProcessView {
    Logger logger;
    JProgressBar progressBar;
    ReaderProcessView(Logger logger){
        this.logger = logger;
        this.progressBar = ((LoggerConsoleFrame)logger).getProgressBar();
    }
    
    void setImagedir(File imagedir){
        ReaderLogMessage.logImageDir(logger, imagedir);
        if(logger instanceof ProgressLoggerConsoleFrame){
            ((ProgressLoggerConsoleFrame)logger).setBorderTitle(imagedir.getAbsolutePath());
        }
    }
    
    void setPDFFile(File pdffile){ 
        ReaderLogMessage.logPDFFile(logger, pdffile);
    }

    public JProgressBar getProgressBar(){
        return progressBar; 
    }
    
    public void setProgressBar(JProgressBar progressbar) {
        this.progressBar = progressbar;
    }

    void error(Exception ex){
        logger.error(ex.getLocalizedMessage());
    }
    
    void errorDialog(Exception ex, boolean throwerror)throws Exception{
        logger.error(ex.getLocalizedMessage());
        JOptionPane.showMessageDialog((Component)logger, ex.getMessage(), "エラー", JOptionPane.ERROR_MESSAGE);
        if(throwerror){
            throw ex;
        }
    }
    
    void errorDialog(TranslatorException ex, boolean throwerror)throws Exception{
        logger.error(ex.getLocalizedMessage());
        JOptionPane.showMessageDialog((Component)logger, ex.getMessages(), "エラー", JOptionPane.ERROR_MESSAGE);
        if(throwerror){
            throw ex;
        }
    }

    void setProgressState(int step){
        setProgressState(step, ProgressLoggerConsoleFrame.PROCESSING);
        progressBar.setStringPainted(true);
        if(step == 1){
            progressBar.setString("(Step 1) initializing...");
        }else if(step == 2){
            progressBar.setString("(Step 2) Find ImageFiles");
        }
    }
    
    void setProgressState(int step, int i, int size){
        if(step != 7){
            setProgressState(step, ProgressLoggerConsoleFrame.PROCESSING);
            progressBar.setStringPainted(true);
            progressBar.setValue(i);
            progressBar.setMaximum(size);
        }
        
        if(step == 3){
            progressBar.setString("(Step 3) Process ImageFiles: "+i+" / "+size+" files");
        }else if(step == 4){
            progressBar.setString("(Step 4) Process ScanFrames: "+i+" / "+size+" files");
        }else if(step == 6){
            progressBar.setString("(Step 6) Process Values: "+i+" / "+size+" samples");
            //((ProgressLoggerConsoleFrame)logger).setProcessedValues(i, size);
        }else if(step == 7){
            progressBar.setString("(ERROR) Process Error Images: "
                    + i + " / " + size
                    + " file(s)");
            //((ProgressLoggerConsoleFrame)logger).setProcessedErrors(errorFileIndex+1, set.size());
        }
    }
    
    void setProgressState(int step, int i, int formElementSize, int validFileIndex, int validFileIndexSize){
        setProgressState(step, ProgressLoggerConsoleFrame.PROCESSING);
        progressBar.setStringPainted(true);
        progressBar.setValue(validFileIndex);
        progressBar.setMaximum(validFileIndexSize);
        if(step == 5){
            progressBar.setString("(Step 5) Process Elements: "+ i + 
                    " / "+ formElementSize +
                    " in File: " + validFileIndex +
                    " / "+validFileIndexSize+""
                    );
        }
    }
    
    void setProgressBarValue(int i){
        progressBar.setValue(i);
    }

    void logScanFrame(int i, String filename){
        ReaderLogMessage.logScanFrame(logger, i, filename);
    }
    
    void logPageError(final Exception ex, final String filename){
        ReaderLogMessage.logPageError(logger, ex, filename);
    	if(logger instanceof Component){
    	    new Thread(){
    	        public void run(){
    	            JOptionPane.showMessageDialog((Component)logger, new Object[]{ex.getMessage(), filename}, "エラー", JOptionPane.ERROR_MESSAGE);
    	        }
    	    }.start();
    	}
    }
    
    void logGuide(Point2D[] scanFrameCorner){
        ReaderLogMessage.logGuide(logger, scanFrameCorner);
	}
    
    void throwIncorrectPDFException(File pdffile)throws TranslatorException{
        throw new TranslatorException(new String[]{"PDFファイルの内容が正しく認識できません．",
                "印刷のときに原稿として利用したPDFファイルをフォルダにコピーしてください．",
                pdffile.getAbsolutePath() });
    }
    void throwNoPDFException()throws TranslatorException{
        throw new TranslatorException(new String[]{"指定されたフォルダの中に「正しいPDFファイル」が見つかりません．",
                "「スキャナで回答用紙を取り込んで作成した画像のPDFファイル」などではなく，",
                "「印刷用の原稿としてSourceEditorで作成したPDFファイル」を入れてください．"
        	});
    }
    void throwManyPDFException()throws TranslatorException{
        throw new TranslatorException(new String[]{"指定されたフォルダの中にPDFファイルが複数あります．",
        "正しいフォルダを指定するか，不要なPDFを削除してください．"});
    }
    
    void showGuideError(String filename){
        ReaderLogMessage.logGuideError(logger, filename);
    	if(logger instanceof Component){
    	    JOptionPane.showMessageDialog((Component)logger, new Object[]{"上下の■マーク列が歪んでいるため，正しく読めません", filename}, "エラー", JOptionPane.ERROR_MESSAGE);
    	}
    }
    
    void setProcessedFile(int i, int numImageFiles){
        ((ProgressLoggerConsoleFrame)logger).setProcessedFile(i, numImageFiles);
    }
    
    void setNumGuideErrorFiles(int error, int numImageFiles){
        ((ProgressLoggerConsoleFrame)logger).setGuideError(error, numImageFiles, numImageFiles);
    }
    
    void setProgressState(int index, int mode){
        ((ProgressLoggerConsoleFrame)logger).setProgressState(index, mode);
    }

    void initProgressBar(int numFiles, int numPages){
        progressBar.setIndeterminate(false);
        progressBar.setMaximum(numFiles);
    }
    
    void setFiles(int numFiles){
        ((ProgressLoggerConsoleFrame)logger).setNumFiles(numFiles);
    }
    
    void setPageSetError(int numFiles, int numPages){
        ((ProgressLoggerConsoleFrame)logger).setPageNumberError(numFiles, numPages);
    }
    
    void setNumTargetFiles(int numTargetFiles){
        ((ProgressLoggerConsoleFrame)logger).setTargetFileCount(numTargetFiles);
    }
    
    void logProcessArea(int pageNumber, int sampleID, String targetFilePath){
        ReaderLogMessage.logFile(logger, pageNumber, sampleID, targetFilePath);
    }
    
    void setProcessedElements(int validFileIndex, int validFileIndexSize){
        ((ProgressLoggerConsoleFrame)logger).setProcessedElements(validFileIndex, validFileIndexSize);
    }
    
    void setProcessedValues(int rowIndex, int numSamples){
        ((ProgressLoggerConsoleFrame)logger).setProcessedValues(rowIndex, numSamples);
    }
        
    void logRectInfo(String id, Rectangle rect, ReaderTranslatedSource source, boolean isMarked, double density){
        ReaderLogMessage.logRectInfo(logger, id, rect, source);
        ReaderLogMessage.logMark(logger, isMarked, density);
    }
    
    void logRectInfo(String id, Rectangle rect, ReaderTranslatedSource source, File imageFile){
        ReaderLogMessage.logRectInfo(logger, id, rect, source);
        ReaderLogMessage.logCropImage(logger, imageFile.getAbsolutePath());
    }
    
    void logInvalidMark(int rowIndex, final String filename, String qid, int size){
        ReaderLogMessage.logInvalidMark(logger, rowIndex, filename, qid, size);
    	if(logger instanceof Component){
    	    new Thread(){
    	        public void run(){
    	    	    JOptionPane.showMessageDialog((Component)logger, new Object[]{"択一選択型設問で複数のマークがされています．",
        	            	"(マークミスではなく，ページ順序間違いの可能性もありますので，",
        	            	" 元ファイル画像の内容を確認してください)",
        	            	filename}, "エラー", JOptionPane.WARNING_MESSAGE);
    	        }
    	    }.start();

        }
    }
    
    void setWaitCursor(){
        progressBar.setValue(0);
        ((Component)logger).setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
    }
    
    void setDefaultCursor(){
        //((ProgressLoggerConsoleFrame)logger).setProcessedErrors(set.size(), set.size());
        ((Component)logger).setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
    }

    void setPDFFileInfo(File pdffile, QuestionnaireSchema qSchema){
        if(logger instanceof ProgressLoggerConsoleFrame){
            ((ProgressLoggerConsoleFrame)logger).setPDFFile(pdffile, qSchema.getNumberOfPages(), qSchema.getNumberOfColumns());
        }
    }

    void throwFileNumberError(int numFiles, int numPages)throws TranslatorException{
        throw new TranslatorException(new String[]{"画像ファイル数に過不足があります．修正してから再実行してください．",
             "総ファイル数: "+numFiles+" (有効サンプル数："+(numFiles/numPages)+")",
            "一人分のページ数: "+numPages+" ページ",
            "  ("+ (numPages - numFiles%numPages)+"個のファイル不足?)"});
    }

    void throwPageError(int errors)throws TranslatorException{
        throw new TranslatorException("上下の「■」の列を正しく認識できない画像ファイルが "
                + errors + "個あります．修正してから再実行してください．");
    }
    
    void showResult(int errors, int warns, int numFiles, int numValidFiles, int numPages, int numSamples){
        if(progressBar != null){
            String status = null;
            if(0 == errors && 0 == warns && 0 < numValidFiles){
                status = "終了";
                progressBar.setString(status+": found "+ numFiles + " files : processed "+numValidFiles+" files = "+numSamples+" samples x "+numPages+" page(s)");
            }else if(errors == -1){
                status = "エラー";
                progressBar.setString("");
            }else{
                if(0 < errors){
                    status = "エラー終了："+errors+"個のエラー";
                }else if(0 < warns){
                    status = "警告終了："+warns+"個の警告";
                }
                progressBar.setString(status+": found "+ numFiles + " files : processed "+numValidFiles+" files ... "+numSamples+" samples x "+numPages+" page(s)");
            }
            //ReaderLogMessage.logResult(logger, status, errors, numFiles, numValidFiles, numPages, numSamples, schema.getNumberOfColumns());
        }

    }
    
    int showFileSaveErrorDialog(IOException ex, File file){
       return JOptionPane.showConfirmDialog((Component)logger, new String[]{"ファイルが保存できません．",
    ex.getLocalizedMessage(),
    file.getAbsolutePath(),
    "....保存をやり直しますか?"}, "File Error", JOptionPane.YES_NO_OPTION);
    }

    public void setSucceed() {
        ((LoggerConsoleFrame)logger).setSucceed();
    }
}
