 /*

 Source.java 
 Copyright 2004 KUBO Hiroya (hiroya@sfc.keio.ac.jp).
 
 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at
 
 http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 
 Created on 2004/07/28

 */
package net.sf.sqs_xml.editor.base.source;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;

/**
 * @author hiroya
 *
 */
public abstract class Source {
    String title = null;
    File file = null;
    URL url = null;
    long mtime = 0L;
    boolean dirty = false;
    boolean readonly = false;
    static int serial = 1;
    
    public Source()throws SourceException{
        this.title = "*Undefined"+(serial++);
        this.file = null;
        dirty = false;
        readonly = false;
    }
    
    public Source(File file)throws SourceException{
        this.title = null;
        this.file = file;
        mtime = file.lastModified();
        readonly = ! file.canWrite();
        dirty = false;
        try{
            initialize(new FileInputStream(file));
        }catch(IOException ex){
            throw new SourceException(ex);
        }
        
    }
    
    public Source(URL url, boolean readonly, String title)throws SourceException{
        this.url = url;
        String readOnlyMark = readonly?"%":""; 
        if(title != null){
            this.title = readOnlyMark+title;
        }else{
            String filePart = url.getFile();
            try{
                String file = filePart.substring(filePart.lastIndexOf('/')+1);
                this.title = readOnlyMark+file;
            }catch(NullPointerException ex){
                this.title = readOnlyMark+filePart;
            }
        }
        if(url.getProtocol().toLowerCase().equals(file)){
            setFile(new File(url.getFile()));
        }else{
            this.file = null;
        }
        this.readonly = readonly;
        dirty = false;
        try{
            initialize(url.openStream());
        }catch(IOException ex){
            throw new SourceException(ex);
        }
    }
    
    public String toString(){
        return "Source["+title+" "+file+"("+url+")]";
    }

    public abstract void initialize(InputStream stream)throws SourceException;

    public abstract InputStream createInputStream()throws IOException;
    
    public void update(){
        mtime = file.lastModified();
    }
    
    public long lastModified(){
        return mtime;
    }
    
    public boolean isModified() {
        if(file != null){
            return mtime < file.lastModified();
        }else{
            return false;
        }
    }

    public void setFile(File file){
        this.file = file;
        if(file != null){
            readonly = ! file.canWrite();
        }
    }
    
    public boolean equals(Object obj){
        try{
            Source source = (Source)obj;
            if(source == null || source.getFile() == null || getFile() == null){
                return false;
            }
            return source.getFile().getAbsoluteFile().equals(getFile().getAbsoluteFile());
        }catch(ClassCastException ex){
            return false;
        }
    }
    
    public File getFile(){
        return file;
    }
    
    public URL getURL(){
        return url;
    }
    
    public String getTitle(){
        String prefix = "";
        if(readonly){
            prefix = "%";
        }else if(dirty){
            prefix = "*";                
        }
        if(file != null){
            return prefix + file.getName();
        }else{
            return prefix + title;
        }
    }
    
    public boolean isDirty(){
        return dirty;
    }

    public boolean isReadOnly(){
        return readonly;
    }

    public void setReadOnly(boolean readonly){
        this.readonly = readonly;
    }

    public void setDirty(boolean dirty){
        this.dirty = dirty;
    }
    
    public void save() throws IOException{
        InputStream in = null;
        //PrintWriter writer = new PrintWriter(new OutputStreamWriter(new BufferedOutputStream(new FileOutputStream(source.getFile())), "UTF-8"));
        OutputStream out = null;
        try{
            in = createInputStream();
            out = new BufferedOutputStream(new FileOutputStream(getFile()));
            byte[] buf = new byte[4096];
            int len;
            while(0 < (len = in.read(buf, 0, 1024))){
                out.write(buf, 0, len);			    
        	}
            setDirty(false);
        }finally{
            try{
                out.close();
                in.close();
                update();
            }catch(Exception ignore){}
        }
    }
    
}

