/*

 EditMenuFeature.java
 
 Copyright 2004 KUBO Hiroya (hiroya@sfc.keio.ac.jp).
 
 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at
 
 http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 
 Created on 2005/02/03

 */
package net.sf.sqs_xml.swing;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import javax.swing.AbstractAction;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.KeyStroke;
import javax.swing.event.UndoableEditEvent;
import javax.swing.event.UndoableEditListener;
import javax.swing.text.Document;
import javax.swing.text.JTextComponent;
import javax.swing.undo.CannotRedoException;
import javax.swing.undo.CannotUndoException;
import javax.swing.undo.UndoManager;
import javax.swing.undo.UndoableEdit;

/**
 * @author hiroya
 *
 */
class EditMenuFeature {
    public static final long serialVersionUID = 0L;

    JPopupMenu popup;

    JMenuItem undoMenu;

    JMenuItem redoMenu;

    UndoManager undo = null;

    JTextComponent textcomp = null;

    public EditMenuFeature(JTextComponent textcomp) {
        this.textcomp = textcomp;
        initPopupMenu();
        initUndoFeature();
        textcomp.addMouseListener(new MouseAdapter() {
            public void mousePressed(MouseEvent e) {
                /*
                 JComponent c = (JComponent)e.getSource();
                 TransferHandler th = c.getTransferHandler();
                 th.exportAsDrag(c, e, TransferHandler.COPY);
                 System.err.println(e.getButton());
                 */
                if (e.getButton() == MouseEvent.BUTTON3 && !popup.isVisible()) {
                    showPopup(e.getX(), e.getY());
                }
            }

            public void mouseReleased(MouseEvent e) {
                if (e.getButton() == MouseEvent.BUTTON3 && !popup.isVisible()) {
                    showPopup(e.getX(), e.getY());
                }
            }
        });

    }

    void showPopup(int x, int y) {
        popup.show(textcomp, x, y);
    }

    public JPopupMenu getPopup() {
        return popup;
    }

    public void setUndoManager(UndoManager undoManager) {
        undo = undoManager;
    }

    private void initPopupMenu() {
        this.popup = new JPopupMenu();
        undoMenu = new JMenuItem("元に戻す  Ctrl-Z");
        redoMenu = new JMenuItem("やり直し  Ctrl-Y");

        JMenuItem cutMenu = new JMenuItem("カット    Ctrl-X");
        JMenuItem copyMenu = new JMenuItem("コピー    Ctrl-C");
        JMenuItem pasteMenu = new JMenuItem("貼り付け  Ctrl-V");
        JMenuItem selectAllMenu = new JMenuItem("全て選択  Ctrl-A");

        undoMenu.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent ev) {
                undo();
            }
        });
        redoMenu.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent ev) {
                redo();
            }
        });
        cutMenu.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent ev) {
                textcomp.cut();
            }
        });
        copyMenu.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent ev) {
                textcomp.copy();
            }
        });
        pasteMenu.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent ev) {
                textcomp.paste();
            }
        });
        selectAllMenu.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent ev) {
                textcomp.selectAll();
            }
        });
        popup.add(undoMenu);
        popup.add(redoMenu);
        popup.addSeparator();
        popup.add(cutMenu);
        popup.add(copyMenu);
        popup.add(pasteMenu);
        popup.addSeparator();
        popup.add(selectAllMenu);

        undoMenu.setEnabled(false);
        redoMenu.setEnabled(false);
    }

    public void addEdit(UndoableEdit edit) {
        if (undo != null) {
            undo.addEdit(edit);
        }
    }

    public void undo() {
        try {
            if (undo != null && undo.canUndo()) {
                undo.undo();
                refreshEnabledStatus();
            }
        } catch (CannotUndoException ignore) {
        }
    }

    public void redo() {
        try {
            if (undo != null && undo.canRedo()) {
                undo.redo();
                refreshEnabledStatus();
            }
        } catch (CannotRedoException ignore) {
        }
    }

    void refreshEnabledStatus() {
        if (undo != null) {
            if (undo.canRedo()) {
                redoMenu.setEnabled(true);
            }
            if (undo.canUndo()) {
                undoMenu.setEnabled(true);
            }
        }
    }

    private void initUndoFeature() {
        Document doc = textcomp.getDocument();

        // Listen for undo and redo events
        doc.addUndoableEditListener(new UndoableEditListener() {
            public void undoableEditHappened(UndoableEditEvent ev) {
                undoMenu.setEnabled(true);
                addEdit(ev.getEdit());
                refreshEnabledStatus();
            }
        });

        // Create an undo action and add it to the text component
        textcomp.getActionMap().put("Undo", new AbstractAction("Undo") {
            public final static long serialVersionUID = 0L;

            public void actionPerformed(ActionEvent ev) {
                undo();
            }
        });

        // Bind the undo action to ctl-Z
        textcomp.getInputMap().put(KeyStroke.getKeyStroke("control Z"), "Undo");

        // Create a redo action and add it to the text component
        textcomp.getActionMap().put("Redo", new AbstractAction("Redo") {
            public final static long serialVersionUID = 0L;

            public void actionPerformed(ActionEvent ev) {
                redo();
            }
        });

        // Bind the redo action to ctl-Y
        textcomp.getInputMap().put(KeyStroke.getKeyStroke("control Y"), "Redo");
    }
}
