/*
 * 
  FileTranslator.java
 
 Copyright 2004 KUBO Hiroya (hiroya@sfc.keio.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at
 
 http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */
package net.sf.sqs_xml.translator.swing;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;


import javax.swing.Box;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JLabel;
import javax.swing.BorderFactory;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import javax.swing.BoxLayout;
import javax.swing.border.EtchedBorder;
import javax.xml.transform.TransformerException;

import net.sf.sqs_xml.editor.base.swing.SuffixBasedFileChooser;
import net.sf.sqs_xml.swing.FileChooserPanel;
import net.sf.sqs_xml.swing.IconTitledPane;
import net.sf.sqs_xml.swing.LoggerConsoleFrame;
import net.sf.sqs_xml.swing.SuffixBasedFileFilter;
import net.sf.sqs_xml.swing.SwingUtil;
import net.sf.sqs_xml.translator.logic.Translator;
import net.sf.sqs_xml.translator.logic.TranslatorException;
import net.sf.sqs_xml.translator.logic.TranslatorResource;
import net.sf.sqs_xml.util.FileUtil;

import java.awt.Cursor;

public abstract class FileTranslator extends JFrame{
    protected Translator translator = null;
	protected FileChooserPanel inputFileChooserPanel;
	protected SuffixBasedFileChooser outputFileChooser;
	protected JPanel outputFilePanel;
	protected JButton fileSaveButton;
	
	public FileTranslator(Icon icon, String title, Translator translator,
	        				SuffixBasedFileChooser inputFileChooser, String inputFileLabel,  
	        				SuffixBasedFileChooser outputFileChooser, String outputFileLabel) throws Exception{
		super();
		addWindowListener(SwingUtil.createConfirmOnExitAdapter(this, "Exit Confirmation", "終了してよろしいですか?"));
		
		this.translator = translator;
		this.setTitle(title);
		this.outputFileChooser = outputFileChooser; 
		this.inputFileChooserPanel = new FileChooserPanel(this, "1: "+inputFileLabel, "入力: ", inputFileChooser){
            public static final long serialVersionUID = 0;
			public void fireFileSelection(File file){
				save(file);
			}
		};
		this.fileSaveButton = createFileSaveButton();
		if(inputFileChooser.getSelectedFile() == null){
		    fileSaveButton.setEnabled(false);
		}else{
		    fileSaveButton.setEnabled(true);
		}
		JPanel basePanel = createBasePanel(icon, title, inputFileChooserPanel,
		        						   createOutputPanel(outputFileLabel, fileSaveButton));
		getContentPane().add(basePanel);
	}
	
	public void setSlectedFile(File file){
	    inputFileChooserPanel.setSelectedFile(file);
	}
	    
    private JButton createFileSaveButton() {
        JButton fileSaveButton = new JButton("保存");
		fileSaveButton.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ev){
				String target = inputFileChooserPanel.getAbsolutePath();
				if(target != null && ! "".equals(target)){
					save(new File(target));
				}
			}
		});
		return fileSaveButton;
    }

    private JPanel createOutputPanel(String outputFileLabel, JButton fileSaveButton) {
        JPanel outputPanel = new JPanel();
		outputPanel.setBorder(BorderFactory.createTitledBorder("2: "+outputFileLabel));		
		outputPanel.setLayout(new BoxLayout(outputPanel, BoxLayout.X_AXIS));
		outputPanel.add(Box.createHorizontalStrut(5));
		outputPanel.add(new JLabel("出力: "));
		outputPanel.add(Box.createHorizontalStrut(5));
		outputPanel.add(fileSaveButton);
		outputPanel.add(Box.createHorizontalStrut(40));
        return outputPanel;
    }

    private JPanel createBasePanel(Icon icon, String title, JPanel inputPanel, JPanel outputPanel) {
        IconTitledPane basePanel = new IconTitledPane();
        basePanel.initialize(icon, title, new EtchedBorder());
		basePanel.setLayout(new BoxLayout(basePanel, BoxLayout.Y_AXIS));
		basePanel.add(inputPanel);
		basePanel.add(outputPanel);
        return basePanel;
    }

    public void setFinished(boolean isFinished){
		if(isFinished){
		    inputFileChooserPanel.setEnabled(true);
			fileSaveButton.setEnabled(true);
			this.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
		}else{
		    inputFileChooserPanel.setEnabled(false);
			fileSaveButton.setEnabled(false);
			this.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		}
	}
	
	class TranslatingTask extends Thread{
		File sourceFile, targetFile;
		TranslatorResource translatorResource;

		TranslatingTask(File sourceFile, TranslatorResource translatorResource, File targetFile){
			this.sourceFile = sourceFile;
			this.targetFile = targetFile;
			this.translatorResource = translatorResource;
		}

		public void run(){
			LoggerConsoleFrame logger = (LoggerConsoleFrame)translatorResource.getLogger();
			logger.setVisible(true);
		    InputStream in = null;
		    OutputStream out = null;
			try{
			    String baseURI = null;//TODO
			    in = new BufferedInputStream(new FileInputStream(sourceFile));
			    out = new BufferedOutputStream(new FileOutputStream(targetFile));
			    translator.translate(in, translatorResource, out, baseURI);
			}catch(IOException ex){
			    ex.printStackTrace();
			    logger.setError("", ex.getCause());
			}catch(TranslatorException ex){
				if( logger != null){
					if(ex.getCause() instanceof TransformerException){//set location 
						logTransformerError(logger, ex);
					}else{
					    logger.setError("", ex.getCause());
					}
					ex.printStackTrace();
				}else{
					ex.printStackTrace();
				}
			}finally{
			    try{
			        in.close();
			        out.close();
			    }catch(Exception ignore){}
				setFinished(true);
				logger.setFinished(true);
			}
		}

        /**
         * @param logger
         * @param ex
         */
        private void logTransformerError(LoggerConsoleFrame logger, TranslatorException ex) {
            TransformerException e = (TransformerException)ex.getCause();
            if(e.getLocator() != null){
            	int lineNumber = e.getLocator().getLineNumber();
            	int columnNumber = e.getLocator().getColumnNumber();
            	logger.setError("File:"+sourceFile+"  Line:"+lineNumber+" Column:"+columnNumber,
            			//	e.getLocationAsString()+":"+e.getLocalizedMessage()+
            			ex.getCause());
            }else{
            	logger.setError("File:"+sourceFile,
            			//	e.getLocationAsString()+":"+e.getLocalizedMessage()+
            			ex.getCause());
            }
        }
	}
	
	public void save(final File sourceFile){
	    fileSaveButton.setEnabled(true);
	    String defaultTargetFile = null;
	    int result = JFileChooser.APPROVE_OPTION;
	    File selectedFile = null;
	    if(outputFileChooser != null){
	        defaultTargetFile = FileUtil.getSuffixReplacedFilePath(sourceFile, outputFileChooser.getSuffix());
	        if(defaultTargetFile != null){
	            outputFileChooser.setSelectedFile(new File(defaultTargetFile));
	        }else if(sourceFile != null){
	            outputFileChooser.setSelectedFile(new File(sourceFile.getAbsolutePath()+outputFileChooser.getSuffix()));
	        }
	        result = outputFileChooser.showSaveDialog(this);
	        selectedFile = outputFileChooser.getSelectedFile();
	    }
		if(result == JFileChooser.APPROVE_OPTION) {
			setFinished(false);
			translate(sourceFile, selectedFile);
		}
	}
	
	public void translate(File sourceFile, File targetFile){
	    new TranslatingTask(sourceFile, createResource(), targetFile).start();
	}
	
	public TranslatorResource createResource(){
	    LoggerConsoleFrame logger = createLoggerConsoleFrame();
	    return new TranslatorResource(logger);
	}
	
    public LoggerConsoleFrame createLoggerConsoleFrame() {
        final LoggerConsoleFrame logger = new LoggerConsoleFrame(getTitle()+" Log",  "Cancel",  "OK");
        logger.setFinished(false);
        logger.getCancelButton().addActionListener(new ActionListener(){
        	public void actionPerformed(ActionEvent ev){
        		logger.setVisible(false);
        		logger.setFinished(true);
        		setFinished(true);
        	}
        });
        return logger;
    }
    
	public abstract SuffixBasedFileChooser createSuffixBasedFileChooser(SuffixBasedFileFilter filter);
	

}
