/*
 * 

 PersistentCacheManager.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */
package net.sqs2.cache;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import net.sf.ehcache.Cache;
import net.sf.ehcache.CacheManager;
import net.sf.ehcache.Ehcache;
import net.sf.ehcache.config.CacheConfiguration;
import net.sf.ehcache.config.Configuration;
import net.sf.ehcache.config.DiskStoreConfiguration;

public class PersistentCacheManager extends CacheManager{

	public PersistentCacheManager(Configuration configuration){
		super(configuration);
	}

	public Ehcache getECache(String name){
		Ehcache cache = this.cacheMap.get(name);
		if(cache != null){
			return cache;
		}
		try{
			cache = super.getEhcache(name);
		}catch(IllegalStateException ex){
			super.removeCache(name);
		}catch(ClassCastException ex){
			super.removeCache(name);
		}
		if(cache == null){
			cache = new Cache(name, 1, null, true, null, true, 1L, 1L, true, 120L, null, null);
			this.addCache(cache);
			this.cacheMap.put(name, cache);
		}
		return cache;
	}

	static Map<File, PersistentCacheManager> instanceMap = new HashMap<File, PersistentCacheManager>();
	Map<String, Ehcache> cacheMap = new HashMap<String, Ehcache>();

	public static void shutdownAll(){
		for(PersistentCacheManager cacheManager: PersistentCacheManager.instanceMap.values()){
			cacheManager.shutdown();
			cacheManager.cacheMap.clear();
		}
	}
	
	@Override
	public void shutdown(){
		super.shutdown();
		this.cacheMap.remove(getName());
	}

	public static Configuration createConfiguration(File directory){
		Configuration configuration = new Configuration();

		DiskStoreConfiguration diskStoreConfiguration = new DiskStoreConfiguration();
		diskStoreConfiguration.setPath(directory.getAbsolutePath());		
		configuration.addDiskStore(diskStoreConfiguration);

		CacheConfiguration defaultCacheConfiguration = new CacheConfiguration();
		defaultCacheConfiguration.setDiskPersistent(true);
		defaultCacheConfiguration.setEternal(true);

		configuration.addDefaultCache(defaultCacheConfiguration);

		return configuration;
	}

	private static File createStorageDirectory(File dir1, File dir2, String dirname)throws IOException{
		for(File dir: new File[]{dir2, dir1}){
			File storageDirectory = new File(dir.getAbsoluteFile()+File.separator+dirname);
			if(! storageDirectory.isDirectory()){
				storageDirectory.mkdirs();
			}
			if(storageDirectory.exists() && storageDirectory.canWrite()){
				return storageDirectory;
			}
		}
		throw new IOException("ReadOnlyFileSystem");
	}

	public synchronized static PersistentCacheManager getInstance(File dir1, File dir2, String dirname)throws IOException{
		File path = createStorageDirectory(dir1, dir2, dirname);
		return PersistentCacheManager.getInstance(path);
	}
	
	public synchronized static PersistentCacheManager getInstance(File directory)throws IOException{
		PersistentCacheManager ret = PersistentCacheManager.instanceMap.get(directory);
		if(ret == null){
			ret = new PersistentCacheManager(directory);
			PersistentCacheManager.instanceMap.put(directory, ret);
		}
		return ret;
	}

	public PersistentCacheManager(File directory)throws IOException{
		super(createConfiguration(directory));
		//this.diskStorePath = directory;
	}

}
