/**
 *  DirectoryChooserPanel.java

 Copyright 2007 KUBO Hiroya (hiroya@cuc.ac.jp).

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Created on 2007/02/12
 Author hiroya
 */
package net.sqs2.swing;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.util.LinkedList;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JTextField;

import org.xnap.commons.gui.DirectoryChooser;
import org.xnap.commons.gui.completion.AutomaticDropDownCompletionMode;
import org.xnap.commons.gui.completion.Completion;
import org.xnap.commons.gui.completion.FileCompletionModel;

public class DirectoryChooserPanel extends JPanel{

	private static final long serialVersionUID = 0L;
	final JTextField textField;
	final DirectoryChooser dchooser;
	final JButton btn;
	Component parent;

	public DirectoryChooserPanel(final Component parent) {
		this.parent = parent;
		this.textField = new JTextField();
		this.dchooser = new DirectoryChooser(){
			private static final long serialVersionUID = 0L;
			public boolean apply(){
				setFile(getSelectedDirectory());
				return super.apply();
			}
		};
		this.textField.setPreferredSize(new Dimension(300,32));
		Completion completion = new Completion(this.textField, new FileCompletionModel(true));
		completion.setMode(new AutomaticDropDownCompletionMode());

		new FileDropTargetDecorator(this.textField){
			public void drop(File file){
				if(textField.isEnabled() && file.exists()){
					DirectoryChooserPanel.this.textField.setText(file.getAbsolutePath());
					setFile(file);
				}
			}
		};

		this.setLayout(new BorderLayout());
		this.add(this.textField, BorderLayout.CENTER);

		this.btn = new JButton("...");
		this.btn.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent ev){
				File file = new File(DirectoryChooserPanel.this.textField.getText());
				if(file.isDirectory()){
					DirectoryChooserPanel.this.dchooser.setSelectedDirectory(file);
				}else if(file.exists()){
					DirectoryChooserPanel.this.dchooser.setSelectedDirectory(file.getParentFile());
				}
				showChooserDialog();
			}
		});

		this.add(btn, BorderLayout.EAST);
	}
	
	public JTextField getTextField(){
		return this.textField;
	}

	public void showChooserDialog(){
		this.dchooser.setSelectedDirectory(getFile());
		this.dchooser.showChooseDialog(this.parent);
	}

	public void setToolTipText(String text){
		this.textField.setToolTipText(text);		
	}

	public void setForeground(Color color){
		if(this.textField != null){
			this.textField.setForeground(color);
		}
	}

	public void setBackground(Color color){
		if(this.textField != null){
			this.textField.setBackground(color);
		}
	}

	public void setEnabled(boolean enabled){
		try{
			this.textField.setEnabled(enabled);
			this.dchooser.setEnabled(enabled);
			this.btn.setEnabled(enabled);
		}catch(NullPointerException ignore){}
	}

	public void setFile(File file){
		if(! file.exists()){
			return;
		}
		this.textField.setText(file.getAbsolutePath());
		for(DirectoryChooserEventListener listener: this.list){
			listener.chooseDirectory(new FileChooserEvent(this, file));
		}
	}

	public File getFile(){
		return new File(this.textField.getText());
	}

	List<DirectoryChooserEventListener> list = new LinkedList<DirectoryChooserEventListener>();
	public void addDirectoryChooserEventListener(DirectoryChooserEventListener listener){
		this.list.add(listener);
	}
}