using System;
using System.Collections.Generic;
using System.Drawing;
using System.Text;

namespace StarEngine.Core
{
    /// <summary>
    /// テクスチャのためのメソッドを提供します。
    /// </summary>
    public interface ITexture : ICloneable, IDisposable
    {
        /// <summary>
        /// 解放されたか否かを示す値を取得します。
        /// </summary>
        bool IsDisposed { get; }

        /// <summary>
        /// 大きさを取得します。
        /// </summary>
        Size Size { get; }

        /// <summary>
        /// 幅を取得します。
        /// </summary>
        int Width { get; }

        /// <summary>
        /// 高さを取得します。
        /// </summary>
        int Height { get; }

        /// <summary>
        /// ビットマップに変換します。
        /// </summary>
        /// <returns>ビットマップ。</returns>
        Bitmap ToBitmap();

        /// <summary>
        /// ピクセルの色を取得します。
        /// </summary>
        /// <param name="x">X 座標。</param>
        /// <param name="y">Y 座標。</param>
        /// <returns>ピクセルの色。</returns>
        Color GetPixel(int x, int y);

        /// <summary>
        /// ピクセルの色を設定します。
        /// </summary>
        /// <param name="x">X 座標。</param>
        /// <param name="y">Y 座標。</param>
        /// <param name="color">ピクセルの色。</param>
        void SetPixel(int x, int y, Color color);

        /// <summary>
        /// 色相を変化させます。
        /// </summary>
        /// <param name="dHue">色相の変化量 (0〜359)。</param>
        void ChangeHue(int dHue);

        /// <summary>
        /// 消去します。
        /// </summary>
        void Clear();

        /// <summary>
        /// 単色で塗りつぶします。
        /// </summary>
        /// <param name="color">描画色。</param>
        void Fill(Color color);

        /// <summary>
        /// 単色で矩形を塗りつぶします。
        /// </summary>
        /// <param name="rectangle">矩形。</param>
        /// <param name="color">描画色。</param>
        void FillRectangle(Rectangle rectangle, Color color);

        /// <summary>
        /// 文字列を描画します。
        /// </summary>
        /// <remarks>
        /// 改行文字、タブ文字はスペースに置き換えられます。
        /// </remarks>
        /// <param name="text">文字列。</param>
        /// <param name="location">描画位置。</param>
        /// <param name="font">フォント。</param>
        /// <param name="color">色。</param>
        void RenderText(string text, Point location, Font font, Color color);

        /// <summary>
        /// テクスチャを描画します。
        /// </summary>
        /// <param name="texture"></param>
        /// <param name="sourceRectangle"></param>
        /// <param name="affineMatrix"></param>
        /// <param name="alpha"></param>
        /// <param name="blendType"></param>
        /// <param name="tone"></param>
        void RenderTexture(ITexture texture,
            Rectangle sourceRectangle,
            AffineMatrix affineMatrix,
            byte alpha,
            BlendType blendType,
            Tone tone);
    }
}
